/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectStreamClass;
import java.lang.ref.WeakReference;

import org.apache.commons.lang3.Validate;

//=============================================================================
/** Class serialization inside an OSGI environment can be some kind of pane.
 *
 *  A typical problem : on deserialization the wrong (default) class loader
 *  is used and is not able to load right class within it's context.
 *
 *  This helper solve the problem by hooking into the class loader hierarchy
 *  and select the right loader instead of using the default class loader.
 *
 *  The tricky part here is the fact :
 *  - an object knows it's class.
 *  - a class knows it's class loader.
 *  - a object/class of an OSGI environment has a class loader set which knows
 *    the rules of OSGI already .-)
 *
 *  So - if we use that class loader of the context object we can be sure
 *  all classes in that OSGI context will be available for loading.
 */
public class ClassContextAwareObjectInputStream extends ObjectInputStream
{
	//-------------------------------------------------------------------------
	/** construct a new instance and define the context it has to use.
	 *
	 *	@param	aObjectData [IN]
	 *			the stream containing the raw data of the serialized object.
	 *
	 *	@param	aContext [IN]
	 *			the context class loading has to be done.
	 */
    public ClassContextAwareObjectInputStream (final InputStream aObjectData,
    										   final Object      aContext   )
    	throws Exception
    {
        super (aObjectData);

        Validate.notNull(aContext, "No context given.");

        final Class< ? >  aContextClass  = aContext.getClass();
        final ClassLoader aContextLoader = aContextClass.getClassLoader();
        m_aContextLoader = new WeakReference< ClassLoader > (aContextLoader);
    }

	//-------------------------------------------------------------------------
    @Override
    protected Class<?> resolveClass (final ObjectStreamClass aClassDescriptor)
    	throws IOException
    		 , ClassNotFoundException
    {
    	final String      sClass         = aClassDescriptor.getName ();
    	final ClassLoader aContextLoader = m_aContextLoader.get();
    	      Class< ? >  aClass         = aContextLoader.loadClass(sClass);

    	if (aClass == null)
    		aClass = super.resolveClass(aClassDescriptor);

        return aClass;
    }

    //-------------------------------------------------------------------------
    private WeakReference< ClassLoader > m_aContextLoader = null;
}