/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.realtime.impl;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;
import org.json.JSONArray;
import org.json.JSONObject;

import com.openexchange.office.DocumentProperties;
import com.openexchange.office.FilterException;
import com.openexchange.office.IImporter;
import com.openexchange.office.IPartImporter;
import com.openexchange.office.tools.doc.DocContextLogger;
import com.openexchange.office.tools.doc.DocumentMetaData;
import com.openexchange.office.tools.doc.OXDocument;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.office.tools.message.MessagePropertyKey;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.realtime.packet.ID;
import com.openexchange.session.Session;

/**
 * Wrapper class to provide a general access to the importer regardless of the
 * features that the importer implements.
 *
 * @author Carsten Driesner
 * @since 7.8.2
 * @see com.openexchange.office.IPartImporter
 */
public class ChunkableDocLoader {
	private final static String PROP_PART_COUNT        = "partCount";
	private final static String PROP_ACTIVE_PART_INDEX = "activePartIndex";
    @SuppressWarnings( "deprecation" )
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(ChunkableDocLoader.class);

	private final OXDocument    docLoader;
	private final IImporter     importer;
	private final IPartImporter partImporter;
	private final DocumentProperties  docProperties;
	private Map<String, Object> metaData = null;
	private JSONObject          importerProps = null;

	/**
	 * Initializes a new chunkable document loader instance.
	 *
	 * @param docLoader a valid document loader instance
	 * @param importer a valid importer instance (supporting chunks or not)
	 * @param importerProps optional importer properties to be used by this loader instance
	 */
	public ChunkableDocLoader(final OXDocument docLoader, final IImporter importer, final JSONObject importerProps) {
		Validate.notNull(docLoader);
		Validate.notNull(importer);

		this.docLoader     = docLoader;
		this.importer      = importer;
		this.docProperties = new DocumentProperties();

		if (importer instanceof IPartImporter) {
			partImporter = (IPartImporter)importer;
		} else {
			partImporter = null;
		}

		this.importerProps = importerProps;
		impl_initDocumentProperties();
	}

	/**
	 * Prepares to load the document stream and create the operations.
	 *
	 * @return TRUE if the preparation was successful, otherwise FALSE.
	 * @throws Exception
	 */
	public boolean prepareLoad() throws Exception {
		impl_initDocumentProperties();

		if (null != partImporter) {
			final Session session = docLoader.getSession();
			final InputStream inStream = new ByteArrayInputStream(docLoader.getDocumentBuffer());
			partImporter.initPartitioning(session, inStream, docProperties);
		} else {
			// nothing to do for old importer interface
		}

		return true;
	}

	/**
	 * Retrieves the file meta data of the document.
	 *
	 * @return
	 * @throws Exception
	 */
	public DocumentMetaData getFileMetaData() throws Exception {
		return docLoader.getDocumentMetaData();
	}

	/**
	 * Retrieves the low-level document loader instance used by
	 * this chunkable document loader.
	 *
	 * @return
	 * @throws Exception
	 */
	public OXDocument getDocumentLoader() throws Exception {
		return docLoader;
	}

	/**
	 * Retrieves the global operations of the document. Global operations are
	 * defined as operations which are connected to the whole document and not
	 * to parts (e.g. page size, number of pages, sheets, slides).
	 *
	 * @returns JSONObject containing a property "operations" which includes
	 *  the global operations.
	 *
	 * @throws Exception
	 */
	public JSONObject getGlobalOperations() throws Exception {
		JSONObject globalOps = new JSONObject();

		if (null != partImporter) {
			metaData = partImporter.getMetaData(docProperties);
			Object value = metaData.get(MessagePropertyKey.KEY_OPERATIONS);

			if (value instanceof JSONArray) {
				globalOps.put(MessagePropertyKey.KEY_OPERATIONS, value);
			}

			metaData.put(PROP_ACTIVE_PART_INDEX, docProperties.get(DocumentProperties.PROP_SPREADHSEET_ACTIVE_SHEET_INDEX));
		} else {
			// empty global operations
			globalOps.put(MessagePropertyKey.KEY_OPERATIONS, new JSONArray());
		}

		return globalOps;
	}

	/**
	 * Provides the active part index of the document. The active part is defined
	 * as the page, sheet or slide which currently visible to the user. The index is
	 * zero based.
	 *
	 * @return
	 * @throws Exception
	 */
	public int getActivePartIndex() throws Exception {
		int index = 1;

		if (metaData != null) {
			Object value = metaData.get(PROP_ACTIVE_PART_INDEX);
			if (value instanceof Integer)
				index = (Integer)value;
		}

		return index;
	}


	/**
	 * Provides the number of parts of which this document consists. This
	 * can be one or greater than one.
	 *
	 * @return
	 * @throws Exception
	 */
	public int getPartsCount() throws Exception {
		int count = 1;

		if (metaData != null) {
			Object value = metaData.get(PROP_PART_COUNT);
			if (value instanceof Integer)
				count = (Integer)value;
		}

		return count;
	}

	/**
	 * Retrieves the active operations of the document. Active operations are
	 * defined as operations which are connected to the active part of the document
	 * (e.g. the part which is visible to the user: page, slide or sheet.
	 *
	 * @returns JSONObject containing a property "operations" which includes
	 *  the global operations.
	 *
	 * @throws Exception
	 */
	public JSONObject getActiveOperations() throws Exception {
		JSONObject opsObject = new JSONObject();

		if (null != partImporter) {
			final Map<String, Object> activePartProps = partImporter.getActivePart(docProperties);
			if (null != activePartProps) {
				Object value = activePartProps.get(MessagePropertyKey.KEY_OPERATIONS);
				if (value instanceof JSONArray) {
					final JSONArray activeOps = (JSONArray)value;
					opsObject.put(MessagePropertyKey.KEY_OPERATIONS, activeOps);
				}
			}
			// sheet count is only available after getActivePart is called!!
			metaData.put(PROP_PART_COUNT, docProperties.get(DocumentProperties.PROP_SPREADSHEET_SHEET_COUNT));
		} else {
			final Session session = docLoader.getSession();
			final InputStream inStream = new ByteArrayInputStream(docLoader.getDocumentBuffer());

			opsObject = importer.createOperations(session, inStream, docProperties);
			if (null == opsObject) {
				throw new FilterException("No operations received from filter!", FilterException.ErrorCode.CRITICAL_ERROR);
			}
		}

		return opsObject;
	}

	/**
	 * Retrieves the remaining operations of the document. Remaining operations consists
	 * of all operations which are not part of the global or active part.
	 *
	 * @returns JSONObject containing a property "operations" which includes
	 *  the remaining operations. The remaining operations count can be zero!
	 *
	 * @throws Exception
	 */
	public JSONObject getRemainingOperations() throws Exception {
		JSONObject remOpsObject = new JSONObject();
		JSONArray remOps = new JSONArray();

		if (null != partImporter) {
			Map<String, Object> nextPartProps = partImporter.getNextPart(docProperties);
			while (nextPartProps != null) {
				final Object value = nextPartProps.get(MessagePropertyKey.KEY_OPERATIONS);
				if (value instanceof JSONArray) {
					JSONHelper.appendArray(remOps, (JSONArray)value);
				}
				nextPartProps = partImporter.getNextPart(docProperties);
			}
		}

		// don't forget to append the operations from a possible rescue document
		JSONHelper.appendArray(remOps, docLoader.getRescueOperations());
		remOpsObject.put(MessagePropertyKey.KEY_OPERATIONS, remOps);

		return remOpsObject;
	}

	public static String getContextStringWithDoc(boolean debugInfo, final ID id, final UserData userData, final ChunkableDocLoader docLoader) {
		String aDocContextString = "";

		try {
            final String idString = (null != id) ? id.toString() : null;
            final String folderId = StringUtils.isEmpty(userData.getFolderId()) ? "???" : userData.getFolderId();
            final String fileId = StringUtils.isEmpty(userData.getFileId()) ? "???" : userData.getFileId();
            final String fileName = docLoader.getFileMetaData().getFileName();

            aDocContextString =  DocContextLogger.getContextStringWithDoc(debugInfo, idString, folderId, fileId, fileName);
		} catch (Throwable e) {
			// nothing to do - provide empty string for context
			LOG.error("RT connection: Couldn't create document context string", e);
		}

		return aDocContextString;
	}

	private void impl_initDocumentProperties() {
		if (importerProps != null) {
			// map values to document properties
			final Object aMemObserver = importerProps.opt(DocumentProperties.PROP_MEMORYOBSERVER);
			if (aMemObserver != null) {
				this.docProperties.put(DocumentProperties.PROP_MEMORYOBSERVER, aMemObserver);
			}
		}
	}

}
