/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.presenter.impl;

import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.lang.StringUtils;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Optional;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.capabilities.CapabilitySet;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.ldap.User;
import com.openexchange.office.presenter.rt.MessageData;
import com.openexchange.office.tools.SessionUtils;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.rt.IDUtils;
import com.openexchange.realtime.ComponentHandle;
import com.openexchange.realtime.group.GroupDispatcher;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.Duration;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;


/**
 * The PresenterConnection class implements the necessary real-time communication of an OX Presenter instance.
 * This includes start/end/pause/join of presentations, slide changes and presenter/participants handling.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @author <a href="mailto:mario.schroeder@open-xchange.com">Mario Schroeder</a>
 */
public @NotThreadSafe class PresenterConnection extends GroupDispatcher implements ComponentHandle {
    private static final Logger LOG = LoggerFactory.getLogger(PresenterConnection.class);

    // An introspecting handler that allows clients to formulate stanzas that call all handle* methods
    private static final ActionHandler handler = new ActionHandler(PresenterConnection.class);

    private final ConnectionStatus m_connectionStatus = new ConnectionStatus();
    private final ServiceLookup m_services;
    private final String m_componentID;
    private final int MAX_NUMBER_OF_PARTICIPANTS;

    // Create a new presenter connection instance
    public PresenterConnection(ID id, final ServiceLookup services, final String componentID, final int maxParticipants) {
        super(id, handler);
        this.m_services = services;
        this.m_componentID = componentID;
        this.MAX_NUMBER_OF_PARTICIPANTS = maxParticipants; 

        LOG.debug("PresenterConnection id = " + id);
    }

    /**
     * Change presentation slide.
     *
     * @param stanza
     *  The Stanza containing the slide information.
     * @throws OXException
     */
    public void handleUpdateSlide(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handleUpdateSlide], id = " + fromId);

        if ((null != fromId) && connectionStatus.isPresenter(fromId.toString())) {
            final JSONObject jsonRequest = MessageData.getJSONRequest(stanza, new ElementPath(m_componentID, "slideInfo"));

            boolean changed = false;
            try {
	            if (jsonRequest.has("activeSlide")) {
	            	changed = connectionStatus.setActiveSlide(jsonRequest.getInt("activeSlide"));
	            }
            } catch (JSONException e) {
                LOG.error("PresenterConnection [handleUpdateSlide], exception caught", e);
            }

            if (changed) {
            	this.sendUpdateToAllExceptSender(fromId);
            }
        }
    }

    /**
     * Start the presentation as presenter
     *
     * @param stanza
     *  The Stanza containing the presenter and slide information.
     * @throws OXException
     */
    public void handleStartPresentation(Stanza stanza) throws OXException {
    	final ID fromId = stanza.getFrom();
    	final ConnectionStatus connectionStatus = m_connectionStatus;
    	Session session = null;

        LOG.debug("PresenterConnection [handleStartPresentation], id = " + fromId);

        try {
    		session = fromId.toSession();
    	} catch (OXException e) {

    	}
        
        final CapabilityService capabilityService = m_services.getService(CapabilityService.class);
        final CapabilitySet capabilitySet = (null != capabilityService) ? capabilityService.getCapabilities(session) : null;

    	if ((null != fromId) && (null != capabilitySet) && capabilitySet.contains("remote_presenter") && ConnectionStatus.isEmptyUser(connectionStatus.getCurrentPresenterId())) {
    		final String rtId = fromId.toString();
    		final JSONObject jsonRequest = MessageData.getJSONRequest(stanza, new ElementPath(m_componentID, "slideInfo"));
            String userDisplayName = ConnectionStatus.EMPTY_USERNAME;
            try {
            	userDisplayName = SessionUtils.getUserDisplayName(session, m_services);
            } catch (Exception e) {
                LOG.error("PresenterConnection [handleStartPresentation], exception caught", e);
            }

            try {
                connectionStatus.setCurrentPresenter(rtId, userDisplayName);
                connectionStatus.setActiveSlide(jsonRequest.getInt("activeSlide"));
                connectionStatus.setParticipantState(rtId, true);
            } catch (JSONException e) {
                LOG.error("PresenterConnection [handleStartPresentation], exception caught", e);
            }
            this.sendUpdateToAll(fromId);
        }
    }

    /**
     * End the presentation as presenter
     *
     * @param stanza
     *  The Stanza containing the presenter information.
     * @throws OXException
     */
    public void handleEndPresentation(Stanza stanza) throws OXException {
    	final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handleEndPresentation], id = " + fromId);

        if ((null != fromId) && (connectionStatus.isPresenter(fromId.toString()))) {
            try {
                connectionStatus.setCurrentPresenter(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                connectionStatus.disjoinAll();
                connectionStatus.setPausedState(false);
            } catch (JSONException e) {
                LOG.error("PresenterConnection [handleEndPresentation], exception caught", e);
            }
        	this.sendUpdateToAll(fromId);
        }
    }

    /**
     * Pause the presentation as presenter
     *
     * @param stanza
     * @throws OXException
     */
    public void handlePausePresentation(Stanza stanza) throws OXException {
    	final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handlePausePresentation], id = " + fromId);

        if ((null != fromId) && (connectionStatus.isPresenter(fromId.toString()) && (!connectionStatus.isPaused()))) {

        	connectionStatus.setPausedState(true);
        	this.sendUpdateToAll(fromId);
        }
    }

    /**
     * Continue the previously paused presentation as presenter.
     *
     * @param stanza
     * @throws OXException
     */
    public void handleContinuePresentation(Stanza stanza) throws OXException {
    	final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handleContinuePresentation], id = " + fromId);

        if ((null != fromId) && (connectionStatus.isPresenter(fromId.toString()) && (connectionStatus.isPaused()))) {

        	connectionStatus.setPausedState(false);
        	this.sendUpdateToAll(fromId);
        }
    }

    /**
     * Join the presentation as participant / listener.
     *
     * @param stanza
     * @throws OXException
     */
    public void handleJoinPresentation(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handleJoinPresentation], id = " + fromId);

        if (null != fromId) {
            if (connectionStatus.getParticipantsCount() < MAX_NUMBER_OF_PARTICIPANTS) {
                boolean changed = false;

                try {
                    changed = connectionStatus.setParticipantState(fromId.toString(), true);
                } catch (JSONException e) {
                    LOG.error("PresenterConnection [handleJoinPresentation], exception caught", e);
                }

                if (changed) {
                    this.sendUpdateToAll(fromId);
                }
            } else {
                this.sendUpdateToID(fromId, ErrorCode.PRESENTER_MAX_PARTICIPANTS_FOR_PRESENTATION_REACHED_ERROR);
            }
        }
    }

    /**
     * Leave the presentation as participant / listener.
     *
     * @param stanza
     * @throws OXException
     */
    public void handleLeavePresentation(Stanza stanza) throws OXException {
        final ID fromId = stanza.getFrom();
        final ConnectionStatus connectionStatus = m_connectionStatus;

        LOG.debug("PresenterConnection [handleLeavePresentation], id = " + fromId);

        if (null != fromId) {
            boolean changed = false;
            try {
				changed = connectionStatus.setParticipantState(fromId.toString(), false);
			} catch (JSONException e) {
			    LOG.error("PresenterConnection [handleLeavePresentation], exception caught", e);
			}

            if (changed) {
            	this.sendUpdateToAll(fromId);
            }
        }
    }

    @Override
    protected void onJoin(ID id, Stanza stanza) {
        LOG.debug("PresenterConnection [onJoin], id = " + id);

        Session session = null;
        try {
            session = id.toSession();
        } catch (OXException e) {

        }

        final String rtId = id.toString();
        String userDisplayName = ConnectionStatus.EMPTY_USERNAME;
        int userId = -1;
        try {
        	userDisplayName = SessionUtils.getUserDisplayName(session, m_services);
    		userId = SessionUtils.getUserId(session);
        } catch (Exception e) {
            LOG.error("PresenterConnection [onJoin], exception caught", e);
        }
        m_connectionStatus.addActiveUser(rtId, userDisplayName, userId);

        // check presenter handling
        impl_handlePresenter(id);

        try {
            this.sendUpdateToAllExceptSender(id);
        } catch (OXException e) {
            LOG.error("PresenterConnection [onJoin], exception caught", e);
        }
    }

    @Override
    protected void onLeave(ID id) {
        LOG.debug("PresenterConnection [onLeave], id = " + id);

        final String rtId = id.toString();
        try {
            if (m_connectionStatus.isPresenter(id.toString())) {
                m_connectionStatus.setCurrentPresenter(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
            }
            m_connectionStatus.removeActiveUser(rtId);
        } catch (Exception e) {
            LOG.error("PresenterConnection [onLeave], exception caught", e);
        }

        try {
            this.sendUpdateToAllExceptSender(id);
        } catch (OXException e) {
            LOG.error("PresenterConnection [onLeave], exception caught", e);
        }
    }

    @Override
    public Stanza getWelcomeMessage(ID onBehalfOf) {
        return createMessage(getId(), onBehalfOf, "join", ErrorCode.NO_ERROR);
    }

    @Override
    //@Asynchronous
    public Stanza getSignOffMessage(ID onBehalfOf) {
        return createMessage(getId(), onBehalfOf, "leave", ErrorCode.NO_ERROR);
    }

    /**
     * Handles the inactivity notice of the real-time framework for a client that is a member of this connection instance. Currently the
     * handler disables the remote selection of inactive clients.
     *
     * @param stanza
     *  The Stanza containing the inactive client identified by ElementPath 'com.openexchange.realtime.client'
     *  and the Duration of inactivity identified by 'com.openexchange.realtime.client.inactivity'.
     */
    @Override
    public void handleInactivityNotice(Stanza stanza) {
        Optional<ID> inactiveClient = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime", "client"), ID.class);
        Optional<Duration> inactivityDuration = stanza.getSinglePayload(new ElementPath("com.openexchange.realtime.client", "inactivity"), Duration.class);

        if (inactiveClient.isPresent() && inactivityDuration.isPresent()) {
            final String id = inactiveClient.get().toString();
            final Duration timeOfDuration = inactivityDuration.get();
            boolean enableActiveUser = false;

            LOG.debug("PresenterConnection [handleInactivityNotice], id = " + inactiveClient.get() + "inactive for " + inactivityDuration.get());
            if (timeOfDuration.equals(Duration.NONE)) {
                enableActiveUser = true; // enable active user, if we have a duration of zero
            }

            ConnectionStatus statusToSend = m_connectionStatus;
            boolean isUserEnabled = m_connectionStatus.isActiveUserEnabled(id);
            if (isUserEnabled != enableActiveUser) {
                if (enableActiveUser) {
                    m_connectionStatus.enableActiveUser(id);
                } else {
                    m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                }
                LOG.debug("PresenterConnection [handleInactivityNotice], User=" + id + " enabled=" + String.valueOf(enableActiveUser));
            } else if (!enableActiveUser) {
                // update inactivity time
                m_connectionStatus.disableActiveUser(id, timeOfDuration.getValueInS());
                // remove presenter after one minutes of inactivity.
                if (timeOfDuration.equals(Duration.ONE_MINUTE)) {
                	if (m_connectionStatus.isPresenter(id)) {
                		try {
							m_connectionStatus.setParticipantState(id, false);
						} catch (JSONException e) {
							LOG.error("PresenterConnection [handleInactivityNotice], exception caught", e);
						}
                		m_connectionStatus.setCurrentPresenter(ConnectionStatus.EMPTY_USERID, ConnectionStatus.EMPTY_USERNAME);
                		m_connectionStatus.setPausedState(false);
                	}
                }
            }

            // The status must now be sent on any possible change
            // (update of collaboration clients).
            statusToSend.setInactivityChanged(id);

            try {
          	    sendUpdateToAll(inactiveClient.get());
            } catch (OXException e) {
                LOG.error("PresenterConnection [handleInactivityNotice], exception caught", e);
            }
        }
    }

    /**
     * Sends an update message to all clients of the presentation except the sender. This method can be used if a update should only be sent to
     * all other clients, because the sender already have all information available.
     *
     * @param fromId The client who sent a request which resulted in this update message. This client won't be part of the clients that will receive the update message.
     * @throws OXException
     */
    private void sendUpdateToAllExceptSender(final ID fromId) throws OXException {
        sendMessage(fromId, ErrorCode.NO_ERROR, true, "update", false);
    }

    /**
     * Sends an update message to all clients of the presentation.
     *
     * @param fromId The client who sent a request which resulted in this update message.
     * @throws OXException
     */
    private void sendUpdateToAll(final ID fromId) throws OXException {
        sendMessage(fromId, ErrorCode.NO_ERROR, false, "update", false);
    }

    /**
     * Sends an update message to a specific client of the presentation.
     *
     * @param receiverId
     * @throws OXException
     */
    private void sendUpdateToID(final ID receiverId, final ErrorCode errorCode) throws OXException {
        sendMessage(receiverId, errorCode, false, "update", true);
    }

    /**
     * Internal method to send an update message to a set of clients of the presentation.
     *
     * @param fromId The client who sent a request which resulted in this update message.
     * @param exceptSender If set to TRUE the sender/fromId won't be part of the clients that will received the message.
     * @param reason The type of the message e.g. "update".
     * @param toFromIDOnly Must be set, if the message should be send to the client with fromId only
     * @throws OXException
     */
    private void sendMessage(final ID fromId, final ErrorCode errorCode, boolean exceptSender, final String reason, boolean toFromIDOnly) throws OXException {
        final Message message = createMessage(fromId, reason, errorCode);

        if (exceptSender) {
        	this.relayToAll(message, fromId);
        } else if (!toFromIDOnly) {
        	this.relayToAll(message);
        } else {
            this.relayToID(message, fromId);
        }
    }

    /**
     * Internal method to create an update message.
     *
     * @param fromId The id of this real-time connection instance.
     * @param reason The type of the message e.g. "update".
     * @return The message.
     */
    private Message createMessage(final ID fromId, final String reason, final ErrorCode errorCode) {
        return createMessage(fromId, null, reason, errorCode);
    }

    /**
     * Internal method to create an update message.
     *
     * @param fromId The id of this real-time connection instance.
     * @param onBehalfOf The id of the client that requested the action.
     * @param reason The type of the message e.g. "update".
     * @return The message.
     */
    private Message createMessage(final ID fromId, ID onBehalfOf, final String reason, final ErrorCode errorCode) {
        final Message message = new Message();

    	message.setFrom(fromId);
    	if (null != onBehalfOf) {
    		message.setTo(onBehalfOf);
    	}

        message.addPayload(
            new PayloadTree(
                PayloadTreeNode.builder()
                    .withPayload(
                        new PayloadElement(new MessageData(m_connectionStatus, errorCode.getAsJSONResultObject()), MessageData.class.getName(), m_componentID, reason)
                    )
                .build()
            )
        );

    	return message;
    }


    @Override
    protected void firstJoined(ID id) {
        LOG.debug("PresenterConnection [firstJoined], id = " + id);
        /*
        Session session = null;
        try {
            session = id.toSession();
        } catch (OXException e) {

        }

        final String rtId = id.toString();
        String userDisplayName = ConnectionStatus.EMPTY_USERNAME;
        try {
        	userDisplayName = SessionUtils.getUserDisplayName(session, m_services);
        } catch (Exception e) {
            LOG.error("PresenterConnection [firstJoined], exception caught", e);
        }
        */
        //m_connectionStatus.setCurrentPresenter(rtId, userDisplayName);
    }

    @Override
    protected void onDispose(ID id) {
        LOG.debug("PresenterConnection [onDispose], id = " + id);
    }

    /**
     * Determines who is/will be the presenter, when a new client joins.
     *
     * @param newId The real-time ID of the joining client.
     */
    private void impl_handlePresenter(final ID newId) {
        boolean joiningClientWillBePresenter = false;

        Session session = null;
        User userInfo = null;
        try {
            session = newId.toSession();
            userInfo = SessionUtils.getUserInfo(session, m_services);
        } catch (Exception e) {
            LOG.error("PresenterConnection [impl_handlePresenter], exception caught while trying to get user info");
        }

        final String currentPresenterId = m_connectionStatus.getCurrentPresenterId();

        try {
            if ((StringUtils.isNotEmpty(currentPresenterId)) && ((null != userInfo) && (!userInfo.isGuest()))) {
                int userId1 = m_connectionStatus.getUserId(currentPresenterId);
                int userId2 = IDUtils.getUserIdFromRealTimeId(newId);

                LOG.debug("PresenterConnection [impl_handlePresenter], id = {}, presenter id = {}, presenter user id = {}, joining user id = {}.", newId, currentPresenterId, userId1, userId2);

                if (userId1 == userId2) {
                	joiningClientWillBePresenter = true;
                }
            }
        } catch (Exception e) {
            // nothing to do -> this is just a heuristic and can be ignored
        }

        if (joiningClientWillBePresenter) {
            try {
            	m_connectionStatus.setCurrentPresenter(newId.toString(), SessionUtils.getUserDisplayName(session, m_services));
            	m_connectionStatus.setParticipantState(newId.toString(), true);
            } catch (Exception e) {
                LOG.error("PresenterConnection [impl_handlePresenter], exception caught", e);
            }
        }
    }

}
