/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.presenter.impl;

import javax.annotation.concurrent.NotThreadSafe;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * {@link ConnectionStatus}
 *
 * The ConnectionStatus class stores important state information about a
 * document connection. The content is used to be sent to connected clients
 * to update their internal state. The state includes the editor name and
 * id, the current server osn, the number of clients, the current selection
 * of every client etc.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @author <a href="mailto:mario.schroeder@open-xchange.com">Mario Schroeder</a>
 */
@NotThreadSafe
public class ConnectionStatus implements Cloneable {
    public static final String EMPTY_USERID = "";
    public static final String EMPTY_USERNAME = "";
    public static final int NO_USERID = -1;

    // - Members ---------------------------------------------------------------

    private String m_currentPresenterName = EMPTY_USERNAME;
    private String m_currentPresenterId = EMPTY_USERID;
    private int m_activeSlide = 0;
    private boolean m_paused = false;
    private JSONArray m_activeUsers = new JSONArray();

    // - Implementation --------------------------------------------------------

    /**
     * Initializes a new {@link ConnectionStatus}.
     */
    public ConnectionStatus() {
        super();
    }

    /**
     * Gets the m_currentPresenterName
     *
     * @return
     *  The m_currentPresenterName
     */
    public void setCurrentPresenter(String currentPresenterId, String currentPresenterName) {
        m_currentPresenterId = currentPresenterId;
        m_currentPresenterName = currentPresenterName;
    }

    /**
     * Gets the m_currentPresenterName
     *
     * @return
     *  The m_currentPresenterName
     */
    public String getCurrentPresenterId() {
        return m_currentPresenterId;
    }

    /**
     * Determine if the provided user is the presenter or not.
     *
     * @param userId
     *  The ID of the user to find out the enabled state.
     *
     * @return
     *  Whether the provided user is the presenter.
     */
    public boolean isPresenter(String userId) {
    	if (ConnectionStatus.isNotEmptyUser(userId) && m_currentPresenterId.equals(userId)) {
    		return true;
    	}
    	return false;
    }

    /**
     * Sets the m_currentPresenterName
     *
     * @param currentPresenterName
     * 	The currentPresenterName to set
     */
    public String getCurrentPresenterName() {
        return m_currentPresenterName;
    }

    /**
     * Provides the number of active clients.
     *
     * @return
     *  The number of active clients.
     */
    public int getActiveClients() {
        return (null != m_activeUsers) ? m_activeUsers.length() : 0;
    }

    /**
     * Provides the number of participants.
     *
     * @return
     *  The number of participants.
     */
    public int getParticipantsCount() {
        int num = 0;

        for (int i = 0; i < m_activeUsers.length(); i++) {
            final JSONObject user = m_activeUsers.optJSONObject(i);
            if ((null != user) && user.optBoolean("joined")) {
                ++num;
            }
        }
        return num;
    }

    /**
     * Gets the activeUsers
     *
     * @return
     *  The activeUsers
     */
    public JSONArray getActiveUsers() {
        return m_activeUsers;
    }

    /**
     * Sets the activeUsers
     *
     * @param activeUsers
     *  The activeUsers to set
     */
    public void setActiveUsers(JSONArray activeUsers) {
        this.m_activeUsers = activeUsers;
    }

    /**
     * Determine if the presentation is paused.
     *
     * @return
     *  Whether the presentation is paused.
     */
    public boolean isPaused() {
    	return this.m_paused;
    }

    /**
     * Set the paused state of the presentation.
     *
     * @param state
     *  The paused state.
     */
    public void setPausedState(boolean state) {
    	this.m_paused = state;
    }

    /**
     * Gets the id of the active slide.
     *
     * @return
     *  The zero based slide id.
     */
    public int getActiveSlide() {
    	return this.m_activeSlide;
    }

    /**
     * Sets the active slide.
     *
     * @param slide
     *  The slide id.
     *
     * @return
     *  Returns true if the previous slide id differs from the new one.
     */
    public boolean setActiveSlide(int slide) {
    	if (this.m_activeSlide != slide) {
            this.m_activeSlide = slide;
            return true;
    	}
    	return false;
    }

    /**
     * Add a user to the activeUsers map
     *
     * @param rtUserId
     *  ID of the user
     * @param userDisplayName
     *  The UI name of the new user
     * @param accountUserId
     * 	The user id associated with the OX Appsuite account.
     * @throws JSONException
     */
    public void addActiveUser(String rtUserId, String userDisplayName, int accountUserId){
        JSONObject userObject = new JSONObject();
        try {
            userObject.put("userId", rtUserId);
            userObject.put("userDisplayName", userDisplayName);
            userObject.put("active", true);
            userObject.put("id", accountUserId);
            userObject.put("durationOfInactivity", 0L);
            userObject.put("joined", false);
        } catch (JSONException e) {
            //LOG.error("Exception caught while adding new active user", e);
        }
        m_activeUsers.put(userObject);
    }

    /**
     * Remove a specified user from the activeUsers map
     *
     * @param userId
     *  ID from the user to be removed
     * @return
     *  true if the user was previously present in the list and removed, otherwise false.
     * @throws JSONException
     */
    public boolean removeActiveUser(String userId) throws JSONException {
        for (int i = 0; i < m_activeUsers.length();i++) {
            JSONObject user = m_activeUsers.getJSONObject(i);
            if (user.get("userId").equals(userId)) {
                m_activeUsers.remove(i);
                return true;
            }
        }
        return false;
    }

    /**
     * Set the participant state of the user.
     *
     * @param userId
     *  ID from the user to set the state for
     * @param state
     *  true if the user joined, false if the user left the presentation
     * @return
     *  true if the user was previously present in the list, otherwise false.
     * @throws JSONException
     */
    public boolean setParticipantState(String userId, boolean state) throws JSONException {
    	for (int i = 0; i < m_activeUsers.length();i++) {
            JSONObject user = m_activeUsers.getJSONObject(i);
            if (user.get("userId").equals(userId)) {
            	user.put("joined", state);
                return true;
            }
        }
    	return false;
    }

    /**
     * Disjoin all users.
     *
     * @throws JSONException
     */
    public void disjoinAll() throws JSONException {
        for (int i = 0; i < m_activeUsers.length();i++) {
            JSONObject user = m_activeUsers.getJSONObject(i);
            user.put("joined", false);
        }
    }

    /**
     * Disables an active user so her data is NOT sent to other clients.
     *
     * @param userId
     *  The ID of the user to be disabled.
     *
     * @return
     *  TRUE if the user is disabled, FALSE if the user couldn't be
     *  found in the list of active users.
     */
    public boolean disableActiveUser(String userId) {
        JSONObject user;

        if (null != userId) {
            for (int i = 0; i < m_activeUsers.length();i++) {
                try {
                    user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        user.put("active", false);
                        return true;
                    }
                } catch (JSONException e) {
                    //LOG.error("Exception while setting updated user data", e);
                }
            }
        }

        return false;
    }

    /**
     * Disables an active user so her data is NOT sent to other clients.
     *
     * @param userId
     *  The ID of the user to be disabled.
     *
     * @param durationOfInactivity
     *  The duration of inactivity in seconds.
     *
     * @return
     *  TRUE if the user is disabled, FALSE if the user couldn't be
     *  found in the list of active users.
     */
    public boolean disableActiveUser(String userId, long durationOfInactivity) {

        if (null != userId) {
            for (int i = 0; i < m_activeUsers.length();i++) {
                try {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        int lastDurationOfInactivity = user.optInt("durationOfInactivity", 0);
                        user.put("active", false);
                        if (lastDurationOfInactivity < durationOfInactivity) {
                            // Don't set a lower inactivity time in case of
                            // disable active user. He/she must be activated
                            // before.
                            user.put("durationOfInactivity", durationOfInactivity);
                        }
                        return true;
                    }
                } catch (JSONException e) {
                    //LOG.error("Exception while setting updated user data", e);
                }
            }
        }

        return false;
    }

    /**
     * Determine if the provided user is enabled or not.
     *
     * @param userId
     *  The ID of the user to find out the enabled state.
     *
     * @return
     *  TRUE if the user is enabled, FALSE if the user couldn't be
     *  found in the list or is not enabled.
     */
    public boolean isActiveUserEnabled(String userId) {

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optBoolean("active", false);
                    }
                }
            } catch (JSONException e) {
                //LOG.error("Exception while retrieving activity state of user", e);
            }
        }

        return false;
    }

    /**
     * Determine if the provided user is enabled or not.
     *
     * @param userId
     *  The ID of the user to find out the enabled state.
     *
     * @return
     *  The duration of inactivity, where 0 means no inactivity and
     *  -1 that the inactivity could not be determined (normally the
     *  user is unknown).
     */
    public long getDurationOfInactivity(String userId) {
        long durationOfInactivity = -1;

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optLong("durationOfInactivity", 0);
                    }
                }
            } catch (JSONException e) {
                //LOG.error("Exception while retrieving duration of inactivity for user", e);
            }
        }

        return durationOfInactivity;
    }

    /**
     * Get session user id of the user associated with the RT-id.
     *
     * @param userId
     *  The real-time user id.
     *
     * @return
     *  The session user id of the user associated width the RT-id or
     *  -1 if the RT-id is unknown.
     */
    public int getUserId(String userId) {
        int id = -1;

        if (null != userId) {
            try {
                for (int i = 0; i < m_activeUsers.length();i++) {
                    final JSONObject user = m_activeUsers.getJSONObject(i);
                    if (userId.equals(user.optString("userId", null))) {
                        return user.optInt("id", -1);
                    }
                }
            } catch (JSONException e) {
                //LOG.error("Exception while retrieving duration of inactivity for user", e);
            }
        }

        return id;
    }

    /**
     * Enables an active user again so her data is sent to other clients.
     *
     * @param userId
     *  The ID of the user to be enabled again.
     * @return
     *  TRUE if the user is active again, FALSE if the user couldn't be
     *  found in the list of active users.
     *
     * @throws JSONException
     */
    public boolean enableActiveUser(String userId) {
        JSONObject user;
        for (int i = 0; i < m_activeUsers.length();i++) {
            try {
                user = m_activeUsers.getJSONObject(i);
                if (user.get("userId").equals(userId)) {
                    user.put("active", true);
                    user.put("durationOfInactivity", 0L);
                    return true;
                }
            } catch (JSONException e) {
                //LOG.error("Exception while setting updated user data", e);
            }
        }
        return false;
    }

    /**
     * Determines if a user id is part of the active users.
     *
     * @param userId
     *  The ID of the user to determine that is a member of this connection.
     *
     * @return
     *  TRUE if the user is member or FALSE if not.
     */
    public boolean hasActiveUser(String userId) {
        JSONObject user;
        if (null != userId) {
	        for (int i = 0; i < m_activeUsers.length();i++) {
                user = m_activeUsers.optJSONObject(i);
                if ((null != user) && (userId.equals(user.optString("userId")))) {
                    return true;
                }
	        }
        }
        return false;
    }

    /**
     * ATTENTION: This method sets a TEMPORARY state. Don't use it on the global
     * connection state object.
     *
     * @param userId
     */
    public void setInactivityChanged(String userId) {
        JSONObject user;
        if (null != userId) {
        	try {
		        for (int i = 0; i < m_activeUsers.length();i++) {
	                user = m_activeUsers.optJSONObject(i);
	                if ((null != user) && (userId.equals(user.optString("userId")))) {
	                    user.put("inactivityChanged", true);
	                }
		        }
        	} catch (JSONException e) {
                //LOG.error("Exception while setting updated user data", e);
        	}
        }
    }

    /**
     * Creates a shallow clone of this connection status
     * object.
     *
     * @return
     *  A shallow clone.
     */
    @Override
    public ConnectionStatus clone() {
        ConnectionStatus clone = new ConnectionStatus();
        clone.setCurrentPresenter(m_currentPresenterId, m_currentPresenterName);
        clone.setActiveSlide(m_activeSlide);
        clone.setPausedState(m_paused);
        JSONArray users = new JSONArray();
        try {
	        for (int i = 0; i < m_activeUsers.length(); i++) {
	        	JSONObject user = m_activeUsers.getJSONObject(i);
	        	if (null != user) {
	                users.put(new JSONObject(user.toString()));
	        	}
	        }
        } catch (JSONException e) {
        	//LOG.debug("Exception while cloning ConnectionStatus", e);
        }
        clone.setActiveUsers(users);
        return clone;
    }

    /**
     * Creates the JSON representation of the current connection status
     * object. The JSON object should only provide properties that are
     * part of the client/server communication. Internal properties should
     * be omitted.
     *
     * @return
     *  A JSON object that contains the properties of the current connection
     *  status. The JSON object can be empty in case of an error.
     */
    public JSONObject toJSON() {
        final JSONObject ret = new JSONObject();

        try {
            ret.put("presenterId", m_currentPresenterId);
            ret.put("presenterName", m_currentPresenterName);
            // send all collaboration user (client must filter/process list)
            ret.put("activeUsers", m_activeUsers);
            ret.put("activeSlide", m_activeSlide);
            ret.put("paused", m_paused);
        } catch (JSONException e) {
            //LOG.error("Exception while creating JSON object of the current connection status", e);
        }

        return ret;
    }

    /**
     * Creates a new connection status object using the provided JSONObject
     * to set the state.
     *
     * @param jsonData
     *  The JSON object containing properties that are used to set the
     *  state of the new connection status object.
     *
     * @return
     *  The new connection status object.
     */
    static public ConnectionStatus createFrom(JSONObject jsonData) {
        ConnectionStatus queueStatus = new ConnectionStatus();
        boolean found = false;

        try {
            if (jsonData.has("presenterId")) {
                queueStatus.m_currentPresenterId = jsonData.getString("presenterId");
                found = true;
            }

            if (jsonData.has("presenterName")) {
                queueStatus.m_currentPresenterName = jsonData.getString("presenterName");
                found = true;
            }

            if (!jsonData.has("activeUsers")) {
                queueStatus.m_activeUsers = jsonData.getJSONArray("activeUsers");
                found = true;
            }

            if (!jsonData.has("activeSlide")) {
                queueStatus.m_activeSlide = jsonData.getInt("activeSlide");
                found = true;
            }

            if (!jsonData.has("paused")) {
            	queueStatus.m_paused = jsonData.getBoolean("paused");
            }

        } catch (JSONException e) {
            //LOG.error("Exception while creating new connection status object", e);
        }

        return (found ? queueStatus : null);
    }

    static public boolean isEmptyUser(final String user) {
        return ((user != null) && EMPTY_USERID.equals(user));
    }

    static public boolean isEmptyUserName(final String userName) {
        return ((userName != null) && EMPTY_USERNAME.equals(userName));
    }

    static public boolean isNotEmptyUser(final String user) {
        return ((user != null) && !(EMPTY_USERID.equals(user)));
    }

    static public boolean isNotEmptyUserName(final String userName) {
        return ((userName != null) && !(EMPTY_USERNAME.equals(userName)));
    }
}
