/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.drawingml;

import java.util.Iterator;
import java.util.List;
import java.util.Map.Entry;

import org.docx4j.dml.CTAdjPoint2D;
import org.docx4j.dml.CTCustomGeometry2D;
import org.docx4j.dml.CTGeomGuide;
import org.docx4j.dml.CTGeomGuideList;
import org.docx4j.dml.CTGeomRect;
import org.docx4j.dml.CTPath2D;
import org.docx4j.dml.CTPath2DArcTo;
import org.docx4j.dml.CTPath2DClose;
import org.docx4j.dml.CTPath2DCubicBezierTo;
import org.docx4j.dml.CTPath2DLineTo;
import org.docx4j.dml.CTPath2DList;
import org.docx4j.dml.CTPath2DMoveTo;
import org.docx4j.dml.CTPath2DQuadBezierTo;
import org.docx4j.dml.CTPresetGeometry2D;
import org.docx4j.dml.CTShapeProperties;
import org.docx4j.dml.IAvList;
import org.docx4j.dml.STPathFillMode;
import org.docx4j.dml.STShapeType;
import org.docx4j.jaxb.Context;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

final public class DMLGeometry {

	public static void applyGeometryPropertiesFromJson(JSONObject geometryAttrs, CTShapeProperties shapeProperties) {

		final String presetShape = geometryAttrs.optString("presetShape", null);
		if(presetShape!=null) {
			shapeProperties.setCustGeom(null);
			shapeProperties.getPrstGeom(true).setPrst(STShapeType.fromValue(presetShape));
		}
		if(shapeProperties.getCustGeom()!=null) {
			applyCustomGeometryPropertiesFromJson(geometryAttrs, shapeProperties.getCustGeom());
		}
		else if(shapeProperties.getPrstGeom(false)!=null) {
			applyPresetGeometryPropertiesFromJson(geometryAttrs, shapeProperties.getPrstGeom(false));
		}
	}

	public static void applyPresetGeometryPropertiesFromJson(JSONObject geometryAttrs, CTPresetGeometry2D presetGeometry) {
		final Object avList = geometryAttrs.optJSONObject("avList");
		if(avList!=null) {
			applyAvListFromJson(avList, presetGeometry);
		}
	}

	public static void applyCustomGeometryPropertiesFromJson(JSONObject geometryAttrs, CTCustomGeometry2D customGeometry) {
		final JSONObject avList = geometryAttrs.optJSONObject("avList");
		if(avList!=null) {
			applyAvListFromJson(avList, customGeometry);
		}
	}

	public static void applyAvListFromJson(Object jsonObject, IAvList avList) {
		if(jsonObject instanceof JSONObject) {
			CTGeomGuideList geometryList = avList.getAvLst();
			if(geometryList==null) {
				geometryList = Context.getDmlObjectFactory().createCTGeomGuideList();
				avList.setAvLst(geometryList);
			}
			final List<CTGeomGuide> gdList = geometryList.getGd();
			final Iterator<Entry<String, Object>> avValueIter = ((JSONObject)jsonObject).entrySet().iterator();
			while(avValueIter.hasNext()) {
				final Entry<String, Object> avValueEntry = avValueIter.next();
				final String avName = avValueEntry.getKey();
				final Object avValue = avValueEntry.getValue();

				int i;
				for(i = 0; i < gdList.size(); i++) {
					final CTGeomGuide geomGuide = gdList.get(i);
					if(geomGuide.getName().equals(avName)) {
						if(avValue instanceof Number) {
							geomGuide.setFmla("val " + Long.valueOf(((Number)avValue).longValue()).toString());
						}
						else if(avValue == JSONObject.NULL) {
							gdList.remove(i);
						}
						break;
					}
				}
				if(i == gdList.size()) {
					if(avValue instanceof Number) {
						final CTGeomGuide geomGuide = Context.getDmlObjectFactory().createCTGeomGuide();
						geomGuide.setName(avName);
						geomGuide.setFmla("val " + Long.valueOf(((Number)avValue).longValue()).toString());
						gdList.add(geomGuide);
					}
				}
			}
		}
		else if(jsonObject == JSONObject.NULL) {
			avList.setAvLst(null);
		}
	}

	public static void createJsonFromPresetGeometry(JSONObject attrs, CTPresetGeometry2D presetGeometry)
    	throws JSONException {

        if(presetGeometry!=null) {

        	final JSONObject initialGeometryAttrs = attrs.optJSONObject("geometry");
            final JSONObject geometryAttrs = initialGeometryAttrs!=null ? initialGeometryAttrs : new JSONObject(5);

            final STShapeType presetShapeType = presetGeometry.getPrst();
            if(presetShapeType!=null) {
            	geometryAttrs.put("presetShape", presetShapeType.value());
            }
            if(initialGeometryAttrs==null&&!geometryAttrs.isEmpty()) {
                attrs.put("geometry", geometryAttrs);
            }
            if(presetShapeType!=null) {
        		createJsonFromAvList(geometryAttrs, presetGeometry.getAvLst());
            }
        }
    }

    public static void createJsonFromCustomGeometry(JSONObject attrs, CTCustomGeometry2D customGeometry)
    	throws JSONException {

    	if(customGeometry!=null) {

    		final JSONObject initialGeometryAttrs = attrs.optJSONObject("geometry");
    		final JSONObject geometryAttrs = initialGeometryAttrs!=null ? initialGeometryAttrs : new JSONObject(4);

    		createJsonFromAvList(geometryAttrs, customGeometry.getAvLst());
    		createJsonFromGuideList(geometryAttrs, customGeometry.getGdLst());
    		createJsonFromPathList(geometryAttrs, customGeometry.getPathLst());
    		createJsonFromRect(geometryAttrs, customGeometry.getRect());
    		if(initialGeometryAttrs==null&&!geometryAttrs.isEmpty()) {
    			attrs.put("geometry", geometryAttrs);
    		}
    	}
    }

    public static JSONObject createJSPresetGeometries()
    	throws JSONException {

    	final JSONObject allPresets = new JSONObject(1);
    	final Iterator<Entry<STShapeType, CTCustomGeometry2D>> customShapePresetIter = Context.getCustomShapePresets().entrySet().iterator();
    	while(customShapePresetIter.hasNext()) {
    		final Entry<STShapeType, CTCustomGeometry2D> customShapePreset = customShapePresetIter.next();
    		final JSONObject preset = new JSONObject(4);
    		final CTCustomGeometry2D customGeometry = customShapePreset.getValue();
    		createJsonFromAvList(preset, customGeometry.getAvLst());
    		createJsonFromGuideList(preset, customGeometry.getGdLst());
    		createJsonFromPathList(preset, customGeometry.getPathLst());
    		createJsonFromRect(preset, customGeometry.getRect());
    		allPresets.put(customShapePreset.getKey().value(), preset);
    	}
    	return allPresets;
    }

    public static void createJsonFromAvList(JSONObject geometryAttrs, CTGeomGuideList geomGuides)
    	throws JSONException {

    	if(geomGuides!=null) {
    		final List<CTGeomGuide> geomGuideList = geomGuides.getGd();
    		JSONObject lst = geometryAttrs.optJSONObject("avList");
    		if(lst==null) {
    			lst = new JSONObject(geomGuideList.size());
        		geometryAttrs.put("avList", lst);
    		}
    		for(CTGeomGuide geomGuide:geomGuideList) {
    			if(geomGuide.getName()!=null&&geomGuide.getFmla()!=null) {
    				lst.put(geomGuide.getName(), Integer.valueOf(geomGuide.getFmla().substring(4)));
    			}
    		}
    	}
    }

    final private static String[] geomParameterNames = { "p0", "p1", "p2" };

    public static void createJsonFromGuideList(JSONObject geometryAttrs, CTGeomGuideList geomGuides)
    	throws JSONException {

    	if(geomGuides!=null) {
    		final List<CTGeomGuide> geomGuideList = geomGuides.getGd();
    		final JSONArray lst = new JSONArray(geomGuideList.size());
    		for(CTGeomGuide geomGuide:geomGuideList) {
    			final Object[] p = geomGuide.getParsedFmla();
    			final JSONObject gd = new JSONObject(1 + p.length);
    			if(geomGuide.getName()!=null) {
    				gd.put("name", geomGuide.getName());
    			}
   				gd.put("op", (String)p[0]);
   				for(int i = 1; i < p.length; i++) {
   					final Object o = p[i];
   					if (o instanceof Number) {
   						gd.put(geomParameterNames[i-1], (Number)o);
   					}
   					else {
   						gd.put(geomParameterNames[i-1], (String)o);
   					}
   				}
   				lst.put(gd);
    		}
    		geometryAttrs.put("gdList", lst);
    	}
    }

    public static void createJsonFromPathList(JSONObject geometryAttrs, CTPath2DList pathes)
    	throws JSONException {

    	if(pathes!=null) {
    		final List<CTPath2D> pathList = pathes.getPath();
    		final JSONArray jsonPathList = new JSONArray(pathList.size());
    		for(CTPath2D path:pathList) {
    			final JSONObject jsonPath = new JSONObject();
    			final STPathFillMode pathFillMode = path.getFill();
    			if(pathFillMode!=null) {
    				jsonPath.put("fillMode", pathFillMode.value());
    			}
    			jsonPath.put("width", path.getW());
    			jsonPath.put("height", path.getH());
    			if(path.isStroke()==false) {
    				jsonPath.put("isStroke", false);
    			}
    			if(path.isExtrusionOk()==false) {
    				jsonPath.put("isExtrusionOk", false);
    			}
    			final List<Object> commands = path.getCloseOrMoveToOrLnTo();
    			final JSONArray jsonCommands = new JSONArray(commands.size());
    			for(Object command:commands) {
    				if(command instanceof CTPath2DLineTo) {
    					final CTPath2DLineTo c = (CTPath2DLineTo)command;
    					final CTAdjPoint2D pt2D = c.getPt();
    					final JSONObject jsonLineTo = new JSONObject(3);
    					jsonLineTo.put("c", "lineTo");
    					jsonLineTo.put("x", pt2D.getParsedX());
    					jsonLineTo.put("y", pt2D.getParsedY());
    					jsonCommands.put(jsonLineTo);
    				}
    				else if(command instanceof CTPath2DArcTo) {
    					final CTPath2DArcTo c = (CTPath2DArcTo)command;
    					final JSONObject jsonArcTo = new JSONObject(5);
    					jsonArcTo.put("c", "arcTo");
    					jsonArcTo.put("hr", c.getParsedHR());
    					jsonArcTo.put("wr", c.getParsedWR());
    					jsonArcTo.put("stAng", c.getParsedStAng());
    					jsonArcTo.put("swAng", c.getParsedSwAng());
    					jsonCommands.put(jsonArcTo);
    				}
    				else if(command instanceof CTPath2DMoveTo) {
    					final CTPath2DMoveTo c = (CTPath2DMoveTo)command;
    					final CTAdjPoint2D pt2D = c.getPt();
    					final JSONObject jsonMoveTo = new JSONObject(3);
    					jsonMoveTo.put("c", "moveTo");
    					jsonMoveTo.put("x", pt2D.getParsedX());
    					jsonMoveTo.put("y", pt2D.getParsedY());
    					jsonCommands.put(jsonMoveTo);
    				}
    				else if(command instanceof CTPath2DClose) {
    					final JSONObject jsonClose = new JSONObject(1);
    					jsonClose.put("c", "close");
    					jsonCommands.put(jsonClose);
    				}
    				else if(command instanceof CTPath2DQuadBezierTo) {
    					final CTPath2DQuadBezierTo c = (CTPath2DQuadBezierTo)command;
    					final List<CTAdjPoint2D> pts2D = c.getPt();
    					final JSONObject jsonQuadBezierTo = new JSONObject(2);
    					jsonQuadBezierTo.put("c", "quadBezierTo");
    					final JSONArray jsonPts = new JSONArray(pts2D.size());
    					for(CTAdjPoint2D pt:pts2D) {
        					final JSONObject jsonPt = new JSONObject(2);
    						jsonPt.put("x", pt.getParsedX());
    						jsonPt.put("y", pt.getParsedY());
    						jsonPts.put(jsonPt);
    					}
    					jsonQuadBezierTo.put("pts", jsonPts);
    					jsonCommands.put(jsonQuadBezierTo);
    				}
    				else if(command instanceof CTPath2DCubicBezierTo) {
    					final CTPath2DCubicBezierTo c = (CTPath2DCubicBezierTo)command;
    					final List<CTAdjPoint2D> pts2D = c.getPt();
    					final JSONObject jsonCubicBezierTo = new JSONObject(2);
    					jsonCubicBezierTo.put("c", "cubicBezierTo");
    					final JSONArray jsonPts = new JSONArray(pts2D.size());
    					for(CTAdjPoint2D pt:pts2D) {
        					final JSONObject jsonPt = new JSONObject(2);
    						jsonPt.put("x", pt.getParsedX());
    						jsonPt.put("y", pt.getParsedY());
    						jsonPts.put(jsonPt);
    					}
    					jsonCubicBezierTo.put("pts", jsonPts);
    					jsonCommands.put(jsonCubicBezierTo);
    				}
    			}
    			jsonPath.put("commands", jsonCommands);
    			jsonPathList.put(jsonPath);
    		}
    		geometryAttrs.put("pathList", jsonPathList);
    	}
    }

    public static void createJsonFromRect(JSONObject geometryAttrs, CTGeomRect rect)
    	throws JSONException {

    	if(rect!=null) {
    		final JSONObject textRect = new JSONObject(4);
    		if(rect.getL()!=null) {
    			textRect.put("l", CTAdjPoint2D.getLongIfPossible(rect.getL()));
    		}
    		if(rect.getT()!=null) {
    			textRect.put("t", CTAdjPoint2D.getLongIfPossible(rect.getT()));
    		}
    		if(rect.getR()!=null) {
    			textRect.put("r", CTAdjPoint2D.getLongIfPossible(rect.getR()));
    		}
    		if(rect.getB()!=null) {
    			textRect.put("b", CTAdjPoint2D.getLongIfPossible(rect.getB()));
    		}
    		if(!textRect.isEmpty()) {
    			geometryAttrs.put("textRect", textRect);
    		}
    	}
    }
}
