/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.operations;

import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.dml.CTGroupShapeProperties;
import org.docx4j.dml.CTGroupTransform2D;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTPoint2D;
import org.docx4j.dml.CTPositiveSize2D;
import org.docx4j.dml.CTTableStyle;
import org.docx4j.dml.CTTableStyleList;
import org.docx4j.dml.CTTextField;
import org.docx4j.dml.ITransform2D;
import org.docx4j.dml.ITransform2DAccessor;
import org.docx4j.jaxb.Context;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.packages.PresentationMLPackage;
import org.docx4j.openpackaging.parts.PresentationML.MainPresentationPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideLayoutPart;
import org.docx4j.openpackaging.parts.PresentationML.SlideMasterPart;
import org.docx4j.openpackaging.parts.PresentationML.SlidePart;
import org.docx4j.openpackaging.parts.PresentationML.TableStylesPart;
import org.docx4j.openpackaging.parts.PresentationML.ViewPropertiesPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart.AddPartBehaviour;
import org.docx4j.relationships.Relationship;
import org.docx4j.wml.ContentAccessor;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.jvnet.jaxb2_commons.ppp.Child;
import org.pptx4j.Pptx4jException;
import org.pptx4j.pml.CTNormalViewPortion;
import org.pptx4j.pml.CTNormalViewProperties;
import org.pptx4j.pml.CommonSlideData;
import org.pptx4j.pml.GroupShape;
import org.pptx4j.pml.GroupShape.NvGrpSpPr;
import org.pptx4j.pml.ObjectFactory;
import org.pptx4j.pml.Presentation.SldIdLst.SldId;
import org.pptx4j.pml.Presentation.SldMasterIdLst;
import org.pptx4j.pml.Presentation.SldMasterIdLst.SldMasterId;
import org.pptx4j.pml.Sld;
import org.pptx4j.pml.SldLayout;
import org.pptx4j.pml.SldMaster;
import org.pptx4j.pml.SlideLayoutIdList.SldLayoutId;
import org.pptx4j.pml.ViewPr;

import com.openexchange.office.FilterException;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.IParagraph;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.components.Component.SplitMode;
import com.openexchange.office.ooxml.components.Component.Type;
import com.openexchange.office.ooxml.operations.ApplyOperationHelper;
import com.openexchange.office.ooxml.pptx.PptxOperationDocument;
import com.openexchange.office.ooxml.pptx.components.RootComponent;
import com.openexchange.office.ooxml.pptx.components.ShapeGraphicComponent;
import com.openexchange.office.ooxml.pptx.components.ShapeGroupComponent;
import com.openexchange.office.ooxml.pptx.components.SlideComponent;
import com.openexchange.office.ooxml.pptx.components.TableRowComponent;
import com.openexchange.office.ooxml.pptx.components.TextFieldComponent;

public class PptxApplyOperationHelper extends ApplyOperationHelper {

    private final PptxOperationDocument operationDocument;
    private final PresentationMLPackage presentationMLPackage;

    public PptxApplyOperationHelper(PptxOperationDocument _operationDocument) {
        super();

        operationDocument = _operationDocument;
        presentationMLPackage = _operationDocument.getPackage();
    }

    @Override
    public PptxOperationDocument getOperationDocument() {
        return operationDocument;
    }

    public void insertParagraph(JSONArray start, String target, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Component.Type.PARAGRAPH);
    }

    public void splitParagraph(JSONArray start, String target)
        throws JSONException {

        ((IParagraph)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1)).splitParagraph(start.getInt(start.length()-1));
    }

    public void mergeParagraph(JSONArray start, String target) {

    	((IParagraph)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length())).mergeParagraph();
    }

    public void insertText(JSONArray start, String target, String text, JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	if(text.length()>0) {
    		final Component component = Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1);
    		((IParagraph)component).insertText(operationDocument, start.getInt(start.length()-1), text, attrs);
    	}
    }

    public void insertTab(JSONArray start, String target, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JAXBException, JSONException {

    	insertText(start, target, "\t", attrs);
    }

    public void delete(JSONArray start, String target, JSONArray end)
        throws JSONException, InvalidFormatException, JAXBException, PartUnrecognisedException {

        if(start==null||start.length()==0)
            return;

        int startComponent = start.getInt(start.length()-1);
        int endComponent = startComponent;
        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endComponent = end.getInt(end.length()-1);
        }
        Component component = Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length());
		component.splitStart(startComponent, SplitMode.DELETE);
        component.delete(getOperationDocument(), (endComponent-startComponent)+1);
    }

    public void setAttributes(JSONArray start, String target, JSONArray end, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	if(attrs==null) {
    		return;
    	}
        int startIndex = start.getInt(start.length()-1);
        int endIndex = startIndex;

        if(end!=null) {
            if(end.length()!=start.length())
                return;
            endIndex = end.getInt(end.length()-1);
        }
    	Component component = Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length());
		component.splitStart(startIndex, SplitMode.ATTRIBUTE);
        while(component!=null&&component.getComponentNumber()<=endIndex) {
        	if(component.getNextComponentNumber()>=endIndex+1) {
        		component.splitEnd(endIndex, SplitMode.ATTRIBUTE);
        	}
        	component.applyAttrsFromJSON(operationDocument, attrs);
            component = component.getNextComponent();
        }
    }

    public void insertSlide(JSONArray start, String target, JSONObject attrs)
    	throws InvalidFormatException, Pptx4jException, JSONException, PartUnrecognisedException, JAXBException {

    	final SlideLayoutPart slideLayoutPart = (SlideLayoutPart)getOperationDocument().getPartByTarget(target);
    	final SlidePart newSlidePart = new SlidePart();
    	final Sld newSlide = new Sld();
    	final CommonSlideData commonSlideData = Context.getpmlObjectFactory().createCommonSlideData();
    	newSlide.setCSld(commonSlideData);
    	commonSlideData.setSpTree(createSpTree());
    	newSlidePart.setJaxbElement(newSlide);
    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
    	mainPresentationPart.addSlide(start.getInt(0), newSlidePart);
    	newSlidePart.addTargetPart(slideLayoutPart, AddPartBehaviour.REUSE_EXISTING);
    	if(attrs!=null) {
    		getOperationDocument().setContextPart(newSlidePart);
    		new SlideComponent(null, new IndexedNode<Object>(newSlidePart), 0).applyAttrsFromJSON(operationDocument, attrs);
    	}
    }

    public void insertMasterSlide(String id, Integer start, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JSONException {

    	final SlideMasterPart newSlideMasterPart = new SlideMasterPart();
    	final SldMaster newSlideMaster = new SldMaster();
    	final CommonSlideData commonSlideData = Context.getpmlObjectFactory().createCommonSlideData();
    	newSlideMaster.setCSld(commonSlideData);
    	commonSlideData.setSpTree(createSpTree());
    	newSlideMasterPart.setJaxbElement(newSlideMaster);
    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
    	final SldMasterIdLst sldMasterIdLst = mainPresentationPart.getJaxbElement().getSldMasterIdLst();

    	final SldMasterId sldMasterId = Context.getpmlObjectFactory().createPresentationSldMasterIdLstSldMasterId();
    	sldMasterId.setId(Long.valueOf(id));
    	if(start!=null) {
    		sldMasterIdLst.getSldMasterId().add(start, sldMasterId);
    	}
    	else {
    		sldMasterIdLst.getSldMasterId().add(sldMasterId);
    	}
    	final Relationship rel = mainPresentationPart.addTargetPart(newSlideMasterPart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
    	sldMasterId.setRid(rel.getId());
    	getOperationDocument().getIdToPartMap().put(id, newSlideMasterPart);
    	getOperationDocument().getRelTargetToIdMap().put("../" + mainPresentationPart.getRelationshipsPart().getRelationshipByID(sldMasterId.getRid()).getTarget(), sldMasterId.getId().toString());
    	if(attrs!=null) {
    		getOperationDocument().setContextPart(newSlideMasterPart);
    		new SlideComponent(null, new IndexedNode<Object>(newSlideMasterPart), 0).applyAttrsFromJSON(operationDocument, attrs);
    	}
    }

    public void insertLayoutSlide(String id, String target, Integer start, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JSONException {

    	final SlideMasterPart slideMasterPart = (SlideMasterPart)getOperationDocument().getPartByTarget(target);
    	final SlideLayoutPart newLayoutSlidePart = new SlideLayoutPart();
    	final SldLayout newLayoutSlide = new SldLayout();
    	final CommonSlideData commonSlideData = Context.getpmlObjectFactory().createCommonSlideData();
    	newLayoutSlide.setCSld(commonSlideData);
    	commonSlideData.setSpTree(createSpTree());
    	newLayoutSlidePart.setJaxbElement(newLayoutSlide);
    	final Relationship rel = slideMasterPart.addTargetPart(newLayoutSlidePart, AddPartBehaviour.RENAME_IF_NAME_EXISTS);
    	final SldMaster sldMaster = slideMasterPart.getJaxbElement();
    	final SldLayoutId sldLayoutId = Context.getpmlObjectFactory().createSlideLayoutIdListSldLayoutId();
    	sldLayoutId.setId(Long.valueOf(id));
    	final List<SldLayoutId> sldLayoutIdLst = sldMaster.getSldLayoutIdLst(true).getSldLayoutId();
    	if(start!=null) {
    		sldLayoutIdLst.add(start, sldLayoutId);
    	}
    	else {
    		sldLayoutIdLst.add(sldLayoutId);
    	}
    	sldLayoutId.setRid(rel.getId());
    	getOperationDocument().getIdToPartMap().put(id, newLayoutSlidePart);
    	getOperationDocument().getRelTargetToIdMap().put(slideMasterPart.getRelationshipsPart().getRelationshipByID(sldLayoutId.getRid()).getTarget(), sldLayoutId.getId().toString());
    	newLayoutSlidePart.addTargetPart(slideMasterPart, AddPartBehaviour.REUSE_EXISTING);
    	if(attrs!=null) {
    		getOperationDocument().setContextPart(newLayoutSlidePart);
    		new SlideComponent(null, new IndexedNode<Object>(newLayoutSlidePart), 0).applyAttrsFromJSON(operationDocument, attrs);
    	}
    }

    public GroupShape createSpTree() {
    	final ObjectFactory pmlFactory = Context.getpmlObjectFactory();
    	final org.docx4j.dml.ObjectFactory dmlFactory = Context.getDmlObjectFactory();
    	final GroupShape spTree = pmlFactory.createGroupShape();
    	final NvGrpSpPr nvGrpSpPr = pmlFactory.createGroupShapeNvGrpSpPr();
    	spTree.setNvGrpSpPr(nvGrpSpPr);
    	final CTNonVisualDrawingProps cNvPr = dmlFactory.createCTNonVisualDrawingProps();
    	cNvPr.setName("");
    	nvGrpSpPr.setCNvPr(cNvPr);
    	nvGrpSpPr.setCNvGrpSpPr(dmlFactory.createCTNonVisualGroupDrawingShapeProps());
    	nvGrpSpPr.setNvPr(pmlFactory.createNvPr());
    	final CTGroupShapeProperties grpSpPr = dmlFactory.createCTGroupShapeProperties();
    	final CTGroupTransform2D xFrm = grpSpPr.getXfrm(true);
    	final CTPoint2D off = dmlFactory.createCTPoint2D();
    	off.setX(0);
    	off.setY(0);
    	xFrm.setOff(off);
    	final CTPositiveSize2D ext = dmlFactory.createCTPositiveSize2D();
    	ext.setCx(0);
    	ext.setCy(0);
    	xFrm.setExt(ext);
    	final CTPoint2D chOff = dmlFactory.createCTPoint2D();
    	chOff.setX(0);
    	chOff.setY(0);
    	xFrm.setChOff(chOff);
    	final CTPositiveSize2D chExt = dmlFactory.createCTPositiveSize2D();
    	chExt.setCx(0);
    	chExt.setCy(0);
    	xFrm.setChExt(chExt);
    	spTree.setGrpSpPr(grpSpPr);
    	return spTree;
    }

    public Component insertDrawing(JSONArray start, String target, String type, JSONObject attrs)
    	throws UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JAXBException, JSONException {

    	Component.Type cType = Component.Type.AC_SHAPE;
    	if(type.equals("IMAGE")) {
    		cType = Component.Type.AC_IMAGE;
    	}
    	else if(type.equals("GROUP")) {
    		cType = Component.Type.AC_GROUP;
    	}
    	else if(type.equals("CONNECTOR")) {
    		cType = Component.Type.AC_CONNECTOR;
    	}
    	else if(type.equals("TABLE")) {
    		cType = Component.Type.TABLE;
    	}
    	return Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1).insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, cType);
    }

    public void setDocumentAttributes(JSONObject attrs)
    	throws InvalidFormatException {

    	final MainPresentationPart mainPresentationPart = presentationMLPackage.getMainPresentationPart();
        final RelationshipsPart mainPresentationRelations = mainPresentationPart.getRelationshipsPart();
    	final JSONObject layoutSettings = attrs.optJSONObject("layout");
    	if(layoutSettings!=null) {
    		ViewPropertiesPart viewPropertiesPart = null;    		
            final Relationship viewRel = mainPresentationRelations.getRelationshipByType(Namespaces.PRESENTATIONML_VIEW_PROPS);
            if(viewRel!=null) {
            	viewPropertiesPart = (ViewPropertiesPart)mainPresentationRelations.getPart(viewRel);
            }
            else {
            	viewPropertiesPart = new ViewPropertiesPart();
            	viewPropertiesPart.setJaxbElement(new ViewPr());
            	mainPresentationPart.addTargetPart(viewPropertiesPart);
            }
            final ViewPr viewPr = viewPropertiesPart.getJaxbElement();
            final Object slidePaneWidth = layoutSettings.opt("slidePaneWidth");
            if(slidePaneWidth instanceof Number) {
            	CTNormalViewProperties normalViewProperties = viewPr.getNormalViewPr();
            	if(normalViewProperties==null) {
            		normalViewProperties = Context.getpmlObjectFactory().createCTNormalViewProperties();
            		viewPr.setNormalViewPr(normalViewProperties);
            	}
            	CTNormalViewPortion leftViewPortion = normalViewProperties.getRestoredLeft();
            	if(leftViewPortion==null) {
            		leftViewPortion = Context.getpmlObjectFactory().createCTNormalViewPortion();
            		normalViewProperties.setRestoredLeft(leftViewPortion);
            	}
            	final Double sz = (((Number)slidePaneWidth).doubleValue() * 1000);
            	leftViewPortion.setSz(sz.intValue());
            }
    	}
    	final JSONObject pageSettings = attrs.optJSONObject("page");
    	if(pageSettings!=null) {
    		final Object width = pageSettings.opt("width");
    		if(width instanceof Number) {
    			mainPresentationPart.getJaxbElement().getSldSz(true).setCx(((Number)width).intValue() * 360);
    		}
    		final Object height = pageSettings.opt("height");
    		if(height instanceof Number) {
    			mainPresentationPart.getJaxbElement().getSldSz(true).setCy(((Number)height).intValue() * 360);
    		}
    	}
    }

    public void insertHardBreak(JSONArray start, String target, JSONObject attrs)
        throws JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

    	Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1)
			.insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Type.HARDBREAK_DEFAULT);
    }

    public void changeLayout(JSONArray start, String target)
    	throws InvalidFormatException {

    	operationDocument.setContextPartByTarget(target);
    	final SlideLayoutPart slideLayoutPart = (SlideLayoutPart)operationDocument.getContextPart();
    	final SlideComponent slideComponent = (SlideComponent)Component.getComponent(new RootComponent(operationDocument, null), start, start.length());
    	final SlidePart slidePart = (SlidePart)slideComponent.getObject();
    	final RelationshipsPart slideRelationshipsPart = slidePart.getRelationshipsPart();
    	Relationship relationship = slideRelationshipsPart.getRelationshipByType(Namespaces.PRESENTATIONML_SLIDE_LAYOUT);
    	if(relationship!=null) {
    		slideRelationshipsPart.removeRelationship(relationship);
    	}
    	slidePart.addTargetPart(slideLayoutPart);
    }

    public void moveSlide(JSONArray start, JSONArray end)
    	throws JSONException {
    
    	final int source = start.getInt(0);
    	final int dest = end.getInt(0);

    	final IndexedNodeList<Object> sldList = presentationMLPackage.getMainPresentationPart().getContent();
    	Object sld = sldList.remove(source);
    	sldList.add(dest, sld);
    	final List<SldId> sldIdList = presentationMLPackage.getMainPresentationPart().getJaxbElement().getSldIdLst(false).getSldId();
    	final SldId sldId = sldIdList.remove(source);
    	sldIdList.add(dest, sldId);
    }

    public void moveLayoutSlide(String id, int start, String target)
    	throws JSONException, InvalidFormatException {

    	operationDocument.setContextPartByTarget(target);
    	final SlideMasterPart targetMasterPart = (SlideMasterPart)getOperationDocument().getContextPart();
    	final SlideLayoutPart sourceLayoutPart = (SlideLayoutPart)getOperationDocument().getPartByTarget(id);
    	final RelationshipsPart sourceLayoutRelationshipsPart = sourceLayoutPart.getRelationshipsPart();
    	final Relationship sourceLayoutTargetRelation = sourceLayoutRelationshipsPart.getRelationshipByType(Namespaces.PRESENTATIONML_SLIDE_MASTER);

    	// retrieving the layout target that has to be used for this slide
		final String sourceLayoutTargetId = getOperationDocument().getRelTargetToIdMap()
			.get(sourceLayoutTargetRelation.getTarget());
		final SlideMasterPart sourceMasterPart = (SlideMasterPart)getOperationDocument().getIdToPartMap().get(sourceLayoutTargetId);

		// removing the layout relation to the source master page
		final List<SldLayoutId> sourceLayoutIdList = sourceMasterPart.getJaxbElement().getSldLayoutIdLst(true).getSldLayoutId();
		SldLayoutId sldLayoutId = null;
		for(int i = 0; i < sourceLayoutIdList.size(); i++) {
			sldLayoutId = sourceLayoutIdList.get(i);
			if(sldLayoutId.getId().toString().equals(id)) {
				sourceLayoutIdList.remove(i);
				sourceMasterPart.getRelationshipsPart().removeRelationship(sourceMasterPart.getRelationshipsPart().getRelationshipByID(sldLayoutId.getRid()));
				break;
			}
		}
		targetMasterPart.getJaxbElement().getSldLayoutIdLst(true).getSldLayoutId().add(start, sldLayoutId);
		sldLayoutId.setRid(targetMasterPart.addTargetPart(sourceLayoutPart, AddPartBehaviour.OVERWRITE_IF_NAME_EXISTS).getId());
		sourceLayoutRelationshipsPart.removeRelationship(sourceLayoutTargetRelation);
		sourceLayoutPart.addTargetPart(targetMasterPart);
	}

    public void insertField(JSONArray start, String target, String type, String representation, JSONObject attrs)
    	throws JSONException, UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JAXBException {

    	final CTTextField textField = (CTTextField)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1)
			.insertChildComponent(operationDocument, start.getInt(start.length()-1), attrs, Type.SIMPLE_FIELD).getObject();

    	textField.setId("{2640DB9D-575D-4FF8-9117-D1A6111C2CD0}");
    	textField.setType(type);
    	textField.setT(representation);
    }

    public void updateField(JSONArray start, String target, String type, String representation, JSONObject attrs)
        	throws JSONException, UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JAXBException {

    	final TextFieldComponent tfComponent = (TextFieldComponent)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length());
    	if(type!=null) {
    		((CTTextField)tfComponent.getObject()).setType(type);
    	}
    	if(representation!=null) {
    		((CTTextField)tfComponent.getObject()).setT(representation);
    	}
    }

    public void group(JSONArray start, String target, JSONArray drawings, JSONObject attrs, JSONArray childAttrs)
    	throws JSONException, InvalidFormatException, UnsupportedOperationException, PartUnrecognisedException, JAXBException {

    	int groupInsertPosition = start.getInt(start.length()-1);
    	final List<Object> drawingChildObjects = new ArrayList<Object>();
    	for(int i = drawings.length()-1; i >= 0; i--) {
    		final int c = drawings.getInt(i);
    		if(c < groupInsertPosition) {
    			groupInsertPosition--;
    		}
    		final Component child = Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1).getChildComponent(c);
    		drawingChildObjects.add(0, child.getObject());
    		child.splitStart(child.getComponentNumber(), SplitMode.DELETE);
    		child.delete(getOperationDocument(), 1);
    	}
    	final JSONArray groupStart = new JSONArray(start.toArray());
    	groupStart.put(groupStart.length()-1, groupInsertPosition);
    	final Component groupComponent = insertDrawing(start, target, "GROUP", attrs);
    	final GroupShape groupShape = (GroupShape)groupComponent.getObject();
    	groupShape.getContent().addAll(drawingChildObjects);

    	final JSONObject drawingAttrs = attrs!=null ? attrs.optJSONObject("drawing") : null;
    	Double rotation = null;
    	double grpCenterX = 0;
    	double grpCenterY = 0;
    	if(drawingAttrs!=null) {
    		final Object rot = drawingAttrs.opt("rotation");
    		if(rot instanceof Number) {
    			rotation = ((Number)rot).doubleValue();
    		}
    		Rectangle childsCenter = new Rectangle();
    		for(Object child:groupShape.getContent()) {
    			final ITransform2D iTransform2D = ((ITransform2DAccessor)child).getXfrm(true);
    			final CTPoint2D chOff = iTransform2D.getOff(true);
    			final CTPositiveSize2D chExt = iTransform2D.getExt(true);
    			final Point point = new Point(Double.valueOf(chExt.getCx()*0.5+chOff.getX()).intValue(), Double.valueOf(chExt.getCy()*0.5+chOff.getY()).intValue());
    			childsCenter.add(point);
    		}
    		if(!childsCenter.isEmpty()) {
    			grpCenterX = childsCenter.getCenterX();
    			grpCenterY = childsCenter.getCenterY();
    		}
    	}
    	Rectangle childsRect = new Rectangle();
    	for(Object child:groupShape.getContent()) {
			final ITransform2D iTransform2D = ((ITransform2DAccessor)child).getXfrm(true);
			final CTPoint2D chOff = iTransform2D.getOff(true);
			final CTPositiveSize2D chExt = iTransform2D.getExt(true);
			double chOffX = chOff.getX();
			double chOffY = chOff.getY();
			double chExtX = chExt.getCx();
			double chExtY = chExt.getCy();
			if(rotation!=null) {
				iTransform2D.setRot(Double.valueOf(iTransform2D.getRot()-rotation.doubleValue()*60000.0).intValue()%21600000);

				double[] pts = rotatePoint(chOffX, chOffY, chOffX + chExtX * 0.5, chOffY + chExtY * 0.5, -(double)iTransform2D.getRot() / 60000.0);
				pts = rotatePoint(pts[0], pts[1], grpCenterX, grpCenterY, -rotation);
				chOff.setX(Double.valueOf(pts[0] + 0.5).longValue());
				chOff.setY(Double.valueOf(pts[1] + 0.5).longValue());
			}
			final Rectangle childRect = new Rectangle((int)chOff.getX(), (int)chOff.getY(), (int)chExtX, (int)chExtY);
			if(childsRect.isEmpty()) {
				childsRect = childRect;
			}
			else {
				childsRect = childsRect.union(childRect);
			}
    	}
    	final CTGroupTransform2D iTransform2D = groupShape.getXfrm(true);
    	final CTPoint2D groupChildOffset = iTransform2D.getChOff(true);
    	groupChildOffset.setX(childsRect.x);
    	groupChildOffset.setY(childsRect.y);
    	final CTPositiveSize2D groupChildExt = iTransform2D.getChExt(true);
    	groupChildExt.setCx(childsRect.width);
    	groupChildExt.setCy(childsRect.height);
    	groupComponent.applyAttrsFromJSON(operationDocument, attrs);
    }

    public void ungroup(JSONArray start, String target, JSONArray drawings)
    	throws InvalidFormatException, JSONException {

    	final int groupPosition = start.getInt(start.length()-1);
    	final Component parentComponent = Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1);
    	final IndexedNodeList<Object> parentContent = (IndexedNodeList<Object>)((ContentAccessor)parentComponent.getObject()).getContent();

    	final ShapeGroupComponent groupComponent = (ShapeGroupComponent)parentComponent.getChildComponent(groupPosition);
    	final GroupShape group = (GroupShape)groupComponent.getObject();
    	final CTGroupTransform2D groupTransform = group.getXfrm(true);
    	final CTPoint2D grpOff = groupTransform.getOff(true);
    	final CTPositiveSize2D grpExt = groupTransform.getExt(true);
    	final CTPoint2D grpChOff = groupTransform.getChOff(true);
    	final CTPositiveSize2D grpChExt = groupTransform.getChExt(true);

    	final double fXScale = (double)grpExt.getCx() / (grpChExt.getCx()==0 ? 1 : grpChExt.getCx());
    	final double fYScale = (double)grpExt.getCy() / (grpChExt.getCy()==0 ? 1 : grpChExt.getCy());

    	final double groupRotation = group.getXfrm(true).getRot() / 60000.0;
    	final IndexedNodeList<Object> groupContent = group.getContent();

		groupComponent.splitStart(groupComponent.getComponentNumber(), SplitMode.DELETE);;
		groupComponent.delete(operationDocument, 1);

		Component referenceComponent = parentComponent.getChildComponent(groupPosition);
		IndexedNode<Object> referenceNode = referenceComponent!=null ? referenceComponent.getNode() : null;

		for(int i = 0; i < groupContent.size(); i++) {
			if(drawings!=null&&i<drawings.length()) {
				referenceComponent = parentComponent.getChildComponent(drawings.getInt(i));
				referenceNode = referenceComponent!=null ? referenceComponent.getNode() : null;
			}
			final Object newChild = groupContent.get(i);
			if(newChild instanceof Child) {
				((Child)newChild).setParent(parentComponent.getObject());
			}
			if(newChild instanceof ITransform2DAccessor) {
				final ITransform2D childTransform = ((ITransform2DAccessor)newChild).getXfrm(true);
				final CTPositiveSize2D chExt = childTransform.getExt(true);
				final CTPoint2D chOff = childTransform.getOff(true);
				double chOffX = chOff.getX();
				double chOffY = chOff.getY();
				chExt.setCx(Double.valueOf(chExt.getCx()*fXScale).longValue());
				chExt.setCy(Double.valueOf(chExt.getCy()*fYScale).longValue());
				chOffX = (chOffX - grpChOff.getX())*fXScale + grpOff.getX();
				chOffY = (chOffY - grpChOff.getY())*fXScale + grpOff.getY();
				if(groupRotation!=0) {
					double[] pts = rotatePoint(chOffX, chOffY, chOffX + chExt.getCx() * 0.5, chOffY + chExt.getCy() * 0.5, -groupRotation);
					pts = rotatePoint(pts[0], pts[1], grpOff.getX() + (grpExt.getCx() * 0.5), grpOff.getY() + (grpExt.getCy() * 0.5), groupRotation);
					chOffX = pts[0];
					chOffY = pts[1];
					childTransform.setRot(Double.valueOf(childTransform.getRot()+groupRotation*60000.0).intValue()%21600000);
				}
				chOff.setX(Double.valueOf(chOffX + 0.5).longValue());
				chOff.setY(Double.valueOf(chOffY + 0.5).longValue());
			}
	        final IndexedNode<Object> newChildNode = new IndexedNode<Object>(newChild);
	        parentContent.addNode(referenceNode, newChildNode, true);
		}
    }

    private double[] rotatePoint(double ptX, double ptY, double refX, double refY, double rotation) {
    	ptX -= refX;
    	ptY -= refY;
    	double x = ptX;
    	double y = ptY;
		double a = (Math.PI * rotation) / 180.0;
		double s = Math.sin(a);
		double c = Math.cos(a);
		ptX = x * c - y * s;
		ptY = x * s + y * c;
		ptX += refX;
		ptY += refY;
		double[] pts = new double[2];
		pts[0] = ptX;
		pts[1] = ptY;
		return pts;
    }

    public void move(JSONArray start, String target, JSONArray end, JSONArray to)
    	throws JSONException {

    	Component.move(operationDocument, start, target, end, to);
    }

    public void insertRows(JSONArray start, String target, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
    	throws UnsupportedOperationException, InvalidFormatException, PartUnrecognisedException, JAXBException, JSONException {

    	((ShapeGraphicComponent)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1))
    		.insertRows(operationDocument, start.getInt(start.length()-1), count, insertDefaultCells, referenceRow, attrs);
    }

    public void insertCells(JSONArray start, String target, int count, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JAXBException, JSONException {

    	((TableRowComponent)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length()-1)).insertCells(operationDocument, start.getInt(start.length()-1), count, attrs);
    }

    public void insertColumn(JSONArray start, String target, JSONArray tableGrid, int gridPosition, String insertMode)
    	throws JSONException {

    	((ShapeGraphicComponent)Component.getComponent(new RootComponent(getOperationDocument(), target), start, start.length())).insertColumn(operationDocument, tableGrid, gridPosition, insertMode);
    }

    public void deleteColumns(JSONArray start, String target, int startGrid, int endGrid)
    	throws JSONException {

    	((ShapeGraphicComponent)Component.getComponent(getOperationDocument().getRootComponent(null), start, start.length())).deleteColumns(operationDocument, startGrid, endGrid);
    }

    public void changeOrInsertStyleSheet(boolean insert, String type, String styleId, String styleName, JSONObject attrs, String parent, boolean hidden,  int uiPriority, boolean defaultStyle)
    	throws InvalidFormatException, FilterException, JSONException, PartUnrecognisedException {

    	if(!type.equals("table")) {
    		throw new RuntimeException("PPTX: StyleSheetOpertions of type table allowed!");
    	}
    	final TableStylesPart tableStylesPart = getOperationDocument().getTableStylesPart(true);
		final Object o = tableStylesPart.getJaxbElement();
		if(o instanceof CTTableStyleList) {
			final CTTableStyleList tableStyles = (CTTableStyleList)o;
			final List<CTTableStyle> tableStyleList = tableStyles.getTblStyle();

			CTTableStyle tableStyle = null;
			if(insert) {
				tableStyle = new CTTableStyle();
				tableStyleList.add(tableStyle);
				tableStyle.setStyleId(styleId);
			}
			else {
				final Iterator<CTTableStyle> tableStyleIter = tableStyleList.iterator();
				while(tableStyleIter.hasNext()) {
					final CTTableStyle ts = tableStyleIter.next();
					if(styleId.equals(ts.getStyleId())) {
						tableStyle = ts;
						break;
					}
				}
			}
			if(styleName!=null&&!styleName.isEmpty()) {
				tableStyle.setStyleName(styleName);
			}
			if(attrs!=null) {
				DMLHelper.applyTableStyleFromJson(tableStyle, attrs, operationDocument, getOperationDocument().getThemePart(false), tableStylesPart);
			}
		}
    }

    public void deleteStyleSheet(String type, String styleId)
    	throws InvalidFormatException {

    	if(!type.equals("table")) {
    		throw new RuntimeException("PPTX: StyleSheetOpertions of type table allowed!");
    	}
    	final TableStylesPart tableStylesPart = getOperationDocument().getTableStylesPart(false);
		final Object o = tableStylesPart.getJaxbElement();
		final CTTableStyleList tableStyles = (CTTableStyleList)o;
		final List<CTTableStyle> tableStyleList = tableStyles.getTblStyle();
		final Iterator<CTTableStyle> tableStyleIter = tableStyleList.iterator();
		while(tableStyleIter.hasNext()) {
			final CTTableStyle tableStyle = tableStyleIter.next();
			if(styleId.equals(tableStyle.getStyleId())) {
				tableStyleIter.remove();
				return;
			}
		}
		throw new RuntimeException("PPTX: deleteStyleSheet, could not find style!");
    }
}
