/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.pptx.components;

import java.util.Iterator;
import java.util.List;

import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;

import org.docx4j.IndexedNode;
import org.docx4j.IndexedNodeList;
import org.docx4j.XmlUtils;
import org.docx4j.dml.CTNonVisualDrawingProps;
import org.docx4j.dml.CTTable;
import org.docx4j.dml.CTTableCell;
import org.docx4j.dml.CTTableCellProperties;
import org.docx4j.dml.CTTableCol;
import org.docx4j.dml.CTTableGrid;
import org.docx4j.dml.CTTableRow;
import org.docx4j.dml.CTTextParagraph;
import org.docx4j.dml.Graphic;
import org.docx4j.dml.GraphicData;
import org.docx4j.mce.AlternateContent;
import org.docx4j.mce.AlternateContent.Choice;
import org.docx4j.mce.AlternateContent.Fallback;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.VMLPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.relationships.Relationship;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.root.Xml;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.pptx4j.pml.CTGraphicalObjectFrame;
import org.pptx4j.pml.CTOleObject;
import org.pptx4j.pml.Pic;

import com.openexchange.office.ooxml.OperationDocument;
import com.openexchange.office.ooxml.components.Component;
import com.openexchange.office.ooxml.components.ComponentContext;
import com.openexchange.office.ooxml.components.IShapeType;
import com.openexchange.office.ooxml.components.ShapeType;
import com.openexchange.office.ooxml.drawingml.DMLHelper;
import com.openexchange.office.ooxml.drawingml.Picture;
import com.openexchange.office.ooxml.operations.CreateOperationHelper;
import com.openexchange.office.ooxml.pptx.tools.PMLShapeHelper;

public class ShapeGraphicComponent extends PptxComponent implements IShapeComponent, IShapeType {

	final private CTGraphicalObjectFrame graphicalObjectFrame;
	private ShapeType shapeType = ShapeType.UNDEFINED;
	private Pic pic = null;
	private CTTable tbl = null;

	public ShapeGraphicComponent(ComponentContext parentContext, IndexedNode<Object> _node, int _componentNumber) {
        super(parentContext, _node, _componentNumber);

        this.graphicalObjectFrame = (CTGraphicalObjectFrame)getObject();

        pic = getPicture(false);
        if(pic!=null) {
        	shapeType = ShapeType.IMAGE;
        }
        else {
        	tbl = getTable();
        	if(tbl!=null) {
        		shapeType = ShapeType.TABLE;
        	}
        }
    }

    @Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
    	if(shapeType==ShapeType.TABLE) {
            final int index = previousChildContext!=null?previousChildContext.getNode().getIndex()+1:0;
            final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;

            Component nextComponent = null;
            final IndexedNodeList<Object> nodeList = tbl.getContent();
            for(int i=index; nextComponent==null&&i<nodeList.size(); i++) {
                final IndexedNode<Object> childNode = nodeList.getNode(i);
                final Object o = getContentModel(childNode, tbl);
                if(o instanceof CTTableRow) {
                	nextComponent = new TableRowComponent(this, childNode, nextComponentNumber);
                }
            }
            return nextComponent;
    	}
    	else {
    		return null;
    	}
    }

    @Override
    public void applyAttrsFromJSON(OperationDocument operationDocument, JSONObject attrs)
    	throws InvalidFormatException, PartUnrecognisedException, JSONException {

    	PMLShapeHelper.applyAttrsFromJSON(operationDocument, attrs, graphicalObjectFrame);
		DMLHelper.applyTransform2DFromJson(graphicalObjectFrame, attrs, true);
        DMLHelper.applyNonVisualDrawingProperties(graphicalObjectFrame, attrs.optJSONObject("drawing"));
        DMLHelper.applyNonVisualDrawingShapeProperties(graphicalObjectFrame, attrs.optJSONObject("drawing"));

        if(pic!=null) {
        	Picture.applyAttrsFromJSON(operationDocument, attrs, operationDocument.getContextPart(), pic, !(getParentComponent() instanceof ShapeGroupComponent));
    	}
    	else if(tbl!=null) {
    		DMLHelper.applyTableFromJson(attrs, tbl, graphicalObjectFrame.getXfrm(true).getExt(true).getCx(), operationDocument, operationDocument.getContextPart());
    	}
    }

    @Override
    public JSONObject createJSONAttrs(CreateOperationHelper createOperationHelper, JSONObject attrs)
    	throws JSONException {

    	PMLShapeHelper.createJSONAttrs(createOperationHelper, attrs, graphicalObjectFrame);
        DMLHelper.createJsonFromTransform2D(attrs, graphicalObjectFrame.getXfrm(false), true);
        DMLHelper.createJsonFromNonVisualDrawingProperties(attrs, graphicalObjectFrame);
    	DMLHelper.createJsonFromNonVisualDrawingShapeProperties(attrs, graphicalObjectFrame);

    	if(pic!=null) {
        	final OperationDocument operationDocument = createOperationHelper.getOperationDocument();
    		Picture.createJSONAttrs(operationDocument, attrs, operationDocument.getContextPart(), pic, !(getParentComponent() instanceof ShapeGroupComponent));
    	}
    	else if(tbl!=null) {
        	DMLHelper.createJsonFromTable(attrs, tbl, createOperationHelper.getOperationDocument().getThemePart(false), createOperationHelper.getOperationDocument().getContextPart());
    	}
    	return attrs;
    }

	@Override
	public Long getId() {
		final CTNonVisualDrawingProps nvdProps = graphicalObjectFrame.getNonVisualDrawingProperties(false);
		return nvdProps!=null ? nvdProps.getId() : null;
	}

	@Override
	public ShapeType getType() {
		return shapeType;
	}

	@Override
	public boolean isPresentationObject() {
		return false;
	}

	private Pic getPicture(boolean removeVML) {

		final Graphic graphic = graphicalObjectFrame.getGraphic();
	    if(graphic!=null&&graphic.getGraphicData()!=null) {
	    	final GraphicData graphicData = graphic.getGraphicData();
	    	final String uri = graphicData.getUri();
	        final List<Object> any = graphicData.getAny();
	    	if(uri!=null&&!any.isEmpty()) {
	            if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/presentationml/2006/ole")) {
	            	CTOleObject oleObject = null;
	                Object anyO = any.get(0);
	                if(anyO instanceof AlternateContent) {
	                	final Fallback fallback = ((AlternateContent)anyO).getFallback();
	                	if(fallback!=null&&!fallback.getAny().isEmpty()) {
	                		oleObject = tryGetOle(fallback.getAny().get(0));
	                		if(oleObject!=null) {
	                		    // this is only for #47483 ... otherwise the ole placeholder representation is doubled after removing all choices :-(
	                		    final ComponentContext rootContext = getRootContext();
	                		    if(rootContext instanceof RootComponent) {
	                		        final Part contextPart = ((RootComponent)rootContext).getOperationDocument().getContextPart();
	                		        if(contextPart!=null) {
	                		            final Relationship relationShip = contextPart.getRelationshipsPart().getRelationshipByType(Namespaces.VML);
	                		            if(relationShip!=null) {
        	                		        // as we remove all choices and we have a drawingVML relation, we need to check if a corresponding vml shape has to be deleted also
	                		                // :( below used spids are not part within the ooxml spec and also a drawingPart relation is missing on the slide... -> double failure
	                		                final VMLPart vmlPart = OperationDocument.getVMLPart(contextPart, null);
	                		                if(vmlPart!=null) {
	                		                    final Xml vml = vmlPart.getJaxbElement();
	                		                    if(vml!=null) {
                	                		        final Iterator<Choice> choiceIter = ((AlternateContent)anyO).getChoice().iterator();
                	                		        while(choiceIter.hasNext()) {
                	                		            final List<Object> choiceAny = choiceIter.next().getAny();
                	                		            if(!choiceAny.isEmpty()) {
                	                		                final CTOleObject oleChoice = tryGetOle(choiceAny.get(0));
                	                		                if(oleChoice!=null) {
                	                		                    final String spid = oleChoice.getSpid();
                	                		                    if(spid!=null&&!spid.isEmpty()) {
                                                                    // we have a spid, now we check if we the corresponding DrawingVmlPart
                	                		                        final Iterator<Object> vmlEntryIter = vml.getAny().iterator();
                	                		                        while(vmlEntryIter.hasNext()) {
                	                		                            final Object o = vmlEntryIter.next();
                	                		                            CTShape shape = null;
                	                		                            if(o instanceof CTShape) {
                	                		                                shape = (CTShape)o;
                	                		                            }
                	                		                            else if(o instanceof JAXBElement<?>&&((JAXBElement<?>) o).getDeclaredType().getName().equals("org.docx4j.vml.CTShape")) {
                	                		                                shape =(CTShape)((JAXBElement<?>)o ).getValue();
                	                		                            }
                	                		                            if(shape!=null) {
                	                		                                final String id = shape.getVmlId();
                	                		                                if(id!=null&&id.equals(spid)) {
                	                		                                    vmlEntryIter.remove();
                	                		                                    break;
                	                		                                }
                	                		                            }
                	                		                        }
                	                		                    }
                	                		                }
                	                		            }
                	                		        }
	                		                    }
	                		                }
	                		            }
	                		        }
	                		    }
	                		}
	                	}
	                }
	                else {
	                	oleObject = tryGetOle(anyO);
	                }
	                if(oleObject!=null) {
	                	any.set(0, oleObject);
	                	return oleObject.getPic();
	                }
	            }        		
	    	}
	    }
	    return null;
	}

	private static CTOleObject tryGetOle(Object o) {
		if(o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.pptx4j.pml.CTOleObject") ) {
        	o = ((JAXBElement<?>)o).getValue();
        }
		if(o instanceof CTOleObject) {
        	return (CTOleObject)o;
        }
		return null;
	}

	private CTTable getTable() {
		final Graphic graphic = graphicalObjectFrame.getGraphic();
	    if(graphic!=null&&graphic.getGraphicData()!=null) {
	    	final GraphicData graphicData = graphic.getGraphicData();
	    	final String uri = graphicData.getUri();
	        final List<Object> any = graphicData.getAny();
	    	if(uri!=null&&!any.isEmpty()) {
	            if(uri.equalsIgnoreCase("http://schemas.openxmlformats.org/drawingml/2006/table")) {
	            	Object o = any.get(0);
	                if(o instanceof JAXBElement && ((JAXBElement<?>)o).getDeclaredType().getName().equals("org.docx4j.dml.CTTable") ) {
	                	o = ((JAXBElement<?>)o).getValue();
	                	any.set(0, o);
	                }
	                if(o instanceof CTTable) {
	                	return (CTTable)o;
	                }
	            }
	    	}
	    }
	    return null;
	}

	public void insertRows(com.openexchange.office.ooxml.OperationDocument operationDocument, int destinationRowComponent, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs)
		throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException {

    	final TableRowComponent trReferenceRow = referenceRow!=-1 ? (TableRowComponent)getChildComponent(referenceRow) : null;
        final IndexedNodeList<Object> tblContent = tbl.getContent();
        final Component oldTr = getChildComponent(destinationRowComponent);
        IndexedNode<Object> referenceNode = oldTr!=null?oldTr.getNode():null;

        for(int i = 0; i<count; i++) {
            final CTTableRow tr = new CTTableRow();
            tr.setParent(tbl);
            final IndexedNode<Object> rowNode = new IndexedNode<Object>(tr);
            tblContent.addNode(referenceNode, rowNode, true);
            referenceNode = rowNode;
            if(insertDefaultCells) {
            	final CTTableGrid tableGrid = tbl.getTblGrid(false);
            	if(tableGrid!=null) {
            		for(CTTableCol tableCol:tableGrid.getGridCol()) {
            			final CTTableCell tc = new CTTableCell();
            			tc.setParent(tr);
            			tr.getContent().add(tc);
            			final CTTextParagraph tp = new CTTextParagraph();
            			tp.setParent(tc);
            			tc.getContent().addNode(new IndexedNode<Object>(tp));
            		}
            	}
            }
            else if(trReferenceRow!=null) {       // we try to create the new row from the referenceRow
                final CTTableRow trReference = (CTTableRow)trReferenceRow.getObject();
                tr.setH(trReference.getH());
                final Iterator<Object> rowIter = trReference.getContent().iterator();
                while(rowIter.hasNext()) {
                    final CTTableCell tcReference = (CTTableCell)rowIter.next();
                    final CTTableCell newTc = new CTTableCell();
                    newTc.setParent(tr);
                    tr.getContent().add(newTc);
                    newTc.setGridSpan(tcReference.getGridSpan());
                    tcReference.setHMerge(tcReference.isHMerge());
                    final CTTableCellProperties tcPrReference = tcReference.getTcPr(false);
                    if(tcPrReference!=null) {
                        final CTTableCellProperties newTcPr = XmlUtils.deepCopy(tcPrReference);
                        newTc.setTcPr(newTcPr);
                    }
        			final CTTextParagraph tp = new CTTextParagraph();
        			tp.setParent(newTc);
        			newTc.getContent().addNode(new IndexedNode<Object>(tp));
                }
            }
        }
        if(attrs!=null) {
        	Component c = getNextChildComponent(null, null).getComponent(destinationRowComponent);
	        for(int i=0; i<count; i++) {
	        	c.applyAttrsFromJSON(operationDocument, attrs);
	        	c = c.getNextComponent();
	        }
        }
        recalcTableHeight();
	}

	public void recalcTableHeight() {
		long newTableHeight = 0;
		final Iterator<Object> rowIter = tbl.getContent().iterator();
		while(rowIter.hasNext()) {
			newTableHeight += ((CTTableRow)rowIter.next()).getH();
		}
		graphicalObjectFrame.getXfrm(true).getExt(true).setCy(newTableHeight);
	}

	public void insertColumn(com.openexchange.office.ooxml.OperationDocument operationDocument, JSONArray tableGrid, int gridPosition, String insertMode)
		throws JSONException {

		boolean before = insertMode.equals("before");
        TableRowComponent trComponent = (TableRowComponent)getNextChildComponent(null, null);
        while(trComponent!=null) {
            TableCellComponent tcReference = (TableCellComponent)trComponent.getNextChildComponent(null, null);
            TableCellComponent destination = null;
            while(tcReference!=null) {
                if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                    destination = tcReference;
                    break;
                }
                tcReference = (TableCellComponent)tcReference.getNextComponent();
            }
            final CTTableCell tc = new CTTableCell();
            if(tcReference!=null) {
            	CTTableCellProperties referenceTcPr = ((CTTableCell)tcReference.getObject()).getTcPr(false);
                tc.setParent(trComponent.getObject());
                if(referenceTcPr!=null) {
                    tc.setTcPr(XmlUtils.deepCopy(referenceTcPr));
                }
            }
			final CTTextParagraph tp = new CTTextParagraph();
			tp.setParent(tc);
			tc.getContent().addNode(new IndexedNode<Object>(tp));
            final IndexedNodeList<Object> rowContent = ((CTTableRow)trComponent.getObject()).getContent();
            if(destination==null) {
            	rowContent.add(tc);
            }
            else {
            	final ComponentContext contextChild = destination.getContextChild(null);
            	rowContent.addNode(contextChild.getNode(), new IndexedNode<Object>(tc), before);
            }
            trComponent = (TableRowComponent)trComponent.getNextComponent();
        }
        DMLHelper.applyTableGridFromJson(tbl, graphicalObjectFrame.getXfrm(true).getExt(true).getCx(), tableGrid);
	}

    public void deleteColumns(com.openexchange.office.ooxml.OperationDocument operationDocument, int gridStart, int gridEnd)
       	throws JSONException {

    	Component trComponent = getNextChildComponent(null, null);
        while(trComponent!=null) {
        	final IndexedNodeList<Object> rowContent = ((CTTableRow)trComponent.getObject()).getContent();
        	for(int i=gridEnd; i>=gridStart; i--) {
        		final CTTableCell cell = (CTTableCell)rowContent.get(i);
        		if(cell.isHMerge()) {
        			for(int j = i-1; j>=0; j--) {
        				final CTTableCell gridSpanCell = (CTTableCell)rowContent.get(j);
        				if(!gridSpanCell.isHMerge()) {
        					if(gridSpanCell.getGridSpan()>1) {
        						gridSpanCell.setGridSpan(gridSpanCell.getGridSpan()-1);
        					}
        					break;
        				}
        			}
        		}
        		if(cell.getGridSpan()>1) {
        			rowContent.remove(i+1);
        			cell.setGridSpan(cell.getGridSpan()-1);
        		}
        		else {
        			rowContent.remove(i);
        		}
        	}
            trComponent = trComponent.getNextComponent();
        }
        final CTTableGrid tableGrid = tbl.getTblGrid(false);
        if(tableGrid!=null) {
        	final List<CTTableCol> tableCol = tableGrid.getGridCol();
        	int count = gridEnd - gridStart;
        	while(count>=0) {
        		tableCol.remove(count--);
        	}
        }
	}
}
