
package com.openexchange.office.ods.dom;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.apache.commons.lang.StringUtils;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfSchemaDocument;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.SaxContextHandler;

public class EntryHolderCondition extends Condition {

    private List<ConditionEntry> entries = new ArrayList<ConditionEntry>();

    private String iconType;

    private String maxLength;
    private String negativeColor;
    private String positiveColor;
    private String axisColor;

    private String entryType;

    private boolean show = true;
    private boolean gradient = true;

    public EntryHolderCondition(String id, Ranges ranges, String entryType) {
        super(id, ranges);

        this.entryType = entryType;
    }

    public EntryHolderCondition(Attributes attributes, Ranges ranges, String entryType, String tmpID) {
        super(attributes, ranges, tmpID);

        show = getBoolean(attributes, "calcext:show-value", true);

        switch (entryType) {
            case "icon-set":
                this.iconType = attributes.getValue("calcext:icon-set-type");
                break;
            case "color-scale":
                //nothing
                break;
            case "data-bar":
                this.maxLength = attributes.getValue("calcext:max-length");
                this.negativeColor = attributes.getValue("calcext:negative-color");
                this.positiveColor = attributes.getValue("calcext:positive-color");
                this.axisColor = attributes.getValue("calcext:axis-color");

                gradient = getBoolean(attributes, "calcext:gradient", true);
                break;
        }

        this.entryType = entryType;

    }

    void addEntry(ConditionEntry entry) {
        this.entries.add(entry);
    }

    @Override
    public void writeObject(SerializationHandler output) throws SAXException {

        switch (entryType) {
            case "icon-set":
                startElement(output, "icon-set", "calcext:icon-set");
                addAttribute(output, "icon-set-type", "calcext:icon-set-type", iconType);
                addBooleanAttribute(output, "show-value", "calcext:show-value", show, true);

                for (ConditionEntry e : entries) {
                    addEntryAttribute(output, e, "formatting-entry", "calcext:formatting-entry");
                }

                endElement(output, "icon-set", "calcext:icon-set");
                break;
            case "color-scale":
                startElement(output, "color-scale", "calcext:color-scale");

                for (ConditionEntry e : entries) {
                    addEntryAttribute(output, e, "color-scale-entry", "calcext:color-scale-entry");
                }

                endElement(output, "color-scale", "calcext:color-scale");
                break;
            case "data-bar":
                startElement(output, "data-bar", "calcext:data-bar");
                addAttribute(output, "max-length", "calcext:max-length", maxLength);
                addAttribute(output, "negative-color", "calcext:negative-color", negativeColor);
                addAttribute(output, "positive-color", "calcext:positive-color", positiveColor);
                addAttribute(output, "axis-color", "calcext:axis-color", axisColor);
                addBooleanAttribute(output, "gradient", "calcext:gradient", gradient, true);
                addBooleanAttribute(output, "show-value", "calcext:show-value", show, true);

                for (ConditionEntry e : entries) {
                    addEntryAttribute(output, e, "formatting-entry", "calcext:formatting-entry");
                }

                endElement(output, "data-bar", "calcext:data-bar");

                break;
        }

    }

    @Override
    JSONObject createCondFormatRuleOperation(OdfSpreadsheetDocument doc, int sheetIndex) throws JSONException, SAXException {
        JSONObject result = super.createCondFormatRuleOperation(doc, sheetIndex);

        switch (entryType) {
            case "icon-set":

                JSONObject iconSet = new JSONObject();
                iconSet.put("is", iconType);
                iconSet.put("s", show);

                JSONArray rules = new JSONArray();
                for (ConditionEntry e : entries) {
                    iconSet.put("p", e.type);

                    JSONObject setRule = new JSONObject();
                    setRule.put("t", e.type);
                    setRule.put("v", e.value);
                    setRule.put("gte", true);

                    rules.put(setRule);
                }
                iconSet.put("ir", rules);

                result.put("iconSet", iconSet);
                result.put("type", "iconSet");
                break;
            case "color-scale":
                JSONArray colorScale = new JSONArray();

                for (ConditionEntry e : entries) {
                    JSONObject setRule = new JSONObject();
                    setRule.put("t", e.type);
                    setRule.put("v", e.value);
                    setRule.put("c", getRealColor(e.color));

                    colorScale.put(setRule);
                }
                result.put("colorScale", colorScale);
                result.put("type", "colorScale");
                break;
            case "data-bar":

                JSONObject dataBar = new JSONObject();
                dataBar.put("c", getRealColor(positiveColor));
                dataBar.put("max", maxLength);
                dataBar.put("b", false);
                dataBar.put("nbs", true);
                dataBar.put("g", gradient);
                dataBar.put("s", show);

                if (StringUtils.isNotEmpty(negativeColor)) {
                    dataBar.put("ncs", false);
                    dataBar.put("nc", getRealColor(negativeColor));
                } else {
                    dataBar.put("ncs", true);
                }

                if (StringUtils.isNotEmpty(axisColor)) {
                    dataBar.put("ap", "automatic");
                    dataBar.put("ac", getRealColor(axisColor));
                } else {
                    dataBar.put("ap", "none");
                }

                ConditionEntry e = entries.get(0);
                JSONObject setRule = new JSONObject();
                setRule.put("t", e.type);
                setRule.put("v", e.value);
                dataBar.put("r1", setRule);

                e = entries.get(1);
                setRule = new JSONObject();
                setRule.put("t", e.type);
                setRule.put("v", e.value);
                dataBar.put("r2", setRule);

                result.put("dataBar", dataBar);
                result.put("type", "dataBar");
                break;
        }
        return result;
    }

    @Override
    void applyCondFormatRuleOperation(OdfSchemaDocument doc, String type, Object value1, String value2, Integer priority, boolean stop, JSONObject dataBar, JSONObject iconSet, JSONArray colorScale, JSONObject attrs)
        throws JSONException, SAXException {
        super.applyCondFormatRuleOperation(doc, type, value1, value2, priority, stop, dataBar, iconSet, colorScale, attrs);

        switch (entryType) {
            case "icon-set":
                if (null == iconSet) {
                    return;
                }
                iconType = iconSet.optString("is", iconType);

                entries.clear();
                for (Object e : iconSet.getJSONArray("ir")) {
                    entries.add(new ConditionEntry((JSONObject) e));
                }
                show = iconSet.optBoolean("s", true);
                break;
            case "color-scale":
                if (null == colorScale) {
                    return;
                }
                entries.clear();
                for (Object e : colorScale) {
                    entries.add(new ConditionEntry((JSONObject) e));
                }
                break;
            case "data-bar":
                if (null == dataBar) {
                    return;
                }
                maxLength = dataBar.optString("max", maxLength);
                positiveColor = fromRealColor(dataBar.getJSONObject("c"));

                if (dataBar.getBoolean("ncs")) {
                    negativeColor = "";
                } else {
                    negativeColor = fromRealColor(dataBar.getJSONObject("nc"));
                }

                if (dataBar.getString("ap").equals("none")) {
                    axisColor = "";
                } else {
                    axisColor = fromRealColor(dataBar.getJSONObject("ac"));
                }

                entries.clear();
                entries.add(new ConditionEntry(dataBar.getJSONObject("r1")));
                entries.add(new ConditionEntry(dataBar.getJSONObject("r2")));

                gradient = dataBar.optBoolean("g", true);
                show = dataBar.optBoolean("s", true);
                break;
        }
    }

    private static String limitTypesXmlToJson(String xml) {
        String json = xml.replace("minimum", "min");
        json = json.replace("maximum", "max");
        json = json.replace("-", "");
        return json.replace("number", "formula");
    }

    private static String limitTypesJsonToXml(String json) {
        String xml = json.replace("auto", "auto-");
        xml = xml.replace("min", "minimum");
        return xml = xml.replace("max", "maximum");
//        return xml.replace("formula", "number");
    }

    private static JSONObject getRealColor(String color) throws JSONException {
        JSONObject res = new JSONObject();
        res.put("value", color.replace("#", ""));
        res.put("type", "rgb");
        return res;
    }

    private static String fromRealColor(JSONObject color) throws JSONException {
        if (color.has("fallbackValue")) {
            return color.getString("fallbackValue");
        }
        return "#" + color.getString("value");
    }

    private static boolean getBoolean(Attributes attributes, String qName, boolean def) {
        String bool = attributes.getValue(qName);
        if (StringUtils.isNotEmpty(bool)) {
            try {
                return Boolean.parseBoolean(bool);
            } catch (NumberFormatException e) {
                Logger.getAnonymousLogger().info("error while parsing string to boolean");
            }
        }
        return def;
    }

    public static void addBooleanAttribute(SerializationHandler output, String localName, String qName, boolean value, boolean def)
        throws SAXException {

        if (value != def) {
            addAttribute(output, localName, qName, Boolean.toString(value));
        }
    }

    public static String getValueByTypeJsonToXml(String value, String type) {
        String val = "";
        if ("formula".equals(type) && value != null && !value.startsWith("=")) {
            val = "=";
        }
        return val += value;
    }

    public static void addEntryAttribute(SerializationHandler output, ConditionEntry e, String localName, String qName)
        throws SAXException {

        startElement(output, localName, qName);
        String type = limitTypesJsonToXml(e.type);
        addAttribute(output, "type", "calcext:type", type);
        addAttribute(output, "value", "calcext:value", getValueByTypeJsonToXml(e.value, type));
        if (StringUtils.isNotEmpty(e.color)) {
            addAttribute(output, "color", "calcext:color", e.color);
        }

        endElement(output, localName, qName);
    }

    public static void addAttribute(SerializationHandler output, String localName, String qName, String value)
        throws SAXException {

        SaxContextHandler.addAttribute(output, Namespaces.CALCEXT, localName, qName, value);
    }

    public static void startElement(SerializationHandler output, String localName, String qName)
        throws SAXException {

        output.startElement(Namespaces.CALCEXT, localName, qName);
    }

    public static void endElement(SerializationHandler output, String localName, String qName)
        throws SAXException {

        output.endElement(Namespaces.CALCEXT, localName, qName);
    }

    static class ConditionEntry {

        private final String value;
        private final String type;
        private final String color;

        public ConditionEntry(Attributes attributes) {
            type = limitTypesXmlToJson(attributes.getValue("calcext:type"));
            String val = attributes.getValue("calcext:value");
            if ("formula".equals(type) && val.charAt(0) == '=') {
                val = val.substring(1);
            }
            value = val;
            color = attributes.getValue("calcext:color");
        }

        public ConditionEntry(JSONObject attributes) throws JSONException {
            value = attributes.optString("v");
            type = attributes.optString("t");
            if (attributes.has("c")) {
                color = fromRealColor(attributes.getJSONObject("c"));
            } else {
                color = null;
            }
        }

        @Override
        public String toString() {
            return "ConditionEntry [value=" + value + ", type=" + type + ", color=" + color + "]";
        }

    }

}
