/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ods.dom;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONException;
import org.json.JSONObject;
import org.odftoolkit.odfdom.component.OdfOperationDocument;
import org.odftoolkit.odfdom.pkg.OdfFileDom;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.Length;
import com.openexchange.office.odf.Length.Unit;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.odf.styles.StyleBase;
import com.openexchange.office.odf.styles.StyleManager;

public class Drawing {

	private DrawingAnchor drawingAnchor;
	private ElementNSImpl element;

	private String type;

	public Drawing(DrawingAnchor anchor) {
		drawingAnchor = anchor;
		element = null;
		type = "undefined";
	}

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}

	public DrawingAnchor getAnchor() {
		return drawingAnchor;
	}

	public ElementNSImpl getElement() {
		return element;
	}

	public void setElement(ElementNSImpl element) {
		this.element = element;
	}

	// changes the DrawingAnchor, if the Drawing is already added to Drawings, then
	// this DrawingAnchor should be changed via Drawings.setDrawingAnchor() only
	public void setDrawingAnchor(DrawingAnchor anchor) {
		drawingAnchor = anchor;
	}

	public void writeDrawing(SerializationHandler output)
		throws SAXException {

		if(element!=null) {
			SaxContextHandler.serializeElement(output, element);
		}
		else {
			// write dummy drawing, this is necessary to ensure the same number
			// of drawings after save/reload
			SaxContextHandler.startElement(output, Namespaces.DRAW, "frame", "draw:frame");
			// TODO: ... create a valid dummy frame
			SaxContextHandler.endElement(output, Namespaces.DRAW, "frame", "draw:frame");
		}
	}

	public JSONObject createAttributes(Sheet sheet, JSONObject attrs)
		throws JSONException, SAXException {

		if(attrs==null) {
			attrs = new JSONObject();
		}
		final JSONObject initialDrawingAttrs = attrs.optJSONObject("drawing");
		final JSONObject drawingAttrs = initialDrawingAttrs!=null?initialDrawingAttrs:new JSONObject();

		if(element!=null) {

			boolean protectedSize = false;
			boolean protectedPosition = false;

			final StyleManager styleManager = ((OdfFileDom)sheet.getOwnerDocument()).getStyleManager();
			final String graphicStyleName = element.getAttributeNS(Namespaces.STYLE, "style-name");
			if(graphicStyleName!="") {
				final StyleBase styleBase = styleManager.getStyle(graphicStyleName, "graphic", true);
				if(styleBase!=null) {
					final String styleProtect = styleBase.getAttribute("style:protect");
					if(styleProtect!="") {
						if(styleProtect.contains("size")) {
							protectedSize = true;
						}
						if(styleProtect.contains("position")) {
							protectedPosition = true;
						}
					}
				}
			}

			Double posX = null;
			Double posY = null;
			Double width = null;
			Double height = null;
			if(element.getLocalName().equals("line")) {
				// nice ... special code for lines...
				final String x1 = element.getAttributeNS(Namespaces.SVG, "x1");
				final String y1 = element.getAttributeNS(Namespaces.SVG, "y1");
				final String x2 = element.getAttributeNS(Namespaces.SVG, "x2");
				final String y2 = element.getAttributeNS(Namespaces.SVG, "y2");
				if(!x1.isEmpty()&&!y1.isEmpty()&&!x2.isEmpty()&&!y2.isEmpty()) {
					final Double posX1 = Length.getLength(x1, Unit.MILLIMETER);
					final Double posY1 = Length.getLength(y1, Unit.MILLIMETER);
					final Double posX2 = Length.getLength(x2, Unit.MILLIMETER);
					final Double posY2 = Length.getLength(y2, Unit.MILLIMETER);
					if(posX1!=null&&posY1!=null&&posX2!=null&&posY2!=null) {
						posX = Math.min(posX1, posX2);
						posY = Math.min(posY1, posY2);
						width = Math.abs(posX2-posX1);
						height = Math.abs(posY2-posY1);
					}
				}
			}
			else {
				final String x = element.getAttributeNS(Namespaces.SVG, "x");
				final String y = element.getAttributeNS(Namespaces.SVG, "y");
				if(!x.isEmpty()) {
					posX = Length.getLength(x, Unit.MILLIMETER);
				}
				if(!y.isEmpty()) {
					posY = Length.getLength(y, Unit.MILLIMETER);
				}
				final String w = element.getAttributeNS(Namespaces.SVG, "width");
				final String h = element.getAttributeNS(Namespaces.SVG, "height");
				if(!w.isEmpty()) {
	    			width = Length.getLength(w, Unit.MILLIMETER);
				}
				if(!h.isEmpty()) {
					height = Length.getLength(h, Unit.MILLIMETER);
				}
			}

			if(drawingAnchor==null) {
				drawingAttrs.put("anchorType", "absolute");
    			if(posX!=null) {
    				drawingAttrs.put("left", posX*100);
    			}
    			if(posY!=null) {
    				drawingAttrs.put("top", posY*100);
    			}
			}
			else {
				drawingAttrs.put("anchorType", "oneCell"); 			// protectedSize ? "oneCell" : "twoCell");
				drawingAttrs.put("startCol", drawingAnchor.getColumn());
				drawingAttrs.put("startRow", drawingAnchor.getRow());
				if(posX!=null) {
					drawingAttrs.put("startColOffset", posX*100);
				}
				if(posY!=null) {
					drawingAttrs.put("startRowOffset", posY*100);
				}
			}
			if(width!=null) {
				drawingAttrs.put("width", width*100);
			}
			if(height!=null) {
				drawingAttrs.put("height", height*100);
			}
		}
		if(!drawingAttrs.isEmpty()&&initialDrawingAttrs!=drawingAttrs) {
			attrs.put("drawing", drawingAttrs);
		}
		return attrs;
	}

	// anchorType and startCol/startRow is already set
	public void applyAttributes(OdfOperationDocument opsDoc, JSONObject attrs)
		throws JSONException {

		if(element==null) {
			return;
		}

		final JSONObject drawingAttrs = attrs.optJSONObject("drawing");
		if(drawingAttrs!=null) {
			final Object width = drawingAttrs.opt("width");;
			final Object height = drawingAttrs.opt("height");;

			if(element.getLocalName().equals("line")) {
				
			}
			else {
				if(drawingAnchor==null) {
					final Object left = drawingAttrs.opt("left");
					final Object top = drawingAttrs.opt("top");
					if(left instanceof Number) {
						element.setAttributeNS(Namespaces.SVG, "svg:x", Double.toString(((Number)left).doubleValue()/100) + "mm");
					}
					else if(left==JSONObject.NULL) {
						element.removeAttributeNS(Namespaces.SVG, "x");
					}
					if(top instanceof Number) {
						element.setAttributeNS(Namespaces.SVG, "svg:y", Double.toString(((Number)top).doubleValue()/100) + "mm");
					}
					else if(top==JSONObject.NULL) {
						element.removeAttributeNS(Namespaces.SVG, "y");
					}
				}
				else {
					final Object startColOffset = drawingAttrs.opt("startColOffset");
					final Object startRowOffset = drawingAttrs.opt("startRowOffset");
					if(startColOffset instanceof Number) {
						element.setAttributeNS(Namespaces.SVG, "svg:x", Double.toString(((Number)startColOffset).doubleValue()/100) + "mm");
					}
					else if(startColOffset==JSONObject.NULL) {
						element.removeAttributeNS(Namespaces.SVG, "x");
					}
					if(startRowOffset instanceof Number) {
						element.setAttributeNS(Namespaces.SVG, "svg:y", Double.toString(((Number)startRowOffset).doubleValue()/100) + "mm");
					}
					else if(startRowOffset==JSONObject.NULL) {
						element.removeAttributeNS(Namespaces.SVG, "y");
					}
/*
					if(!drawingAttrs.has("anchorType")||drawingAttrs.getString("anchorType").startsWith("twoCell")) {
						if(drawingAttrs.has("endColOffset")) {
							
						}
						if(drawingAttrs.has("endRowOffset")) {
							
						}
					}
*/					
				}
				if(width instanceof Number) {
					element.setAttributeNS(Namespaces.SVG, "svg:width", Double.toString(((Number)width).doubleValue()/100) + "mm");
				}
				else if(width==JSONObject.NULL) {
					element.removeAttributeNS(Namespaces.SVG, "width");
				}
				if(height instanceof Number) {
					element.setAttributeNS(Namespaces.SVG, "svg:height", Double.toString(((Number)height).doubleValue()/100) + "mm");
				}
				else if(height==JSONObject.NULL) {
					element.removeAttributeNS(Namespaces.SVG, "height");
				}
				if(width instanceof Number||height instanceof Number) {
					element.removeAttributeNS(Namespaces.TABLE, "end-cell-address");
					element.removeAttributeNS(Namespaces.TABLE, "end-x");
					element.removeAttributeNS(Namespaces.TABLE, "end-y");
				}
			}
		}
	}
}
