/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.odf.styles;

import java.util.HashMap;
import java.util.Iterator;

import org.apache.xml.serializer.SerializationHandler;
import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;
import org.xml.sax.SAXException;

import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.DLList;
import com.openexchange.office.odf.Length;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.SaxContextHandler;
import com.openexchange.office.odf.listlevel.ListLevelLabelAlignment;
import com.openexchange.office.odf.listlevel.ListLevelProperties;
import com.openexchange.office.odf.listlevel.ListLevelStyleBullet;
import com.openexchange.office.odf.listlevel.ListLevelStyleImage;
import com.openexchange.office.odf.listlevel.ListLevelStyleNumber;
import com.openexchange.office.odf.properties.PropertyHelper;
import com.openexchange.office.odf.properties.StylePropertiesBase;

final public class TextListStyle extends StyleBase {

	private DLList<IElementWriter> content;

	public TextListStyle(String name, boolean automaticStyle, boolean contentStyle) {
		super(null, name, automaticStyle, contentStyle);
	}

	public TextListStyle(String name, AttributesImpl attributesImpl, boolean automaticStyle, boolean contentStyle) {
		super(name, attributesImpl, false, automaticStyle, contentStyle);
	}

	public String getFamily() {
		return "list-style";
	}

	@Override
	public String getQName() {
		return "text:list-style";
	}

	@Override
	public String getLocalName() {
		return "list-style";
	}

	@Override
	public String getNamespace() {
		return Namespaces.TEXT;
	}

	public DLList<IElementWriter> getContent() {
		if(content==null) {
			content = new DLList<IElementWriter>();
		}
		return content;
	}

	@Override
	public void writeObject(SerializationHandler output)
		throws SAXException {

		SaxContextHandler.startElement(output, getNamespace(), getLocalName(), getQName());
		writeAttributes(output);
		final Iterator<IElementWriter> iter = getContent().iterator();
		while(iter.hasNext()) {
			iter.next().writeObject(output);
		}
		SaxContextHandler.endElement(output, getNamespace(), getLocalName(), getQName());
	}

	@Override
	public void mergeAttrs(StyleBase styleBase) {
		
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject listDefinition) {
        for (int i = 0; i < 9; i++) {
        	addListDefinition(listDefinition.optJSONObject("listLevel" + i), i);
        }
	}

	@Override
	public void createAttrs(StyleManager styleManager, OpAttrs attrs) {
		final String consecutiveNumbering = attributes.getValue("text:consecutive-numbering");
		if(consecutiveNumbering!=null&&!consecutiveNumbering.isEmpty()) {
			attrs.put("listUnifiedNumbering", Boolean.parseBoolean(consecutiveNumbering));
		}
		if(content!=null) {
			final Iterator<IElementWriter> listStyleIter = content.iterator();
			while(listStyleIter.hasNext()) {
				final IElementWriter listStyleEntry = listStyleIter.next();
				if(listStyleEntry instanceof StylePropertiesBase) {
					((StylePropertiesBase)listStyleEntry).createAttrs(styleManager, isContentStyle(), attrs);
				}
			}
		}
	}

	@Override
	protected int _hashCode() {
		return content==null||content.isEmpty() ? 0 : content.hashCode();
	}

	@Override
	protected boolean _equals(StyleBase e) {
		final TextListStyle other = (TextListStyle)e;
		if((content!=null&&!content.isEmpty())&&(other.content!=null&&!other.content.isEmpty())) {
			if(!content.equals(other.content)) {
				return false;
			}
		}
		return true;
	}

	@Override
	public TextListStyle clone() {
		return (TextListStyle)_clone();
	}

	public void addListDefinition(JSONObject listLevelDefinition, int listLevel) {

		if (listLevelDefinition != null) {
			StylePropertiesBase listLevelStyle = null;
	
			// numberFormat: One of 'none', 'bullet', 'decimal', 'lowerRoman', 'upperRoman', 'lowerLetter', or 'upperLetter'.
			String numberFormat = listLevelDefinition.optString("numberFormat");
			String levelText = listLevelDefinition.optString("levelText");
			String numPrefix = null;
			String numSuffix = null;
			if (numberFormat.equals("bullet")) {
				// if there is also a suffix appended
				if (levelText.length() > 1) {
					// num-suffix
					numSuffix = levelText.substring(1);
					// bullet-char
					levelText = levelText.substring(0, 1);
					// ToDo: API FIX to split prefix & suffix from bullet char, a single levelText will not be able to round-trip
				}
				String levelPicBulletUri = listLevelDefinition.optString("levelPicBulletUri");
				if (levelText != null) { // if there is no text label, we have to create an image list
					if (levelPicBulletUri != null && !levelPicBulletUri.isEmpty()) {
						listLevelStyle = createListLevelStyleImage(listLevel, levelPicBulletUri);
					} else {
						listLevelStyle = new ListLevelStyleBullet(new AttributesImpl());
						listLevelStyle.getAttributes().setValue(Namespaces.TEXT, "bullet-char", "text:bullet-char", levelText);
						listLevelStyle.getAttributes().setValue(Namespaces.TEXT, "level", "text:level", Integer.toString(listLevel + 1));
					}
				} else {
					listLevelStyle = createListLevelStyleImage(listLevel, levelPicBulletUri);
				}
			} else { // *** NUMBERED LIST ***
				final AttributesImpl attributes = new AttributesImpl();
				listLevelStyle = new ListLevelStyleNumber(attributes);
				attributes.setValue(Namespaces.STYLE, "num-format", "style:num-format", getNumFormat(numberFormat));
				attributes.setIntValue(Namespaces.TEXT, "level", "text:level", listLevel + 1);
				attributes.setValue(Namespaces.TEXT, "display-levels", "text:display-levels", Integer.toString(countOccurrences(levelText, '%')));
	
				// if there is prefix
				if (!levelText.startsWith("%")) {
					// num-prefix
					int prefixEnd = levelText.indexOf('%');
					numPrefix = levelText.substring(0, prefixEnd);
					levelText = levelText.substring(prefixEnd);
				}
				// num-suffix
				int suffixStart = levelText.lastIndexOf('%') + 2;
				if (levelText.length() >= suffixStart) {
					numSuffix = levelText.substring(suffixStart);
				}
				int listStartValue = listLevelDefinition.optInt("listStartValue", -1);
				if (listStartValue != -1) {
					attributes.setValue(Namespaces.TEXT, "start-value", "text:start-value", Integer.toString(listStartValue));
				}
	       	}
	       	if (numPrefix != null && !numPrefix.isEmpty()) {
	       		listLevelStyle.getAttributes().setValue(Namespaces.STYLE, "num-prefix", "style:num-prefix", numPrefix);
	       	}
	       	if (numSuffix != null && !numSuffix.isEmpty()) {
	       		listLevelStyle.getAttributes().setValue(Namespaces.STYLE, "num-suffix", "style:num-suffix", numSuffix);
	       	}
	       	getContent().add(listLevelStyle);
	       	// Add common list style properties
	       	addCommonListStyles(listLevelStyle, listLevelDefinition);
		}
	}

	private static int countOccurrences(String haystack, char needle) {
        int count = 0;
        for (int i = 0; i < haystack.length(); i++) {
            if (haystack.charAt(i) == needle) {
                count++;
            }
        }
        return count;
    }

	private static ListLevelStyleImage createListLevelStyleImage(int listLevel, String levelPicBulletUri) {
		final AttributesImpl attributes = new AttributesImpl();
		final ListLevelStyleImage listLevelStyle = new ListLevelStyleImage(attributes);
		attributes.setIntValue(Namespaces.TEXT, "level", "text:level", listLevel + 1);
        if(levelPicBulletUri != null && !levelPicBulletUri.isEmpty()) {
            attributes.setValue(Namespaces.XLINK, "href", "xlink:href", levelPicBulletUri);
            attributes.setValue(Namespaces.XLINK, "actuate", "xlink:actuate", "onLoad");
            attributes.setValue(Namespaces.XLINK, "show", "xlink:show", "embed");
            attributes.setValue(Namespaces.XLINK, "type", "xlink:type", "simple");
        }
        return listLevelStyle;
    }

	private void addCommonListStyles(StylePropertiesBase listLevelStyle, JSONObject listLevelDefinition) {
		final AttributesImpl attributes = new AttributesImpl();
		final ListLevelProperties styleListLevelProperties = new ListLevelProperties(attributes);
		listLevelStyle.getContent().add(styleListLevelProperties);
		
        addListLabelAlignment(styleListLevelProperties, listLevelDefinition);

        if (listLevelDefinition.has("height")) {
        	attributes.setValue(Namespaces.FO, "height", "fo:height", listLevelDefinition.optInt("height") / 100 + "mm");
        }
        if (listLevelDefinition.has("textAlign")) {
        	attributes.setValue(Namespaces.FO, "text-align", "fo:text-align", listLevelDefinition.optString("textAlign"));
        }
        if (listLevelDefinition.has("width")) {
        	attributes.setValue(Namespaces.FO, "width", "fo:width", listLevelDefinition.optInt("width") / 100 + "mm");
        }
        if (listLevelDefinition.has("font-name")) {
        	attributes.setValue(Namespaces.STYLE, "font-name", "style:font-name", listLevelDefinition.optString("fontName"));
        }
        if (listLevelDefinition.has("vertical-pos")) {
        	attributes.setValue(Namespaces.STYLE, "verical-pos", "style:vertical-pos", listLevelDefinition.optString("verticalPos"));
        }
        if (listLevelDefinition.has("vertical-rel")) {
        	attributes.setValue(Namespaces.STYLE, "vertical-rel", "style:vertical-rel", listLevelDefinition.optString("verticalRel"));
        }
        if (listLevelDefinition.has("y")) {
        	attributes.setValue(Namespaces.SVG, "y", "svg:y", listLevelDefinition.optString("y"));
        }
        if (listLevelDefinition.has("listLevelPositionAndSpaceMode")) {
        	attributes.setValue(Namespaces.TEXT, "list-level-position-and-space-mode", "text:list-level-position-and-space-mode", listLevelDefinition.optString("listLevelPositionAndSpaceMode"));
        }
        if (listLevelDefinition.has("minLabelDistance")) {
        	attributes.setValue(Namespaces.TEXT, "min-label-distance", "text:min-label-distance", listLevelDefinition.optInt("minLabelDistance") / 100 + "mm");
        }
        if (listLevelDefinition.has("minLabelWidth")) {
        	attributes.setValue(Namespaces.TEXT, "min-label-width", "text:min-label-width", listLevelDefinition.optInt("minLabelWidth") / 100 + "mm");
        }
        if (listLevelDefinition.has("spaceBefore")) {
        	attributes.setValue(Namespaces.TEXT, "space-before", "text:space-before", listLevelDefinition.optInt("spaceBefore") / 100 + "mm");
        }
    }

    private static void addListLabelAlignment(ListLevelProperties styleListLevelProperties, JSONObject listLevelDefinition) {
    	if (listLevelDefinition.has("indentLeft") || listLevelDefinition.has("indentFirstLine")) {
    		styleListLevelProperties.getAttributes().setValue(Namespaces.TEXT, "list-level-position-and-space-mode", "text:list-level-position-and-space-mode", "label-alignment");
            String labelFollowedBy = listLevelDefinition.optString("labelFollowedBy", "listtab");
            final AttributesImpl labelAttributes = new AttributesImpl();
            final ListLevelLabelAlignment levelAlignment = new ListLevelLabelAlignment(labelAttributes);
            labelAttributes.setValue(Namespaces.TEXT, "label-followed-by", "text:label-followed-by", labelFollowedBy);            
            styleListLevelProperties.getContent().add(levelAlignment);
            if (listLevelDefinition.has("indentLeft")) {
                int indentLeft = listLevelDefinition.optInt("indentLeft");
                labelAttributes.setValue(Namespaces.FO, "margin-left", "fo:margin-left", indentLeft / 100.0 + "mm");
            }
            if (listLevelDefinition.has("indentFirstLine")) {
                int indentFirstLine = listLevelDefinition.optInt("indentFirstLine");
                labelAttributes.setValue(Namespaces.FO, "text-indent", "fo:text-indent", indentFirstLine / 100.0 + "mm");
            }
            if (listLevelDefinition.has("tabStopPosition")) {
                int tabStopPosition = listLevelDefinition.optInt("tabStopPosition");
                labelAttributes.setValue(Namespaces.TEXT, "list-tab-stop-position", "text:list-tab-stop-position", tabStopPosition / 100.0 + "mm");
                labelAttributes.setValue(Namespaces.TEXT, "label-followed-by", "text:label-followed-by", "listtab");
            }
        }
    }

    private static String getNumFormat(String numberFormat) {
        String numFormat = ""; // "none" is nothing set
        if (numberFormat.equals("decimal")) {
            numFormat = "1";
        } else if (numberFormat.equals("lowerRoman")) {
            numFormat = "i";
        } else if (numberFormat.equals("upperRoman")) {
            numFormat = "I";
        } else if (numberFormat.equals("lowerLetter")) {
            numFormat = "a";
        } else if (numberFormat.equals("upperLetter")) {
            numFormat = "A";
        }
        return numFormat;
    }

    public static void createListLevelDefinition(StylePropertiesBase listLevelStyle, OpAttrs attrs, int listLevel)  {
		final HashMap<String, Object> listLevelDefinition = attrs.getMap("listLevel" + Integer.toString(listLevel-1), true);
        if (listLevelStyle instanceof ListLevelStyleNumber) {
        	ListLevelStyleNumber listLevelNumberStyle = (ListLevelStyleNumber) listLevelStyle;
            listLevelDefinition.put("levelText", getLabel(listLevelNumberStyle, listLevel-1));
            final String startValue = listLevelStyle.getAttribute("text:start-value");
            if (startValue!=null) {
                listLevelDefinition.put("listStartValue", Integer.parseInt(startValue));
            }
            // There is always the number format set
            listLevelDefinition.put("numberFormat", getNumberFormat(listLevelNumberStyle));
        } else if (listLevelStyle instanceof ListLevelStyleBullet) {
        	ListLevelStyleBullet listLevelBulletStyle = (ListLevelStyleBullet) listLevelStyle;
            listLevelDefinition.put("levelText", getLabel(listLevelBulletStyle, listLevel));
            final String buRelSize = listLevelStyle.getAttribute("text:bullet-relative-size");
            if (buRelSize!=null) {
                listLevelDefinition.put("bulletRelativeSize", buRelSize);
            }
            listLevelDefinition.put("numberFormat", "bullet");
        } else if (listLevelStyle instanceof ListLevelStyleImage) {
            listLevelDefinition.put("levelPicBulletUri", listLevelStyle.getAttribute("xlink:href"));
            listLevelDefinition.put("numberFormat", "bullet");
        }
        final String styleName = listLevelStyle.getAttribute("text:style-name");
        if(styleName!=null) {
            listLevelDefinition.put("styleId", styleName);
        }

        final Iterator<IElementWriter> listLevelStyleContentIter = listLevelStyle.getContent().iterator();
        while(listLevelStyleContentIter.hasNext()) {
        	final IElementWriter listLevelContentEntry = listLevelStyleContentIter.next();
        	if(listLevelContentEntry instanceof ListLevelProperties) {
        		final ListLevelProperties styleListLevelProperties = (ListLevelProperties)listLevelContentEntry;
            	final String height = styleListLevelProperties.getAttribute("fo:height");
                if(height!=null) {
                    listLevelDefinition.put("height", AttributesImpl.normalizeLength(height));
                }
                final String textAlign = styleListLevelProperties.getAttribute("fo:text-align");
                if(textAlign!=null) {
                    listLevelDefinition.put("textAlign", PropertyHelper.mapFoTextAlign(textAlign));
                }
                final String width = styleListLevelProperties.getAttribute("fo:width");
                if(width!=null) {
                    listLevelDefinition.put("width", AttributesImpl.normalizeLength(width));
                }
            	final String fontName = styleListLevelProperties.getAttribute("style:font-name");
                if(fontName!=null) {
                    listLevelDefinition.put("fontName", fontName);
                }
                final String verticalPos = styleListLevelProperties.getAttribute("style:vertical-pos");
                if(verticalPos!=null) {
                    listLevelDefinition.put("verticalPos", verticalPos);
                }
                final String verticalRel = styleListLevelProperties.getAttribute("style:vertical-rel");
                if(verticalRel!=null) {
                    listLevelDefinition.put("verticalRel", verticalRel);
                }
                final String svgY = styleListLevelProperties.getAttribute("svg:y");
                if(svgY!=null) {
                    listLevelDefinition.put("y", svgY);
                }
				final String positionAndSpaceMode = styleListLevelProperties.getAttribute("text:list-level-position-and-space-mode");
				if(positionAndSpaceMode!=null) {
                    listLevelDefinition.put("listLevelPositionAndSpaceMode", positionAndSpaceMode);
                }
				final String minLabelDistance = styleListLevelProperties.getAttribute("text:min-label-distance");
				if(minLabelDistance!=null&&!minLabelDistance.isEmpty()) {
                    listLevelDefinition.put("minLabelDistance", AttributesImpl.normalizeLength(minLabelDistance));
                }
				final String minLabelWidth = styleListLevelProperties.getAttribute("text:min-label-width");
                if (minLabelWidth!=null&&!minLabelWidth.isEmpty()) {
                    listLevelDefinition.put("minLabelWidth", AttributesImpl.normalizeLength(minLabelWidth));
                }
                final String spaceBeforeValue = styleListLevelProperties.getAttribute("text:space-before");
                if(spaceBeforeValue!=null&&!spaceBeforeValue.isEmpty()) {
                    listLevelDefinition.put("spaceBefore", AttributesImpl.normalizeLength(spaceBeforeValue));
                }

                // Mapping list XML ODF 1.1 to ODF 1.2: Adding @text:min-label-width & @text:space-before to margin-left
                mapIndent(minLabelWidth, spaceBeforeValue, listLevelDefinition);

                final Iterator<IElementWriter> listLevelStyleContentChildIter = styleListLevelProperties.getContent().iterator();
                while(listLevelStyleContentChildIter.hasNext()) {
                	final IElementWriter levelStyleContentChildEntry = listLevelStyleContentChildIter.next();
                	if(levelStyleContentChildEntry instanceof ListLevelLabelAlignment) {
                		final ListLevelLabelAlignment listLevelLabelAlignment = (ListLevelLabelAlignment)levelStyleContentChildEntry;
                		final String marginLeft = listLevelLabelAlignment.getAttribute("fo:margin-left");
                        if(marginLeft!=null&&!marginLeft.isEmpty()) {
                            listLevelDefinition.put("indentLeft", AttributesImpl.normalizeLength(marginLeft));
                        } else {
                            mapIndent(minLabelWidth, spaceBeforeValue, listLevelDefinition);
                        }
                        final String textIndent = listLevelLabelAlignment.getAttribute("fo:text-indent");
                        if(textIndent!=null&&!textIndent.isEmpty()) {
                            listLevelDefinition.put("indentFirstLine", AttributesImpl.normalizeLength(textIndent));
                        }
                        final String listTabStopPosition = listLevelLabelAlignment.getAttribute("text:list-tab-stop-position");
                        if(listTabStopPosition!=null&&!listTabStopPosition.isEmpty()) {
    	                	listLevelDefinition.put("tabStopPosition", AttributesImpl.normalizeLength(listTabStopPosition));
                        }
                        final String labelFollowedBy = listLevelLabelAlignment.getAttribute("text:label-followed-by");
                        if(labelFollowedBy!=null) {
                            listLevelDefinition.put("labelFollowedBy", labelFollowedBy);
                        }
                	}
            	}
        	}
        }
    }

    private static String getLabel(StylePropertiesBase listLevelStyle, int listLevel) {
        StringBuilder levelText = new StringBuilder();

        // creating label prefix
        String labelPrefix = listLevelStyle.getAttribute("style:num-prefix");
        if (labelPrefix != null && !labelPrefix.isEmpty()) {
            levelText.append(labelPrefix);
        }

        // creating label number
        if (listLevelStyle instanceof ListLevelStyleNumber) {
            String displayLevels = listLevelStyle.getAttribute("text:display-levels");
            if (displayLevels != null && !displayLevels.isEmpty()) {
                int showLevels = Integer.parseInt(displayLevels);
                // Creating the label, in ODF always adding the low levelText first, adding each follow up level for display level
                // Custom string with one of the placeholders from '%1' to '%9') for numbered lists.
                for (int i = showLevels; i > 0; i--) {
                    levelText.append("%").append(listLevel + 2 - i);
                    // Although not commented in the specification a "." is being added to the text level
                    if (i != 1) {
                        levelText.append('.');
                    }
                }
            } else {
                levelText.append("%").append(listLevel + 1);
            }
            // creating label suffix
            String labelSuffix = listLevelStyle.getAttribute("style:num-suffix");
            if (labelSuffix != null && !labelSuffix.isEmpty()) {
                levelText.append(labelSuffix);
            }
        } else {
            String bulletChar = listLevelStyle.getAttribute("text:bullet-char");
            if (bulletChar != null && !bulletChar.isEmpty()) {
                levelText.append(bulletChar);
            }
        }
        return levelText.toString();
    }

    private static void mapIndent(String minLabelWidthValue, String spaceBeforeValue, HashMap<String, Object> listLevelDefinition) {
        int minLabelWidth = 0;
        boolean isValidMinLabelWidth = Length.isValid(minLabelWidthValue);
        if (isValidMinLabelWidth) {
            minLabelWidth = AttributesImpl.normalizeLength(minLabelWidthValue);
        }
        int spaceBefore = 0;
        boolean isValidSpaceBefore = Length.isValid(spaceBeforeValue);
        if (isValidSpaceBefore) {
            spaceBefore = AttributesImpl.normalizeLength(spaceBeforeValue);
        }
        if (isValidMinLabelWidth || isValidSpaceBefore) {
            listLevelDefinition.put("indentLeft", minLabelWidth + spaceBefore);
        }
    }

    private static String getNumberFormat(StylePropertiesBase listLevelStyle) {
        String numberFormat = listLevelStyle.getAttribute("style:num-format");
        String numFormat;
        if (numberFormat == null || numberFormat.isEmpty()) {
            numFormat = "none";
        } else if (numberFormat.equals("1")) {
            numFormat = "decimal";
        } else if (numberFormat.equals("i")) {
            numFormat = "lowerRoman";
        } else if (numberFormat.equals("I")) {
            numFormat = "upperRoman";
        } else if (numberFormat.equals("a")) {
            numFormat = "lowerLetter";
        } else if (numberFormat.equals("A")) {
            numFormat = "upperLetter";
        } else {
            // a value of type string 18.2. (COMPLEX NUMBERING ie. ASIAN oder ERSTENS..)
            numFormat = numberFormat;
        }
        return numFormat;
    }
}
