/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odf.styles;

import java.util.Iterator;

import org.json.JSONObject;
import org.odftoolkit.odfdom.IElementWriter;

import com.openexchange.office.odf.AttributesImpl;
import com.openexchange.office.odf.Namespaces;
import com.openexchange.office.odf.OpAttrs;
import com.openexchange.office.odf.properties.NumberAmPm;
import com.openexchange.office.odf.properties.NumberDay;
import com.openexchange.office.odf.properties.NumberDayOfWeek;
import com.openexchange.office.odf.properties.NumberEra;
import com.openexchange.office.odf.properties.NumberHours;
import com.openexchange.office.odf.properties.NumberMinutes;
import com.openexchange.office.odf.properties.NumberMonth;
import com.openexchange.office.odf.properties.NumberQuarter;
import com.openexchange.office.odf.properties.NumberSeconds;
import com.openexchange.office.odf.properties.NumberText;
import com.openexchange.office.odf.properties.NumberWeekOfYear;
import com.openexchange.office.odf.properties.NumberYear;

final public class NumberDateStyle extends NumberStyleBase {

	public NumberDateStyle(String name, boolean automaticStyle, boolean contentStyle) {
		super(null, name, automaticStyle, contentStyle);
	}

	public NumberDateStyle(String name, AttributesImpl attributesImpl, boolean automaticStyle, boolean contentStyle) {
		super(name, attributesImpl, false, automaticStyle, contentStyle);
	}

	@Override
	public String getQName() {
		return "number:date-style";
	}

	@Override
	public String getLocalName() {
		return "date-style";
	}

	@Override
	public String getNamespace() {
		return Namespaces.NUMBER;
	}

	@Override
	public String getFormat(StyleManager styleManager, boolean contentAutoStyle) {
		String result = "";
		final Iterator<IElementWriter> propertiesIter = getContent().iterator();
		while(propertiesIter.hasNext()) {
			final IElementWriter properties = propertiesIter.next();
			if (properties instanceof NumberDay) {
				final String numberstyle = ((NumberDay)properties).getAttribute("number:style");;
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "DD";;
				}
				else {
					result += "D";
				}
			}
			else if (properties instanceof NumberMonth) {
				final String numberstyle = ((NumberMonth)properties).getAttribute("number:style");
				if(((NumberMonth)properties).getBoolean("number:textual", false)) {
					if ((numberstyle != null) && numberstyle.equals("long")) {
						result += "MMMM";
					} else {
						result += "MMM";
					}
				}
				else {
					if((numberstyle != null) && numberstyle.equals("long")) {
						result += "MM";
					}
					else {
						result += "M";
					}
				}
			}
			else if (properties instanceof NumberYear) {
				final String numberstyle = ((NumberYear)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "YYYY";
				}
				else {
					result += "YY";
				}
			}
			else if (properties instanceof NumberText) {
				final String content = ((NumberText)properties).getTextContent();
				if((content == null) || (content.equals(""))) {
					result += " ";
				}
				else {
					result += content;
				}
			}
			else if (properties instanceof NumberEra) {
				final String numberstyle = ((NumberEra)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "GG";
				}
				else {
					result += "G";
				}
			}
			else if (properties instanceof NumberHours) {
				final String numberstyle = ((NumberHours)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "HH";
				}
				else {
					result += "H";
				}
			}
			else if (properties instanceof NumberMinutes) {
				final String numberstyle = ((NumberMinutes)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "mm";
				}
				else {
					result += "m";
				}
			}
			else if (properties instanceof NumberSeconds) {
				final String numberstyle = ((NumberSeconds)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "SS";
				}
				else {
					result += "S";
				}
				final Integer decimals = ((NumberSeconds)properties).getInteger("number:decimal-places", null);
                if(decimals!=null&&decimals.intValue()>0){
                    result += '.';
                    for( int i = 0; i < decimals.intValue(); i++){
                        result += '0';
                    }
                }
			}
			else if (properties instanceof NumberQuarter) {
				final String numberstyle = ((NumberQuarter)properties).getAttribute("number:style");
				if((numberstyle != null) && numberstyle.equals("long")) {
					result += "QQ";
				}
				else {
					result += "Q";
				}
			}
			else if (properties instanceof NumberDayOfWeek) {
				final String numberstyle = ((NumberDayOfWeek)properties).getAttribute("number:style");
				if ((numberstyle != null) && numberstyle.equals("long")) {
					result += "NNN";
				}
				else {
					result += "NN";
				}
			}
			else if (properties instanceof NumberAmPm) {
				result += "AM/PM";
            }
			else if (properties instanceof NumberWeekOfYear) {
                result += "WW";
			}
		}
		return result;
	}

	@Override
	public void setFormat(String formatCode) {
		final String actionChars = "GgYyQqMWwDdNnEeHhmSs";
		int actionCount = 0;

		char ch;
		String textBuffer = "";
		boolean endQuote = false;

		int i = 0;

		attributes.setValue(Namespaces.NUMBER, "format-source", "number:format-source", "language");

		while (i < formatCode.length()) {
			ch = formatCode.charAt(i);
			if (actionChars.indexOf(ch) >= 0) {
				if(!textBuffer.isEmpty()) {
					getContent().add(new NumberText(textBuffer));
				}
				textBuffer = "";
				actionCount = 0;
				while (i < formatCode.length() && formatCode.charAt(i) == ch) {
					actionCount++;
					i++;
				}
                int decimalCount = 0;
                if(i < formatCode.length() - 1 && formatCode.charAt(i) == '.' && formatCode.charAt(i + 1) == '0'){
                    decimalCount = 1;
                    i += 2;
                    while (i < formatCode.length() && formatCode.charAt(i) == '0') {
                        decimalCount++;
                        i++;
                    }
                }
                //special case: a single 'w' is not an action char
                if(actionCount > 1 || (ch != 'w' && ch!= 'W')){
                    processChar(ch, actionCount, decimalCount);
                } else {
                    textBuffer += ch;
                }
                    
			} else if (ch == '\'') {
				endQuote = false;
				i++;
				while (i < formatCode.length() && (!endQuote)) {
					ch = formatCode.charAt(i);
					if (ch == '\'') // check to see if this is really the end
					{
						if (i + 1 < formatCode.length() && formatCode.charAt(i + 1) == '\'') {
							i++;
							textBuffer += "'";
						} else {
							endQuote = true;
						}
					} else {
						textBuffer += ch;
					}
					i++;
				}
			} else {
                //special handling "AM/PM"
                if(ch=='A' && formatCode.startsWith("AM/PM", i)) {
    				if(!textBuffer.isEmpty()) {
    					getContent().add(new NumberText(textBuffer));
    				}
                    textBuffer = "";
                    getContent().add(new NumberAmPm(new AttributesImpl()));
                    i+=5;
                }
                else {
	                textBuffer += ch;
	                i++;
                }
			}
		}
		if(!textBuffer.isEmpty()) {
			getContent().add(new NumberText(textBuffer));
		}
	}

	/**
	 * Process a formatting character.
	 * These elements are built "by hand" rather than
	 * @param ch the formatting character to process
	 * @param count the number of occurrences of this character
	 */
	private void processChar(char ch, int count, int decimalCount) {
		switch (ch) {
			case 'G': {
				final NumberEra numberEra = new NumberEra(new AttributesImpl());
				numberEra.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 3));
				getContent().add(numberEra);
				break;
			}
			case 'y':
            case 'Y': {
            	final NumberYear numberYear = new NumberYear(new AttributesImpl());
				numberYear.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 3));
				getContent().add(numberYear);
				break;
            }
			case 'Q': {
				final NumberQuarter numberQuarter = new NumberQuarter(new AttributesImpl());
				numberQuarter.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 1));
				getContent().add(numberQuarter);
				break;
			}
			case 'M': {
				final NumberMonth numberMonth = new NumberMonth(new AttributesImpl());
				numberMonth.getAttributes().setBooleanValue(Namespaces.NUMBER, "textual", "number:textual", count > 2);
				numberMonth.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count % 2 == 0));
				getContent().add(numberMonth);
				break;
			}
			case 'w':
            case 'W': {
            	getContent().add(new NumberWeekOfYear(new AttributesImpl()));
				break;
            }
			case 'd':
            case 'D': {
				if(count > 2) {
					final NumberDayOfWeek numberDayOfWeek = new NumberDayOfWeek(new AttributesImpl());
					numberDayOfWeek.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 3));
                    getContent().add(numberDayOfWeek);
				} else {
					final NumberDay numberDayOfWeek = new NumberDay(new AttributesImpl());
					numberDayOfWeek.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 1));
					getContent().add(numberDayOfWeek);
				}
				break;
            }
			case 'N': {
				final NumberDayOfWeek numberDayOfWeek = new NumberDayOfWeek(new AttributesImpl());
				numberDayOfWeek.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 3));
				if(count > 3) {
					getContent().add(new NumberText(", "));
				}
				getContent().add(numberDayOfWeek);
				break;
			}
			case 'H':
			case 'h': {
				final NumberHours numberHours = new NumberHours(new AttributesImpl());
				numberHours.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 1));
				getContent().add(numberHours);
				break;
			}
			case 'm': {
				final NumberMinutes numberMinutes = new NumberMinutes(new AttributesImpl());
				numberMinutes.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 1));
				getContent().add(numberMinutes);
				break;
			}
			case 's':
            case 'S': {
            	final NumberSeconds numberSeconds = new NumberSeconds(new AttributesImpl());
            	numberSeconds.getAttributes().setValue(Namespaces.NUMBER, "style", "number:style", isLongIf(count > 1));
                if(decimalCount > 0){
                	numberSeconds.getAttributes().setIntValue(Namespaces.NUMBER, "decimal-places", "number:decimal-places", decimalCount);
                }
                getContent().add(numberSeconds);
                break;
            }
        }
	}

	/**
	 * Add long or short style to an element.
	 * @param isLong true if this is number:style="long"; false if number:style="short"
	 * @return the string "long" or "short"
	 */
	private String isLongIf(boolean isLong) {
		return ((isLong) ? "long" : "short");
	}

	@Override
	public void mergeAttrs(StyleBase styleBase) {
		
	}

	@Override
	public void applyAttrs(StyleManager styleManager, JSONObject attrs) {
		
	}

	@Override
	public void createAttrs(StyleManager styleManager, OpAttrs attrs) {
		// TODO Auto-generated method stub
		
	}
}
