/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.odf;

import java.util.HashMap;
import java.util.Map;

import org.json.JSONException;
import org.json.JSONObject;

public class Border implements Cloneable {

    // 100thmm
	private Integer width;
    // rgb color value
    private String color;
    // fo:style
    private String style;

    public Border() {
    	width = null;
    	color = null;
    	style = null;
    }

    public boolean isBorder() {
    	return style!=null;
    }

    @Override
	public Border clone() throws CloneNotSupportedException {
		return (Border)super.clone();
	}

    public HashMap<String, String> getColorMap() {
    	if(color==null) {
    		return null;
    	}
    	final HashMap<String, String> colorMap = new HashMap<String, String>(2);
        colorMap.put("type", "rgb");
        colorMap.put("value", color);
    	return colorMap;
    }

    public void setColor(JSONObject color)
    	throws JSONException {

    	if(color==null) {
    	    return;
    	}
    	else if(color==JSONObject.NULL) {
    		this.color = null;
    	}
    	else {
            String type = color.getString("type");
            if (type.equals("auto")) {
                this.color = "000000";
            } else if (type.equals("rgb")) {
                this.color = color.getString("value");
            } else if (color.has("fallbackValue")) {
            	this.color = color.getString("fallbackValue");
            }
            else {
            	this.color = null;
            }
    	}
    }

    // returns border width in 100th/mm or zero
    public Integer getWidth() {
    	return width;
    }

    public void setWidth(Integer width) {
    	this.width = width;
    }

    // returns the style (as specified for the operations) or null if no border
    public String getStyle() {
    	if(style==null) {
    		return null;
    	}
    	switch(style) {
            case "dotted" : return "dotted";
            case "dashed" : return "dashed";
            case "groove" :
            case "ridge" :
            case "inset" :
            case "outset" :
            case "solid" : return "single";
            case "double" : return "double";
            default:
            case "none" :
            case "hidden" : return "none";
    	}
    }

    public void setStyle(String style) {
    	if(!(style instanceof String)) {
    		this.style = "none";
    	}
    	else switch((String)style) {
    		default:
    		case "none": this.style = "none"; break;
    		case "single": this.style = "solid"; break;
    		case "double": this.style = "double"; break;
    		case "dotted": this.style = "dotted"; break;
    		case "dashDot":
    		case "dashDotDot":
    		case "dashed": this.style = "dashed"; break;
    	}
    }

    public void applyFoBorder(String foBorder) {
        if(foBorder==null||foBorder.isEmpty()) {
            return;
        }
        final String[] tokens = foBorder.split("\\s+");
        for(String token:tokens) {
            switch(token) {
                // WIDTH
                case "thin" : {             // A thin border    1px
                    width = Integer.valueOf(35);
                    continue;
                }
                case "medium" : {           // A medium border  3px
                    width = Integer.valueOf(106);
                    continue;
                }
                case "thick" : {            // A thick border   5px
                    width = Integer.valueOf(176);
                    continue;
                }

                // STYLE
                case "none" :               // No border; the computed border width is zero.
                case "hidden" :             // Same as 'none', except in terms of border conflict resolution for table elements.
                case "dotted" :             // A series of dots.
                case "dashed" :             // A series of short line segments.
                case "solid" :              // A single line segment.
                case "double" :             // Two solid lines. The sum of the two lines and the space between them equals the value of 'border-width'.
                case "groove" :             // The border looks as though it were carved into the canvas.
                case "ridge" :              // The opposite of 'groove': the border looks as though it were coming out of the canvas.
                case "inset" :              // The border makes the box look as though it were embedded in the canvas.
                case "outset" : {           // The opposite of 'inset': the border makes the box look as though it were coming out of the canvas.
                    style = token;
                    continue;
                }

                // COLOR
                case "inherit" :
                case "transparent" : {      // The border is transparent (though it may have width).
                    color = null;
                    continue;
                }
                case "aqua" : {
                    color = "00FFFF";
                    continue;
                }
                case "black" : {
                    color = "000000";
                    continue;
                }
                case "blue" : {
                    color = "0000FF";
                    continue;
                }
                case "fuchsia" : {
                    color = "FF00FF";
                    continue;
                }
                case "gray" : {
                    color = "808080";
                    continue;
                }                    
                case "green" : {
                    color = "008000";
                    continue;
                }
                case "lime" : {
                    color = "00FF00";
                    continue;
                }
                case "maroon" : {
                    color = "800000";
                    continue;
                }
                case "navy" : {
                    color = "000080";
                    continue;
                }
                case"olive" : {
                    color = "808000";
                    continue;
                }
                case "purple" : {
                    color = "800080";
                    continue;
                }
                case "red" : {
                    color = "FF0000";
                    continue;
                }
                case "silver" :  {
                    color = "C0C0C0";
                    continue;
                }
                case "teal" : {
                    color = "008080";
                    continue;
                }
                case "white": {
                    color = "FFFFFF";
                    continue;
                }
                case "yellow" : {
                    color = "FFFF00";
                    continue;
                }
            }

            // check for a color
            if(token.startsWith("#")) {
                if(token.length()==7) {
                    color = token.substring(1).toUpperCase();
                }
                continue;
            }

            try {
                width = AttributesImpl.normalizeLength(token);
            }
            catch(NumberFormatException e) {}
        }
    }

    public void applyJsonBorder(JSONObject borderAttrs)
    	throws JSONException {

    	setStyle(borderAttrs.optString("style", null));
        final Object width = borderAttrs.opt("width");
        setWidth(width instanceof Number ? ((Number)width).intValue() : null);
        setColor(borderAttrs.optJSONObject("color"));
    }

    public void applyJsonLine(JSONObject lineAttrs)
    	throws JSONException {

    	if(lineAttrs==null||lineAttrs==JSONObject.NULL) {
    		setStyle("none");
    	}
    	else {
    		final Object type = lineAttrs.opt("type");
    		if(!(type instanceof String)||((String)type).equals("none")) {
    			setStyle("none");
    		}
    		else {
    			final Object style = lineAttrs.opt("style");
    			if(!(style instanceof String)) {
    				setStyle("single");
    			}
    			else switch((String)style) {
    				case "dashDotDot":    			
    				case "dotted": setStyle("dotted"); break;
    				case "dashDot":
    				case "dashed": setStyle("dashed"); break;
    				default:
    				case "solid": setStyle("single"); break;
    			
    			}
    	        final Object width = lineAttrs.opt("width");
    	        if(width!=null) {
    	            setWidth(width instanceof Number ? ((Number)width).intValue() : null);
    	        }
    	        setColor(lineAttrs.optJSONObject("color"));
	        }
    	}
    }
    
    /* returns the border in fo:border style as it is stored in the odf format */
    @Override
    public String toString() {
    	if(style==null) {
    		return "";
    	}
    	if(style.equals("none") || style.equals("hidden")) {
    		return style;
    	}
    	final StringBuffer border = new StringBuffer();
    	if(width!=null) {
    		border.append(Double.valueOf(width.doubleValue() / 100).toString());
    		border.append("mm ");
    	}
    	border.append(style);
    	if(color==null||color.length()!=6) {
    		border.append(" transparent");
    	}
    	else {
    		border.append(" #");
    		border.append(color);
    	}
    	return border.toString();
    }

    public static void createDefaultBorderMapAttrs(HashMap<String, Object> attrs, Integer defaultPadding, AttributesImpl sourceAttrs) {
    	final Map<String, Object> defaultBorderMap = createBorderMap(sourceAttrs.getValue("fo:border"), defaultPadding);
        if(defaultBorderMap!=null) {
        	attrs.put("borderLeft", defaultBorderMap);
        	attrs.put("borderTop", defaultBorderMap);
        	attrs.put("borderRight", defaultBorderMap);
        	attrs.put("borderBottom", defaultBorderMap);
        }
    }

    public static Map<String, Object> createBorderMap(String value, Integer defaultPadding) {
    	if(!(value instanceof String)||((String)value).isEmpty()) {
    		return null;
    	}
    	final Border border = new Border();
    	border.applyFoBorder((String)value);
    	if(!border.isBorder()) {
    		return null;
    	}
    	final Map<String, Object> borderMap = new HashMap<String, Object>();
    	final String style = border.getStyle();
    	borderMap.put("style", style);
    	if(!style.equals("none")) {
    		if(defaultPadding!=null) {
    			borderMap.put("space", defaultPadding);
    		}
    	}
    	final HashMap<String, String> colorMap = border.getColorMap();
    	if(colorMap!=null) {
    		borderMap.put("color", colorMap);
    	}
    	final Integer width = border.getWidth();
    	if(width!=null) {
    		borderMap.put("width", width);
    	}
    	return borderMap;
    }

    public static void applyFoBorderToSingleBorder(AttributesImpl attributes) {
		final AttributeImpl attribute = attributes.remove("fo:border");
		if(attribute!=null) {
			if(attributes.getValue("fo:border-left")==null) {
				attributes.setValue(new AttributeImpl(Namespaces.FO, "fo:border-left", "border-left", attribute.getValue()));
			}
			if(attributes.getValue("fo:border-top")==null) {
				attributes.setValue(new AttributeImpl(Namespaces.FO, "fo:border-top", "border-top", attribute.getValue()));
			}
			if(attributes.getValue("fo:border-right")==null) {
				attributes.setValue(new AttributeImpl(Namespaces.FO, "fo:border-right", "border-right", attribute.getValue()));
			}
			if(attributes.getValue("fo:border-bottom")==null) {
				attributes.setValue(new AttributeImpl(Namespaces.FO, "fo:border-bottom", "border-bottom", attribute.getValue()));
			}
		}
	}
}
