/**
 * **********************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************************
 */
package org.odftoolkit.odfdom.pkg;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xerces.dom.ParentNode;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.text.TextSElement;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

abstract public class OdfElement extends ElementNSImpl {

    private static final long serialVersionUID = -4939293285696678939L;

    /**
     * Creates a new instance of OdfElement
     */
    public OdfElement(OdfFileDom ownerDocument, String namespaceURI,
        String qualifiedName) throws DOMException {
        super(ownerDocument, namespaceURI, qualifiedName);
    }

    /**
     * Creates a new instance of OdfElement
     */
    public OdfElement(OdfFileDom ownerDocument, OdfName aName)
        throws DOMException {
        super(ownerDocument, aName.getUri(), aName.getQName());
    }

    abstract public OdfName getOdfName();

    protected <T extends OdfElement> T getParentAs(Class<T> clazz) {
        Node parent = getParentNode();
        if (parent != null && clazz.isInstance(parent)) {
            return clazz.cast(parent);
        } else {
            return null;
        }
    }

    protected <T extends OdfElement> T getAncestorAs(Class<T> clazz) {
        Node node = getParentNode();
        while (node != null) {
            if (clazz.isInstance(node)) {
                return clazz.cast(node);
            }
            node = node.getParentNode();
        }
        return null;
    }

    /**
     * @returns true if the given potentialParent is an ancestor of this element
     */
    public boolean hasAncestor(Node potentialParent) {
        Node parentNode = this.getParentNode();
        boolean isParent = false;
        do {
            if (parentNode == null) {
                break;
            } else if (parentNode.equals(potentialParent)) {
                isParent = true;
                break;
            }
        } while ((parentNode = parentNode.getParentNode()) != null);
        return isParent;
    }

    @Override
    public String toString() {
        return mapNode(this, new StringBuilder()).toString();
    }

    /**
     * Only Siblings will be traversed by this method as Children
     */
    static private StringBuilder mapNodeTree(Node node, StringBuilder xml) {
        while (node != null) {
            // mapping node and this mapping include always all descendants
            xml = mapNode(node, xml);
            // next sibling will be mapped to XML
            node = node.getNextSibling();
        }
        return xml;
    }

    private static StringBuilder mapNode(Node node, StringBuilder xml) {
        if (node instanceof Element) {
            xml = mapElementNode(node, xml);
        } else if (node instanceof Text) {
            xml = mapTextNode(node, xml);
        }
        return xml;
    }

    private static StringBuilder mapTextNode(Node node, StringBuilder xml) {
        if (node != null) {
            xml = xml.append(node.getTextContent());
        }
        return xml;
    }

    private static StringBuilder mapElementNode(Node node, StringBuilder xml) {
        if (node != null) {
            xml = xml.append("<");
            xml = xml.append(node.getNodeName());
            xml = mapAttributeNode(node, xml);
            xml = xml.append(">");
            xml = mapNodeTree(node.getFirstChild(), xml);
            xml = xml.append("</");
            xml = xml.append(node.getNodeName());
            xml = xml.append(">");
        }
        return xml;
    }

    private static StringBuilder mapAttributeNode(Node node, StringBuilder xml) {
        NamedNodeMap attrs = null;
        int length;
        if ((attrs = node.getAttributes()) != null
            && (length = attrs.getLength()) > 0) {
            for (int i = 0; length > i; i++) {
                xml = xml.append(" ");
                xml = xml.append(attrs.item(i).getNodeName());
                xml = xml.append("=\"");
                xml = xml.append(attrs.item(i).getNodeValue());
                xml = xml.append("\"");
            }
        }
        return xml;
    }

    /**
     * Set the value of an ODF attribute by <code>OdfName</code>.
     *
     * @param name The qualified name of the ODF attribute.
     * @param value The value to be set in <code>String</code> form
     */
    public void setOdfAttributeValue(OdfName name, String value) {
        setAttributeNS(name.getUri(), name.getQName(), value);
    }

    /**
     * Set an ODF attribute to this element
     *
     * @param attribute the attribute to be set
     */
    public void setOdfAttribute(OdfAttribute attribute) {
        setAttributeNodeNS(attribute);
    }

    /**
     * Retrieves a value of an ODF attribute by <code>OdfName</code>.
     *
     * @param name The qualified name of the ODF attribute.
     * @return The value of the attribute as <code>String</code> or
     * <code>null</code> if the attribute does not exist.
     */
    public String getOdfAttributeValue(OdfName name) {
        return getAttributeNS(name.getUri(), name.getLocalName());
    }

    /**
     * Retrieves an ODF attribute by <code>OdfName</code>.
     *
     * @param name The qualified name of the ODF attribute.
     * @return The <code>OdfAttribute</code> or <code>null</code> if the
     * attribute does not exist.
     */
    public OdfAttribute getOdfAttribute(OdfName name) {
        return (OdfAttribute) getAttributeNodeNS(name.getUri(), name.getLocalName());
    }

    /**
     * Retrieves an ODF attribute by <code>NamespaceName</code>, and local name.
     *
     * @param namespace The namespace of the ODF attribute.
     * @param localname The local name of the ODF attribute.
     * @return The <code>OdfAttribute</code> or <code>null</code> if the
     * attribute does not exist.
     */
    public OdfAttribute getOdfAttribute(NamespaceName namespace, String localname) {
        return (OdfAttribute) getAttributeNodeNS(namespace.getUri(),
            localname);
    }

    /**
     * Determines if an ODF attribute exists.
     *
     * @param name The qualified name of the ODF attribute.
     * @return True if the attribute exists.
     */
    public boolean hasOdfAttribute(OdfName name) {
        return hasAttributeNS(name.getUri(), name.getLocalName());
    }

    /**
     * returns the first child node that implements the given class.
     *
     * @param <T> The type of the ODF element to be found.
     * @param clazz is a class that extends OdfElement.
     * @param parentNode is the parent O of the children to be found.
     * @return the first child node of the given parentNode that is a clazz or
     * null if none is found.
     */
    @SuppressWarnings("unchecked")
    static public <T extends OdfElement> T findFirstChildNode(Class<T> clazz,
        Node parentNode) {
        if (parentNode != null && parentNode instanceof ParentNode) {
            Node node = ((ParentNode) parentNode).getFirstChild();
            while ((node != null) && !clazz.isInstance(node)) {
                node = node.getNextSibling();
            }

            if (node != null) {
                return (T) node;
            }
        }

        return null;
    }

    /**
     * returns the first sibling after the given reference node that implements
     * the given class.
     *
     * @param <T> The type of the ODF element to be found.
     * @param clazz is a class that extends OdfElement.
     * @param refNode the reference node of the siblings to be found.
     * @return the first sibling of the given reference node that is a class or
     * null if none is found.
     */
    @SuppressWarnings("unchecked")
    static public <T extends OdfElement> T findNextChildNode(Class<T> clazz,
        Node refNode) {
        if (refNode != null) {
            Node node = refNode.getNextSibling();
            while (node != null && !clazz.isInstance(node)) {
                node = node.getNextSibling();
            }

            if (node != null) {
                return (T) node;
            }
        }

        return null;
    }

    /**
     * returns the first previous sibling before the given reference node that
     * implements the given class.
     *
     * @param clazz is a class that extends OdfElement.
     * @param refNode the reference node which siblings are to be searched.
     * @return the first previous sibling of the given reference node that is a
     * class or null if none is found.
     */
    @SuppressWarnings("unchecked")
    static public <T extends OdfElement> T findPreviousChildNode(
        Class<T> clazz, Node refNode) {
        if (refNode != null) {
            Node node = refNode.getPreviousSibling();
            while (node != null && !clazz.isInstance(node)) {
                node = node.getPreviousSibling();
            }

            if (node != null) {
                return (T) node;
            }
        }

        return null;
    }

    /**
     * Clones this complete element with all descendants.
     *
     * @return the cloned element
     */
    public OdfElement cloneElement() {
        OdfElement cloneElement = ((OdfFileDom) this.ownerDocument).newOdfElement(this.getClass());
        // if it is an unknown ODF element
        if (cloneElement == null) {
            cloneElement = new OdfAlienElement((OdfFileDom) this.getOwnerDocument(), OdfName.newName(OdfNamespace.getNamespace(this.getNamespaceURI()), this.getTagName()));
        }

        if (attributes != null) {
            for (int i = 0; i < attributes.getLength(); i++) {
                Node item = attributes.item(i);
                String qname = null;
                String prefix = item.getPrefix();
                if (prefix == null) {
                    qname = item.getLocalName();
                    cloneElement.setAttribute(qname, item.getNodeValue());
                } else {
                    qname = prefix + ":" + item.getLocalName();
                    cloneElement.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
                }
            }
        }

        Node childNode = getFirstChild();
        while (childNode != null) {
            cloneElement.appendChild(childNode.cloneNode(true));
            childNode = childNode.getNextSibling();
        }
        return cloneElement;
    }

    @Override
    /**
     * Clones this element but without cloning xml:id (and office:value)
     * attributes
     *
     * @param deep if a deep copy should happen. If False, only the given
     * element with attributes and no content is copied
     */
    public Node cloneNode(boolean deep) {
        OdfElement cloneElement = ((OdfFileDom) this.ownerDocument).newOdfElement(this.getClass());
        // if it is an unknown ODF element
        if (cloneElement == null) {
            cloneElement = new OdfAlienElement((OdfFileDom) this.getOwnerDocument(), OdfName.newName(OdfNamespace.getNamespace(this.getNamespaceURI()), this.getTagName()));
        }

        if (attributes != null) {
            for (int i = 0; i < attributes.getLength(); i++) {
                Node item = attributes.item(i);
                String qname = null;
                String prefix = item.getPrefix();
                if (prefix == null) {
                    qname = item.getLocalName();
                    cloneElement.setAttribute(qname, item.getNodeValue());
                } else {
                    qname = prefix + ":" + item.getLocalName();
                    if (!qname.equals("xml:id") && !qname.equals("office:value") && !qname.equals("calcext:value-type") && !qname.equals("office:value-type")) {
                        cloneElement.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
                    }
                }
            }
        }
        if (deep) {
            Node childNode = getFirstChild();
            while (childNode != null) {
                cloneElement.appendChild(childNode.cloneNode(true));
                childNode = childNode.getNextSibling();
            }
        }
        return cloneElement;
    }

    /**
     * @param depth how many levels of children should be considered
     * @return the cloned node (element)
     * @depth level of children to be cloned All attributes except xml:id and
     * office:value attributes will not be cloned.
     */
    public Node cloneNode(int depth) {
        OdfElement cloneElement = ((OdfFileDom) this.ownerDocument).newOdfElement(this.getClass());
        // if it is an unknown ODF element
        if (cloneElement == null) {
            cloneElement = new OdfAlienElement((OdfFileDom) this.getOwnerDocument(), OdfName.newName(OdfNamespace.getNamespace(this.getNamespaceURI()), this.getTagName()));
        }

        if (attributes != null) {
            for (int i = 0; i < attributes.getLength(); i++) {
                Node item = attributes.item(i);
                String qname = null;
                String prefix = item.getPrefix();
                if (prefix == null) {
                    qname = item.getLocalName();
                    cloneElement.setAttribute(qname, item.getNodeValue());
                } else {
                    qname = prefix + ":" + item.getLocalName();
                    // cell value & type handling might as well overwritten in cell class
                    if (!qname.equals("xml:id") && !qname.equals("office:value") && !qname.equals("calcext:value-type") && !qname.equals("office:value-type")) {
                        cloneElement.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
                    }
                }
            }
        }
        if (depth > 0) {
            Node childNode = getFirstChild();
            while (childNode != null) {
                if (childNode instanceof OdfElement) {
                    cloneElement.appendChild(((OdfElement) childNode).cloneNode(depth - 1));
                }
                childNode = childNode.getNextSibling();
            }
        }
        return cloneElement;
    }

    /**
     * Clones the content of the source element including attributes even xml:id
     * to the target element. Helpful when changing a <text:h> to a <text:p> and
     * vice versa, when outline attribute changes.
     *
     * @param source the element to copy the content & attributes from.
     * @param target the element to copy the content & attributes into.
     * @param deep if a deep copy should happen. If false only the source
     * element attributes will be copied, otherwise all descendants.
     * @return the target element with all new nodes
     */
    // ToDo: Test if a parameter by reference isn't working here!
    public static OdfElement cloneNode(OdfElement source, OdfElement target, boolean deep) {
        NamedNodeMap attributes = source.getAttributes();
        if (attributes != null) {
            for (int i = 0; i < attributes.getLength(); i++) {
                Node item = attributes.item(i);
                String qname;
                String prefix = item.getPrefix();
                if (prefix == null) {
                    qname = item.getLocalName();
                    target.setAttribute(qname, item.getNodeValue());
                } else {
                    qname = prefix + ":" + item.getLocalName();
                    target.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
                }
            }
        }
        if (deep) {
            Node childNode = source.getFirstChild();
            Node newNode;
            while (childNode != null) {
                if (childNode instanceof OdfElement) {
                    newNode = ((OdfElement) childNode).cloneNode(true);
                } else {
                    newNode = childNode.cloneNode(true);
                }
                target.appendChild(newNode);

                childNode = childNode.getNextSibling();
            }
        }
        return target;
    }

    @Override
    public Node appendChild(Node node) {
        // No Counting necessary as appendChild() will call insertBefore()
        return super.appendChild(node);
    }

    /**
     * indicates if some other object is equal to this one.
     *
     * @param obj - the reference object with which to compare.
     * @return true if this object is the same as the obj argument; false
     * otherwise.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }

        if ((obj == null) || !(obj instanceof OdfElement)) {
            return false;
        }

        OdfElement compare = (OdfElement) obj;

        // compare node name
        if (!localName.equals(compare.localName)) {
            return false;
        }

        if (!this.namespaceURI.equals(compare.namespaceURI)) {
            return false;
        }

        // compare node attributes
        if (attributes == compare.attributes) {
            return true;
        }

        if ((attributes == null) || (compare.attributes == null)) {
            return false;
        }

        int attr_count1 = attributes.getLength();
        int attr_count2 = compare.attributes.getLength();

        List<Node> attr1 = new ArrayList<Node>();
        for (int i = 0; i < attr_count1; i++) {
            Node node = attributes.item(i);
            if (node.getNodeValue().length() == 0) {
                continue;
            }
            attr1.add(node);
        }

        List<Node> attr2 = new ArrayList<Node>();
        for (int i = 0; i < attr_count2; i++) {
            Node node = compare.attributes.item(i);
            if (node.getNodeValue().length() == 0) {
                continue;
            }
            attr2.add(node);
        }

        if (attr1.size() != attr2.size()) {
            return false;
        }

        for (int i = 0; i < attr1.size(); i++) {
            Node n1 = attr1.get(i);
            if (n1.getLocalName().equals("name")
                && n1.getNamespaceURI().equals(
                    OdfDocumentNamespace.STYLE.getUri())) {
                continue; // do not compare style names
            }
            Node n2 = null;
            int j = 0;
            for (j = 0; j < attr2.size(); j++) {
                n2 = attr2.get(j);
                if (n1.getLocalName().equals(n2.getLocalName())) {
                    String ns1 = n1.getNamespaceURI();
                    String ns2 = n2.getNamespaceURI();
                    if (ns1 != null && ns2 != null && ns1.equals(ns2)) {
                        break;
                    }
                }
            }
            if (j == attr2.size()) {
                return false;
            }

            if (!n1.getTextContent().equals(n2.getTextContent())) {
                return false;
            }
        }

        // now compare child elements
        NodeList childs1 = this.getChildNodes();
        NodeList childs2 = compare.getChildNodes();

        int child_count1 = childs1.getLength();
        int child_count2 = childs2.getLength();
        if ((child_count1 == 0) && (child_count2 == 0)) {
            return true;
        }

        List<Node> nodes1 = new ArrayList<Node>();
        for (int i = 0; i < child_count1; i++) {
            Node node = childs1.item(i);
            if (node.getNodeType() == Node.TEXT_NODE) {
                if (node.getNodeValue().trim().length() == 0) {
                    continue; // skip whitespace text nodes
                }
            }
            nodes1.add(node);
        }

        List<Node> nodes2 = new ArrayList<Node>();
        for (int i = 0; i < child_count2; i++) {
            Node node = childs2.item(i);
            if (node.getNodeType() == Node.TEXT_NODE) {
                if (node.getNodeValue().trim().length() == 0) {
                    continue; // skip whitespace text nodes
                }
            }
            nodes2.add(node);
        }

        if (nodes1.size() != nodes2.size()) {
            return false;
        }

        for (int i = 0; i < nodes1.size(); i++) {
            Node n1 = nodes1.get(i);
            Node n2 = nodes2.get(i);
            if (!n1.equals(n2)) {
                return false;
            }
        }
        return true;
    }

    protected void onRemoveNode(Node node) {
        if (node != null) {
            Node child = node.getFirstChild();
            while (child != null) {
                this.onRemoveNode(child);
                child = child.getNextSibling();
            }

            if (OdfElement.class.isInstance(node)) {
                ((OdfElement) node).onRemoveNode();
            }
        }
    }

    protected void onInsertNode(Node node) {
        Node child = node.getFirstChild();
        while (child != null) {
            this.onInsertNode(child);
            child = child.getNextSibling();
        }

        if (OdfElement.class.isInstance(node)) {
            ((OdfElement) node).onInsertNode();
        }
    }

    protected void onRemoveNode() {
    }

    protected void onInsertNode() {
    }

    @Override
    public Node insertBefore(Node newChild, Node refChild) throws DOMException {
        Node n = null;
        onInsertNode(newChild);
        n = super.insertBefore(newChild, refChild);
        return n;
    }


    /* Removes the element from the DOM tree, but keeping its ancestors by moving its children in its place */
    public static Element removeSingleElement(Element oldElement) throws DOMException {
        Element parent = (Element) oldElement.getParentNode();
        if (parent != null) {
            NodeList children = oldElement.getChildNodes();
            int childCount = children.getLength();
            Node lastChild = children.item(childCount - 1);
            parent.replaceChild(lastChild, oldElement);
            Node newChild;
            for (int i = childCount - 2; i >= 0; i--) {
                newChild = children.item(i);
                parent.insertBefore(newChild, lastChild);
                lastChild = newChild;
            }
        }
        return parent;
    }

    @Override
    public Node removeChild(Node oldChild) throws DOMException {
        onRemoveNode(oldChild);
        return super.removeChild(oldChild);
    }

    /**
     * @return true if the element does represent multiple instances. (only
     * applicable for some elements as cell or row).
     */
    public boolean isRepeatable() {
        return Boolean.FALSE;
    }

    /**
     * @return the repetition the element represents, by default it is 1
     */
    public int getRepetition() {
        return 1;
    }

    /**
     * Removes all the content from the element
     */
    public void removeContent() {
        NodeList children = this.getChildNodes();
        for (int i = children.getLength() - 1; i >= 0; i--) {
            Node child = children.item(i);
            if (child != null) {
                this.removeChild(child);
            }
        }
    }

    @Override
    public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
        onRemoveNode(oldChild);
        onInsertNode(newChild);
        //Currently the replace only reduces
        //reduceComponentSize(oldChild);
        return super.replaceChild(newChild, oldChild);
    }

    /**
     * Accept an visitor instance to allow the visitor to do some operations.
     * Refer to visitor design pattern to get a better understanding.
     *
     * @param visitor	an instance of DefaultElementVisitor
     */
    public void accept(ElementVisitor visitor) {
        visitor.visit(this);
    }

    /**
     * If the string is inserted into a text:p/text:h element and it will be
     * inserted in the start/end all spaces are replaced by <text:s/>
     * element(s). tabulator and linefeeds are being removed.
     *
     * If both the previous text node ends with a space and newString starts
     * with a space, we would need to encode the single leading space as an
     * element, otherwise it would be stripped. Same occurs for the next text
     * node and an ending space. For Example:
     * <span> text </span><text:s c="7"/><span> text2 </span> <== SAVE when
     * starting ending a span as well with space element independent of
     * preceding
     */
    static protected void appendUsingWhitespaceHandling(Node precedingNode, OdfElement parent, Node followingNode, String newString) {
//		addTextNode(precedingNode, parent, followingNode, newString);
        // only addChild text, if parent exists
        if (parent != null) {
// DEFAULT EMPTY PARAGRAPH TEXT STYLE -- https://bugs.open-xchange.com/show_bug.cgi?id=35136
//            // if there is only one span in a paragraph/heading
//            if(parent instanceof TextParagraphElementBase){
//                // add the new content to this span
//                NodeList children = parent.getChildNodes();
//                if(children.getLength() == 1){
//                    Node child = children.item(0);
//                    if(child instanceof TextSpanElement){
//                        parent = (OdfElement) child;
//                    }
//                }
//            }
            int spaceCount = 0;
            // Note: The delta between startPosition and endPosition marks the text to be written out
            // startPosition will only be raised to endposition, when characters have to be skipped!
            int startPos = 0;
            int endPos = 0;
            // check if first character is a white space
            for (int i = 0; i < newString.length(); i++) {
                char c = newString.charAt(i);
                if (c == '\u0020' // space
                    || c == '\t' // \t (tabulator = 0x09)
                    // \r (carriage return = 0x0D)
                    || c == '\r'
                    // \n (line feed =' 0x0A)
                    || c == '\n') {
                    spaceCount++;

                    if (spaceCount > 1) {
                        // if there are more than one space a space element have to be inserted, write out the previous spaces
                        if (endPos - startPos > 0) {
                            precedingNode = addTextNode(precedingNode, parent, followingNode, newString.substring(startPos, endPos));
                            // for the single whitespace not written
                        }
                        // NOT including the additional whitespace character
                        startPos = endPos;
                    }
                } else { // else if there was no whitespace character found or at the beginning
                    if (spaceCount > 1 || i == 1 && spaceCount == 1) {
                        TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
                        // if there were multiple preceing whitespace, write out a space Element
                        if (spaceCount > 1) {
                            s.setTextCAttribute(spaceCount);
                        }

                        // write out space element
                        precedingNode = addElementNode(precedingNode, parent, followingNode, s);
                        endPos += spaceCount;
                        startPos = endPos;
                        spaceCount = 0;

                        // reset space count to zero as now a character was found
                    } else if (spaceCount == 1) {
                        endPos++;
                        spaceCount = 0;
                    }

                    // write out character
                    endPos++;// including this character
                }
            }
            // reset space count to zero as now a character was found
            if (spaceCount > 1) {
                TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
                // if there were multiple preceing whitespace, write out a space Element
                if (spaceCount > 1) {
                    s.setTextCAttribute(spaceCount);
                }
                // write out space element
                precedingNode = addElementNode(precedingNode, parent, followingNode, s);
                endPos += spaceCount;
                startPos = endPos;
                spaceCount = 0;
            }
            if (endPos - startPos > 0) {
                precedingNode = addTextNode(precedingNode, parent, followingNode, newString.substring(startPos, endPos));
            }
            if (spaceCount == 1) {
                TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
                addElementNode(precedingNode, parent, followingNode, s);
            }
        } else {
            Logger.getLogger(OdfElement.class.getName()).log(Level.SEVERE, "Node parent should not be NULL!");
        }
    }

    private static Node addTextNode(Node precedingNode, OdfElement parent, Node followingNode, String newString) {
        Node newNode = null;
        // APPEND: if there is no text node to expand and no element that follows
        if (precedingNode == null && followingNode == null || precedingNode != null && precedingNode instanceof Element && followingNode == null) {
            newNode = parent.appendChild(parent.getOwnerDocument().createTextNode(newString));
        } else {
            // INSERT:
            if (precedingNode != null && precedingNode instanceof Text) {
                // insert at the end of the given text
                ((Text) precedingNode).appendData(newString);
                newNode = precedingNode;
            } else if (followingNode != null) {
                // insert before the given element
                newNode = parent.insertBefore(parent.getOwnerDocument().createTextNode(newString), followingNode);
            }
        }
        return newNode;
    }

    private static Node addElementNode(Node precedingNode, OdfElement parent, Node followingNode, OdfElement newElement) {
        Node newNode = null;
        // APPEND: if there is no text node to expand and no element that follows
        if (followingNode == null) {
            newNode = parent.appendChild(newElement);
        } else {
            // INSERT:
            // insert before the given following-node
            newNode = parent.insertBefore(newElement, followingNode);
        }
        return newNode;
    }

    /**
     * Copy attributes from one element to another, existing attributes will be
     * overwritten
     */
    public static void copyAttributes(OdfElement from, OdfElement to) {
        NamedNodeMap attributes = from.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++) {
            Attr node = (Attr) attributes.item(i);
            to.setAttributeNS(node.getNamespaceURI(), node.getNodeName(), node.getValue());
        }
    }

    /**
     * @return the first child element of a given parent
     */
    public Element getChildElement(String uri, String localName) {
        return getChildElement(uri, localName, 0);
    }

    /**
     * @return the child element of a given parent from a given position (starting with 0)
     */
    public Element getChildElement(String uri, String localName, int position) {
        NodeList childList = this.getElementsByTagNameNS(uri, localName);
        return (Element) childList.item(position);
    }

    /**
     * @returns the next element sibling of the given node or null if none
     * exists
     */
    public static OdfElement getNextSiblingElement(Node node) {
        OdfElement nextElement = null;
        Node _nextSibling = node.getNextSibling();
        if (_nextSibling instanceof OdfElement) {
            nextElement = (OdfElement) _nextSibling;
        } else if (_nextSibling instanceof Text) {
            nextElement = getNextSiblingElement(_nextSibling);
        }
        return nextElement;
    }

    /**
     * @returns the next element sibling of the given node or null if none
     * exists
     */
    public static OdfElement getPreviousSiblingElement(Node node) {
        OdfElement previousElement = null;
        Node _previousElement = node.getPreviousSibling();
        if (_previousElement instanceof OdfElement) {
            previousElement = (OdfElement) _previousElement;
        } else if (_previousElement instanceof Text) {
            previousElement = getPreviousSiblingElement(_previousElement);
        }
        return previousElement;
    }

    /**
     * @returns the first element child of the this or null if none exists
     */
    public OdfElement getFirstChildElement() {
        OdfElement firstElementChild = null;
        NodeList nodeList = this.getChildNodes();
        Node node = nodeList.item(0);
        if (node != null) {
            if (node instanceof OdfElement) {
                firstElementChild = (OdfElement) node;
            } else {
                firstElementChild = getNextSiblingElement(node);
            }
        }
        return firstElementChild;
    }

    /**
     * @returns the last element child of the this or null if none exists
     */
    public OdfElement getLastChildElement() {
        OdfElement lastElementChild = null;
        NodeList nodeList = this.getChildNodes();
        Node node = nodeList.item(0);
        for (int i = nodeList.getLength(); i >= 0; i--) {
            if (node instanceof OdfElement) {
                lastElementChild = (OdfElement) node;
                break;
            }
        }
        return lastElementChild;
    }

    public int countPrecedingSiblingElements() {
        int i = 0;
        Node node = this.getPreviousSibling();
        while (node != null) {
            node = node.getPreviousSibling();
            if (node instanceof Element) {
                i++;
            }
        }
        return i;
    }

    // ToDo: Move this to a intermediate class, e.g. ComponentRootElement
    /**
     * @return the component size of a heading, which is always 1
     */
    public void setRepetition(int repetition) {
        // does not work for all classes
    }
}
