/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package org.odftoolkit.odfdom.component;

import org.json.JSONObject;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelLabelAlignmentElement;
import org.odftoolkit.odfdom.dom.element.style.StyleListLevelPropertiesElement;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleElementBase;
import org.odftoolkit.odfdom.dom.element.text.TextListLevelStyleImageElement;
import org.odftoolkit.odfdom.incubator.doc.text.OdfTextListStyle;

public class ListStyleHelper {

    private static int countOccurrences(String haystack, char needle) {
        int count = 0;
        for (int i = 0; i < haystack.length(); i++) {
            if (haystack.charAt(i) == needle) {
                count++;
            }
        }
        return count;
    }
    
	/**
	*
	* <
	* text:list-level-style-bullet text:level="1"
	* text:style-name="Bullet_20_Symbols" text:bullet-char="\u2022">
	* <style:list-level-properties
	* text:list-level-position-and-space-mode="label-alignment">
	* <style:list-level-label-alignment text:label-followed-by="listtab"
	* text:list-tab-stop-position="1.27cm" fo:text-indent="-0.635cm"
	* fo:margin-left="1.27cm"/>
	* </style:list-level-properties>
	* </text:list-level-style-bullet>
	*
	*/
	public static void addListDefinition(OdfTextListStyle listStyle, JSONObject listLevelDefinition, int listLevel) {

		if (listLevelDefinition != null) {
			TextListLevelStyleElementBase listLevelStyle = null;
	
			// numberFormat: One of 'none', 'bullet', 'decimal', 'lowerRoman', 'upperRoman', 'lowerLetter', or 'upperLetter'.
			String numberFormat = listLevelDefinition.optString("numberFormat");
			String levelText = listLevelDefinition.optString("levelText");
			String numPrefix = null;
			String numSuffix = null;
			if (numberFormat.equals("bullet")) {
				// if there is also a suffix appended
				if (levelText.length() > 1) {
					// num-suffix
					numSuffix = levelText.substring(1);
					// bullet-char
					levelText = levelText.substring(0, 1);
					// ToDo: API FIX to split prefix & suffix from bullet char, a single levelText will not be able to round-trip
				}
				String levelPicBulletUri = listLevelDefinition.optString("levelPicBulletUri");
				if (levelText != null) { // if there is no text label, we have to create an image list
					if (levelPicBulletUri != null && !levelPicBulletUri.isEmpty()) {
						listLevelStyle = createListLevelStyleImage(listStyle, listLevel, levelPicBulletUri);
					} else {
						listLevelStyle = listStyle.newTextListLevelStyleBulletElement(levelText, listLevel + 1);
					}
				} else {
					listLevelStyle = createListLevelStyleImage(listStyle, listLevel, levelPicBulletUri);
				}
			} else { // *** NUMBERED LIST ***
				listLevelStyle = listStyle.newTextListLevelStyleNumberElement(getNumFormat(numberFormat), listLevel + 1);
				listLevelStyle.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:display-levels", Integer.toString(countOccurrences(levelText, '%')));
	
				// if there is prefix
				if (!levelText.startsWith("%")) {
					// num-prefix
					int prefixEnd = levelText.indexOf('%');
					numPrefix = levelText.substring(0, prefixEnd);
					levelText = levelText.substring(prefixEnd);
				}
				// num-suffix
				int suffixStart = levelText.lastIndexOf('%') + 2;
				if (levelText.length() >= suffixStart) {
					numSuffix = levelText.substring(suffixStart);
				}
				int listStartValue = listLevelDefinition.optInt("listStartValue", -1);
				if (listStartValue != -1) {
					listLevelStyle.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:start-value", Integer.toString(listStartValue));
				}
	       	}
	       	if (numPrefix != null && !numPrefix.isEmpty()) {
	    	   listLevelStyle.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:num-prefix", numPrefix);
	       	}
	       	if (numSuffix != null && !numSuffix.isEmpty()) {
	    	   listLevelStyle.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:num-suffix", numSuffix);
	       	}
	       	// Add common list style properties
	       	addCommonListStyles(listLevelStyle, listLevelDefinition);
		}
	}

	private static TextListLevelStyleElementBase createListLevelStyleImage(OdfTextListStyle listStyle, int listLevel, String levelPicBulletUri) {
		TextListLevelStyleElementBase listLevelStyle = listStyle.newTextListLevelStyleImageElement(listLevel + 1);

        if (levelPicBulletUri != null && !levelPicBulletUri.isEmpty()) {
            listLevelStyle.setAttributeNS(OdfDocumentNamespace.XLINK.getUri(), "xlink:href", levelPicBulletUri);
            TextListLevelStyleImageElement listLevelStyleImage = (TextListLevelStyleImageElement) listLevelStyle;
            listLevelStyleImage.setXlinkActuateAttribute("onLoad");
            listLevelStyleImage.setXlinkShowAttribute("embed");
            listLevelStyleImage.setXlinkTypeAttribute("simple");
        }
        return listLevelStyle;
    }

    /**
     * The <style:list-level-properties> element has the following attributes:
     * <ul>
     * <li>fo:height</li>
     * <li>fo:text-align</li>
     * <li>fo:width</li>
     * <li>style:font-name</li>
     * <li>style:vertical-pos</li>
     * <li>style:vertical-rel</li>
     * <li>svg:y</li>
     * <li>text:list-level-position-and-space-mode</li>
     * <li>text:min-label-distance</li>
     * <li>text:min-label-width</li>
     * <li>text:space-before</li>
     * </ul>
     */
	private static void addCommonListStyles(TextListLevelStyleElementBase listLevelStyle, JSONObject listLevelDefinition) {
		StyleListLevelPropertiesElement styleListLevelProperties = listLevelStyle.newStyleListLevelPropertiesElement();
        addListLabelAlignment(styleListLevelProperties, listLevelDefinition);

        if (listLevelDefinition.has("height")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:height", listLevelDefinition.optInt("height") / 100 + "mm");
        }
        if (listLevelDefinition.has("textAlign")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:text-align", listLevelDefinition.optString("textAlign"));
        }
        if (listLevelDefinition.has("width")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.FO.getUri(), "fo:width", listLevelDefinition.optInt("width") / 100 + "mm");
        }
        if (listLevelDefinition.has("font-name")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:font-name", listLevelDefinition.optString("fontName"));
        }
        if (listLevelDefinition.has("vertical-pos")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:vertical-pos", listLevelDefinition.optString("verticalPos"));
        }
        if (listLevelDefinition.has("vertical-rel")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.STYLE.getUri(), "style:vertical-rel", listLevelDefinition.optString("verticalRel"));
        }
        if (listLevelDefinition.has("y")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.SVG.getUri(), "svg:y", listLevelDefinition.optString("y"));
        }
        if (listLevelDefinition.has("listLevelPositionAndSpaceMode")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:list-level-position-and-space-mode", listLevelDefinition.optString("listLevelPositionAndSpaceMode"));
        }
        if (listLevelDefinition.has("minLabelDistance")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:min-label-distance", listLevelDefinition.optInt("minLabelDistance") / 100 + "mm");
        }
        if (listLevelDefinition.has("minLabelWidth")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:min-label-width", listLevelDefinition.optInt("minLabelWidth") / 100 + "mm");
        }
        if (listLevelDefinition.has("spaceBefore")) {
            styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:space-before", listLevelDefinition.optInt("spaceBefore") / 100 + "mm");
        }
    }

    private static void addListLabelAlignment(StyleListLevelPropertiesElement styleListLevelProperties, JSONObject listLevelDefinition) {
    	if (listLevelDefinition.has("indentLeft") || listLevelDefinition.has("indentFirstLine")) {
    		styleListLevelProperties.setAttributeNS(OdfDocumentNamespace.TEXT.getUri(), "text:list-level-position-and-space-mode", "label-alignment");
            String labelFollowedBy = listLevelDefinition.optString("labelFollowedBy", "listtab");
            StyleListLevelLabelAlignmentElement listLevelLabelAlignmentElement = styleListLevelProperties.newStyleListLevelLabelAlignmentElement(labelFollowedBy);

            if (listLevelDefinition.has("indentLeft")) {
                int indentLeft = listLevelDefinition.optInt("indentLeft");
                listLevelLabelAlignmentElement.setFoMarginLeftAttribute(indentLeft / 100.0 + "mm");
            }
            if (listLevelDefinition.has("indentFirstLine")) {
                int indentFirstLine = listLevelDefinition.optInt("indentFirstLine");
                listLevelLabelAlignmentElement.setFoTextIndentAttribute(indentFirstLine / 100.0 + "mm");
            }
            if (listLevelDefinition.has("tabStopPosition")) {
                int tabStopPosition = listLevelDefinition.optInt("tabStopPosition");
                listLevelLabelAlignmentElement.setTextListTabStopPositionAttribute(tabStopPosition / 100.0 + "mm");
                listLevelLabelAlignmentElement.setTextLabelFollowedByAttribute("listtab");
            }
        }
    }

    /**
     * The style:num-format attribute specifies a numbering sequence. The
     * defined values for the style:num-format attribute are: 1: Hindu-Arabic
     * number sequence starts with 1. a: number sequence of lowercase Modern
     * Latin basic alphabet characters starts with "a". A: number sequence of
     * uppercase Modern Latin basic alphabet characters starts with "A". i:
     * number sequence of lowercase Roman numerals starts with "i". I: number
     * sequence of uppercase Roman numerals start with "I". a value of type
     * string. an empty string: no number sequence displayed. If no value is
     * given, no number sequence is displayed.
     */
    private static String getNumFormat(String numberFormat) {
        String numFormat = ""; // "none" is nothing set
        if (numberFormat.equals("decimal")) {
            numFormat = "1";
        } else if (numberFormat.equals("lowerRoman")) {
            numFormat = "i";
        } else if (numberFormat.equals("upperRoman")) {
            numFormat = "I";
        } else if (numberFormat.equals("lowerLetter")) {
            numFormat = "a";
        } else if (numberFormat.equals("upperLetter")) {
            numFormat = "A";
        }
        return numFormat;
    }
}
