/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.backup.distributed;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.lang.Validate;

import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.log.Log;
import com.openexchange.office.hazelcast.doc.DocumentResourcesDirectory;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentResources;
import com.openexchange.office.tools.directory.IStreamIDCollection;
import com.openexchange.office.tools.resource.Resource;
import com.openexchange.server.ServiceLookup;

/**
 * Manages the document resources a document references images and
 * controls the life-cycle of them in the distributed managed file management.
 *
 * @author Carsten Driesner
 * @since 7.8.1
 *
 */
public class DistributedDocumentResourceManager implements IDistributedManageFileCollection {

    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(DistributedDocumentStreamManager.class);
    private final ManagedFileManagement fileManager;
    private final DocumentResourcesDirectory globalDocumentResourcesMap;
    private final HashMap<DocRestoreID, DocumentResources> docResourcesMap = new HashMap<DocRestoreID, DocumentResources>();
    private final String uniqueSingletonID;
    private final String uniqueID;
    private final DistributedManagedFileLocker locker;

    /**
     * Initializes a DistributedDocumentStreamManager
     *
     * @param services the service lookup to be used for this instance
     */
    public DistributedDocumentResourceManager(final ServiceLookup services, final DocumentResourcesDirectory globalDocumentResourcesMap, final String uniqueSingletonID) {
        Validate.notNull(services);
        Validate.notNull(globalDocumentResourcesMap);

        fileManager = services.getService(ManagedFileManagement.class);
        locker = DistributedManagedFileLocker.get();
        this.globalDocumentResourcesMap = globalDocumentResourcesMap;
        this.uniqueSingletonID = uniqueSingletonID;
        this.uniqueID = UUID.randomUUID().toString();

        locker.addCollectionToLock(this);
    }

    public boolean addResourceRef(final DocRestoreID docRestoreID, final Resource resource) {
        Validate.notNull(docRestoreID);
        Validate.notNull(resource);

        boolean result = false;

        try {
            DocumentResources docResources = globalDocumentResourcesMap.get(docRestoreID);

            if (null == docResources) {
                final long timeStamp = new Date().getTime();
                final Set<String> resources = new HashSet<String>();

                docResources = new DocumentResources(docRestoreID, timeStamp, uniqueSingletonID, resources);
            }

            if (null != docResources) {
                Set<String> resourcesSet = docResources.getResources();
                if (resourcesSet.add(resource.getManagedId())) {
                    // set new resource in set and write back
                    docResources.setResources(resourcesSet);
                    globalDocumentResourcesMap.set(docRestoreID, docResources);

                    // update our local hash map
                    synchronized(docResourcesMap) {
                        final DocumentResources localDocResources = docResourcesMap.get(docRestoreID);
                        if (null == localDocResources) {
                            docResourcesMap.put(docRestoreID, docResources);
                        } else {
                            Set<String> set = localDocResources.getResources();
                            if (null == set) { set = new HashSet<String>(); }
                            set.add(resource.getManagedId());
                            localDocResources.setResources(set);
                            docResourcesMap.put(docRestoreID, localDocResources);
                        }
                    }
                }
                result = true;
            }
        } catch (Exception e) {
            LOG.warn("RT connection: Exception caught while trying to set new resource to global hazelcast doc resource directory", e);
        }

        return result;
    }

    public boolean removeResourceRef(final DocRestoreID docRestoreID, final Resource resource) {
        Validate.notNull(docRestoreID);
        Validate.notNull(resource);

        boolean result = false;

        try {
            final DocumentResources docResources = globalDocumentResourcesMap.get(docRestoreID);

            if (null != docResources) {
                Set<String> resourcesSet = docResources.getResources();
                if (resourcesSet.remove(resource.getManagedId())) {
                    // set new resource in set and write back
                    docResources.setResources(resourcesSet);
                    globalDocumentResourcesMap.set(docRestoreID, docResources);

                    // update our local hash map
                    synchronized(docResourcesMap) {
                        final DocumentResources localDocResources = docResourcesMap.get(docRestoreID);
                        if (null != localDocResources) {
                            Set<String> set = localDocResources.getResources();
                            if (null != set) {
                                set.remove(resource.getManagedId());
                                localDocResources.setResources(set);
                                docResourcesMap.put(docRestoreID, localDocResources);
                            }
                        }
                    }
                }
                result = true;
            }
        } catch (Exception e) {
            LOG.warn("RT connection: Exception caught while trying to set new resource to global hazelcast doc resource directory", e);
        }

        return result;
    }

    public void removeDocResources(final DocRestoreID docRestoreID) {
        Validate.notNull(docRestoreID);

        try {
            final DocumentResources docResources = globalDocumentResourcesMap.get(docRestoreID);

            if (null != docResources) {
                globalDocumentResourcesMap.remove(docRestoreID);
            }

            // remove the resource ids from our local map
            synchronized(docResourcesMap) {
                docResourcesMap.remove(docRestoreID);
            }
        } catch (Exception e) {
            LOG.warn("RT connection: Exception caught while trying to set new resource to global hazelcast doc resource directory", e);
        }
    }

    public Set<String> getDocResources(final DocRestoreID docRestoreID) {
        Validate.notNull(docRestoreID);

        Set<String> result = null;

        synchronized(docResourcesMap) {
            final DocumentResources docResources = docResourcesMap.get(docRestoreID);

            if (null != docResources) {
                // first, provide the local resources map
                result = docResources.getResources();
            }
        }

        // Try the global hazelcast map now - it's possible that a client asks us
        // but the document was first opened on a different node.
        try {
            if (null == result) {
                final DocumentResources docResources = globalDocumentResourcesMap.get(docRestoreID);

                if (null != docResources) {
                    result = docResources.getResources();
                }
            }
        } catch (final OXException e) {
            LOG.warn("RT connection: Exception caught while trying to get resources from global hazelcast doc resource directory", e);
        }

        return result;
    }

    @Override
    public Set<IStreamIDCollection> getStreamIDCollectionCollection() {
        Set<IStreamIDCollection> result = null;

        synchronized(docResourcesMap) {
            if (docResourcesMap.size() > 0) {
                final HashSet<IStreamIDCollection> copy = new HashSet<IStreamIDCollection>();
                final Collection<DocumentResources> docResources = docResourcesMap.values();

                // make a deep copy of the map to enable parallel access
                for (DocumentResources docRes : docResources) {
                    copy.add(docRes.clone());
                }

                result = copy;
            }
        }

        return result;
    }

    @Override
    public void touch(final DocRestoreID uniqueCollectionID, String streamID) {
        fileManager.contains(streamID);
    }

    @Override
    public void updateStates(Set<IStreamIDCollection> updatedCollections) {
        final Collection<DocRestoreID> docResIds = new HashSet<DocRestoreID>(updatedCollections.size());
        final HashMap<DocRestoreID, IStreamIDCollection> tmpMap = new HashMap<DocRestoreID, IStreamIDCollection>(updatedCollections.size());

        for (final IStreamIDCollection updated : updatedCollections) {
            final DocRestoreID docResId = updated.getUniqueCollectionID();
            tmpMap.put(docResId, updated);
            docResIds.add(docResId);
        }

        try {
            final Map<DocRestoreID, DocumentResources> documentResourcesMap = globalDocumentResourcesMap.get(docResIds);

            for (final DocRestoreID docRestoreId : docResIds) {
                final DocumentResources docResources = documentResourcesMap.get(docRestoreId);
                final IStreamIDCollection streamIDCollection = tmpMap.get(docRestoreId);

                if ((null != docResources) && (null != streamIDCollection)) {
                   docResources.setTimeStamp(streamIDCollection.getTimeStamp());
                   globalDocumentResourcesMap.set(docRestoreId, docResources);
                }
            }
        } catch (Exception e) {
            LOG.warn("RT connection: Exception caught while trying to update time stamp of global hazelcast document resources", e);
        }

        synchronized(docResourcesMap) {

            for (final IStreamIDCollection updated : updatedCollections) {
                final DocumentResources docResources = docResourcesMap.get(updated.getUniqueCollectionID());
                if (null != docResources) {
                    docResources.touch(updated.getTimeStamp());
                }
            }
        }
    }

    @Override
    public String getUniqueID() {
        return this.uniqueID;
    }

    public void dispose() {
        DistributedManagedFileLocker.destroy();
    }

}
