/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

window.chai.use(function (chai, utils) {

    'use strict';

    var EPSILON = Math.pow(2, -51);

    // private global functions ===============================================

    /**
     * Adds a new property to Chai's message chain.
     *
     * @param {String} name
     *  The name of the new property. May be a space-separated list of
     *  different property names.
     *
     * @param {Function} handler
     *  The callback function invoked for the property. Will be called in the
     *  context of the assertion instance.
     */
    function addProperty(name, handler) {
        name.split(/\s+/).forEach(function (property) {
            utils.addProperty(chai.Assertion.prototype, property, handler);
        });
    }

    /**
     * Adds a new method to Chai's message chain.
     *
     * @param {String} name
     *  The name of the new method. May be a space-separated list of different
     *  methods names.
     *
     * @param {Function} handler
     *  The callback function invoked for the method. Will be called in the
     *  context of the assertion instance. Receives the parameters passed to
     *  the method in the language chain of the assertion.
     */
    function addMethod(name, handler) {
        name.split(/\s+/).forEach(function (method) {
            utils.addMethod(chai.Assertion.prototype, method, handler);
        });
    }

    /**
     * Returns whether the passed value is a finite number.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed value is a finite number. Does NOT return true for
     *  arrays containing a number element, as the native function isFinite()
     *  does.
     */
    function isFiniteNumber(value) {
        return (typeof value === 'number') && isFinite(value);
    }

    // properties =============================================================

    /**
     * A property asserting whether the actual value is a jQuery collection
     * containing exactly one DOM node, regardless of its type.
     *
     * @example
     *  expect($('<div>')).to.be.a.jqNode;
     */
    addProperty('jqNode', function () {
        this.assert(
            (this._obj instanceof $) && (this._obj.length === 1),
            'expected #{this} to be a jQuery collection with a single DOM node',
            'expected #{this} to not be a jQuery collection with a single DOM node'
        );
    });

    /**
     * A property asserting whether the actual value is a jQuery collection
     * containing exactly one DOM element node.
     *
     * @example
     *  expect($('<div>')).to.be.a.jqElementNode;
     */
    addProperty('jqElementNode', function () {
        this.assert(
            (this._obj instanceof $) && (this._obj.length === 1) && (this._obj[0].nodeType === 1),
            'expected #{this} to be a jQuery collection with a single DOM element node',
            'expected #{this} to not be a jQuery collection with a single DOM element node'
        );
    });

    /**
     * A property asserting whether the actual value is a jQuery collection
     * containing exactly one DOM text node.
     *
     * @example
     *  expect($('<div>text</div>').contents()).to.be.a.jqTextNode;
     */
    addProperty('jqTextNode', function () {
        this.assert(
            (this._obj instanceof $) && (this._obj.length === 1) && (this._obj[0].nodeType === 3),
            'expected #{this} to be a jQuery collection with a single DOM text node',
            'expected #{this} to not be a jQuery collection with a single DOM text node'
        );
    });

    /**
     * A property asserting whether the actual value is zero, or almost zero.
     * Precisely, returns whether the absolute value is less than or equal to
     * 2^-51 which is twice of Number.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI)).to.be.almostZero;
     */
    addProperty('almostZero', function () {
        this.assert(
            isFiniteNumber(this._obj) && (Math.abs(this._obj) <= EPSILON),
            'expected #{this} to be almost zero',
            'expected #{this} to not be almost zero'
        );
    });

    /**
     * A property asserting whether the actual value is a finite number.
     *
     * @example
     *  expect(Math.sin(0)).to.be.finite;
     */
    addProperty('finite', function () {
        this.assert(
            isFiniteNumber(this._obj),
            'expected #{this} to be finite',
            'expected #{this} to not be finite'
        );
    });

    /**
     * A property asserting whether the actual value is NaN.
     *
     * @example
     *  expect(Math.log(-1)).to.be.NaN;
     */
    addProperty('NaN', function () {
        this.assert(
            // isNaN() would return true for the array [NaN]
            (typeof this._obj === 'number') && isNaN(this._obj),
            'expected #{this} to be NaN',
            'expected #{this} to not be NaN'
        );
    });

    // methods ================================================================

    /**
     * A method asserting whether the tested number and the specified number
     * are almost equal, with a relative error less than or equal to 2^-51
     * which is twice of Number.EPSILON.
     *
     * @example
     *  expect(Math.sin(Math.PI/2)).to.almostEqual(1);
     *
     * @param {Number} expected
     *  The expected number that will be compared with the tested number. MUST
     *  NOT be zero (use the property 'almostZero' for that).
     */
    addMethod('almostEqual', function (expected) {
        this.assert(
            isFiniteNumber(this._obj) && (Math.abs((this._obj - expected) / expected) <= EPSILON),
            'expected #{this} to almost equal #{exp}',
            'expected #{this} to not almost equal #{exp}',
            expected
        );
    });

    /**
     * A method asserting whether the tested value, converted to a string,
     * equals the passed expectation.
     *
     * @example
     *  expect([1, 2]).to.stringifyTo('1,2');
     *
     * @param {String} expected
     *  The expected string that will be compared with the result of passing
     *  the tested value to the String() constructor.
     */
    addMethod('stringifyTo stringifiesTo', function (expected) {
        var result = String(this._obj);
        this.assert(
            result === expected,
            'expected #{this} to stringify to #{exp} but got \'' + result + '\'',
            'expected #{this} to not stringify to #{exp}',
            expected
        );
    });

    /**
     * Method asserting whether the tested value has passed class name,
     * using jQuery's hasClass method.
     *
     * @example
     *  expect($('<div>').addClass('test')).to.have.className('test');
     *
     * @param {String} expected
     *  The expected string class name that will be compared with jQuery's hasClass method.
     */
    addMethod('className', function (expected) {
        this.assert(
            (this._obj instanceof $) && this._obj.hasClass(expected),
            'expected #{this} to have class name #{exp}',
            'expected #{this} not to have class name #{exp}',
            expected
        );
    });

    // helpers ================================================================

    /**
     * Wrap the expect() function and check the parameter count. This helps to
     * prevent typos in the unit tests that may hide failing tests.
     */
    window.expect = (function (expect) {
        return function (result) {
            if (arguments.length !== 1) {
                throw new Error('expect() MUST be called with exactly one parameter');
            }
            return expect(result);
        };
    }(window.expect));

    // ========================================================================
});
