/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/render/colorutils'
], function (ColorUtils) {

    'use strict';

    // class Color ============================================================

    describe('Toolkit module ColorUtils', function () {

        it('should exist', function () {
            expect(ColorUtils).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        // verifies a color channel in the passed color model
        function verifyChannel(colorModel, channelName, exp) {
            expect(colorModel).to.have.a.property(channelName).that.is.closeTo(exp, 0.001);
        }

        // checks the 'rgb' property of the passed color descriptor
        function verifyRGB(colorDesc, expR, expG, expB, expA) {
            expect(colorDesc).to.have.a.property('rgb').that.is.an('object');
            verifyChannel(colorDesc.rgb, 'r', expR);
            verifyChannel(colorDesc.rgb, 'g', expG);
            verifyChannel(colorDesc.rgb, 'b', expB);
            verifyChannel(colorDesc.rgb, 'a', expA);
        }

        // checks the 'hsl' property of the passed color descriptor
        function verifyHSL(colorDesc, expH, expS, expL, expA) {
            expect(colorDesc).to.have.a.property('hsl').that.is.an('object');
            expect(colorDesc.hsl).to.have.a.property('h').that.is.closeTo(expH, 0.5);
            verifyChannel(colorDesc.hsl, 's', expS);
            verifyChannel(colorDesc.hsl, 'l', expL);
            verifyChannel(colorDesc.hsl, 'a', expA);
        }

        // verifies the passed color descriptor
        function verifyColorDescRGB(colorDesc, expR, expG, expB, expA, expHex, expCSS, expType) {
            verifyRGB(colorDesc, expR, expG, expB, expA);
            verifyChannel(colorDesc, 'a', expA);
            expect(colorDesc).to.have.a.property('hex', expHex.toUpperCase());
            expect(colorDesc).to.have.a.property('css', expCSS);
            expect(colorDesc).to.have.a.property('type', expType);
        }

        // verifies the passed color descriptor
        function verifyColorDescHSL(colorDesc, expH, expS, expL, expA) {
            verifyHSL(colorDesc, expH, expS, expL, expA);
            verifyChannel(colorDesc, 'a', expA);
            expect(colorDesc).to.have.a.property('type', 'hsl');
        }

        // static methods -----------------------------------------------------

        describe('method "getPresetColor"', function () {
            it('should exist', function () {
                expect(ColorUtils).itself.to.respondTo('getPresetColor');
            });
            it('should return the settings of a preset color', function () {
                expect(ColorUtils.getPresetColor('LightYellow')).to.match(/^[0-9a-f]{6}$/i);
                expect(ColorUtils.getPresetColor('_invalid_')).to.equal(null);
            });
        });

        describe('method "getSystemColor"', function () {
            it('should exist', function () {
                expect(ColorUtils).itself.to.respondTo('getSystemColor');
            });
            it('should return the settings of a system color', function () {
                expect(ColorUtils.getSystemColor('WindowText')).to.match(/^[0-9a-f]{6}$/i);
                expect(ColorUtils.getSystemColor('_invalid_')).to.equal(null);
            });
        });

        describe('method "parseColor"', function () {
            it('should exist', function () {
                expect(ColorUtils).itself.to.respondTo('parseColor');
            });
            it('should return null for empty or invalid string', function () {
                expect(ColorUtils.parseColor('')).to.equal(null);
                expect(ColorUtils.parseColor('_invalid_')).to.equal(null);
            });
            it('should parse "transparent" keyword', function () {
                verifyColorDescRGB(ColorUtils.parseColor('transparent'), 0, 0, 0, 0, '000000', 'transparent', 'transparent');
            });
            it('should parse preset color names', function () {
                verifyColorDescRGB(ColorUtils.parseColor('black'), 0, 0, 0, 1, '000000', '#000000', 'preset');
                verifyColorDescRGB(ColorUtils.parseColor('white'), 1, 1, 1, 1, 'FFFFFF', '#ffffff', 'preset');
            });
            it('should parse system color names', function () {
                expect(ColorUtils.parseColor('window')).to.have.a.property('type', 'system');
            });
            it('should parse hexadecimal RGB color', function () {
                verifyColorDescRGB(ColorUtils.parseColor('#0066cc'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#3399FF'), 0.2, 0.6, 1, 1, '3399FF', '#3399ff', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#06c'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#39F'), 0.2, 0.6, 1, 1, '3399FF', '#3399ff', 'rgb');
            });
            it('should parse hexadecimal RGBA color', function () {
                verifyColorDescRGB(ColorUtils.parseColor('#0066cc00'), 0, 0.4, 0.8, 0, '0066CC', 'transparent', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#0066cccc'), 0, 0.4, 0.8, 0.8, '0066CC', 'rgba(0,102,204,0.8)', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#0066ccFF'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#06c0'), 0, 0.4, 0.8, 0, '0066CC', 'transparent', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#06cc'), 0, 0.4, 0.8, 0.8, '0066CC', 'rgba(0,102,204,0.8)', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('#06cF'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
            });
            it('should reject invalid hexadecimal RGB color', function () {
                expect(ColorUtils.parseColor('#')).to.equal(null);
                expect(ColorUtils.parseColor('#4')).to.equal(null);
                expect(ColorUtils.parseColor('#48')).to.equal(null);
                expect(ColorUtils.parseColor('#4080c')).to.equal(null);
                expect(ColorUtils.parseColor('#4080c04')).to.equal(null);
                expect(ColorUtils.parseColor('#4080c0400')).to.equal(null);
                expect(ColorUtils.parseColor('#4080cg')).to.equal(null);
            });
            it('should parse function-style RGB(A) color', function () {
                verifyColorDescRGB(ColorUtils.parseColor('rgb(0,102,204)'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('rgb( 51 , 153 , 255 )'), 0.2, 0.6, 1, 1, '3399FF', '#3399ff', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('rgba(0,102,204,1)'), 0, 0.4, 0.8, 1, '0066CC', '#0066cc', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('rgba( 51 , 153 , 255 , 0.8)'), 0.2, 0.6, 1, 0.8, '3399FF', 'rgba(51,153,255,0.8)', 'rgb');
                verifyColorDescRGB(ColorUtils.parseColor('rgba(0,102,204,0)'), 0, 0.4, 0.8, 0, '0066CC', 'transparent', 'rgb');
                expect(ColorUtils.parseColor('rgb(64,128)')).to.equal(null);
                expect(ColorUtils.parseColor('rgb(64,128,192,1)')).to.equal(null);
                verifyColorDescRGB(ColorUtils.parseColor('rgb(0,102,1000)'), 0, 0.4, 1, 1, '0066FF', '#0066ff', 'rgb');
                expect(ColorUtils.parseColor('rgb(64,128,ff)')).to.equal(null);
                expect(ColorUtils.parseColor('rgba(64,128,192)')).to.equal(null);
                expect(ColorUtils.parseColor('rgba(64,128,192,1,1)')).to.equal(null);
                expect(ColorUtils.parseColor('rgba(64,128,192,2)')).to.equal(null);
            });
            it('should parse function-style HSL(A) color', function () {
                verifyColorDescHSL(ColorUtils.parseColor('hsl(60,100%,50%)'), 60, 1, 0.5, 1);
                verifyColorDescHSL(ColorUtils.parseColor('hsl( 0 , 0% , 0% )'), 0, 0, 0, 1);
                verifyColorDescHSL(ColorUtils.parseColor('hsla(60,100%,50%,1)'), 60, 1, 0.5, 1);
                verifyColorDescHSL(ColorUtils.parseColor('hsla( 60 , 100% , 50% , 0.42 )'), 60, 1, 0.5, 0.42);
                verifyColorDescHSL(ColorUtils.parseColor('hsla( 60 , 100% , 50% , 0 )'), 60, 1, 0.5, 0);
                expect(ColorUtils.parseColor('hsl(60,100%)')).to.equal(null);
                expect(ColorUtils.parseColor('hsl(60,100%,50%,1)')).to.equal(null);
                expect(ColorUtils.parseColor('hsl(60,100%,50)')).to.equal(null);
                expect(ColorUtils.parseColor('hsl(60%,100%,50%)')).to.equal(null);
                verifyColorDescHSL(ColorUtils.parseColor('hsl(420,200%,50%)'), 60, 1, 0.5, 1);
                expect(ColorUtils.parseColor('hsl(60,100%,ff%)')).to.equal(null);
                expect(ColorUtils.parseColor('hsla(60,100%,50%)')).to.equal(null);
                expect(ColorUtils.parseColor('hsla(60,100%,50%,1,1)')).to.equal(null);
                expect(ColorUtils.parseColor('hsla(60,100%,50%,2)')).to.equal(null);
            });
        });
    });

    // ========================================================================
});
