/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser'
], function (LocaleData, Parser) {

    'use strict';

    // static class Parser ====================================================

    describe('Toolkit module Parser', function () {

        it('should exist', function () {
            expect(Parser).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        function utcDate(y, m, d) {
            return new Date(Date.UTC(y, m, d));
        }

        function utcTime(h, m, s, ms) {
            return new Date(Date.UTC(1970, 0, 1, h, m, s, ms || 0));
        }

        var CENTURY = Math.floor(new Date().getFullYear() / 100) * 100;

        // constants ----------------------------------------------------------

        describe('constant "GENERAL"', function () {
            it('should exist', function () {
                expect(Parser).to.have.a.property('GENERAL', 'GENERAL');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "numberToString"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('numberToString');
            });
            it('should convert numbers to string using local decimal separator', function () {
                expect(Parser.numberToString(0)).to.equal('0');
                expect(Parser.numberToString(42)).to.equal('42');
                expect(Parser.numberToString(-1)).to.equal('-1');
                expect(Parser.numberToString(0.5)).to.equal('0,5');
                expect(Parser.numberToString(-12.5)).to.equal('-12,5');
            });
        });

        describe('method "stringToNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('stringToNumber');
            });
            it('should convert strings to number using local decimal separator', function () {
                expect(Parser.stringToNumber('0')).to.equal(0);
                expect(Parser.stringToNumber('42')).to.equal(42);
                expect(Parser.stringToNumber('-1')).to.equal(-1);
                expect(Parser.stringToNumber('0,5')).to.equal(0.5);
                expect(Parser.stringToNumber('-12,5')).to.equal(-12.5);
            });
            it('should return NaN for invalid strings', function () {
                expect(Parser.stringToNumber('')).to.be.nan;
                expect(Parser.stringToNumber('a')).to.be.nan;
            });
        });

        describe('method "parseLeadingNumber"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingNumber');
            });
            it('should parse numbers', function () {
                expect(Parser.parseLeadingNumber('42')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: '' });
                expect(Parser.parseLeadingNumber('-42', { sign: true })).to.deep.equal({ number: -42, text: '-42', sign: '-', dec: false, scientific: false, grouped: false, remaining: '' });

                expect(Parser.parseLeadingNumber('42 \u20ac')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: ' \u20ac' });
                expect(Parser.parseLeadingNumber('42 EUR')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: ' EUR' });
                expect(Parser.parseLeadingNumber('-42 \u20ac', { sign: true })).to.deep.equal({ number: -42, text: '-42', sign: '-', dec: false, scientific: false, grouped: false, remaining: ' \u20ac' });

                expect(Parser.parseLeadingNumber('42 %')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: ' %' });
                expect(Parser.parseLeadingNumber('-42 %', { sign: true })).to.deep.equal({ number: -42, text: '-42', sign: '-', dec: false, scientific: false, grouped: false, remaining: ' %' });
            });
            it('should parse numbers using local decimal separator', function () {
                expect(Parser.parseLeadingNumber('42abc')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,abc')).to.deep.equal({ number: 42, text: '42,', sign: '', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5abc')).to.deep.equal({ number: 42.5, text: '42,5', sign: '', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('0,5abc')).to.deep.equal({ number: 0.5, text: '0,5', sign: '', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber(',5abc')).to.deep.equal({ number: 0.5, text: ',5', sign: '', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber(',5,5')).to.deep.equal({ number: 0.5, text: ',5', sign: '', dec: true, scientific: false, grouped: false, remaining: ',5' });
                expect(Parser.parseLeadingNumber('42,5E3abc')).to.deep.equal({ number: 42500, text: '42,5E3', sign: '', dec: true, scientific: true, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e0abc')).to.deep.equal({ number: 42.5, text: '42,5e0', sign: '', dec: true, scientific: true, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e+03abc')).to.deep.equal({ number: 42500, text: '42,5e+03', sign: '', dec: true, scientific: true, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e-03abc')).to.deep.equal({ number: 0.0425, text: '42,5e-03', sign: '', dec: true, scientific: true, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5e3,5')).to.deep.equal({ number: 42500, text: '42,5e3', sign: '', dec: true, scientific: true, grouped: false, remaining: ',5' });
                expect(Parser.parseLeadingNumber('42e3,5')).to.deep.equal({ number: 42000, text: '42e3', sign: '', dec: false, scientific: true, grouped: false, remaining: ',5' });
                expect(Parser.parseLeadingNumber('42.5')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: '.5' });
            });
            it('should reject invalid text', function () {
                expect(Parser.parseLeadingNumber('abc')).to.equal(null);
                expect(Parser.parseLeadingNumber(',')).to.equal(null);
                expect(Parser.parseLeadingNumber('e3')).to.equal(null);
                expect(Parser.parseLeadingNumber(',e3')).to.equal(null);
            });
            it('should handle leading sign according to option', function () {
                expect(Parser.parseLeadingNumber('+42abc')).to.equal(null);
                expect(Parser.parseLeadingNumber('+42abc', { sign: true })).to.deep.equal({ number: 42, text: '+42', sign: '+', dec: false, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-42abc')).to.equal(null);
                expect(Parser.parseLeadingNumber('-42abc', { sign: true })).to.deep.equal({ number: -42, text: '-42', sign: '-', dec: false, scientific: false, grouped: false, remaining: 'abc' });
            });
            it('should use custom decimal separator', function () {
                expect(Parser.parseLeadingNumber('42#5abc')).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: '#5abc' });
                expect(Parser.parseLeadingNumber('42#5abc', { dec: '#' })).to.deep.equal({ number: 42.5, text: '42#5', sign: '', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('42,5abc', { dec: '#' })).to.deep.equal({ number: 42, text: '42', sign: '', dec: false, scientific: false, grouped: false, remaining: ',5abc' });
                expect(Parser.parseLeadingNumber('+42#5abc', { dec: '#' })).to.equal(null);
                expect(Parser.parseLeadingNumber('+42#5abc', { dec: '#', sign: true })).to.deep.equal({ number: 42.5, text: '+42#5', sign: '+', dec: true, scientific: false, grouped: false, remaining: 'abc' });
                expect(Parser.parseLeadingNumber('-42#5abc', { dec: '#', sign: true })).to.deep.equal({ number: -42.5, text: '-42#5', sign: '-', dec: true, scientific: false, grouped: false, remaining: 'abc' });
            });
            it('should reject remaining garbage after number', function () {
                expect(Parser.parseLeadingNumber('42abc', { complete: true })).to.equal(null);
            });
        });

        describe('method "parseLeadingDate"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingDate');
            });
            it('should parse valid dates', function () {
                expect(Parser.parseLeadingDate('21/3/1999')).to.deep.equal({ D: 21, M: 2, Y: 1999, date: utcDate(1999, 2, 21), text: '21/3/1999', remaining: '' });
                expect(Parser.parseLeadingDate('21/3/99')).to.deep.equal({ D: 21, M: 2, Y: 99, date: utcDate(CENTURY + 99, 2, 21), text: '21/3/99', remaining: '' });
            });
            it('shouldn\'t parse invalid dates', function () {
                expect(Parser.parseLeadingDate('31/4/1999')).to.equal(null);
                expect(Parser.parseLeadingDate('31/4/99')).to.equal(null);
                expect(Parser.parseLeadingDate('31/April/99')).to.equal(null);
            });
            it('should parse valid dates with month-names', function () {
                expect(Parser.parseLeadingDate('21/April/1999')).to.deep.equal({ D: 21, M: 'April', Y: 1999, date: utcDate(1999, 3, 21), text: '21/April/1999', remaining: '' });
                expect(Parser.parseLeadingDate('21/Apri/99')).to.deep.equal({ D: 21, M: 'Apri', Y: 99, date: utcDate(CENTURY + 99, 3, 21), text: '21/Apri/99', remaining: '' });
                expect(Parser.parseLeadingDate('21. April 1999')).to.deep.equal({ D: 21, M: 'April', Y: 1999, date: utcDate(1999, 3, 21), text: '21. April 1999', remaining: '' });
                expect(Parser.parseLeadingDate('21.April 2016')).to.deep.equal({ D: 21, M: 'April', Y: 2016, date: utcDate(2016, 3, 21), text: '21.April 2016', remaining: '' });
                expect(Parser.parseLeadingDate('21. April 16')).to.deep.equal({ D: 21, M: 'April', Y: 16, date: utcDate(2016, 3, 21), text: '21. April 16', remaining: '' });
                expect(Parser.parseLeadingDate('21. Apr. 16')).to.deep.equal({ D: 21, M: 'Apr', Y: 16, date: utcDate(2016, 3, 21), text: '21. Apr. 16', remaining: '' });
                expect(Parser.parseLeadingDate('21. Apr 16')).to.deep.equal({ D: 21, M: 'Apr', Y: 16, date: utcDate(2016, 3, 21), text: '21. Apr 16', remaining: '' });
                expect(Parser.parseLeadingDate('21.Apr 16')).to.deep.equal({ D: 21, M: 'Apr', Y: 16, date: utcDate(2016, 3, 21), text: '21.Apr 16', remaining: '' });
                expect(Parser.parseLeadingDate('21 Apr 16')).to.deep.equal({ D: 21, M: 'Apr', Y: 16, date: utcDate(2016, 3, 21), text: '21 Apr 16', remaining: '' });
            });
            it('should parse valid dates (DMY)', function () {
                expect(Parser.parseLeadingDate('9 November 2014')).to.deep.equal({ D: 9, M: 'November', Y: 2014, date: utcDate(2014, 10, 9), text: '9 November 2014', remaining: '' });
                expect(Parser.parseLeadingDate('9. November 2014')).to.deep.equal({ D: 9, M: 'November', Y: 2014, date: utcDate(2014, 10, 9), text: '9. November 2014', remaining: '' });
                expect(Parser.parseLeadingDate('9/11/2014')).to.deep.equal({ D: 9, M: 10, Y: 2014, date: utcDate(2014, 10, 9), text: '9/11/2014', remaining: '' });
                expect(Parser.parseLeadingDate('09.11.2014')).to.deep.equal({ D: 9, M: 10, Y: 2014, date: utcDate(2014, 10, 9), text: '09.11.2014', remaining: '' });
                expect(Parser.parseLeadingDate('9-11-2014')).to.deep.equal({ D: 9, M: 10, Y: 2014, date: utcDate(2014, 10, 9), text: '9-11-2014', remaining: '' });
                expect(Parser.parseLeadingDate('09-Nov-2014')).to.deep.equal({ D: 9, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 9), text: '09-Nov-2014', remaining: '' });
                expect(Parser.parseLeadingDate('09 Nov 14')).to.deep.equal({ D: 9, M: 'Nov', Y: 14, date: utcDate(2014, 10, 9), text: '09 Nov 14', remaining: '' });
                expect(Parser.parseLeadingDate('09 Nov 2014')).to.deep.equal({ D: 9, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 9), text: '09 Nov 2014', remaining: '' });
                expect(Parser.parseLeadingDate('9 Nov 14')).to.deep.equal({ D: 9, M: 'Nov', Y: 14, date: utcDate(2014, 10, 9), text: '9 Nov 14', remaining: '' });
                expect(Parser.parseLeadingDate('9 Nov 2014')).to.deep.equal({ D: 9, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 9), text: '9 Nov 2014', remaining: '' });
                expect(Parser.parseLeadingDate('09/Nov/2014')).to.deep.equal({ D: 9, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 9), text: '09/Nov/2014', remaining: '' });
            });
            it('should parse valid dates (YMD)', function () {
                expect(Parser.parseLeadingDate('2003-11-09')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003-11-09', remaining: '' });
                expect(Parser.parseLeadingDate('2003 November 9')).to.deep.equal({ D: 9, M: 'November', Y: 2003, date: utcDate(2003, 10, 9), text: '2003 November 9', remaining: '' });
                expect(Parser.parseLeadingDate('2003 Nov 9')).to.deep.equal({ D: 9, M: 'Nov', Y: 2003, date: utcDate(2003, 10, 9), text: '2003 Nov 9', remaining: '' });
                expect(Parser.parseLeadingDate('2003 Nov 09')).to.deep.equal({ D: 9, M: 'Nov', Y: 2003, date: utcDate(2003, 10, 9), text: '2003 Nov 09', remaining: '' });
                expect(Parser.parseLeadingDate('2003-Nov-9')).to.deep.equal({ D: 9, M: 'Nov', Y: 2003, date: utcDate(2003, 10, 9), text: '2003-Nov-9', remaining: '' });
                expect(Parser.parseLeadingDate('2003-Nov-09')).to.deep.equal({ D: 9, M: 'Nov', Y: 2003, date: utcDate(2003, 10, 9), text: '2003-Nov-09', remaining: '' });
                expect(Parser.parseLeadingDate('2003.11.9')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003.11.9', remaining: '' });
                expect(Parser.parseLeadingDate('2003.11.09')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003.11.09', remaining: '' });
                expect(Parser.parseLeadingDate('2003/11/09')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003/11/09', remaining: '' });
                expect(Parser.parseLeadingDate('2003/11/9')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003/11/9', remaining: '' });
                expect(Parser.parseLeadingDate('2003-11-09')).to.deep.equal({ D: 9, M: 10, Y: 2003, date: utcDate(2003, 10, 9), text: '2003-11-09', remaining: '' });

                // remaining point at the end
                // expect(Parser.parseLeadingDate('2003. november 9.')).to.deep.equal({ D: 9, M: 'november', Y: 2003, date: utcDate(2003, 10, 9), text: '2003. november 9.', remaining: '' });
                // expect(Parser.parseLeadingDate('2003. nov. 9.')).to.deep.equal('');
                // expect(Parser.parseLeadingDate('2003. 11. 9.')).to.deep.equal('');

                // unknown month-name
                // expect(Parser.parseLeadingDate('2003. XI. 9.')).to.deep.equal('');

                // unknown format
                // expect(Parser.parseLeadingDate('20031109')).to.deep.equal('');
            });
            it('should parse valid dates (MY)', function () {
                expect(Parser.parseLeadingDate('November 2014')).to.deep.equal({ D: null, M: 'November', Y: 2014, date: utcDate(2014, 10, 1), text: 'November 2014', remaining: '' });
                expect(Parser.parseLeadingDate('November 14')).to.deep.equal({ D: null, M: 'November', Y: 14, date: utcDate(2014, 10, 1), text: 'November 14', remaining: '' });
                expect(Parser.parseLeadingDate('Nov. 2014')).to.deep.equal({ D: null, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 1), text: 'Nov. 2014', remaining: '' });
                expect(Parser.parseLeadingDate('Nov. 14')).to.deep.equal({ D: null, M: 'Nov', Y: 14, date: utcDate(2014, 10, 1), text: 'Nov. 14', remaining: '' });
                expect(Parser.parseLeadingDate('Nov 2014')).to.deep.equal({ D: null, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 1), text: 'Nov 2014', remaining: '' });
                expect(Parser.parseLeadingDate('Nov 14')).to.deep.equal({ D: null, M: 'Nov', Y: 14, date: utcDate(2014, 10, 1), text: 'Nov 14', remaining: '' });
                expect(Parser.parseLeadingDate('11/2014')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '11/2014', remaining: '' });
                expect(Parser.parseLeadingDate('11/14')).to.deep.equal({ D: null, M: 10, Y: 14, date: utcDate(2014, 10, 1), text: '11/14', remaining: '' });
                expect(Parser.parseLeadingDate('11.2014')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '11.2014', remaining: '' });
                expect(Parser.parseLeadingDate('11.14')).to.deep.equal({ D: null, M: 10, Y: 14, date: utcDate(2014, 10, 1), text: '11.14', remaining: '' });
                expect(Parser.parseLeadingDate('11-2014')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '11-2014', remaining: '' });
                expect(Parser.parseLeadingDate('11-14')).to.deep.equal({ D: null, M: 10, Y: 14, date: utcDate(2014, 10, 1), text: '11-14', remaining: '' });
            });
            it('should parse valid dates (YM)', function () {
                expect(Parser.parseLeadingDate('2014 November')).to.deep.equal({ D: null, M: 'November', Y: 2014, date: utcDate(2014, 10, 1), text: '2014 November', remaining: '' });
                expect(Parser.parseLeadingDate('2014 Nov')).to.deep.equal({ D: null, M: 'Nov', Y: 2014, date: utcDate(2014, 10, 1), text: '2014 Nov', remaining: '' });
                expect(Parser.parseLeadingDate('2014/11')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '2014/11', remaining: '' });
                expect(Parser.parseLeadingDate('2014.11')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '2014.11', remaining: '' });
                expect(Parser.parseLeadingDate('2014-11')).to.deep.equal({ D: null, M: 10, Y: 2014, date: utcDate(2014, 10, 1), text: '2014-11', remaining: '' });
            });
            it('should parse valid dates (DM)', function () {
                expect(Parser.parseLeadingDate('11/09')).to.deep.equal({ D: 11, M: 8, Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11/09', remaining: '' });
                expect(Parser.parseLeadingDate('11/September')).to.deep.equal({ D: 11, M: 'September', Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11/September', remaining: '' });
                expect(Parser.parseLeadingDate('11-09')).to.deep.equal({ D: 11, M: 8, Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11-09', remaining: '' });
                expect(Parser.parseLeadingDate('11-September')).to.deep.equal({ D: 11, M: 'September', Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11-September', remaining: '' });
                expect(Parser.parseLeadingDate('11.09')).to.deep.equal({ D: 11, M: 8, Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11.09', remaining: '' });
                expect(Parser.parseLeadingDate('11.September')).to.deep.equal({ D: 11, M: 'September', Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11.September', remaining: '' });
                expect(Parser.parseLeadingDate('11. September')).to.deep.equal({ D: 11, M: 'September', Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11. September', remaining: '' });
                expect(Parser.parseLeadingDate('11 September')).to.deep.equal({ D: 11, M: 'September', Y: null, date: utcDate(new Date().getFullYear(), 8, 11), text: '11 September', remaining: '' });
            });
            // TODO: more
        });

        describe('method "parseLeadingTime"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingTime');
            });
            it('should parse valid times', function () {
                expect(Parser.parseLeadingTime('0:0:0')).to.deep.equal({ h: 0, m: 0, s: 0, ms: null, time: utcTime(0, 0, 0), text: '0:0:0', remaining: '', serial: 0 });
                expect(Parser.parseLeadingTime('23:59:59')).to.deep.equal({ h: 23, m: 59, s: 59, ms: null, time: utcTime(23, 59, 59), text: '23:59:59', remaining: '', serial: 0.999988425925926 });
            });
            it('should parse valid am/pm times', function () {
                expect(Parser.parseLeadingTime('9 am')).to.deep.equal({ h: 9, m: null, s: null, ms: null, time: utcTime(9, 0, 0), ampm: true, text: '9 am', remaining: '', serial: 0.375 });
                expect(Parser.parseLeadingTime('9:30 am')).to.deep.equal({ h: 9, m: 30, s: null, ms: null, time: utcTime(9, 30, 0), ampm: true, text: '9:30', remaining: '', serial: 0.3958333333333333 });
                expect(Parser.parseLeadingTime('9 pm')).to.deep.equal({ h: 21, m: null, s: null, ms: null, time: utcTime(21, 0, 0), ampm: true, text: '9 pm', remaining: '', serial: 0.875 });
                expect(Parser.parseLeadingTime('9:30 pm')).to.deep.equal({ h: 21, m: 30, s: null, ms: null, time: utcTime(21, 30, 0), ampm: true, text: '9:30', remaining: '', serial: 0.8958333333333334 });
                expect(Parser.parseLeadingTime('9 a')).to.deep.equal({ h: 9, m: null, s: null, ms: null, time: utcTime(9, 0, 0), ampm: true, text: '9 a', remaining: '', serial: 0.375 });
                expect(Parser.parseLeadingTime('9:30 a')).to.deep.equal({ h: 9, m: 30, s: null, ms: null, time: utcTime(9, 30, 0), ampm: true, text: '9:30', remaining: '', serial: 0.3958333333333333 });
                expect(Parser.parseLeadingTime('9 p')).to.deep.equal({ h: 21, m: null, s: null, ms: null, time: utcTime(21, 0, 0), ampm: true, text: '9 p', remaining: '', serial: 0.875 });
                expect(Parser.parseLeadingTime('9:30 p')).to.deep.equal({ h: 21, m: 30, s: null, ms: null, time: utcTime(21, 30, 0), ampm: true, text: '9:30', remaining: '', serial: 0.8958333333333334 });
            });
            it('should parse valid 12 am time', function () {
                expect(Parser.parseLeadingTime('12 am')).to.deep.equal({ h: 0, m: null, s: null, ms: null, time: utcTime(0, 0, 0), ampm: true, text: '12 am', remaining: '', serial: 0 });
                expect(Parser.parseLeadingTime('12 a')).to.deep.equal({ h: 0, m: null, s: null, ms: null, time: utcTime(0, 0, 0), ampm: true, text: '12 a', remaining: '', serial: 0 });
            });
            it('should parse valid 12 pm time', function () {
                expect(Parser.parseLeadingTime('12 pm')).to.deep.equal({ h: 12, m: null, s: null, ms: null, time: utcTime(12, 0, 0), ampm: true, text: '12 pm', remaining: '', serial: 0.5 });
                expect(Parser.parseLeadingTime('12 p')).to.deep.equal({ h: 12, m: null, s: null, ms: null, time: utcTime(12, 0, 0), ampm: true, text: '12 p', remaining: '', serial: 0.5 });
            });
            // TODO: more
        });

        describe('method "parseLeadingCurrency"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingCurrency');
            });
            it('should parse currency numbers using local decimal separator', function () {
                expect(Parser.parseLeadingCurrency(LocaleData.CURRENCY + '42')).to.deep.equal(['\u20ac42', '42', '\u20ac']);
                expect(Parser.parseLeadingCurrency(LocaleData.ISO_CURRENCY + '42')).to.deep.equal(['EUR42', '42', 'EUR']);
                expect(Parser.parseLeadingCurrency(LocaleData.CURRENCY + '42' + LocaleData.DEC + '33')).to.deep.equal(['\u20ac42,33', '42,33', '\u20ac']);
                expect(Parser.parseLeadingCurrency(LocaleData.ISO_CURRENCY + '42' + LocaleData.DEC + '33')).to.deep.equal(['EUR42,33', '42,33', 'EUR']);

                expect(Parser.parseLeadingCurrency('-' + LocaleData.CURRENCY + '42')).to.deep.equal(['-\u20ac42', '-42', '\u20ac']);
                expect(Parser.parseLeadingCurrency('-' + LocaleData.ISO_CURRENCY + '42')).to.deep.equal(['-EUR42', '-42', 'EUR']);
                expect(Parser.parseLeadingCurrency('-' + LocaleData.CURRENCY + '42' + LocaleData.DEC + '33')).to.deep.equal(['-\u20ac42,33', '-42,33', '\u20ac']);
                expect(Parser.parseLeadingCurrency('-' + LocaleData.ISO_CURRENCY + '42' + LocaleData.DEC + '33')).to.deep.equal(['-EUR42,33', '-42,33', 'EUR']);

                expect(Parser.parseLeadingCurrency(LocaleData.CURRENCY + '-42')).to.deep.equal(['\u20ac-42', '-42', '\u20ac']);
                expect(Parser.parseLeadingCurrency(LocaleData.ISO_CURRENCY + '-42')).to.deep.equal(['EUR-42', '-42', 'EUR']);
                expect(Parser.parseLeadingCurrency(LocaleData.CURRENCY + '-42' + LocaleData.DEC + '33')).to.deep.equal(['\u20ac-42,33', '-42,33', '\u20ac']);
                expect(Parser.parseLeadingCurrency(LocaleData.ISO_CURRENCY + '-42' + LocaleData.DEC + '33')).to.deep.equal(['EUR-42,33', '-42,33', 'EUR']);
            });
            // TODO: more
        });

        describe('method "parseLeadingPercent"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseLeadingPercent');
            });
            it('should parse currency numbers using local decimal separator', function () {
                expect(Parser.parseLeadingPercent('% 42')).to.deep.equal(['% 42', '42']);
                expect(Parser.parseLeadingPercent('% 42' + LocaleData.DEC + '33')).to.deep.equal(['% 42,33', '42,33']);

                expect(Parser.parseLeadingPercent('- % 42')).to.deep.equal(['- % 42', '-42']);
                expect(Parser.parseLeadingPercent('- % 42' + LocaleData.DEC + '33')).to.deep.equal(['- % 42,33', '-42,33']);

                expect(Parser.parseLeadingPercent('%-42')).to.deep.equal(['%-42', '-42']);
                expect(Parser.parseLeadingPercent('%-42' + LocaleData.DEC + '33')).to.deep.equal(['%-42,33', '-42,33']);
            });
            // TODO: more
        });

        describe('method "checkFollowingCurrency"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('checkFollowingCurrency');
            });
            it('should check currency symbol at the end of the string', function () {
                expect(Parser.checkFollowingCurrency(' ' + LocaleData.CURRENCY)).to.equal(true);
                expect(Parser.checkFollowingCurrency(' ' + LocaleData.ISO_CURRENCY)).to.equal(true);
                expect(Parser.checkFollowingCurrency('sample string')).to.equal(false);
            });
            // TODO: more
        });

        describe('method "parseBrackets"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseBrackets');
            });
            it('should check brackets of the string', function () {
                expect(Parser.parseBrackets('(8)')).to.equal('-8');
                expect(Parser.parseBrackets('(8)%')).to.equal('-8%');
                expect(Parser.parseBrackets('(8%)')).to.equal('-8%');
                expect(Parser.parseBrackets('%(8)')).to.equal('%-8');
                expect(Parser.parseBrackets('(%8)')).to.equal('%-8');
            });
            // TODO: more
        });

        describe('method "parseFormatCode"', function () {
            it('should exist', function () {
                expect(Parser).itself.to.respondTo('parseFormatCode');
            });
            it('should cache the parsed format codes', function () {
                var parsedFormat = Parser.parseFormatCode('ooxml', 'op', 'General');
                expect(Parser.parseFormatCode('ooxml', 'op', 'General')).to.equal(parsedFormat);
                expect(Parser.parseFormatCode('ooxml', 'ui', 'General')).to.not.equal(parsedFormat);
                expect(Parser.parseFormatCode('odf', 'op', 'General')).to.not.equal(parsedFormat);
                expect(Parser.parseFormatCode('odf', 'ui', 'General')).to.not.equal(parsedFormat);
            });
            it('should recognize "standard" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'General')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('ooxml', 'ui', 'Standard')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('odf', 'op', 'General')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('odf', 'ui', 'Standard')).to.have.a.property('category', 'standard');
                // bug 46211: empty format code falls back to standard format
                expect(Parser.parseFormatCode('ooxml', 'op', '')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('ooxml', 'ui', '')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('odf', 'op', '')).to.have.a.property('category', 'standard');
                expect(Parser.parseFormatCode('odf', 'ui', '')).to.have.a.property('category', 'standard');
            });
            it('should recognize "number" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '0')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('ooxml', 'op', '0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00;[Red]-#,##0.00')).to.have.a.property('category', 'number');
                expect(Parser.parseFormatCode('ooxml', 'op', '0;@')).to.have.a.property('category', 'number');
            });
            it('should recognize "scientific" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '0E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '0.00E-00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00E+00;[Red]-#,##0.00E+00')).to.have.a.property('category', 'scientific');
                expect(Parser.parseFormatCode('ooxml', 'op', '0E+00;@')).to.have.a.property('category', 'scientific');
            });
            it('should recognize "percent" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '0%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('ooxml', 'op', '0.00%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('ooxml', 'op', '0.00%;[Red]-0.00%')).to.have.a.property('category', 'percent');
                expect(Parser.parseFormatCode('ooxml', 'op', '0%;@')).to.have.a.property('category', 'percent');
            });
            it('should recognize "fraction" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '# ?/?')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('ooxml', 'op', '# ??/??')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('ooxml', 'op', '# ??/10')).to.have.a.property('category', 'fraction');
                expect(Parser.parseFormatCode('ooxml', 'op', '# ??/16')).to.have.a.property('category', 'fraction');
            });
            it('should recognize "currency" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '$#,##0.00')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00 \u20ac')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('ooxml', 'op', '[$$-409]#,##0.00')).to.have.a.property('category', 'currency');
                expect(Parser.parseFormatCode('ooxml', 'op', '#,##0.00 [$\u20ac-407]')).to.have.a.property('category', 'currency');
            });
            it('should recognize "date" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'MM/DD/YYYY')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('ooxml', 'op', 'D. MMM. YYYY')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('ooxml', 'op', 'mmm')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('ooxml', 'op', 'mm')).to.have.a.property('category', 'date');
            });
            it('should recognize "time" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh:mm:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh:mm:ss AM/PM')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', '[hh]:mm:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'mm:ss.00')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh:MM:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'mm:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh:mm')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'MM:ss')).to.have.a.property('category', 'time');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh:M')).to.have.a.property('category', 'time');
            });
            it('should recognize "datetime" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'MM/DD/YYYY hh:mm:ss')).to.have.a.property('category', 'datetime');
                expect(Parser.parseFormatCode('ooxml', 'op', 'DD. MMM. hh:mm AM/PM')).to.have.a.property('category', 'datetime');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh/mmm')).to.have.a.property('category', 'datetime');
                expect(Parser.parseFormatCode('ooxml', 'op', 'hh/mmmm')).to.have.a.property('category', 'datetime');
            });
            it('should recognize "text" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '@')).to.have.a.property('category', 'text');
            });
            it('should recognize "custom" category', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', '-General')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', 'General;-General')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', 'General;-0')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', '0;0E+00')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', '0;0%')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', '0E+00%')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', '$0%')).to.have.a.property('category', 'custom');
            });
            it('should parse format codes in UI grammar', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'YYYY-DD')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('ooxml', 'ui', 'YYYY-DD')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', 'JJJJ-TT')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'ui', 'JJJJ-TT')).to.have.a.property('category', 'date');
            });
            it('should parse special ODF date tokens', function () {
                expect(Parser.parseFormatCode('ooxml', 'op', 'QQ')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', 'WW')).to.have.a.property('category', 'custom');
                expect(Parser.parseFormatCode('ooxml', 'op', 'NN')).to.have.a.property('syntaxError', true);
                expect(Parser.parseFormatCode('odf', 'op', 'QQ')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('odf', 'op', 'WW')).to.have.a.property('category', 'date');
                expect(Parser.parseFormatCode('odf', 'op', 'NN')).to.have.a.property('category', 'date');
            });
        });
    });

    // class ParsedFormat =====================================================

    describe('Toolkit class ParsedParser', function () {

        var stdFormat, percentFormat1, percentFormat2, dateFormat, timeFormat, dateTimeFormat;

        before(function () {
            stdFormat = Parser.parseFormatCode('ooxml', 'op', 'General');
            percentFormat1 = Parser.parseFormatCode('ooxml', 'op', '0%');
            percentFormat2 = Parser.parseFormatCode('ooxml', 'op', '0.00%');
            dateFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD');
            timeFormat = Parser.parseFormatCode('ooxml', 'op', 'hh:mm:ss');
            dateTimeFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD hh:mm:ss');
        });

        describe('method "isStandard"', function () {
            it('should exist', function () {
                expect(stdFormat).to.respondTo('isStandard');
            });
            it('should return true for standard formats only', function () {
                expect(stdFormat.isStandard()).to.equal(true);
                expect(percentFormat1.isStandard()).to.equal(false);
                expect(percentFormat2.isStandard()).to.equal(false);
                expect(dateFormat.isStandard()).to.equal(false);
                expect(timeFormat.isStandard()).to.equal(false);
                expect(dateTimeFormat.isStandard()).to.equal(false);
            });
        });

        describe('method "isAnyDate"', function () {
            it('should exist', function () {
                expect(stdFormat).to.respondTo('isAnyDate');
            });
            it('should return true for date and date/time formats only', function () {
                expect(stdFormat.isAnyDate()).to.equal(false);
                expect(percentFormat1.isAnyDate()).to.equal(false);
                expect(percentFormat2.isAnyDate()).to.equal(false);
                expect(dateFormat.isAnyDate()).to.equal(true);
                expect(timeFormat.isAnyDate()).to.equal(false);
                expect(dateTimeFormat.isAnyDate()).to.equal(true);
            });
        });

        describe('method "isAnyTime"', function () {
            it('should exist', function () {
                expect(stdFormat).to.respondTo('isAnyTime');
            });
            it('should return true for time and date/time formats only', function () {
                expect(stdFormat.isAnyTime()).to.equal(false);
                expect(percentFormat1.isAnyTime()).to.equal(false);
                expect(percentFormat2.isAnyTime()).to.equal(false);
                expect(dateFormat.isAnyTime()).to.equal(false);
                expect(timeFormat.isAnyTime()).to.equal(true);
                expect(dateTimeFormat.isAnyTime()).to.equal(true);
            });
        });

        describe('method "isAnyDateTime"', function () {
            it('should exist', function () {
                expect(stdFormat).to.respondTo('isAnyDateTime');
            });
            it('should return true for date and time formats only', function () {
                expect(stdFormat.isAnyDateTime()).to.equal(false);
                expect(percentFormat1.isAnyDateTime()).to.equal(false);
                expect(percentFormat2.isAnyDateTime()).to.equal(false);
                expect(dateFormat.isAnyDateTime()).to.equal(true);
                expect(timeFormat.isAnyDateTime()).to.equal(true);
                expect(dateTimeFormat.isAnyDateTime()).to.equal(true);
            });
        });

        describe('method "hasEqualCategory"', function () {
            it('should exist', function () {
                expect(stdFormat).to.respondTo('hasEqualCategory');
            });
            it('should return true for equal categories only', function () {
                expect(stdFormat.hasEqualCategory(stdFormat)).to.equal(true);
                expect(stdFormat.hasEqualCategory(percentFormat1)).to.equal(false);
                expect(stdFormat.hasEqualCategory(percentFormat2)).to.equal(false);
                expect(stdFormat.hasEqualCategory(dateFormat)).to.equal(false);
                expect(stdFormat.hasEqualCategory(timeFormat)).to.equal(false);
                expect(stdFormat.hasEqualCategory(dateTimeFormat)).to.equal(false);
                expect(percentFormat1.hasEqualCategory(percentFormat2)).to.equal(true);
                expect(dateFormat.hasEqualCategory(timeFormat)).to.equal(false);
                expect(timeFormat.hasEqualCategory(dateFormat)).to.equal(false);
                expect(dateFormat.hasEqualCategory(dateTimeFormat)).to.equal(false);
                expect(timeFormat.hasEqualCategory(dateTimeFormat)).to.equal(false);
            });
            it('should treat date and time categories differently', function () {
                expect(stdFormat.hasEqualCategory(dateFormat, { anyDateTime: true })).to.equal(false);
                expect(dateFormat.hasEqualCategory(timeFormat, { anyDateTime: true })).to.equal(true);
                expect(timeFormat.hasEqualCategory(dateFormat, { anyDateTime: true })).to.equal(true);
                expect(dateFormat.hasEqualCategory(dateTimeFormat, { anyDateTime: true })).to.equal(true);
                expect(timeFormat.hasEqualCategory(dateTimeFormat, { anyDateTime: true })).to.equal(true);
            });
        });
    });

    // ========================================================================
});
