/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'json!io.ox/office/tk/locale/localedata'
], function (LOCALE_DATA) {

    'use strict';

    // LocaleData JSON data ===================================================

    describe('Toolkit JSON locale data', function () {

        it('should exist', function () {
            expect(LOCALE_DATA).to.be.an('object');
            expect(LOCALE_DATA).to.not.be.empty;
        });

        // contents -----------------------------------------------------------

        describe('entry', function () {
            it('en_US should exist', function () {
                expect(LOCALE_DATA).to.have.a.property('en_US');
            });
            it('de_DE should exist', function () {
                expect(LOCALE_DATA).to.have.a.property('de_DE');
            });
        });

        describe('entries', function () {
            it('should have a valid locale code', function () {
                Object.keys(LOCALE_DATA).forEach(function (locale) {
                    expect(locale).to.match(/^[a-z]{2,}_[A-Z]{2,}$/);
                });
            });
            it('should be objects', function () {
                Object.keys(LOCALE_DATA).forEach(function (locale) {
                    expect(LOCALE_DATA).to.have.a.property(locale).that.is.an('object');
                });
            });
            it('should contain a valid MS locale identifier', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('lcid').that.is.a('number');
                    expect(data.lcid).to.be.above(0);
                });
            });
            it('should contain valid separators', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('dec').that.is.a('string');
                    expect(data.dec).to.have.length(1);
                    expect(data).to.have.a.property('group').that.is.a('string');
                    expect(data.group).to.have.length(1);
                });
            });
            it('should contain a valid writing direction', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('dir').that.is.a('string');
                    expect(data.dir).to.match(/^(ltr|rtl)$/);
                });
            });
            it('should contain a valid CJK flag', function () {
                _.each(LOCALE_DATA, function (data) {
                    if ('cjk' in data) { expect(data.cjk).to.equal(true); }
                });
            });
            it('should contain a valid measurement unit', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('unit').that.is.a('string');
                    expect(data.unit).to.match(/^(cm|mm|in)$/);
                });
            });
            it('should contain valid currency symbols', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('currency').that.is.a('string');
                    expect(data.currency).to.have.length.at.least(1);
                    expect(data).to.have.a.property('isoCurrency').that.is.a('string');
                    expect(data.isoCurrency).to.match(/^[A-Z]{3}$/);
                });
            });
            it('should contain valid date/time formats', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('shortDate').that.is.a('string');
                    expect(data.shortDate).to.have.length.at.least(1);
                    expect(data).to.have.a.property('longDate').that.is.a('string');
                    expect(data.longDate).to.have.length.at.least(1);
                    expect(data).to.have.a.property('shortTime').that.is.a('string');
                    expect(data.shortTime).to.have.length.at.least(1);
                    expect(data).to.have.a.property('longTime').that.is.a('string');
                    expect(data.longTime).to.have.length.at.least(1);
                });
            });
            it('should contain translated format code tokens', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('stdToken').that.is.a('string');
                    expect(data).to.have.a.property('dateTokens').that.is.a('string');
                    expect(data.dateTokens).to.have.length(9);
                    // TODO: ESLint 3.9.0 does not recognize backrefs https://github.com/eslint/eslint/issues/7472
                    /*eslint no-useless-escape: 0*/ expect(data.dateTokens).to.not.match(/(.).*\1/); // no duplicate characters
                });
            });
            it('should contain weekday names', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('shortWeekdays').that.is.an('array');
                    expect(data.shortWeekdays).to.have.length(7);
                    data.shortWeekdays.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                    expect(data).to.have.a.property('longWeekdays').that.is.an('array');
                    expect(data.longWeekdays).to.have.length(7);
                    data.longWeekdays.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                });
            });
            it('should contain month names', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('shortMonths').that.is.an('array');
                    expect(data.shortMonths).to.have.length(12);
                    data.shortMonths.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                    expect(data).to.have.a.property('longMonths').that.is.an('array');
                    expect(data.longMonths).to.have.length(12);
                    data.longMonths.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                });
            });
            it('should contain quarter names', function () {
                _.each(LOCALE_DATA, function (data) {
                    expect(data).to.have.a.property('shortQuarters').that.is.an('array');
                    expect(data.shortQuarters).to.have.length(4);
                    data.shortQuarters.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                    expect(data).to.have.a.property('longQuarters').that.is.an('array');
                    expect(data.longQuarters).to.have.length(4);
                    data.longQuarters.forEach(function (value) {
                        expect(value).to.be.a('string');
                        expect(value).to.have.length.at.least(1);
                    });
                });
            });
        });
    });

    // ========================================================================
});
