/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/tk/container/indexset',
    'io.ox/office/tk/container/simplemap'
], function (IndexSet, SimpleMap) {

    'use strict';

    // class IndexSet =========================================================

    describe('Toolkit class IndexSet', function () {

        it('should exist', function () {
            expect(IndexSet).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        function expectIntervals(set) {
            var intervals = [];
            for (var i = 1; i < arguments.length; i += 2) {
                intervals.push({ a: arguments[i], b: arguments[i + 1] });
            }
            expect(set._intervals).to.deep.equal(intervals);
        }

        // static methods -----------------------------------------------------

        describe('method "from"', function () {
            it('should exist', function () {
                expect(IndexSet).itself.to.respondTo('from');
            });
            it('should create an index set from an array', function () {
                var set = IndexSet.from([1, 5, 2, 1]);
                expect(set).to.be.an.instanceof(IndexSet);
                expectIntervals(set, 1, 2, 5, 5);
            });
            var OBJ = { a: 1, b: 5, c: 2, d: 1 };
            it('should create an index set from an object', function () {
                var set = IndexSet.from(OBJ);
                expectIntervals(set, 1, 2, 5, 5);
            });
            it('should create an index set from a map', function () {
                var set = IndexSet.from(SimpleMap.from(OBJ));
                expectIntervals(set, 1, 2, 5, 5);
            });
        });

        describe('method "pluck"', function () {
            it('should exist', function () {
                expect(IndexSet).itself.to.respondTo('pluck');
            });
            it('should create an index set from an array', function () {
                var set = IndexSet.pluck([{ p: 1 }, { p: 5 }, { p: 2 }, { p: 1, q: 42 }], 'p');
                expect(set).to.be.an.instanceof(IndexSet);
                expectIntervals(set, 1, 2, 5, 5);
            });
            var OBJ = { a: { p: 1 }, b: { p: 5 }, c: { p: 2 }, d: { p: 1, q: 42 } };
            it('should create an index set from an object', function () {
                var set = IndexSet.pluck(OBJ, 'p');
                expectIntervals(set, 1, 2, 5, 5);
            });
            it('should create an index set from a map', function () {
                var set = IndexSet.pluck(SimpleMap.from(OBJ), 'p');
                expectIntervals(set, 1, 2, 5, 5);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "empty"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('empty');
            });
            it('should return whether the set is empty', function () {
                var set = new IndexSet();
                expect(set.empty()).to.equal(true);
                set.insert(1);
                expect(set.empty()).to.equal(false);
            });
        });

        describe('method "clear"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('clear');
            });
            it('should clear the set', function () {
                var set = new IndexSet();
                set.insert(1);
                expect(set.empty()).to.equal(false);
                expect(set.clear()).to.equal(set);
                expect(set.empty()).to.equal(true);
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('has');
            });
            it('should return whether the index exists', function () {
                var set = new IndexSet();
                expect(set.has(1)).to.equal(false);
                set.insert(1);
                expect(set.has(0)).to.equal(false);
                expect(set.has(1)).to.equal(true);
                expect(set.has(2)).to.equal(false);
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('clone');
            });
            it('should create a clone', function () {
                var set = new IndexSet();
                set.insert(1);
                var clone = set.clone();
                expect(clone.has(1)).to.equal(true);
                set.insert(2);
                expect(clone.has(2)).to.equal(false);
            });
        });

        describe('method "insert"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('insert');
            });
            it('should insert indexes', function () {
                var set = new IndexSet();
                expect(set._intervals).to.deep.equal([]);
                expect(set.insert(4)).to.equal(4);
                expectIntervals(set, 4, 4);
                set.insert(5);
                expectIntervals(set, 4, 5);
                set.insert(4);
                expectIntervals(set, 4, 5);
                set.insert(3);
                expectIntervals(set, 3, 5);
                set.insert(1);
                expectIntervals(set, 1, 1, 3, 5);
                set.insert(9);
                expectIntervals(set, 1, 1, 3, 5, 9, 9);
                set.insert(7);
                expectIntervals(set, 1, 1, 3, 5, 7, 7, 9, 9);
                set.insert(8);
                expectIntervals(set, 1, 1, 3, 5, 7, 9);
            });
        });

        describe('method "remove"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('remove');
            });
            it('should remove indexes', function () {
                var set = IndexSet.from([1, 3, 4, 5, 7, 8, 9]);
                expectIntervals(set, 1, 1, 3, 5, 7, 9);
                expect(set.remove(1)).to.equal(1);
                expectIntervals(set, 3, 5, 7, 9);
                set.remove(1);
                expectIntervals(set, 3, 5, 7, 9);
                set.remove(3);
                expectIntervals(set, 4, 5, 7, 9);
                set.remove(5);
                expectIntervals(set, 4, 4, 7, 9);
                set.remove(8);
                expectIntervals(set, 4, 4, 7, 7, 9, 9);
            });
        });

        describe('methods "firstFree" and "reserve"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('firstFree');
                expect(IndexSet).to.respondTo('reserve');
            });
            it('should find first free index', function () {
                var set = new IndexSet();
                expect(set.firstFree(-2)).to.equal(-2);
                expect(set.firstFree(0)).to.equal(0);
                expect(set.firstFree(2)).to.equal(2);
                expect(set.firstFree()).to.equal(0);
                set.insert(1);
                set.insert(4);
                set.insert(5);
                set.insert(6);
                expect(set.firstFree(0)).to.equal(0);
                expect(set.firstFree()).to.equal(0);
                expect(set.firstFree(1)).to.equal(2);
                expect(set.firstFree(2)).to.equal(2);
                expect(set.firstFree(3)).to.equal(3);
                expect(set.firstFree(4)).to.equal(7);
                expect(set.firstFree(5)).to.equal(7);
                expect(set.firstFree(6)).to.equal(7);
                expect(set.firstFree(7)).to.equal(7);
                expect(set.firstFree(8)).to.equal(8);
                expect(set.reserve(0)).to.equal(0);
                expect(set.firstFree(0)).to.equal(2);
                expect(set.reserve()).to.equal(2);
                expect(set.firstFree(0)).to.equal(3);
                expect(set.reserve(5)).to.equal(7);
                expect(set.firstFree(0)).to.equal(3);
                expect(set.firstFree(4)).to.equal(8);
            });
        });

        describe('method "forEach"', function () {
            it('should exist', function () {
                expect(IndexSet).to.respondTo('forEach');
            });
            it('should visit all elements', function () {
                var set = new IndexSet();
                set.insert(1);
                set.insert(4);
                set.insert(5);
                set.insert(6);
                var spy = sinon.spy(), context = {};
                expect(set.forEach(spy, context)).to.equal(set);
                sinon.assert.callCount(spy, 4);
                sinon.assert.alwaysCalledOn(spy, context);
                sinon.assert.calledWithExactly(spy.getCall(0), 1, set);
                sinon.assert.calledWithExactly(spy.getCall(1), 4, set);
                sinon.assert.calledWithExactly(spy.getCall(2), 5, set);
                sinon.assert.calledWithExactly(spy.getCall(3), 6, set);
            });
        });
    });

    // ========================================================================
});
