/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, Position) {

    'use strict';

    // static class Position ==================================================

    describe('Text module Position', function () {

        it('should exist', function () {
            expect(Position).to.be.an('object');
        });

        // private helpers ----------------------------------------------------

        var paraNode1 = $('<div>').addClass('p'),
            paraNode2 = $('<div>').addClass('p'),
            span1 = $('<span>').text('Hello'),
            span2 = $('<span>').text(' World'),
            span3 = $('<span>').text(' Test'),
            span4 = $('<span>').text('Hello'),
            span5 = $('<span>').text(' World'),
            span6 = $('<span>').text('Test'),
            listLabel1 = $('<div>').addClass('helper list-label').append($('<span>').text('123.)')),
            hardBreak = $('<div>').addClass('inline hardbreak');

        paraNode1.append(span1);
        paraNode1.append(span2);
        paraNode1.append(span3);

        paraNode2.append(listLabel1);
        paraNode2.append(span4);
        paraNode2.append(span5);
        paraNode2.append(hardBreak);
        paraNode2.append(span6);

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { defaultTabStop: 1270, zoom: { value: 100 } }, page: { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 } } },
            { name: 'insertParagraph', start: [0] },
            { name: 'insertText', start: [0, 0], text: 'Hello World!' },
            { name: 'insertRange', start: [0, 2], id: 'cmt0', type: 'comment', position: 'start' },
            { name: 'insertRange', start: [0, 5], id: 'cmt0', type: 'comment', position: 'end' },
            { name: 'insertComment', start: [0, 6], author: 'a b', date: '2015-11-28T22:10:00Z', /*uid: 111, */ id: 'cmt0' },
            { name: 'insertParagraph', start: [0], target: 'cmt0' },
            { name: 'insertText', start: [0, 0], target: 'cmt0', text: 'I am inside the comment' }
        ];

        var model, rootNode;
        AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            rootNode = model.getNode();
        });

        // static methods -----------------------------------------------------

        describe('method "appendNewIndex"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('appendNewIndex');
            });
            it('should append index to logical position', function () {
                expect(Position.appendNewIndex([1, 0], 5)).to.be.an('array');
                expect(Position.appendNewIndex([1, 0], 6)).to.have.length(3);
                expect(Position.appendNewIndex([1, 0], 7)).to.deep.equal([1, 0, 7]);
            });
        });

        describe('method "increaseLastIndex"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('increaseLastIndex');
            });
            it('should increase the last index with default value', function () {
                expect(Position.increaseLastIndex([1, 0])).to.be.an('array');
                expect(Position.increaseLastIndex([1, 0])).to.have.length(2);
                expect(Position.increaseLastIndex([1, 0])).to.deep.equal([1, 1]);
            });
            it('should increase the last index with specified value', function () {
                expect(Position.increaseLastIndex([1, 0], 3)).to.be.an('array');
                expect(Position.increaseLastIndex([1, 0], 3)).to.have.length(2);
                expect(Position.increaseLastIndex([1, 0], 3)).to.deep.equal([1, 3]);
            });
        });

        describe('method "decreaseLastIndex"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('decreaseLastIndex');
            });
            it('should decrease the last index with default value', function () {
                expect(Position.decreaseLastIndex([1, 5])).to.be.an('array');
                expect(Position.decreaseLastIndex([1, 5])).to.have.length(2);
                expect(Position.decreaseLastIndex([1, 5])).to.deep.equal([1, 4]);
            });
            it('should decrease the last index with specified value', function () {
                expect(Position.decreaseLastIndex([1, 5], 3)).to.be.an('array');
                expect(Position.decreaseLastIndex([1, 5], 3)).to.have.length(2);
                expect(Position.decreaseLastIndex([1, 5], 3)).to.deep.equal([1, 2]);
            });
            it('should not decrease the last index to negative values', function () {
                expect(Position.decreaseLastIndex([1, 1], 3)).to.be.an('array');
                expect(Position.decreaseLastIndex([1, 1], 3)).to.have.length(2);
                expect(Position.decreaseLastIndex([1, 1], 3)).to.deep.equal([1, 0]);
            });
        });

        describe('method "getOxoPosition"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getOxoPosition');
            });
            it('should find the correct span position inside a paragraph', function () {
                expect(Position.getOxoPosition(paraNode1, span2)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode1, span2)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode1, span2)).to.deep.equal([5]);
                expect(Position.getOxoPosition(paraNode1, span3)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode1, span3)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode1, span3)).to.deep.equal([11]);
            });
            it('should find the correct span position inside a paragraph with offset', function () {
                expect(Position.getOxoPosition(paraNode1, span2, 3)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode1, span2, 3)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode1, span2, 3)).to.deep.equal([8]);
            });
            it('should not fail even with invalid offset inside node', function () {
                expect(Position.getOxoPosition(paraNode1, span3, 9)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode1, span3, 9)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode1, span3, 9)).to.deep.equal([20]);
            });
            it('should find the correct span position and ignore list labels', function () {
                expect(Position.getOxoPosition(paraNode2, span5, 2)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode2, span5, 2)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode2, span5, 2)).to.deep.equal([7]);
            });
            it('should find the correct span position and use length 1 for hard break nodes', function () {
                expect(Position.getOxoPosition(paraNode2, span6, 2)).to.be.an('array');
                expect(Position.getOxoPosition(paraNode2, span6, 2)).to.have.length(1);
                expect(Position.getOxoPosition(paraNode2, span6, 2)).to.deep.equal([14]);
            });
        });

        describe('method "getDOMPosition"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getDOMPosition');
            });
            it('should find the correct dom point position inside a paragraph', function () {
                expect(Position.getDOMPosition(paraNode1, [8])).to.be.an('object');
                expect(Position.getDOMPosition(paraNode1, [8])).to.have.property('offset', 3);
                expect(Position.getDOMPosition(paraNode1, [8])).to.have.property('node');
                expect(Position.getDOMPosition(paraNode1, [8]).node.nodeValue).to.equal(' World');
            });
            it('should not find a valid dom point because of big offset', function () {
                expect(Position.getDOMPosition(paraNode1, [100])).to.be.undefined;
            });
            it('should find the correct dom point including list label in paragraph', function () {
                expect(Position.getDOMPosition(paraNode2, [8])).to.be.an('object');
                expect(Position.getDOMPosition(paraNode2, [8])).to.have.property('offset', 3);
                expect(Position.getDOMPosition(paraNode2, [8])).to.have.property('node');
                expect(Position.getDOMPosition(paraNode2, [8]).node.nodeValue).to.equal(' World');
            });
            it('should find the correct hard break dom point using forcePositionCounting', function () {
                expect(Position.getDOMPosition(paraNode2, [11], true)).to.be.an('object');
                expect(Position.getDOMPosition(paraNode2, [11], true)).to.have.property('offset', 0);
                expect(Position.getDOMPosition(paraNode2, [11], true)).to.have.property('node');
                expect(Position.getDOMPosition(paraNode2, [11], true).node.className).include('hardbreak');
            });
            it('should find the correct text dom point before a hard break node in a paragraph', function () {
                expect(Position.getDOMPosition(paraNode2, [11])).to.be.an('object');
                expect(Position.getDOMPosition(paraNode2, [11])).to.have.property('offset', 6);
                expect(Position.getDOMPosition(paraNode2, [11])).to.have.property('node');
                expect(Position.getDOMPosition(paraNode2, [11]).node.nodeValue).to.equal(' World');
            });
        });

        describe('method "getWordSelection"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getWordSelection');
            });
            it('should find the correct word inside a paragraph with default boundaries', function () {
                expect(Position.getWordSelection(paraNode1, 0)).to.deep.equal({ start: 0, end: 5, text: 'Hello', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 5)).to.deep.equal({ start: 0, end: 5, text: 'Hello', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 6)).to.deep.equal({ start: 6, end: 11, text: 'World', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 16)).to.deep.equal({ start: 12, end: 16, text: 'Test', node: null, nodes: null });
            });
            it('should find the correct word inside a paragraph with user specified boundaries', function () {
                expect(Position.getWordSelection(paraNode1, 1, ['n', 'o', 'p'])).to.deep.equal({ start: 0, end: 4, text: 'Hell', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 5, ['n', 'o', 'p'])).to.deep.equal({ start: 5, end: 7, text: ' W', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 1, ['x', 'y', 'z'])).to.deep.equal({ start: 0, end: 16, text: 'Hello World Test', node: null, nodes: null });
            });
            it('should not find the searched word in the paragraph', function () {
                expect(Position.getWordSelection(paraNode1, 17)).to.equal(null);
            });
            it('should evalute option "addFinalSpaces" correctly', function () {
                expect(Position.getWordSelection(paraNode1, 8, undefined, { addFinalSpaces: true })).to.deep.equal({ start: 6, end: 12, text: 'World ', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 14, undefined, { addFinalSpaces: true })).to.deep.equal({ start: 12, end: 16, text: 'Test', node: null, nodes: null });
            });
            it('should evalute option "ignoreText" correctly', function () {
                expect(Position.getWordSelection(paraNode1, 8, undefined, { ignoreText: true })).to.deep.equal({ start: 6, end: 11, text: '', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 14, undefined, { ignoreText: true })).to.deep.equal({ start: 12, end: 16, text: '', node: null, nodes: null });
            });
            it('should evalute option "onlyLeft" correctly', function () {
                expect(Position.getWordSelection(paraNode1, 8, undefined, { onlyLeft: true })).to.deep.equal({ start: 6, end: 8, text: 'Wo', node: null, nodes: null });
                expect(Position.getWordSelection(paraNode1, 14, undefined, { onlyLeft: true })).to.deep.equal({ start: 12, end: 14, text: 'Te', node: null, nodes: null });
            });
        });

        describe('method "getParagraphElement"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getParagraphElement');
            });
            it('should find a paragraph node', function () {
                expect(Position.getParagraphElement(rootNode, [0])).to.not.be.null;
                expect(Position.getParagraphElement(rootNode, [0])).to.be.a.instanceof(Object);
                expect(Position.getParagraphElement(rootNode, [0])).to.be.an('htmldivelement');
                expect(Position.getParagraphElement(rootNode, [0])).to.have.property('nodeName');
                expect(Position.getParagraphElement(rootNode, [0]).nodeName).to.equal('DIV');
                expect(Position.getParagraphElement(rootNode, [0])).to.have.property('className');
                expect($(Position.getParagraphElement(rootNode, [0])).hasClass('p')).to.equal(true);
            });
        });

        describe('method "getParagraphLength"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getParagraphLength');
            });
            it('should return the paragraph length of paragraph specified by logical position', function () {
                expect(Position.getParagraphLength(rootNode, [0])).to.equal(15);
            });
        });

        describe('method "getParagraphNodeLength"', function () {
            it('should exist', function () {
                expect(Position).itself.to.respondTo('getParagraphNodeLength');
            });
            it('should return the paragraph length of paragraph specified by paragraph node', function () {
                expect(Position.getParagraphNodeLength(Position.getParagraphElement(rootNode, [0]))).to.equal(15);
            });
        });

    });

    // ========================================================================
});
