/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/range',
    'io.ox/office/spreadsheet/utils/movedescriptor'
], function (SheetHelper, Range, MoveDescriptor) {

    'use strict';

    // class MoveDescriptor ===================================================

    describe('Spreadsheet class MoveDescriptor', function () {

        it('should exist', function () {
            expect(MoveDescriptor).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;
        var a = SheetHelper.a;
        var r = SheetHelper.r;
        var ia = SheetHelper.ia;
        var aa = SheetHelper.aa;
        var ra = SheetHelper.ra;

        var docModel = { getMaxIndex: function (columns) { return columns ? 255 : 65535; } };

        var insColMoveDesc = new MoveDescriptor(docModel, i('2:3'), ia('B:C E:F'), true, true);
        var insRowMoveDesc = new MoveDescriptor(docModel, i('B:C'), ia('2:3 5:6'), false, true);
        var delColMoveDesc = new MoveDescriptor(docModel, i('2:3'), ia('B:C E:F'), true, false);
        var delRowMoveDesc = new MoveDescriptor(docModel, i('B:C'), ia('2:3 5:6'), false, false);

        function adjustRow(address) {
            if (address[1] > 128) { address[1] += (65536 - 256); }
            return address;
        }

        function adjustRows(range) {
            adjustRow(range.start);
            adjustRow(range.end);
            return range;
        }

        function expectAddressTransformation(address, insert, expected) {

            address = a(address);
            expected = expected ? a(expected) : null;
            var colMoveDesc = insert ? insColMoveDesc : delColMoveDesc;
            expect(colMoveDesc.transformAddress(address)).to.deep.equal(expected);

            address = adjustRow(address.mirror());
            expected = expected ? adjustRow(expected.mirror()) : null;
            var rowMoveDesc = insert ? insRowMoveDesc : delRowMoveDesc;
            expect(rowMoveDesc.transformAddress(address)).to.deep.equal(expected);
        }

        function expectRangeTransformation(range, options, insert, expected) {

            range = r(range);
            expected = expected ? r(expected) : null;
            var colMoveDesc = insert ? insColMoveDesc : delColMoveDesc;
            expect(colMoveDesc.transformRange(range, options)).to.deep.equal(expected);

            range = adjustRows(range.mirror());
            expected = expected ? adjustRows(expected.mirror()) : null;
            var rowMoveDesc = insert ? insRowMoveDesc : delRowMoveDesc;
            expect(rowMoveDesc.transformRange(range, options)).to.deep.equal(expected);
        }

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a move descriptor for inserting a single column index at the beginning of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), i('A:A'), true, true);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('A:A'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('A:IU'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('B:IV'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('IV:IV'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(true);
                expect(moveDesc.maxIndex).to.equal(255);
            });
            it('should create a move descriptor for inserting a single column index at the end of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), i('IV:IV'), true, true);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('IV:IV'));
                expect(moveDesc.moveFromIntervals).to.be.empty;
                expect(moveDesc.moveToIntervals).to.be.empty;
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('IV:IV'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('IV1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(true);
                expect(moveDesc.maxIndex).to.equal(255);
            });
            it('should create a move descriptor for inserting multiple columns', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), ia('F:I B:C C:D C:C IK:IR IT:IU'), true, true);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('B:C E:F J:M IS:IV'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('B:B C:E F:IJ'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('D:D G:I N:IR'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('IK:IV'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('B1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(true);
                expect(moveDesc.maxIndex).to.equal(255);
            });

            it('should create a move descriptor for inserting a single row index at the beginning of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('A:C'), i('1:1'), false, true);
                expect(moveDesc.bandInterval).to.deep.equal(i('A:C'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('1:1'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('1:65535'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('2:65536'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('65536:65536'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A1:C65536'));
                expect(moveDesc.columns).to.equal(false);
                expect(moveDesc.insert).to.equal(true);
                expect(moveDesc.maxIndex).to.equal(65535);
            });
            it('should create a move descriptor for inserting a single row index at the end of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('A:C'), i('65536:65536'), false, true);
                expect(moveDesc.bandInterval).to.deep.equal(i('A:C'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('65536:65536'));
                expect(moveDesc.moveFromIntervals).to.be.empty;
                expect(moveDesc.moveToIntervals).to.be.empty;
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('65536:65536'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A65536:C65536'));
                expect(moveDesc.columns).to.equal(false);
                expect(moveDesc.insert).to.equal(true);
                expect(moveDesc.maxIndex).to.equal(65535);
            });

            it('should create a move descriptor for deleting a single column index at the beginning of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), i('A:A'), true, false);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('A:A'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('B:IV'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('A:IU'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('A:A'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(false);
                expect(moveDesc.maxIndex).to.equal(255);
            });
            it('should create a move descriptor for deleting a single column index at the end of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), i('IV:IV'), true, false);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('IV:IV'));
                expect(moveDesc.moveFromIntervals).to.be.empty;
                expect(moveDesc.moveToIntervals).to.be.empty;
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('IV:IV'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('IV1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(false);
                expect(moveDesc.maxIndex).to.equal(255);
            });
            it('should create a move descriptor for deleting multiple columns', function () {
                var moveDesc = new MoveDescriptor(docModel, i('1:3'), ia('F:I B:C C:D C:C IK:IR IU:IV'), true, false);
                expect(moveDesc.bandInterval).to.deep.equal(i('1:3'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('B:D F:I IK:IR IU:IV'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('E:E J:IJ IS:IT'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('B:B C:IC ID:IE'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('B:D F:I IK:IR IU:IV'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('B1:IV3'));
                expect(moveDesc.columns).to.equal(true);
                expect(moveDesc.insert).to.equal(false);
                expect(moveDesc.maxIndex).to.equal(255);
            });

            it('should create a move descriptor for deleting a single row index at the beginning of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('A:C'), i('1:1'), false, false);
                expect(moveDesc.bandInterval).to.deep.equal(i('A:C'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('1:1'));
                expect(moveDesc.moveFromIntervals).to.deep.equal(ia('2:65536'));
                expect(moveDesc.moveToIntervals).to.deep.equal(ia('1:65535'));
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('1:1'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A1:C65536'));
                expect(moveDesc.columns).to.equal(false);
                expect(moveDesc.insert).to.equal(false);
                expect(moveDesc.maxIndex).to.equal(65535);
            });
            it('should create a move descriptor for deleting a single row index at the end of the sheet', function () {
                var moveDesc = new MoveDescriptor(docModel, i('A:C'), i('65536:65536'), false, false);
                expect(moveDesc.bandInterval).to.deep.equal(i('A:C'));
                expect(moveDesc.targetIntervals).to.deep.equal(ia('65536:65536'));
                expect(moveDesc.moveFromIntervals).to.be.empty;
                expect(moveDesc.moveToIntervals).to.be.empty;
                expect(moveDesc.deleteIntervals).to.deep.equal(ia('65536:65536'));
                expect(moveDesc.dirtyRange).to.deep.equal(r('A65536:C65536'));
                expect(moveDesc.columns).to.equal(false);
                expect(moveDesc.insert).to.equal(false);
                expect(moveDesc.maxIndex).to.equal(65535);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "containsAddress"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('containsAddress');
            });
            it('should return whether the address is covered', function () {
                expect(insColMoveDesc.containsAddress(a('A1'))).to.equal(false);
                expect(insColMoveDesc.containsAddress(a('A2'))).to.equal(true);
                expect(insColMoveDesc.containsAddress(a('A3'))).to.equal(true);
                expect(insColMoveDesc.containsAddress(a('A4'))).to.equal(false);

                expect(insRowMoveDesc.containsAddress(a('A1'))).to.equal(false);
                expect(insRowMoveDesc.containsAddress(a('B1'))).to.equal(true);
                expect(insRowMoveDesc.containsAddress(a('C1'))).to.equal(true);
                expect(insRowMoveDesc.containsAddress(a('D1'))).to.equal(false);
            });
        });

        describe('method "containsRange"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('containsRange');
            });
            it('should return whether the range is covered', function () {
                expect(insColMoveDesc.containsRange(r('A1:A4'))).to.equal(false);
                expect(insColMoveDesc.containsRange(r('A1:A2'))).to.equal(false);
                expect(insColMoveDesc.containsRange(r('A2:A3'))).to.equal(true);
                expect(insColMoveDesc.containsRange(r('A3:A4'))).to.equal(false);
                expect(insColMoveDesc.containsRange(r('A1:A1'))).to.equal(false);
                expect(insColMoveDesc.containsRange(r('A2:A2'))).to.equal(true);
                expect(insColMoveDesc.containsRange(r('A3:A3'))).to.equal(true);
                expect(insColMoveDesc.containsRange(r('A4:A4'))).to.equal(false);

                expect(insRowMoveDesc.containsRange(r('A1:D1'))).to.equal(false);
                expect(insRowMoveDesc.containsRange(r('A1:B1'))).to.equal(false);
                expect(insRowMoveDesc.containsRange(r('B1:C1'))).to.equal(true);
                expect(insRowMoveDesc.containsRange(r('C1:D1'))).to.equal(false);
                expect(insRowMoveDesc.containsRange(r('A1:A1'))).to.equal(false);
                expect(insRowMoveDesc.containsRange(r('B1:B1'))).to.equal(true);
                expect(insRowMoveDesc.containsRange(r('C1:C1'))).to.equal(true);
                expect(insRowMoveDesc.containsRange(r('D1:D1'))).to.equal(false);
            });
        });

        describe('method "createLineRange"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('createLineRange');
            });
            it('should create a range address for columns', function () {
                expect(insColMoveDesc.createLineRange(0)).to.deep.equal(r('A2:A3'));
                expect(insColMoveDesc.createLineRange(8)).to.deep.equal(r('I2:I3'));
            });
            it('should create a range address for rows', function () {
                expect(insRowMoveDesc.createLineRange(0)).to.deep.equal(r('B1:C1'));
                expect(insRowMoveDesc.createLineRange(8)).to.deep.equal(r('B9:C9'));
            });
        });

        describe('method "createRange"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('createRange');
            });
            it('should create a range address for columns', function () {
                expect(insColMoveDesc.createRange(i('A:B'))).to.deep.equal(r('A2:B3'));
                expect(insColMoveDesc.createRange(i('H:I'))).to.deep.equal(r('H2:I3'));
            });
            it('should create a range address for rows', function () {
                expect(insRowMoveDesc.createRange(i('1:2'))).to.deep.equal(r('B1:C2'));
                expect(insRowMoveDesc.createRange(i('8:9'))).to.deep.equal(r('B8:C9'));
            });
        });

        describe('method "createRanges"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('createRanges');
            });
            it('should create an array of range addresses for columns', function () {
                expect(insColMoveDesc.createRanges(ia('A:B C:D'))).to.deep.equal(ra('A2:B3 C2:D3'));
                expect(insColMoveDesc.createRanges(i('H:I'))).to.deep.equal(ra('H2:I3'));
            });
            it('should create an array of range addresses for rows', function () {
                expect(insRowMoveDesc.createRanges(ia('1:2 3:4'))).to.deep.equal(ra('B1:C2 B3:C4'));
                expect(insRowMoveDesc.createRanges(i('8:9'))).to.deep.equal(ra('B8:C9'));
            });
        });

        describe('method "transformAddress"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('transformAddress');
            });
            it('should transform address when inserting columns or rows', function () {
                expectAddressTransformation('A1', true, 'A1');
                expectAddressTransformation('G1', true, 'G1');
                expectAddressTransformation('A2', true, 'A2');
                expectAddressTransformation('B2', true, 'D2');
                expectAddressTransformation('C2', true, 'E2');
                expectAddressTransformation('D2', true, 'F2');
                expectAddressTransformation('E2', true, 'I2');
                expectAddressTransformation('F2', true, 'J2');
                expectAddressTransformation('G2', true, 'K2');
                expectAddressTransformation('IR2', true, 'IV2');
                expectAddressTransformation('IS2', true, null);
                expectAddressTransformation('IR3', true, 'IV3');
                expectAddressTransformation('IR4', true, 'IR4');
            });
            it('should transform address when deleting columns or rows', function () {
                expectAddressTransformation('A1', false, 'A1');
                expectAddressTransformation('G1', false, 'G1');
                expectAddressTransformation('A2', false, 'A2');
                expectAddressTransformation('B2', false, null);
                expectAddressTransformation('C2', false, null);
                expectAddressTransformation('D2', false, 'B2');
                expectAddressTransformation('E2', false, null);
                expectAddressTransformation('F2', false, null);
                expectAddressTransformation('G2', false, 'C2');
                expectAddressTransformation('IV2', false, 'IR2');
                expectAddressTransformation('IV3', false, 'IR3');
                expectAddressTransformation('IV4', false, 'IV4');
            });
        });

        describe('method "transformAddresses"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('transformAddresses');
            });
            it('should transform addresses when inserting columns', function () {
                expect(insColMoveDesc.transformAddresses(aa('A1 IS2 G2 B2 A2'))).to.deep.equal(aa('A1 K2 D2 A2'));
                expect(insColMoveDesc.transformAddresses(a('G2'))).to.deep.equal(aa('K2'));
                expect(insColMoveDesc.transformAddresses(a('IS2'))).to.deep.equal(aa());
            });
            it('should transform addresses when deleting columns', function () {
                expect(delColMoveDesc.transformAddresses(aa('A1 G2 D2 B2 A2'))).to.deep.equal(aa('A1 C2 B2 A2'));
                expect(delColMoveDesc.transformAddresses(a('G2'))).to.deep.equal(aa('C2'));
                expect(delColMoveDesc.transformAddresses(a('B2'))).to.deep.equal(aa());
            });
            it('should transform addresses when inserting rows', function () {
                expect(insRowMoveDesc.transformAddresses(aa('A1 B65533 B7 B2 B1'))).to.deep.equal(aa('A1 B11 B4 B1'));
                expect(insRowMoveDesc.transformAddresses(a('B7'))).to.deep.equal(aa('B11'));
                expect(insRowMoveDesc.transformAddresses(a('B65533'))).to.deep.equal(aa());
            });
            it('should transform addresses when deleting rows', function () {
                expect(delRowMoveDesc.transformAddresses(aa('A1 B7 B4 B2 B1'))).to.deep.equal(aa('A1 B3 B2 B1'));
                expect(delRowMoveDesc.transformAddresses(a('B7'))).to.deep.equal(aa('B3'));
                expect(delRowMoveDesc.transformAddresses(a('B2'))).to.deep.equal(aa());
            });
        });

        describe('method "transformRange"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('transformRange');
            });
            it('should transform range address when inserting columns or rows', function () {
                expectRangeTransformation('A1:G1', null, true, 'A1:G1');
                expectRangeTransformation('A1:G2', null, true, 'A1:G2');
                expectRangeTransformation('A1:G4', null, true, 'A1:G4');
                expectRangeTransformation('A3:G4', null, true, 'A3:G4');

                expectRangeTransformation('A2:A3', null, true, 'A2:A3');
                expectRangeTransformation('A2:B3', null, true, 'A2:D3');
                expectRangeTransformation('A2:C3', null, true, 'A2:E3');
                expectRangeTransformation('A2:D3', null, true, 'A2:F3');
                expectRangeTransformation('A2:E3', null, true, 'A2:I3');
                expectRangeTransformation('A2:F3', null, true, 'A2:J3');
                expectRangeTransformation('A2:G3', null, true, 'A2:K3');

                expectRangeTransformation('B2:B3', null, true, 'D2:D3');
                expectRangeTransformation('B2:C3', null, true, 'D2:E3');
                expectRangeTransformation('B2:D3', null, true, 'D2:F3');
                expectRangeTransformation('B2:E3', null, true, 'D2:I3');
                expectRangeTransformation('B2:F3', null, true, 'D2:J3');
                expectRangeTransformation('B2:G3', null, true, 'D2:K3');

                expectRangeTransformation('C2:C3', null, true, 'E2:E3');
                expectRangeTransformation('C2:D3', null, true, 'E2:F3');
                expectRangeTransformation('C2:E3', null, true, 'E2:I3');
                expectRangeTransformation('C2:F3', null, true, 'E2:J3');
                expectRangeTransformation('C2:G3', null, true, 'E2:K3');

                expectRangeTransformation('D2:D3', null, true, 'F2:F3');
                expectRangeTransformation('D2:G3', null, true, 'F2:K3');
                expectRangeTransformation('E2:E3', null, true, 'I2:I3');
                expectRangeTransformation('E2:G3', null, true, 'I2:K3');
                expectRangeTransformation('F2:F3', null, true, 'J2:J3');
                expectRangeTransformation('F2:G3', null, true, 'J2:K3');
                expectRangeTransformation('G2:G3', null, true, 'K2:K3');

                expectRangeTransformation('IP2:IR3', null, true, 'IT2:IV3');
                expectRangeTransformation('IQ2:IS3', null, true, 'IU2:IV3');
                expectRangeTransformation('IR2:IT3', null, true, 'IV2:IV3');
                expectRangeTransformation('IS2:IU3', null, true, null);

                expectRangeTransformation('B2:C3', { expandEnd: true }, true, 'D2:E3');
                expectRangeTransformation('B2:D3', { expandEnd: true }, true, 'D2:H3');
            });
            it('should transform range address when deleting columns or rows', function () {
                expectRangeTransformation('A1:G1', null, false, 'A1:G1');
                expectRangeTransformation('A1:G2', null, false, 'A1:G2');
                expectRangeTransformation('A1:G4', null, false, 'A1:G4');
                expectRangeTransformation('A3:G4', null, false, 'A3:G4');

                expectRangeTransformation('A2:A3', null, false, 'A2:A3');
                expectRangeTransformation('A2:B3', null, false, 'A2:A3');
                expectRangeTransformation('A2:C3', null, false, 'A2:A3');
                expectRangeTransformation('A2:D3', null, false, 'A2:B3');
                expectRangeTransformation('A2:E3', null, false, 'A2:B3');
                expectRangeTransformation('A2:F3', null, false, 'A2:B3');
                expectRangeTransformation('A2:G3', null, false, 'A2:C3');
                expectRangeTransformation('A2:H3', null, false, 'A2:D3');

                expectRangeTransformation('B2:B3', null, false, null);
                expectRangeTransformation('B2:C3', null, false, null);
                expectRangeTransformation('B2:D3', null, false, 'B2:B3');
                expectRangeTransformation('B2:E3', null, false, 'B2:B3');
                expectRangeTransformation('B2:F3', null, false, 'B2:B3');
                expectRangeTransformation('B2:G3', null, false, 'B2:C3');
                expectRangeTransformation('B2:H3', null, false, 'B2:D3');

                expectRangeTransformation('C2:C3', null, false, null);
                expectRangeTransformation('C2:D3', null, false, 'B2:B3');
                expectRangeTransformation('C2:E3', null, false, 'B2:B3');
                expectRangeTransformation('C2:F3', null, false, 'B2:B3');
                expectRangeTransformation('C2:G3', null, false, 'B2:C3');
                expectRangeTransformation('C2:H3', null, false, 'B2:D3');

                expectRangeTransformation('D2:D3', null, false, 'B2:B3');
                expectRangeTransformation('D2:E3', null, false, 'B2:B3');
                expectRangeTransformation('D2:F3', null, false, 'B2:B3');
                expectRangeTransformation('D2:G3', null, false, 'B2:C3');
                expectRangeTransformation('D2:H3', null, false, 'B2:D3');

                expectRangeTransformation('E2:E3', null, false, null);
                expectRangeTransformation('E2:F3', null, false, null);
                expectRangeTransformation('E2:G3', null, false, 'C2:C3');
                expectRangeTransformation('E2:H3', null, false, 'C2:D3');

                expectRangeTransformation('F2:F3', null, false, null);
                expectRangeTransformation('F2:G3', null, false, 'C2:C3');
                expectRangeTransformation('F2:H3', null, false, 'C2:D3');

                expectRangeTransformation('G2:G3', null, false, 'C2:C3');
                expectRangeTransformation('G2:H3', null, false, 'C2:D3');
            });
        });

        describe('method "transformRanges"', function () {
            it('should exist', function () {
                expect(MoveDescriptor).to.respondTo('transformRanges');
            });
            it('should transform ranges when inserting columns', function () {
                expect(insColMoveDesc.transformRanges(ra('A2:B3 IS2:IU3 A1:G4'))).to.deep.equal(ra('A2:D3 A1:G4'));
                expect(insColMoveDesc.transformRanges(r('A2:B3'))).to.deep.equal(ra('A2:D3'));
                expect(insColMoveDesc.transformRanges(r('IS2:IU3'))).to.deep.equal(ra());
            });
            it('should transform ranges when deleting columns', function () {
                expect(delColMoveDesc.transformRanges(ra('A2:B3 B2:C3 A1:G4'))).to.deep.equal(ra('A2:A3 A1:G4'));
                expect(delColMoveDesc.transformRanges(r('A2:B3'))).to.deep.equal(ra('A2:A3'));
                expect(delColMoveDesc.transformRanges(r('B2:C3'))).to.deep.equal(ra());
            });
            it('should transform ranges when inserting rows', function () {
                expect(insRowMoveDesc.transformRanges(ra('B1:C2 B65533:C65535 A1:D7'))).to.deep.equal(ra('B1:C4 A1:D7'));
                expect(insRowMoveDesc.transformRanges(r('B1:C2'))).to.deep.equal(ra('B1:C4'));
                expect(insRowMoveDesc.transformRanges(r('B65533:C65535'))).to.deep.equal(ra());
            });
            it('should transform ranges when deleting rows', function () {
                expect(delRowMoveDesc.transformRanges(ra('B1:C2 B2:C3 A1:D7'))).to.deep.equal(ra('B1:C1 A1:D7'));
                expect(delRowMoveDesc.transformRanges(r('B1:C2'))).to.deep.equal(ra('B1:C1'));
                expect(delRowMoveDesc.transformRanges(r('B2:C3'))).to.deep.equal(ra());
            });
        });
    });

    // ========================================================================
});
