/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/interval'
], function (SheetHelper, Interval) {

    'use strict';

    // class Interval =========================================================

    describe('Spreadsheet class Interval', function () {

        it('should exist', function () {
            expect(Interval).to.be.a('function');
        });
        it('should be extendable', function () {
            expect(Interval).itself.to.respondTo('extend');
        });

        // private helpers ----------------------------------------------------

        var i = SheetHelper.i;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create an interval', function () {
                var i1 = new Interval(1, 3);
                expect(i1).to.have.property('first', 1);
                expect(i1).to.have.property('last', 3);
            });
            it('should create an interval from a single parameter', function () {
                var i1 = new Interval(1);
                expect(i1).to.have.property('first', 1);
                expect(i1).to.have.property('last', 1);
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(Interval).itself.to.respondTo('create');
            });
            var i1 = new Interval(0, 1);
            it('should return the adjusted interval', function () {
                expect(Interval.create(0, 1)).to.deep.equal(i1);
                expect(Interval.create(1, 0)).to.deep.equal(i1);
            });
        });

        describe('method "parseAsCols"', function () {
            it('should exist', function () {
                expect(Interval).itself.to.respondTo('parseAsCols');
            });
            it('should return the correct column interval', function () {
                expect(Interval.parseAsCols('A:A')).to.deep.equal(new Interval(0, 0));
                expect(Interval.parseAsCols('A:B')).to.deep.equal(new Interval(0, 1));
                expect(Interval.parseAsCols('A:C')).to.deep.equal(new Interval(0, 2));
                expect(Interval.parseAsCols('B:C')).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsCols('C:C')).to.deep.equal(new Interval(2, 2));
                expect(Interval.parseAsCols('AA:ZZ')).to.deep.equal(new Interval(26, 701));
                expect(Interval.parseAsCols('ZZZZ:ZZZZ')).to.deep.equal(new Interval(475253, 475253));
            });
            it('should return the adjusted column interval', function () {
                expect(Interval.parseAsCols('C:A')).to.deep.equal(new Interval(0, 2));
            });
            it('should accept lower-case columns', function () {
                expect(Interval.parseAsCols('aA:Aa')).to.deep.equal(new Interval(26, 26));
                expect(Interval.parseAsCols('aa:AA')).to.deep.equal(new Interval(26, 26));
            });
            it('should return null for invalid intervals', function () {
                expect(Interval.parseAsCols('')).to.equal(null);
                expect(Interval.parseAsCols('A')).to.equal(null);
                expect(Interval.parseAsCols('1:1')).to.equal(null);
                expect(Interval.parseAsCols('$')).to.equal(null);
                expect(Interval.parseAsCols('$A:$A')).to.equal(null);
                expect(Interval.parseAsCols('AAAAA:AAAAA')).to.equal(null);
            });
            it('should return null if custom maximum does not fit', function () {
                expect(Interval.parseAsCols('B:C', 2)).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsCols('C:B', 2)).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsCols('B:C', 1)).to.equal(null);
                expect(Interval.parseAsCols('C:B', 1)).to.equal(null);
            });
        });

        describe('method "parseAsRows"', function () {
            it('should exist', function () {
                expect(Interval).itself.to.respondTo('parseAsRows');
            });
            it('should return the correct column interval', function () {
                expect(Interval.parseAsRows('1:1')).to.deep.equal(new Interval(0, 0));
                expect(Interval.parseAsRows('1:2')).to.deep.equal(new Interval(0, 1));
                expect(Interval.parseAsRows('1:3')).to.deep.equal(new Interval(0, 2));
                expect(Interval.parseAsRows('2:3')).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsRows('3:3')).to.deep.equal(new Interval(2, 2));
                expect(Interval.parseAsRows('10:999')).to.deep.equal(new Interval(9, 998));
                expect(Interval.parseAsRows('99999999:99999999')).to.deep.equal(new Interval(99999998, 99999998));
            });
            it('should return the adjusted row interval', function () {
                expect(Interval.parseAsRows('3:1')).to.deep.equal(new Interval(0, 2));
            });
            it('should accept leading zeros', function () {
                expect(Interval.parseAsRows('01:000999')).to.deep.equal(new Interval(0, 998));
            });
            it('should return null for invalid intervals', function () {
                expect(Interval.parseAsRows('')).to.equal(null);
                expect(Interval.parseAsRows('1')).to.equal(null);
                expect(Interval.parseAsRows('A:A')).to.equal(null);
                expect(Interval.parseAsRows('$')).to.equal(null);
                expect(Interval.parseAsRows('$1:$1')).to.equal(null);
                expect(Interval.parseAsRows('100000000:100000000')).to.equal(null);
            });
            it('should return null if custom maximum does not fit', function () {
                expect(Interval.parseAsRows('2:3', 2)).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsRows('3:2', 2)).to.deep.equal(new Interval(1, 2));
                expect(Interval.parseAsRows('2:3', 1)).to.equal(null);
                expect(Interval.parseAsRows('3:2', 1)).to.equal(null);
            });
        });

        describe('method "parseAs"', function () {
            it('should exist', function () {
                expect(Interval).itself.to.respondTo('parseAs');
            });
            it('should return the correct column interval', function () {
                expect(Interval.parseAs('A:C', true)).to.deep.equal(new Interval(0, 2));
                expect(Interval.parseAs('C:A', true)).to.deep.equal(new Interval(0, 2));
            });
            it('should return the correct row interval', function () {
                expect(Interval.parseAs('1:3', false)).to.deep.equal(new Interval(0, 2));
                expect(Interval.parseAs('3:1', false)).to.deep.equal(new Interval(0, 2));
            });
            it('should return null if custom maximum does not fit', function () {
                expect(Interval.parseAs('B:C', true, 1)).to.equal(null);
                expect(Interval.parseAs('2:3', false, 1)).to.equal(null);
            });
        });

        describe('method "compare"', function () {
            var i1 = i('3:5');
            it('should exist', function () {
                expect(Interval).itself.to.respondTo('compare');
            });
            it('should return 0 for equal intervals', function () {
                expect(Interval.compare(i1, i1)).to.equal(0);
            });
            it('should return negative value for intervals in order', function () {
                expect(Interval.compare(i1, i('4:4'))).to.be.below(0);
                expect(Interval.compare(i1, i('4:5'))).to.be.below(0);
                expect(Interval.compare(i1, i('4:6'))).to.be.below(0);
                expect(Interval.compare(i1, i('3:6'))).to.be.below(0);
            });
            it('should return positive value for intervals in reversed order', function () {
                expect(Interval.compare(i1, i('2:4'))).to.be.above(0);
                expect(Interval.compare(i1, i('2:5'))).to.be.above(0);
                expect(Interval.compare(i1, i('2:6'))).to.be.above(0);
                expect(Interval.compare(i1, i('3:4'))).to.be.above(0);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "key"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('key');
            });
            it('should return unique key for intervals', function () {
                expect(i('1:1').key()).to.equal('0:0');
                expect(i('2:3').key()).to.equal('1:2');
            });
        });

        describe('method "clone"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('clone');
            });
            it('should return a clone', function () {
                var i1 = i('2:4'), i2 = i1.clone();
                expect(i2).to.be.an.instanceof(Interval);
                expect(i2).to.not.equal(i1);
                expect(i2).to.deep.equal(i1);
            });
        });

        describe('method "equals"', function () {
            var i1 = i('2:4');
            it('should exist', function () {
                expect(Interval).to.respondTo('equals');
            });
            it('should return true for equal intervals', function () {
                expect(i1.equals(i1)).to.equal(true);
                expect(i1.equals(i1.clone())).to.equal(true);
            });
            it('should return false for different intervals', function () {
                expect(i1.equals(i('3:4'))).to.equal(false);
                expect(i1.equals(i('2:3'))).to.equal(false);
                expect(i1.equals(i('3:3'))).to.equal(false);
            });
        });

        describe('method "differs"', function () {
            var i1 = i('2:4');
            it('should exist', function () {
                expect(Interval).to.respondTo('differs');
            });
            it('should return false for equal intervals', function () {
                expect(i1.differs(i1)).to.equal(false);
                expect(i1.differs(i1.clone())).to.equal(false);
            });
            it('should return true for different intervals', function () {
                expect(i1.differs(i('3:4'))).to.equal(true);
                expect(i1.differs(i('2:3'))).to.equal(true);
                expect(i1.differs(i('3:3'))).to.equal(true);
            });
        });

        describe('method "compareTo"', function () {
            var i1 = i('3:5');
            it('should exist', function () {
                expect(Interval).to.respondTo('compareTo');
            });
            it('should return 0 for equal intervals', function () {
                expect(i1.compareTo(i1)).to.equal(0);
                expect(i('1:3').compareTo(i('1:3'))).to.equal(0);
            });
            it('should return negative value for intervals in order', function () {
                expect(i1.compareTo(i('4:4'))).to.be.below(0);
                expect(i1.compareTo(i('4:5'))).to.be.below(0);
                expect(i1.compareTo(i('4:6'))).to.be.below(0);
                expect(i1.compareTo(i('3:6'))).to.be.below(0);
            });
            it('should return positive value for intervals in reversed order', function () {
                expect(i1.compareTo(i('2:4'))).to.be.above(0);
                expect(i1.compareTo(i('2:5'))).to.be.above(0);
                expect(i1.compareTo(i('2:6'))).to.be.above(0);
                expect(i1.compareTo(i('3:4'))).to.be.above(0);
            });
        });

        describe('method "size"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('size');
            });
            it('should return the correct interval size', function () {
                expect(i('1:1').size()).to.equal(1);
                expect(i('1:2').size()).to.equal(2);
                expect(i('1:7').size()).to.equal(7);
                expect(i('1:8').size()).to.equal(8);
                expect(i('2:8').size()).to.equal(7);
                expect(i('7:8').size()).to.equal(2);
                expect(i('8:8').size()).to.equal(1);
            });
        });

        describe('method "single"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('single');
            });
            it('should return the correct result', function () {
                expect(i('1:1').single()).to.equal(true);
                expect(i('3:3').single()).to.equal(true);
                expect(i('3:4').single()).to.equal(false);
            });
        });

        describe('method "containsIndex"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('containsIndex');
            });
            var i1 = i('3:5');
            it('should return false for indexes outside the interval', function () {
                expect(i1.containsIndex(0)).to.equal(false);
                expect(i1.containsIndex(1)).to.equal(false);
                expect(i1.containsIndex(5)).to.equal(false);
                expect(i1.containsIndex(6)).to.equal(false);
            });
            it('should return true for indexes inside the interval', function () {
                expect(i1.containsIndex(2)).to.equal(true);
                expect(i1.containsIndex(3)).to.equal(true);
                expect(i1.containsIndex(4)).to.equal(true);
            });
        });

        describe('method "contains"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('contains');
            });
            var i1 = i('3:5');
            it('should return false for distinct intervals', function () {
                expect(i1.contains(i('1:2'))).to.equal(false);
                expect(i1.contains(i('6:7'))).to.equal(false);
                expect(i1.contains(i('8:9'))).to.equal(false);
                expect(i('1:2').contains(i1)).to.equal(false);
                expect(i('6:7').contains(i1)).to.equal(false);
                expect(i('8:9').contains(i1)).to.equal(false);
            });
            it('should return false for partly overlapping intervals', function () {
                expect(i1.contains(i('1:3'))).to.equal(false);
                expect(i1.contains(i('2:4'))).to.equal(false);
                expect(i1.contains(i('4:6'))).to.equal(false);
                expect(i1.contains(i('5:7'))).to.equal(false);
            });
            it('should return true for intervals containing the other interval', function () {
                expect(i1.contains(i('3:3'))).to.equal(true);
                expect(i1.contains(i('3:4'))).to.equal(true);
                expect(i1.contains(i('3:5'))).to.equal(true);
                expect(i1.contains(i('4:5'))).to.equal(true);
                expect(i1.contains(i('5:5'))).to.equal(true);
            });
            it('should return false for intervals contained by the other interval', function () {
                expect(i('3:3').contains(i1)).to.equal(false);
                expect(i('3:4').contains(i1)).to.equal(false);
                expect(i('4:5').contains(i1)).to.equal(false);
                expect(i('5:5').contains(i1)).to.equal(false);
            });
        });

        describe('method "overlaps"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('overlaps');
            });
            var i1 = i('3:5');
            it('should return false for distinct intervals', function () {
                expect(i1.overlaps(i('1:2'))).to.equal(false);
                expect(i1.overlaps(i('6:7'))).to.equal(false);
                expect(i1.overlaps(i('8:9'))).to.equal(false);
                expect(i('1:2').overlaps(i1)).to.equal(false);
                expect(i('6:7').overlaps(i1)).to.equal(false);
                expect(i('8:9').overlaps(i1)).to.equal(false);
            });
            it('should return true for partly overlapping intervals', function () {
                expect(i1.overlaps(i('1:3'))).to.equal(true);
                expect(i1.overlaps(i('2:4'))).to.equal(true);
                expect(i1.overlaps(i('4:6'))).to.equal(true);
                expect(i1.overlaps(i('5:7'))).to.equal(true);
            });
            it('should return true for intervals containing each other', function () {
                expect(i1.overlaps(i1)).to.equal(true);
                expect(i1.overlaps(i('3:3'))).to.equal(true);
                expect(i('3:3').overlaps(i1)).to.equal(true);
                expect(i1.overlaps(i('3:4'))).to.equal(true);
                expect(i('3:4').overlaps(i1)).to.equal(true);
                expect(i1.overlaps(i('4:5'))).to.equal(true);
                expect(i('4:5').overlaps(i1)).to.equal(true);
                expect(i1.overlaps(i('5:5'))).to.equal(true);
                expect(i('5:5').overlaps(i1)).to.equal(true);
            });
        });

        describe('method "boundary"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('boundary');
            });
            it('should return the bounding interval', function () {
                expect(i('1:3').boundary(i('5:7'))).to.deep.equal(i('1:7'));
                expect(i('5:7').boundary(i('1:3'))).to.deep.equal(i('1:7'));
                expect(i('1:7').boundary(i('3:5'))).to.deep.equal(i('1:7'));
                expect(i('3:5').boundary(i('1:7'))).to.deep.equal(i('1:7'));
            });
        });

        describe('method "intersect"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('intersect');
            });
            var i1 = i('3:5');
            it('should return null for distinct intervals', function () {
                expect(i1.intersect(i('1:2'))).to.equal(null);
                expect(i1.intersect(i('6:7'))).to.equal(null);
                expect(i1.intersect(i('8:9'))).to.equal(null);
                expect(i('1:2').intersect(i1)).to.equal(null);
                expect(i('6:7').intersect(i1)).to.equal(null);
                expect(i('8:9').intersect(i1)).to.equal(null);
            });
            it('should return intersection for overlapping intervals', function () {
                expect(i1.intersect(i('1:3'))).to.deep.equal(i('3:3'));
                expect(i1.intersect(i('2:4'))).to.deep.equal(i('3:4'));
                expect(i1.intersect(i('4:6'))).to.deep.equal(i('4:5'));
                expect(i1.intersect(i('5:7'))).to.deep.equal(i('5:5'));
            });
            it('should return intersection for intervals containing each other', function () {
                expect(i1.intersect(i1)).to.deep.equal(i1);
                expect(i1.intersect(i('3:3'))).to.deep.equal(i('3:3'));
                expect(i('3:3').intersect(i1)).to.deep.equal(i('3:3'));
                expect(i1.intersect(i('3:4'))).to.deep.equal(i('3:4'));
                expect(i('3:4').intersect(i1)).to.deep.equal(i('3:4'));
                expect(i1.intersect(i('4:5'))).to.deep.equal(i('4:5'));
                expect(i('4:5').intersect(i1)).to.deep.equal(i('4:5'));
                expect(i1.intersect(i('5:5'))).to.deep.equal(i('5:5'));
                expect(i('5:5').intersect(i1)).to.deep.equal(i('5:5'));
            });
        });

        describe('method "move"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('move');
            });
            it('should move the indexes in the interval', function () {
                expect(i('5:6').move(-3)).to.deep.equal(i('2:3'));
                expect(i('5:6').move(-2)).to.deep.equal(i('3:4'));
                expect(i('5:6').move(-1)).to.deep.equal(i('4:5'));
                expect(i('5:6').move(0)).to.deep.equal(i('5:6'));
                expect(i('5:6').move(1)).to.deep.equal(i('6:7'));
                expect(i('5:6').move(2)).to.deep.equal(i('7:8'));
                expect(i('5:6').move(3)).to.deep.equal(i('8:9'));
            });
            it('should return itself', function () {
                var i1 = i('5:6');
                expect(i1.move(1)).to.equal(i1);
            });
        });

        describe('method "iterator"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('iterator');
            });
            it('should return an index iterator', function () {
                var it = i('2:4').iterator();
                expect(it).to.respondTo('next');
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: true });
            });
            it('should return a reverse index iterator', function () {
                var it = i('2:4').iterator({ reverse: true });
                expect(it.next()).to.deep.equal({ done: false, value: 3 });
                expect(it.next()).to.deep.equal({ done: false, value: 2 });
                expect(it.next()).to.deep.equal({ done: false, value: 1 });
                expect(it.next()).to.deep.equal({ done: true });
            });
        });

        describe('method "stringifyAsCols"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('stringifyAsCols');
            });
            it('should stringify to column interval', function () {
                expect(i('A:A').stringifyAsCols()).to.equal('A:A');
                expect(i('B:C').stringifyAsCols()).to.equal('B:C');
                expect(i('A:ZZ').stringifyAsCols()).to.equal('A:ZZ');
            });
        });

        describe('method "stringifyAsRows"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('stringifyAsRows');
            });
            it('should stringify to row interval', function () {
                expect(i('1:1').stringifyAsRows()).to.equal('1:1');
                expect(i('2:3').stringifyAsRows()).to.equal('2:3');
                expect(i('1:1000').stringifyAsRows()).to.equal('1:1000');
            });
        });

        describe('method "stringifyAs"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('stringifyAs');
            });
            it('should stringify to column interval', function () {
                expect(i('2:3').stringifyAs(true)).to.equal('B:C');
            });
            it('should stringify to row interval', function () {
                expect(i('B:C').stringifyAs(false)).to.equal('2:3');
            });
        });

        describe('method "toString"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('toString');
            });
            it('should stringify to row interval', function () {
                expect(i('B:C').toString()).to.equal('2:3');
            });
            it('should stringify implicitly', function () {
                expect('<' + i('B:C') + '>').to.equal('<2:3>');
            });
        });

        describe('method "toJSON"', function () {
            it('should exist', function () {
                expect(Interval).to.respondTo('toJSON');
            });
            var i1 = i('B:C'), i2 = { first: 1, last: 2 };
            it('should convert to JSON data', function () {
                expect(i1.toJSON()).to.deep.equal(i2);
            });
            it('should stringify to JSON', function () {
                expect(JSON.parse(JSON.stringify(i1))).to.deep.equal(i2);
            });
        });
    });

    // ========================================================================
});
