/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/spreadsheet/model/listcollection'
], function (AppHelper, ListCollection) {

    'use strict';

    // class CellCollection ===================================================

    describe('Spreadsheet class ListCollection', function () {

        it('should exist', function () {
            expect(ListCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // initialize test document
        var docModel = null, listCollection = null;
        AppHelper.createSpreadsheetApp('ooxml').done(function (app) {
            docModel = app.getModel();
            listCollection = docModel.getListCollection();
        });

        var newList = ['hier', 'meine', 'liste'];
        var newListLength = null;

        // public methods -----------------------------------------------------

        describe('method "getLists"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('getLists');
            });
            it('should return four predefined lists', function () {
                expect(listCollection.getLists().length).to.be.equal(4);
            });
        });

        describe('method "addList"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('addList');
            });
            it('should add the given list and return new array length', function () {
                newListLength = listCollection.addList(newList);
                expect(newListLength).to.equal(5);
            });
            it('should return false if the given value is no array', function () {
                var returnValue = listCollection.addList('value');
                expect(returnValue).to.equal(false);
            });
        });

        describe('method "getList"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('getList');
            });
            it('should return a specific list by key', function () {
                expect(listCollection.getList((newListLength - 1))).to.deep.equal(newList);
            });
            it('should return null, if the given key does not exist', function () {
                expect(listCollection.getList((listCollection.getLists().length + 1))).to.deep.equal(null);
            });
        });

        describe('method "listContains"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('listContains');
            });
            it('should return true if a value exists in any list', function () {
                expect(listCollection.listContains('hier')).to.equal(true);
            });
            it('should return false if a value does not exist in any list', function () {
                expect(listCollection.listContains('failure')).to.equal(false);
            });
        });

        describe('method "each"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('each');
            });
            it('should run through all lists and invoke the callback', function () {
                var callback = sinon.spy();
                listCollection.each(callback);
                expect(callback.callCount).to.equal(listCollection.getLists().length);
            });
        });

        describe('method "some"', function () {
            it('should exist', function () {
                expect(listCollection).to.respondTo('some');
            });

            var listNr = null, myList = null;
            before(function () {
                listNr = _.random(0, (listCollection.getLists().length - 1));
                myList = listCollection.getList(listNr);
            });

            it('should find the list with the value we are searching for', function () {
                var predicate   = { method: function (list) { return _.contains(list, myList[0]); } },
                    callback    = sinon.spy(predicate, 'method');

                expect(listCollection.some(callback)).to.equal(true);
                expect(callback.callCount).to.equal(listNr + 1);
            });

            it('should return false if there were no list with the value we are searching for', function () {
                var predicate   = { method: function (list) { return _.contains(list, 'failure'); } },
                    callback    = sinon.spy(predicate, 'method');

                expect(listCollection.some(callback)).to.equal(false);
                expect(callback.callCount).to.equal(listCollection.getLists().length);
            });
        });
    });

    // ========================================================================
});
