/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/spreadsheet/model/formula/parser/formularesource'
], function (FormulaResource) {

    'use strict';

    // class FormulaResource ==================================================

    describe('Spreadsheet class FormulaResource', function () {

        it('should exist', function () {
            expect(FormulaResource).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var ooxResource = null;
        var odfResource = null;
        var engResource = null;
        var resources = null;

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a formula resource', function () {
                ooxResource = new FormulaResource('ooxml', ox.language);
                expect(ooxResource).to.be.an('object');
            });
        });

        // static methods -----------------------------------------------------

        describe('method "create"', function () {
            it('should exist', function () {
                expect(FormulaResource).itself.to.respondTo('create');
            });
            it('should create a new resource', function () {
                odfResource = FormulaResource.create('odf', ox.language);
                expect(odfResource).to.be.an.instanceof(FormulaResource);
                expect(odfResource).to.not.equal(ooxResource);
                engResource = FormulaResource.create('ooxml', 'en_US');
                expect(engResource).to.not.equal(ooxResource);
                expect(engResource).to.not.equal(odfResource);
                resources = [ooxResource, odfResource, engResource];
            });
            it('should return an existing configuration', function () {
                var resource = FormulaResource.create('odf', ox.language);
                expect(resource).to.equal(odfResource);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "getFileFormat"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getFileFormat');
            });
            it('should return the file format', function () {
                expect(ooxResource.getFileFormat()).to.equal('ooxml');
                expect(odfResource.getFileFormat()).to.equal('odf');
                expect(engResource.getFileFormat()).to.equal('ooxml');
            });
        });

        describe('method "getLocale"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getLocale');
            });
            it('should return the locale identifier', function () {
                expect(ooxResource.getLocale()).to.equal(ox.language);
                expect(odfResource.getLocale()).to.equal(ox.language);
                expect(engResource.getLocale()).to.equal('en_US');
            });
        });

        describe('method "getDec"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getDec');
            });
            it('should return the correct decimal separator', function () {
                expect(ooxResource.getDec(false)).to.equal('.');
                expect(ooxResource.getDec(true)).to.equal(',');
                expect(odfResource.getDec(false)).to.equal('.');
                expect(odfResource.getDec(true)).to.equal(',');
                expect(engResource.getDec(false)).to.equal('.');
                expect(engResource.getDec(true)).to.equal('.');
            });
        });

        describe('method "getGroup"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getGroup');
            });
            it('should return the correct group separator', function () {
                expect(ooxResource.getGroup(false)).to.equal(',');
                expect(ooxResource.getGroup(true)).to.equal('.');
                expect(odfResource.getGroup(false)).to.equal(',');
                expect(odfResource.getGroup(true)).to.equal('.');
                expect(engResource.getGroup(false)).to.equal(',');
                expect(engResource.getGroup(true)).to.equal(',');
            });
        });

        describe('method "getSeparator"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getSeparator');
            });
            it('should return the correct list separator', function () {
                expect(ooxResource.getSeparator(false)).to.equal(',');
                expect(ooxResource.getSeparator(true)).to.equal(';');
                expect(odfResource.getSeparator(false)).to.equal(';');
                expect(odfResource.getSeparator(true)).to.equal(';');
                expect(engResource.getSeparator(false)).to.equal(',');
                expect(engResource.getSeparator(true)).to.equal(',');
            });
        });

        describe('method "getRCPrefixChars"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getRCPrefixChars');
            });
            it('should return the correct characters', function () {
                expect(ooxResource.getRCPrefixChars(false)).to.equal('RC');
                expect(ooxResource.getRCPrefixChars(true)).to.equal('ZS');
                expect(odfResource.getRCPrefixChars(false)).to.equal('RC');
                expect(odfResource.getRCPrefixChars(true)).to.equal('RC');
                expect(engResource.getRCPrefixChars(false)).to.equal('RC');
                expect(engResource.getRCPrefixChars(true)).to.equal('RC');
            });
        });

        describe('method "getBooleanCollection"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getBooleanCollection');
            });
            it('should return the boolean collection', function () {
                resources.forEach(function (resource) {
                    var collection = resource.getBooleanCollection();
                    expect(collection).to.be.an('object');
                    expect(collection).to.respondTo('get');
                    expect(collection.get('f')).to.be.an('object');
                    expect(collection.get('t')).to.be.an('object');
                });
            });
        });

        describe('method "getErrorCollection"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getErrorCollection');
            });
            it('should return the error collection', function () {
                resources.forEach(function (resource) {
                    var collection = resource.getErrorCollection();
                    expect(collection).to.be.an('object');
                    expect(collection).to.respondTo('get');
                    expect(collection.get('NULL')).to.be.an('object');
                    expect(collection.get('DIV0')).to.be.an('object');
                    expect(collection.get('VALUE')).to.be.an('object');
                    expect(collection.get('REF')).to.be.an('object');
                    expect(collection.get('NAME')).to.be.an('object');
                    expect(collection.get('NUM')).to.be.an('object');
                    expect(collection.get('NA')).to.be.an('object');
                    expect(collection.get('DATA')).to.be.an('object');
                    expect(collection.get('null')).to.equal(null);
                    expect(collection.get('#NULL!')).to.equal(null);
                    expect(collection.get('add')).to.equal(null);
                    expect(collection.get('ABS')).to.equal(null);
                });
            });
        });

        describe('method "getRegionCollection"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getRegionCollection');
            });
            it('should return the region collection', function () {
                resources.forEach(function (resource) {
                    var collection = resource.getRegionCollection();
                    expect(collection).to.be.an('object');
                    expect(collection).to.respondTo('get');
                    expect(collection.get('ALL')).to.be.an('object');
                    expect(collection.get('HEADERS')).to.be.an('object');
                    expect(collection.get('DATA')).to.be.an('object');
                    expect(collection.get('TOTALS')).to.be.an('object');
                    expect(collection.get('ROW')).to.be.an('object');
                });
            });
        });

        describe('method "getOperatorCollection"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getOperatorCollection');
            });
            it('should return the operator collection', function () {
                resources.forEach(function (resource) {
                    var collection = resource.getOperatorCollection();
                    expect(collection).to.be.an('object');
                    expect(collection).to.respondTo('get');
                    expect(collection.get('lt')).to.be.an('object');
                    expect(collection.get('le')).to.be.an('object');
                    expect(collection.get('gt')).to.be.an('object');
                    expect(collection.get('ge')).to.be.an('object');
                    expect(collection.get('eq')).to.be.an('object');
                    expect(collection.get('ne')).to.be.an('object');
                    expect(collection.get('add')).to.be.an('object');
                    expect(collection.get('sub')).to.be.an('object');
                    expect(collection.get('mul')).to.be.an('object');
                    expect(collection.get('div')).to.be.an('object');
                    expect(collection.get('pow')).to.be.an('object');
                    expect(collection.get('con')).to.be.an('object');
                    expect(collection.get('pct')).to.be.an('object');
                    expect(collection.get('list')).to.be.an('object');
                    expect(collection.get('isect')).to.be.an('object');
                    expect(collection.get('range')).to.be.an('object');
                    expect(collection.get('+')).to.equal(null);
                });
            });
            it('should provide differences between file formats', function () {
                var ooxCollection = ooxResource.getOperatorCollection();
                var odfCollection = odfResource.getOperatorCollection();
                var engCollection = engResource.getOperatorCollection();
                expect(ooxCollection.getName('list', false)).to.equal(',');
                expect(odfCollection.getName('list', false)).to.equal('~');
                expect(engCollection.getName('list', false)).to.equal(',');
                expect(ooxCollection.getName('list', true)).to.equal(';');
                expect(odfCollection.getName('list', true)).to.equal(';');
                expect(engCollection.getName('list', true)).to.equal(',');
                expect(ooxCollection.getName('isect', false)).to.equal(' ');
                expect(odfCollection.getName('isect', false)).to.equal('!');
                expect(engCollection.getName('isect', false)).to.equal(' ');
                expect(ooxCollection.getName('isect', true)).to.equal(' ');
                expect(odfCollection.getName('isect', true)).to.equal(' ');
                expect(engCollection.getName('isect', true)).to.equal(' ');
            });
        });

        describe('method "getFunctionCollection"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getFunctionCollection');
            });
            it('should return the function collection', function () {
                resources.forEach(function (resource) {
                    var collection = resource.getFunctionCollection();
                    expect(collection).to.be.an('object');
                    expect(collection).to.respondTo('get');
                    expect(collection.get('ABS')).to.be.an('object');
                    expect(collection.get('SUM')).to.be.an('object');
                    expect(collection.get('ERROR.TYPE')).to.be.an('object');
                    expect(collection.get('FORMULATEXT')).to.be.an('object');
                    expect(collection.get('sum')).to.equal(null);
                    expect(collection.get('SUMME')).to.equal(null);
                    expect(collection.getName('SUM', false)).to.equal('SUM');
                    expect(collection.getKey('SUM', false)).to.equal('SUM');
                    expect(collection.getKey('sum', false)).to.equal('SUM');
                    expect(collection.getKey('SUMME', false)).to.equal(null);
                });
                var ooxCollection = ooxResource.getFunctionCollection();
                var odfCollection = odfResource.getFunctionCollection();
                var engCollection = engResource.getFunctionCollection();
                expect(ooxCollection.getName('SUM', true)).to.equal('SUMME');
                expect(odfCollection.getName('SUM', true)).to.equal('SUMME');
                expect(engCollection.getName('SUM', true)).to.equal('SUM');
                expect(ooxCollection.getKey('SUMME', true)).to.equal('SUM');
                expect(odfCollection.getKey('SUMME', true)).to.equal('SUM');
                expect(engCollection.getKey('SUMME', true)).to.equal(null);
                expect(ooxCollection.getKey('summe', true)).to.equal('SUM');
                expect(odfCollection.getKey('summe', true)).to.equal('SUM');
                expect(engCollection.getKey('summe', true)).to.equal(null);
                expect(ooxCollection.getKey('SUM', true)).to.equal(null);
                expect(odfCollection.getKey('SUM', true)).to.equal(null);
                expect(engCollection.getKey('SUM', true)).to.equal('SUM');
            });
            it('should provide differences between file formats', function () {
                var ooxCollection = ooxResource.getFunctionCollection();
                var odfCollection = odfResource.getFunctionCollection();
                var engCollection = engResource.getFunctionCollection();
                expect(ooxCollection.get('ERRORTYPE.ODF')).to.equal(null);
                expect(odfCollection.get('ERRORTYPE.ODF')).to.be.an('object');
                expect(engCollection.get('ERRORTYPE.ODF')).to.equal(null);
                expect(ooxCollection.get('PHONETIC')).to.be.an('object');
                expect(odfCollection.get('PHONETIC')).to.equal(null);
                expect(engCollection.get('PHONETIC')).to.be.an('object');
                expect(ooxCollection.getName('FORMULATEXT', false)).to.equal('_xlfn.FORMULATEXT');
                expect(odfCollection.getName('FORMULATEXT', false)).to.equal('FORMULA');
                expect(engCollection.getName('FORMULATEXT', false)).to.equal('_xlfn.FORMULATEXT');
                expect(ooxCollection.getName('FORMULATEXT', true)).to.equal('FORMELTEXT');
                expect(odfCollection.getName('FORMULATEXT', true)).to.equal('FORMEL');
                expect(engCollection.getName('FORMULATEXT', true)).to.equal('FORMULATEXT');
            });
        });

        describe('method "getFunctionHelp"', function () {
            it('should exist', function () {
                expect(ooxResource).to.respondTo('getFunctionHelp');
            });
            it('should return a function help descriptor', function () {
                resources.forEach(function (resource) {
                    expect(resource.getFunctionHelp('ABS')).to.be.an('object');
                    expect(resource.getFunctionHelp('SUM')).to.be.an('object');
                });
            });
            it('should return null for invalid keys', function () {
                resources.forEach(function (resource) {
                    expect(resource.getFunctionHelp('123')).to.equal(null);
                    expect(resource.getFunctionHelp('abc')).to.equal(null);
                    expect(resource.getFunctionHelp('sum')).to.equal(null);
                    expect(resource.getFunctionHelp('SUMME')).to.equal(null);
                });
            });
        });
    });

    // ========================================================================
});
