/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/textfuncs'
], function (AppHelper, SheetHelper, TextFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;
    var mat = SheetHelper.mat;

    // pattern map for BAHTTEXT tests
    var BAHTTEXT_MAP = {
        0: '\u0E28\u0E39\u0E19\u0E22\u0E4C',
        1: '\u0E2B\u0E19\u0E36\u0E48\u0E07',
        2: '\u0E2A\u0E2D\u0E07',
        3: '\u0E2A\u0E32\u0E21',
        4: '\u0E2A\u0E35\u0E48',
        5: '\u0E2B\u0E49\u0E32',
        6: '\u0E2B\u0E01',
        7: '\u0E40\u0E08\u0E47\u0E14',
        8: '\u0E41\u0E1B\u0E14',
        9: '\u0E40\u0E01\u0E49\u0E32',
        d: '\u0E2A\u0E34\u0E1A',
        I: '\u0E40\u0E2D\u0E47\u0E14',              // trailing one
        T: '\u0E22\u0E35\u0E48',                    // 'two' for twenty
        h: '\u0E23\u0E49\u0E2D\u0E22',              // hundred
        k: '\u0E1E\u0E31\u0E19',                    // thousand
        D: '\u0E2B\u0E21\u0E37\u0E48\u0E19',        // ten thousand
        H: '\u0E41\u0E2A\u0E19',                    // hundred thousand
        M: '\u0E25\u0E49\u0E32\u0E19',              // million
        B: '\u0E1A\u0E32\u0E17',                    // Baht
        S: '\u0E2A\u0E15\u0E32\u0E07\u0E04\u0E4C',  // Satang
        '~': '\u0E16\u0E49\u0E27\u0E19',            // missing (zero) Satang
        '!': '\u0E16\u0E49',                        // missing (zero) Satang for negative zero
        '-': '\u0E25\u0E1A'                         // minus
    };

    function getExpectedBahtText(pattern) {
        return _.reduce(pattern, function (exp, char) { return exp + BAHTTEXT_MAP[char]; }, '');
    }

    function getEscapeSeq(string) {
        return _.reduce(string, function (str, char) {
            return str + '\\u' + ('000' + char.charCodeAt(0).toString(16)).substr(-4);
        }, '');
    }

    // module TextFuncs =======================================================

    describe('Spreadsheet module TextFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576, activeSheet: 0 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A2', contents: [['abc', '', 123], [null, true]] }
        ];

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(TextFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('BAHTTEXT', function (BAHTTEXT) {

            function check() {
                for (var i = 0; i < arguments.length; i += 2) {
                    var number = arguments[i];
                    var expected = getExpectedBahtText(arguments[i + 1]);
                    var result = BAHTTEXT(number);
                    if (result !== expected) {
                        console.warn('BAHTTEXT failed for input value ' + number);
                        console.warn('  exp=' + getEscapeSeq(expected));
                        console.warn('  got=' + getEscapeSeq(result));
                    }
                    expect(result).to.equal(expected);
                }
            }

            it('should return the correct result for integers between 0 and 9', function () {
                check(0, '0B~', 1, '1B~', 2, '2B~', 3, '3B~', 4, '4B~', 5, '5B~', 6, '6B~', 7, '7B~', 8, '8B~', 9, '9B~');
            });
            it('should return the correct result for integers between 10 and 99', function () {
                check(10, 'dB~', 11, 'dIB~', 12, 'd2B~', 13, 'd3B~', 14, 'd4B~', 15, 'd5B~', 16, 'd6B~', 17, 'd7B~', 18, 'd8B~', 19, 'd9B~');
                check(20, 'TdB~', 21, 'TdIB~', 22, 'Td2B~', 23, 'Td3B~', 24, 'Td4B~', 25, 'Td5B~', 26, 'Td6B~', 27, 'Td7B~', 28, 'Td8B~', 29, 'Td9B~');
                check(30, '3dB~', 31, '3dIB~', 32, '3d2B~', 39, '3d9B~');
                check(40, '4dB~', 41, '4dIB~', 42, '4d2B~', 49, '4d9B~');
                check(50, '5dB~', 60, '6dB~', 70, '7dB~', 80, '8dB~', 90, '9dB~', 99, '9d9B~');
            });
            it('should return the correct result for integers between 100 and 999', function () {
                check(100, '1hB~', 101, '1hIB~', 102, '1h2B~', 109, '1h9B~');
                check(110, '1hdB~', 111, '1hdIB~', 112, '1hd2B~', 119, '1hd9B~');
                check(120, '1hTdB~', 121, '1hTdIB~', 122, '1hTd2B~', 129, '1hTd9B~');
                check(130, '1h3dB~', 131, '1h3dIB~', 132, '1h3d2B~', 139, '1h3d9B~');
                check(140, '1h4dB~', 141, '1h4dIB~', 142, '1h4d2B~', 149, '1h4d9B~');
                check(200, '2hB~', 201, '2hIB~', 202, '2h2B~', 209, '2h9B~', 210, '2hdB~', 211, '2hdIB~', 299, '2h9d9B~');
                check(300, '3hB~', 401, '4hIB~', 502, '5h2B~', 609, '6h9B~', 710, '7hdB~', 811, '8hdIB~', 999, '9h9d9B~');
            });
            it('should return the correct result for integers between 1,000 and 9,999', function () {
                check(1000, '1kB~', 1001, '1kIB~', 1002, '1k2B~', 1009, '1k9B~');
                check(1010, '1kdB~', 1011, '1kdIB~', 1012, '1kd2B~', 1019, '1kd9B~');
                check(1020, '1kTdB~', 1021, '1kTdIB~', 1022, '1kTd2B~', 1029, '1kTd9B~');
                check(1030, '1k3dB~', 1031, '1k3dIB~', 1032, '1k3d2B~', 1039, '1k3d9B~');
                check(1100, '1k1hB~', 1101, '1k1hIB~', 1102, '1k1h2B~', 1109, '1k1h9B~');
                check(1210, '1k2hdB~', 2311, '2k3hdIB~', 3422, '3k4hTd2B~', 9999, '9k9h9d9B~');
            });
            it('should return the correct result for integers between 10,000 and 99,999', function () {
                check(10000, '1DB~', 10001, '1DIB~', 10002, '1D2B~', 10009, '1D9B~');
                check(10010, '1DdB~', 10011, '1DdIB~', 10012, '1Dd2B~', 10019, '1Dd9B~');
                check(10120, '1D1hTdB~', 10221, '1D2hTdIB~', 10322, '1D3hTd2B~', 10929, '1D9hTd9B~');
                check(11210, '1D1k2hdB~', 23411, '2D3k4hdIB~', 34522, '3D4k5hTd2B~', 99999, '9D9k9h9d9B~');
            });
            it('should return the correct result for integers between 100,000 and 999,999', function () {
                check(100000, '1HB~', 100001, '1HIB~', 100002, '1H2B~', 100009, '1H9B~');
                check(100010, '1HdB~', 100011, '1HdIB~', 100012, '1Hd2B~', 100019, '1Hd9B~');
                check(100120, '1H1hTdB~', 100221, '1H2hTdIB~', 100322, '1H3hTd2B~', 100929, '1H9hTd9B~');
                check(101120, '1H1k1hTdB~', 101221, '1H1k2hTdIB~', 101322, '1H1k3hTd2B~', 101929, '1H1k9hTd9B~');
                check(111210, '1H1D1k2hdB~', 234511, '2H3D4k5hdIB~', 345622, '3H4D5k6hTd2B~', 999999, '9H9D9k9h9d9B~');
            });
            it('should return the correct result for integers between 1,000,000 and 9,999,999', function () {
                check(1000000, '1MB~', 1000001, '1MIB~', 1000002, '1M2B~', 1000009, '1M9B~', 1999999, '1M9H9D9k9h9d9B~');
                check(2000000, '2MB~', 3000001, '3MIB~', 4567890, '4M5H6D7k8h9dB~', 9999999, '9M9H9D9k9h9d9B~');
            });
            it('should return the correct result for integers between 10,000,000 and 99,999,999', function () {
                check(10000000, 'dMB~', 10000001, 'dMIB~', 11000002, 'dIM2B~', 20000009, 'TdM9B~', 99999999, '9d9M9H9D9k9h9d9B~');
            });
            it('should return the correct result for integers starting from 100,000,000', function () {
                check(100000000, '1hMB~', 100000001, '1hMIB~', 101000000, '1hIMB~', 1000000000000, '1MMB~', 1000001000000, '1MIMB~', 1000001000001, '1MIMIB~');
            });
            it('should return the correct result for negative integers', function () {
                check(-1, '-1B~', -2, '-2B~', -9999999, '-9M9H9D9k9h9d9B~');
            });
            it('should return the correct result for fractionals between -1 and 1', function () {
                check(0.01, '1S', 0.02, '2S', 0.10, 'dS', 0.11, 'dIS', 0.20, 'TdS', 0.21, 'TdIS', 0.99, '9d9S');
                check(-0.01, '-1S', -0.99, '-9d9S');
                check(0.014, '1S', 0.015, '2S', -0.014, '-1S', -0.015, '-2S');
                check(0.004, '0B~', 0.005, '1S', -0.004, '-0B!', -0.005, '-1S');
            });
        });

        moduleTester.testFunction('CONCAT', function (CONCAT) {
            it('should return the concatenation of the operands', function () {
                expect(CONCAT('abc', 123, null, true)).to.equal('abc123WAHR');
            });
            it('should concatenate matrix elements', function () {
                expect(CONCAT(mat([['abc', 123], ['', true]]))).to.equal('abc123WAHR');
            });
            it('should concatenate cell contents', function () {
                expect(CONCAT(r3d('0:0!A2:C3'))).to.equal('abc123WAHR');
            });
        });

        moduleTester.testFunction('CONCATENATE', function (CONCATENATE) {
            it('should return the concatenation of the operands', function () {
                expect(CONCATENATE('abc', '123', '', 'WAHR')).to.equal('abc123WAHR');
            });
        });

        moduleTester.testFunction('DOLLAR', function (DOLLAR) {
            it('should return the currency formatted string for a number', function () {
                expect(DOLLAR(1234.567, 2)).to.equal('1.234,57 \u20AC');
                expect(DOLLAR(1234.567, -2)).to.equal('1.200 \u20AC');
                expect(DOLLAR(-1234.567, -2)).to.equal('-1.200 \u20AC');
                expect(DOLLAR(-0.123, 4)).to.equal('-0,1230 \u20AC');
                expect(DOLLAR(99.888)).to.equal('99,89 \u20AC');

                expect(DOLLAR(1234.567, 1)).to.equal('1.234,6 \u20AC');
                expect(DOLLAR(1234.567, -1)).to.equal('1.230 \u20AC');

                expect(DOLLAR(4444.332)).to.equal('4.444,33 \u20AC');
                expect(DOLLAR(44.332)).to.equal('44,33 \u20AC');

                expect(DOLLAR(1234567.332)).to.equal('1.234.567,33 \u20AC');

                var number = 123456789.987;
                expect(DOLLAR(number, 5)).to.equal('123.456.789,98700 \u20AC');
                expect(DOLLAR(number, 2)).to.equal('123.456.789,99 \u20AC');
                expect(DOLLAR(number, 1)).to.equal('123.456.790,0 \u20AC');
                expect(DOLLAR(number, 0)).to.equal('123.456.790 \u20AC');
                expect(DOLLAR(number, -1)).to.equal('123.456.790 \u20AC');
                expect(DOLLAR(number, -2)).to.equal('123.456.800 \u20AC');
                expect(DOLLAR(number, -5)).to.equal('123.500.000 \u20AC');
                expect(DOLLAR(number, -8)).to.equal('100.000.000 \u20AC');
                expect(DOLLAR(number, -9)).to.equal('0 \u20AC');
                expect(DOLLAR(number, -1000)).to.equal('0 \u20AC');
            });
            it('should throw #VALUE! error for wrong second argument', function () {
                expect(DOLLAR(10, 128)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('FIXED', function (FIXED) {
            it('should return the fixed formated string of assigned number', function () {
                expect(FIXED(1234.567, 1)).to.equal('1.234,6');
                expect(FIXED(1234.567, -1)).to.equal('1.230');
                expect(FIXED(1234.567, -1, true)).to.equal('1230');
                expect(FIXED(-1234.567, -1, true)).to.equal('-1230');

                expect(FIXED(44.332, 2, true)).to.equal('44,33');
                expect(FIXED(4444.332)).to.equal('4.444,33');
                expect(FIXED(44.332)).to.equal('44,33');

                expect(FIXED(1234567.332, 2, true)).to.equal('1234567,33');
                expect(FIXED(1234567.332)).to.equal('1.234.567,33');

                var number = 123456789.987;
                expect(FIXED(number, 5)).to.equal('123.456.789,98700');
                expect(FIXED(number, 2)).to.equal('123.456.789,99');
                expect(FIXED(number, 1)).to.equal('123.456.790,0');
                expect(FIXED(number, 0)).to.equal('123.456.790');
                expect(FIXED(number, -1)).to.equal('123.456.790');
                expect(FIXED(number, -2)).to.equal('123.456.800');
                expect(FIXED(number, -5)).to.equal('123.500.000');
                expect(FIXED(number, -8)).to.equal('100.000.000');
                expect(FIXED(number, -9)).to.equal('0');
                expect(FIXED(number, -1000)).to.equal('0');
            });
            it('should throw #VALUE! error for wrong second argument', function () {
                expect(FIXED(10, 128)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('NUMBERVALUE', function (NUMBERVALUE) {
            it('should return the detected number assigned string', function () {
                expect(NUMBERVALUE('3,5%')).to.equal(0.035);
                expect(NUMBERVALUE('2,500a27', 'a', ',')).to.equal(2500.27);
                expect(NUMBERVALUE('2,500a27', 'abc', ',')).to.equal(2500.27);
                expect(NUMBERVALUE('1.1.2007')).to.equal(112007);
                expect(NUMBERVALUE('1.1.2007', 'd', 'g')).to.equal(39083);
                expect(NUMBERVALUE('1.1,1')).to.equal(11.1);

                expect(NUMBERVALUE('3,5%%')).to.almostEqual(3.5e-4);
                expect(NUMBERVALUE('3,5%%%')).to.almostEqual(3.5e-6);
                expect(NUMBERVALUE('-3,5%%%%')).to.almostEqual(-3.5e-8);
            });
            it('should throw #VALUE! if no valid number was passed', function () {
                expect(NUMBERVALUE('1.1d2007', 'd', 'g')).to.equal(ErrorCode.VALUE);
                expect(NUMBERVALUE('1,1,1')).to.equal(ErrorCode.VALUE);
                expect(NUMBERVALUE('1,1.1')).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('REPT', function (REPT) {
            it('should return the repeated string', function () {
                expect(REPT('abc', 0)).to.equal('');
                expect(REPT('abc', 1)).to.equal('abc');
                expect(REPT('abc', 2)).to.equal('abcabc');
                expect(REPT('abc', 3)).to.equal('abcabcabc');
                expect(REPT('abc', 4)).to.equal('abcabcabcabc');
            });
            it('should throw #VALUE! for invalid string length', function () {
                expect(REPT('abc', -1)).to.equal(ErrorCode.VALUE);
                expect(REPT('a', 32767)).to.be.a('string');
                expect(REPT('a', 32768)).to.equal(ErrorCode.VALUE);
                expect(REPT('abcd', 8191)).to.be.a('string');
                expect(REPT('abcd', 8192)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('ROT13', function (ROT13) {
            it('should return the ROW13 encoding of a string', function () {
                expect(ROT13('hello')).to.equal('uryyb');
                expect(ROT13('HELLO')).to.equal('URYYB');
                expect(ROT13('uryyb')).to.equal('hello');
                expect(ROT13('123')).to.equal('123');
            });
        });

        moduleTester.testFunction('TEXT', function (TEXT) {
            it('should return the particular wrong formated Text in English', function () {
                expect(TEXT(2800, '$0.00')).to.equal('$2.800');
                expect(TEXT(0.4, '0%')).to.equal('40%');
                expect(TEXT(39300.63, 'yyyy-MM-dd')).to.equal('yyyy-08-dd');
                expect(TEXT(39300.63, 'm/d/yyyy h:mm AM/PM')).to.equal('8/d/yyyy 3:07 pm');
                expect(TEXT(39300.63, '0.00E+00')).to.equal('04E+04');
                expect(TEXT(39300.63, '$#,##0.00')).to.equal('$39300,63000');
            });
            it('should return the correct formated Text in German', function () {
                expect(TEXT(2800, '0,00 \u20AC')).to.equal('2800,00 \u20AC');
                expect(TEXT(0.4, '0%')).to.equal('40%');
                expect(TEXT(39300.63, 'h:MM')).to.equal('15:07');
                expect(TEXT(39300.63, 'h:mm')).to.equal('15:07');
                expect(TEXT(39300.63, 'MM:ss')).to.equal('07:12');
                expect(TEXT(39300.63, 'mm:ss')).to.equal('07:12');
                expect(TEXT(39300.63, 'MM')).to.equal('08');
                expect(TEXT(39300.63, 'mm')).to.equal('08');
                expect(TEXT(39300.63, 'jjjj-MM-tt')).to.equal('2007-08-06');
                expect(TEXT(39300.63, 'jjjj-mm-tt')).to.equal('2007-08-06');
                expect(TEXT(39300.63, 't.M.jjjj h:mm AM/PM')).to.equal('6.8.2007 3:07 pm');
                expect(TEXT(39300.63, 't.m.jjjj h:mm AM/PM')).to.equal('6.8.2007 3:07 pm');
                expect(TEXT(39300.63, '0,00E+00')).to.equal('3,93E+04');
                expect(TEXT(39300.63, '#.##0,00 \u20ac')).to.equal('39.300,63 \u20ac');
            });
        });

        moduleTester.testFunction('TEXTJOIN', function (TEXTJOIN) {
            it('should return the concatenation of the operands', function () {
                expect(TEXTJOIN('', true, 'abc', '', 123, null, true)).to.equal('abc123WAHR');
                expect(TEXTJOIN(',', true, 'abc', '', 123, null, true)).to.equal('abc,123,WAHR');
                expect(TEXTJOIN(',', false, 'abc', '', 123, null, true)).to.equal('abc,,123,,WAHR');
            });
            it('should concatenate matrix elements', function () {
                expect(TEXTJOIN('', true, mat([['abc', 123], ['', true]]))).to.equal('abc123WAHR');
                expect(TEXTJOIN(',', true, mat([['abc', 123], ['', true]]))).to.equal('abc,123,WAHR');
                expect(TEXTJOIN(',', false, mat([['abc', 123], ['', true]]))).to.equal('abc,123,,WAHR');
            });
            it('should concatenate cell contents', function () {
                expect(TEXTJOIN('', true, r3d('0:0!A2:C3'))).to.equal('abc123WAHR');
                expect(TEXTJOIN(',', true, r3d('0:0!A2:C3'))).to.equal('abc,123,WAHR');
                expect(TEXTJOIN(',', false, r3d('0:0!A2:C3'))).to.equal('abc,,123,,WAHR,');
                expect(TEXTJOIN(',', true, r3d('0:0!A8:B9'))).to.equal('');
                expect(TEXTJOIN(',', false, r3d('0:0!A8:B9'))).to.equal(',,,');
            });
        });

        moduleTester.testFunction('VALUE', function (VALUE) {
            it('should return numbes', function () {
                expect(VALUE(0)).to.equal(0);
                expect(VALUE(-42)).to.equal(-42);
            });
            it('should convert strings to numbes', function () {
                expect(VALUE('0')).to.equal(0);
                expect(VALUE('-42')).to.equal(-42);
                expect(VALUE('200%')).to.equal(2);
            });
            it('should throw #VALUE! error code', function () {
                expect(VALUE('')).to.equal(ErrorCode.VALUE);
                expect(VALUE(true)).to.equal(ErrorCode.VALUE);
            });
        });
    });

    // ========================================================================
});
