/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils',
    'io.ox/office/spreadsheet/model/formula/funcs/operators'
], function (AppHelper, SheetHelper, MathUtils, Operators) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;
    var r3da = SheetHelper.r3da;

    // module Operators =======================================================

    describe('Spreadsheet module Operators', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(Operators, appPromise);

        // operator implementations -------------------------------------------

        moduleTester.testFunction('add', function (ADD, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 1);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the unconverted value', function () {
                expect(ADD(10)).to.equal(10);
                expect(ADD(true)).to.equal(true);
                expect(ADD('10')).to.equal('10');
                expect(ADD('abc')).to.equal('abc');
            });
            it('should return the sum', function () {
                expect(ADD(10, 2)).to.equal(12);
                expect(ADD(true, 2)).to.equal(3);
                expect(ADD('10', 2)).to.equal(12);
            });
            it('should throw #VALUE! for invalid first argument', function () {
                expect(ADD('abc', 2)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('sub', function (SUB, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 1);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the negated number', function () {
                expect(SUB(10)).to.equal(-10);
            });
            it('should return the difference', function () {
                expect(SUB(10, 2)).to.equal(8);
            });
        });

        moduleTester.testFunction('mul', function (MUL, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the product', function () {
                expect(MUL(10, 2)).to.equal(20);
            });
        });

        moduleTester.testFunction('div', function (DIV, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the quotient', function () {
                expect(DIV(10, 2)).to.equal(5);
            });
            it('should throw #DIV/0! for zero divisor', function () {
                expect(DIV(10, 0)).to.equal(ErrorCode.DIV0);
            });
        });

        moduleTester.testFunction('pow', function (POW, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should provide the correct resolver', function () {
                expect(descriptor).to.have.a.property('resolve', MathUtils.POW);
            });
        });

        moduleTester.testFunction('pct', function (PERCENT, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 1);
                expect(descriptor).to.have.a.property('maxParams', 1);
            });
            it('should return a hundreth of the argument', function () {
                expect(PERCENT(1)).to.equal(0.01);
                expect(PERCENT(123)).to.equal(1.23);
                expect(PERCENT(-1)).to.equal(-0.01);
                expect(PERCENT(-123)).to.equal(-1.23);
                expect(PERCENT(0)).to.equal(0);
            });
        });

        moduleTester.testFunction('con', function (CONCAT, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the concatenated strings', function () {
                expect(CONCAT('ab', 'cd')).to.equal('abcd');
                expect(CONCAT('ab', '')).to.equal('ab');
                expect(CONCAT('', 'cd')).to.equal('cd');
                expect(CONCAT('', '')).to.equal('');
            });
        });

        moduleTester.testFunction('lt', function (LT, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(LT(1, 2)).to.equal(true);
                expect(LT(2, 1)).to.equal(false);
                expect(LT(2, 2)).to.equal(false);
                expect(LT('a', 'b')).to.equal(true);
                expect(LT('b', 'a')).to.equal(false);
                expect(LT('b', 'b')).to.equal(false);
                expect(LT(1, '1')).to.equal(true);
                expect(LT('1', 1)).to.equal(false);
            });
        });

        moduleTester.testFunction('le', function (LE, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(LE(1, 2)).to.equal(true);
                expect(LE(2, 1)).to.equal(false);
                expect(LE(2, 2)).to.equal(true);
                expect(LE('a', 'b')).to.equal(true);
                expect(LE('b', 'a')).to.equal(false);
                expect(LE('b', 'b')).to.equal(true);
                expect(LE(1, '1')).to.equal(true);
                expect(LE('1', 1)).to.equal(false);
            });
        });

        moduleTester.testFunction('gt', function (GT, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(GT(1, 2)).to.equal(false);
                expect(GT(2, 1)).to.equal(true);
                expect(GT(2, 2)).to.equal(false);
                expect(GT('a', 'b')).to.equal(false);
                expect(GT('b', 'a')).to.equal(true);
                expect(GT('b', 'b')).to.equal(false);
                expect(GT(1, '1')).to.equal(false);
                expect(GT('1', 1)).to.equal(true);
            });
        });

        moduleTester.testFunction('ge', function (GE, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(GE(1, 2)).to.equal(false);
                expect(GE(2, 1)).to.equal(true);
                expect(GE(2, 2)).to.equal(true);
                expect(GE('a', 'b')).to.equal(false);
                expect(GE('b', 'a')).to.equal(true);
                expect(GE('b', 'b')).to.equal(true);
                expect(GE(1, '1')).to.equal(false);
                expect(GE('1', 1)).to.equal(true);
            });
        });

        moduleTester.testFunction('eq', function (EQ, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(EQ(1, 2)).to.equal(false);
                expect(EQ(2, 1)).to.equal(false);
                expect(EQ(2, 2)).to.equal(true);
                expect(EQ('a', 'b')).to.equal(false);
                expect(EQ('b', 'a')).to.equal(false);
                expect(EQ('b', 'b')).to.equal(true);
                expect(EQ(1, '1')).to.equal(false);
                expect(EQ('1', 1)).to.equal(false);
            });
        });

        moduleTester.testFunction('ne', function (NE, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should compare the passed values', function () {
                expect(NE(1, 2)).to.equal(true);
                expect(NE(2, 1)).to.equal(true);
                expect(NE(2, 2)).to.equal(false);
                expect(NE('a', 'b')).to.equal(true);
                expect(NE('b', 'a')).to.equal(true);
                expect(NE('b', 'b')).to.equal(false);
                expect(NE(1, '1')).to.equal(true);
                expect(NE('1', 1)).to.equal(true);
            });
        });

        moduleTester.testFunction('list', function (LIST, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should concatenate range lists', function () {
                expect(LIST(r3da('0:0!A1:A1'), r3da('1:1!A1:A1'))).to.deep.equal(r3da('0:0!A1:A1 1:1!A1:A1'));
                expect(LIST(r3da('0:0!A1:B2 0:0!B2:C3'), r3da('1:1!A1:A1'))).to.deep.equal(r3da('0:0!A1:B2 0:0!B2:C3 1:1!A1:A1'));
            });
        });

        moduleTester.testFunction('isect', function (ISECT, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the intersection', function () {
                expect(ISECT(r3da('0:0!A1:B2'), r3da('0:0!B2:C3'))).to.deep.equal(r3da('0:0!B2:B2'));
                expect(ISECT(r3da('0:0!A1:C3 0:0!B2:D4'), r3da('0:0!C3:D5 0:0!D4:E6'))).to.deep.equal(r3da('0:0!C3:C3 0:0!C3:D4 0:0!D4:D4'));
            });
            it('should throw #VALUE! error code for ranges from different sheets', function () {
                expect(ISECT(r3da('0:0!A1:A1'), r3da('1:1!A1:A1'))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('range', function (RANGE, descriptor) {
            it('should provide correct parameter count', function () {
                expect(descriptor).to.have.a.property('minParams', 2);
                expect(descriptor).to.have.a.property('maxParams', 2);
            });
            it('should return the union', function () {
                expect(RANGE(r3da('0:0!A1:B2'), r3da('0:0!B2:C3'))).to.deep.equal(r3d('0:0!A1:C3'));
                expect(RANGE(r3da('0:0!A1:C3 0:0!B2:D4'), r3da('0:0!C3:D5 0:0!D4:E6'))).to.deep.equal(r3d('0:0!A1:E6'));
            });
            it('should throw #VALUE! error code for ranges from different sheets', function () {
                expect(RANGE(r3da('0:0!A1:A1'), r3da('1:1!A1:A1'))).to.equal(ErrorCode.VALUE);
            });
        });
    });

    // ========================================================================
});
