/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/informationfuncs'
], function (AppHelper, SheetHelper, InformationFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;

    // module InformationFuncs ================================================

    describe('Spreadsheet module InformationFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'A2', contents: [
                [2, { v: 3, f: '=SUM(1,2)' }, 'abc'],
                [{ v: 2, f: 'A2', si: 0, sr: 'A3:B3' }, { v: 3, si: 0 }],
                [{ v: 4, f: 'A3:B3*2', mr: 'A4:B4' }, 6]
            ] }
        ];

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(InformationFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('ERROR.TYPE', function (ERROR_TYPE) {
            it('should return the numeric index of an error', function () {
                expect(ERROR_TYPE(ErrorCode.NULL)).to.equal(1);
                expect(ERROR_TYPE(ErrorCode.DIV0)).to.equal(2);
                expect(ERROR_TYPE(ErrorCode.VALUE)).to.equal(3);
                expect(ERROR_TYPE(ErrorCode.REF)).to.equal(4);
                expect(ERROR_TYPE(ErrorCode.NAME)).to.equal(5);
                expect(ERROR_TYPE(ErrorCode.NUM)).to.equal(6);
                expect(ERROR_TYPE(ErrorCode.NA)).to.equal(7);
                expect(ERROR_TYPE(ErrorCode.DATA)).to.equal(8);
            });
            it('should throw #N/A for non-error values', function () {
                expect(ERROR_TYPE(1)).to.equal(ErrorCode.NA);
                expect(ERROR_TYPE('abc')).to.equal(ErrorCode.NA);
                expect(ERROR_TYPE(true)).to.equal(ErrorCode.NA);
                expect(ERROR_TYPE(null)).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('ISFORMULA', function (ISFORMULA) {
            it('should return whether a cell is a formula cell', function () {
                expect(ISFORMULA(r3d('0:0!A2:A2'))).to.equal(false); // number cell
                expect(ISFORMULA(r3d('0:0!B2:B2'))).to.equal(true); // formula cell
                expect(ISFORMULA(r3d('0:0!C2:C2'))).to.equal(false); // string cell
                expect(ISFORMULA(r3d('0:0!IV99:IV99'))).to.equal(false); // blank cell
            });
            it('should return true for all shared formula cells', function () {
                expect(ISFORMULA(r3d('0:0!A3:A3'))).to.equal(true); // shared formula (anchor)
                expect(ISFORMULA(r3d('0:0!B3:B3'))).to.equal(true); // shared formula
            });
            it('should return true for all matrix formula cells', function () {
                expect(ISFORMULA(r3d('0:0!A4:A4'))).to.equal(true); // matrix formula (anchor)
                expect(ISFORMULA(r3d('0:0!B4:B4'))).to.equal(true); // matrix formula
            });
            it('should return true for the reference address of the formula context', function () {
                expect(ISFORMULA(r3d('0:0!A1:A1'))).to.equal(true);
            });
            it('should use the start address of a range', function () {
                expect(ISFORMULA(r3d('0:0!B2:C3'))).to.equal(true);
            });
        });

        moduleTester.testFunction('CELL', function (CELL) {
            var r1 = r3d('0:0!B20:B20');
            it('should return information about the formatting, location, or contents of a cell', function () {
                expect(CELL('address', r1)).to.equal('$B$20');
                expect(CELL('col', r1)).to.equal(2);
                expect(CELL('color', r1)).to.equal(0);
                expect(CELL('contents', r1)).to.equal(null);
                expect(CELL('filename', r1)).to.equal('/path/to/[testfile.xyz]Sheet1');
                expect(CELL('format', r1)).to.equal('G');
                //expect(CELL('parentheses', r1)).to.equal(0);
                expect(CELL('prefix', r1)).to.equal('');
                expect(CELL('protect', r1)).to.equal(1);
                expect(CELL('row', r1)).to.equal(20);
                expect(CELL('type', r1)).to.equal('b');
                //expect(CELL('width', r1)).to.equal(8);
            });
            it('should throw #VAL for non-recognized values', function () {
                expect(CELL('__invalid__', r1)).to.equal(ErrorCode.VALUE);
            });
        });
    });

    // ========================================================================
});
