/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/model/formula/funcs/databasefuncs'
], function (AppHelper, SheetHelper, DatabaseFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;
    var r3d = SheetHelper.r3d;

    function TestContainer(ooxResolver, odfResolver, database, field) {

        this.testValue = function (criteria, ooxResult, odfResult) {
            expect(ooxResolver(database, field, criteria)).to.equal(ooxResult);
            expect(odfResolver(database, field, criteria)).to.equal(odfResult ? odfResult : ooxResult);
        };
    }

    // module DatabaseFuncs ===================================================

    describe('Spreadsheet module DatabaseFuncs', function () {

        // the operations to be applied at the test document
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { fileFormat: 'ooxml', cols: 16384, rows: 1048576, activeSheet: 0 } } },

            { name: 'insertSheet', sheet: 0, sheetName: 'DAverageSheet' },
            { name: 'changeCells', sheet: 0, start: 'A1', contents: [
                ['Tree', 'Height', 'Height', 'Age', 'Profit', 'Height', 'Height', 'Height', 'Height'],
                ['Apple', {}, '>8', null, null, '<16', {}, '>8', '<8'],
                ['Pear', null, null, '>15', {}, '>8', '>8'],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit'],
                ['Apple', 18, 20, 40, 105],
                ['Apple', 12, 12, 31, 96],
                ['Cherry', 13, 14, 9, 105],
                ['Apple', 14, 15, 5, 75],
                ['Pear', 9, 8, 8, 76.8],
                ['Apple', 8, 9, 6, 45]
            ] },

            { name: 'insertSheet', sheet: 1, sheetName: 'DCountSheet' },
            { name: 'changeCells', sheet: 1, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height', 'Height'],
                ['=Apple', '>10', { r: 3 }, '<16', '>30'],
                ['*e*'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit'],
                ['Apple', 18, 20, 14, '$105'],
                ['Pear', 12, '\xfc', 10, '$96'],
                ['Cherry', 13, 14, 9, '$105'],
                ['Apple', 14, 'N/A', 10, '$75'],
                ['Pear', 9, 8, 8, '$77'],
                ['Apple', 12, 11, 6, '$45']
            ] },

            { name: 'insertSheet', sheet: 2, sheetName: 'DCountASheet' },
            { name: 'changeCells', sheet: 2, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 1, 'Height', 'Height'],
                ['=Apple', '>10', '>10', {}, '>70', '<16', '>30'],
                ['*e*'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 1],
                ['Apple', 18, 20, 14, 105],
                ['Pear', 12, '\xfc', 10, 96],
                ['Cherry', 13, 14, 9, 105],
                ['Apple', 14, 'N/A', 10, 75],
                ['Pear', 9, 8, 8, 77],
                ['Apple', 12, 11, 6, 45]
            ] },

            { name: 'insertSheet', sheet: 3, sheetName: 'DGetSheet' },
            { name: 'changeCells', sheet: 3, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height', 'Yield'],
                ['=Apple', '>10', { r: 3 }, '<16', '>100'],
                ['Pear', '>12'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit'],
                ['Apple', 18, 20, 14, '$105'],
                ['Pear', 12, 12, 10, '$96'],
                ['Cherry', 13, 14, 9, '$105'],
                ['Apple', 14, 15, 10, '$75'],
                ['Pear', 9, 8, 8, '$77'],
                ['Apple', 8, 9, 6, '$45']
            ] },

            { name: 'insertSheet', sheet: 4, sheetName: 'DMaxSheet' },
            { name: 'changeCells', sheet: 4, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height', 'Height'],
                ['=Apple', '>10', { r: 3 }, '<16', '>100'],
                ['=Pear'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Empty'],
                ['Apple', 18, 20, 14, 105],
                ['Pear', 12, 12, 10, 96],
                ['Cherry', 13, 14, 9, 105],
                ['Apple', 14, 15, 10, 75],
                ['Pear', 9, 8, 8, 77],
                ['Apple', 8, 9, 6, 45]
            ] },

            { name: 'insertSheet', sheet: 5, sheetName: 'DMinSheet' },
            { name: 'changeCells', sheet: 5, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height', 'Height'],
                ['=Apple', '>10', { r: 3 }, '<16', '>100'],
                ['=Pear'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Empty'],
                ['Apple', 18, 20, 14, 105],
                ['Pear', 12, 12, 10, 96],
                ['Cherry', 13, 14, 9, 105],
                ['Apple', 14, 15, 10, 75],
                ['Pear', 9, 8, 8, 77],
                ['Apple', 8, 9, 6, 45]
            ] },

            { name: 'insertSheet', sheet: 6, sheetName: 'DProductSheet' },
            { name: 'changeCells', sheet: 6, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height', 'Height'],
                ['Apple', '>10', { r: 3 }, '<16', 8],
                ['Pear'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Width'],
                ['Apple', 18, 20, 14, 105, 100],
                ['Pear', 12, {}, -10, 96, 0],
                ['Cherry', 13, {}, 9, -1, 88],
                ['Apple', 14, 15, 10, 75, 10],
                ['Pear', 9, 10, 8, 77, 20],
                ['Apple', 8, 9, 6, 45, 52]
            ] },

            { name: 'insertSheet', sheet: 7, sheetName: 'DefaultSheet' },
            { name: 'changeCells', sheet: 7, start: 'A1', contents: [
                ['Tree', 'Height', 'Age', 'Yield', 'Profit', 'Height'],
                ['=Apple', '>10', { r: 3 }, '<16'],
                ['=Pear'],
                [],
                ['Tree', 'Height', 'Age', 'Yield', 'Profit'],
                ['Apple', 18, 20, 14, 105],
                ['Pear', 12, 12, 'A', 96],
                ['Cherry', 13, 14, 9, 105],
                ['Apple', 14, 15, 10, 75],
                ['Pear', 9, 8, 8, 77],
                ['Apple', 8, 9, 6, 45]
            ] }
        ];

        // initialize the test
        var ooxAppPromise = AppHelper.createSpreadsheetApp('ooxml', OPERATIONS);
        var odfAppPromise = AppHelper.createSpreadsheetApp('odf', OPERATIONS);
        var moduleTester = SheetHelper.createFunctionModuleTester(DatabaseFuncs, ooxAppPromise, odfAppPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('DAVERAGE', function (DAVERAGE, DAVERAGE_ODF) {
            var database = r3d('0:0!A4:E10');
            var testContainer = new TestContainer(DAVERAGE, DAVERAGE_ODF, database, 4);
            it('should return the averageof the defined column with the criteria', function () {
                testContainer.testValue(r3d('0:0!B1:B3'), 16.5);
                testContainer.testValue(r3d('0:0!C1:C2'), 18.6);
                testContainer.testValue(r3d('0:0!C1:C3'), 16.5, 18.6);
                testContainer.testValue(r3d('0:0!B1:C2'), 18.6);
                testContainer.testValue(r3d('0:0!B1:F2'), 13.25);
                testContainer.testValue(r3d('0:0!F1:F3'), 16.5);
                testContainer.testValue(r3d('0:0!G1:G3'), 16.5, 18.6);
                testContainer.testValue(r3d('0:0!C1:D3'), 18.6);
            });
            it('should return error codes', function () {
                // result for average is empty
                testContainer.testValue(r3d('0:0!H1:I2'), ErrorCode.DIV0);
                // database column labels are incorrect
                expect(DAVERAGE(r3d('0:0!A5:E10'), 'Yield', r3d('0:0!A1:F3'))).to.equal(ErrorCode.VALUE);
                // criteria column labels are incorrect
                testContainer.testValue(r3d('0:0!A2:G3'), ErrorCode.DIV0);
                // the field or db column name for average is not valid
                expect(DAVERAGE(database, 'WrongFieldName', r3d('0:0!B1:B3'))).to.equal(ErrorCode.VALUE);
                // criteria row size is lower than 2, first row are the labels, second row is the criteria
                testContainer.testValue(r3d('0:0!A1:A1'), ErrorCode.VALUE);
                // database row size is lower than 2, first row are the labels, second row is the criteria
                expect(DAVERAGE(r3d('0:0!A5:E5'), 4, r3d('0:0!A1:F3'))).to.equal(ErrorCode.VALUE);
                // test null field name
                expect(DAVERAGE(database, null, r3d('0:0!B1:B3'))).to.equal(ErrorCode.NAME);
            });
        });

        moduleTester.testFunction('DCOUNT', function (DCOUNT, DCOUNT_ODF) {
            var database = r3d('1:1!A5:E11');
            var testContainer = new TestContainer(DCOUNT, DCOUNT_ODF, database, 'age');
            it('should return the count the defined column with the criteria', function () {
                testContainer.testValue(r3d('1:1!A1:F2'), 1);
                testContainer.testValue(r3d('1:1!G1:G2'), 0);
                // if the odf regex feature is implemented change the test to testValue(r3d('0:0!A1:A3'), 4, 2);
                testContainer.testValue(r3d('1:1!A1:A3'), 4);
                // test null field name
                expect(DCOUNT(database, null, r3d('1:1!A1:F2'))).to.equal(2);
            });
        });

        moduleTester.testFunction('DCOUNTA', function (DCOUNTA, DCOUNTA_ODF) {
            var database = r3d('2:2!A5:E11');
            var testContainer = new TestContainer(DCOUNTA, DCOUNTA_ODF, database, 'age');
            it('should return the count the defined column with the criteria', function () {
                testContainer.testValue(r3d('2:2!A1:F2'), 0);
                testContainer.testValue(r3d('2:2!G1:G2'), 0);
                // without field value count all rows
                expect(DCOUNTA(database, null, r3d('2:2!B1:C2'))).to.equal(3);
                // Labels are numbers
                expect(DCOUNTA(database, 1, r3d('2:2!E1:E2'))).to.equal(5);
            });
        });

        moduleTester.testFunction('DGET', function (DGET, DGET_ODF) {
            var database = r3d('3:3!A5:E11');
            var testContainer = new TestContainer(DGET, DGET_ODF, database, 'yieLd');
            it('should return the count the defined column with the criteria', function () {
                // More than one match
                testContainer.testValue(r3d('3:3!A1:A3'), ErrorCode.NUM);
                // Exactly one match
                testContainer.testValue(r3d('3:3!A1:F2'), 10);
                // Without a match
                testContainer.testValue(r3d('3:3!G1:G2'), ErrorCode.VALUE);
                // One match with word
                expect(DGET(database, 1, r3d('3:3!A1:F3'))).to.equal('Apple');
            });
        });

        moduleTester.testFunction('DMAX', function (DMAX, DMAX_ODF) {
            var database = r3d('4:4!A5:E11');
            var testContainer = new TestContainer(DMAX, DMAX_ODF, database, 5);
            it('should return the count the defined column with the criteria', function () {
                testContainer.testValue(r3d('4:4!A1:F3'), 96);
                // No match
                testContainer.testValue(r3d('4:4!G1:G2'), 0);
                // Only Text
                expect(DMAX(database, 'TRee', r3d('4:4!A1:F3'))).to.equal(0);
                // Only empty cells
                expect(DMAX(r3d('4:4!A5:F11'), 'EmPty', r3d('4:4!A1:F3'))).to.equal(0);
            });
        });

        moduleTester.testFunction('DMIN', function (DMIN, DMIN_ODF) {
            var database = r3d('5:5!A5:E11');
            var testContainer = new TestContainer(DMIN, DMIN_ODF, database, 5);
            it('should return the count the defined column with the criteria', function () {
                testContainer.testValue(r3d('5:5!A1:F3'), 75);
                // No match
                testContainer.testValue(r3d('5:5!G1:G2'), 0);
                // Only Text
                expect(DMIN(database, 'TRee', r3d('5:5!A1:F3'))).to.equal(0);
                // Only empty cells
                expect(DMIN(r3d('5:5!A5:F11'), 'EmPty', r3d('5:5!A1:F3'))).to.equal(0);
            });
        });

        moduleTester.testFunction('DPRODUCT', function (DPRODUCT, DPRODUCT_ODF) {
            var database = r3d('6:6!A5:F11');
            var testContainer = new TestContainer(DPRODUCT, DPRODUCT_ODF, database, 'width');
            it('should return the count the defined column with the criteria', function () {
                expect(DPRODUCT(database, 'Yield', r3d('6:6!A1:F3'))).to.equal(-800);
                expect(DPRODUCT(database, 'Profit', r3d('6:6!A1:F3'))).to.equal(554400);
                expect(DPRODUCT(database, 'age', r3d('6:6!A1:F3'))).to.equal(150);
                expect(DPRODUCT(database, 1, r3d('6:6!A1:F3'))).to.equal(0);
                testContainer.testValue(r3d('6:6!A1:F3'), 0);
                testContainer.testValue(r3d('6:6!G1:G2'), 52);
            });
        });

        moduleTester.testFunction('DSTDEV', function (DSTDEV) {
            var database = r3d('7:7!A5:F11');
            it('should return the count the defined column with the criteria', function () {
                expect(DSTDEV(database, 'yield', r3d('7:7!A1:A3'))).to.be.closeTo(3.415650255, 5e-10);
                expect(DSTDEV(database, 1, r3d('7:7!B1:B2'))).to.equal(ErrorCode.DIV0);
                expect(DSTDEV(database, 'yield', r3d('7:7!B1:B2'))).to.be.closeTo(2.645751311, 5e-10);
            });
        });

        moduleTester.testFunction('DSTDEVP', function (DSTDEVP) {
            var database = r3d('7:7!A5:F11');
            it('should return the count the defined column with the criteria', function () {
                expect(DSTDEVP(database, 'yield', r3d('7:7!A1:A3'))).to.be.closeTo(2.958039892, 5e-10);
                expect(DSTDEVP(database, 1, r3d('7:7!B1:B2'))).to.equal(ErrorCode.DIV0);
                expect(DSTDEVP(database, 'yield', r3d('7:7!B1:B2'))).to.be.closeTo(2.160246899, 5e-10);
            });
        });

        moduleTester.testFunction('DSUM', function (DSUM, DSUM_ODF) {
            var database = r3d('7:7!A5:F11');
            var testContainer = new TestContainer(DSUM, DSUM_ODF, database, 'yield');
            it('should return the count the defined column with the criteria', function () {
                testContainer.testValue(r3d('7:7!A1:A3'), 38);
                expect(DSUM(database, 1, r3d('7:7!B1:B2'))).to.equal(0);
                testContainer.testValue(r3d('7:7!B1:B2'), 33);
            });
        });

        moduleTester.testFunction('DVAR', function (DVAR) {
            var database = r3d('7:7!A5:F11');
            it('should return the count the defined column with the criteria', function () {
                expect(DVAR(database, 'yield', r3d('7:7!A1:A3'))).to.be.closeTo(11.66666667, 1e-8);
                expect(DVAR(database, 1, r3d('7:7!B1:B2'))).to.equal(ErrorCode.DIV0);
                expect(DVAR(database, 'yield', r3d('7:7!B1:B2'))).to.equal(7);
            });
        });

        moduleTester.testFunction('DVARP', function (DVARP) {
            var database = r3d('7:7!A5:F11');
            it('should return the count the defined column with the criteria', function () {
                expect(DVARP(database, 'yield', r3d('7:7!A1:A3'))).to.equal(8.75);
                expect(DVARP(database, 1, r3d('7:7!B1:B2'))).to.equal(ErrorCode.DIV0);
                expect(DVARP(database, 'yield', r3d('7:7!B1:B2'))).to.be.closeTo(4.666666667, 5e-10);
            });
        });
    });

    // ========================================================================
});
