/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

/* eslint new-cap: 0 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/model/formula/funcs/conversionfuncs'
], function (AppHelper, SheetHelper, Utils, ConversionFuncs) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetHelper.ErrorCode;

    // module ConversionFuncs =================================================

    describe('Spreadsheet module ConversionFuncs', function () {

        // initialize the test
        var appPromise = AppHelper.createSpreadsheetApp('ooxml');
        var moduleTester = SheetHelper.createFunctionModuleTester(ConversionFuncs, appPromise);

        // function implementations -------------------------------------------

        moduleTester.testFunction('ARABIC', function (ARABIC) {
            it('should convert roman to decimal', function () {
                expect(ARABIC('')).to.equal(0);
                expect(ARABIC('I')).to.equal(1);
                expect(ARABIC('V')).to.equal(5);
                expect(ARABIC('X')).to.equal(10);
                expect(ARABIC('L')).to.equal(50);
                expect(ARABIC('C')).to.equal(100);
                expect(ARABIC('D')).to.equal(500);
                expect(ARABIC('M')).to.equal(1000);
                expect(ARABIC('II')).to.equal(2);
                expect(ARABIC('III')).to.equal(3);
                expect(ARABIC('IIII')).to.equal(4);
                expect(ARABIC('IIIII')).to.equal(5);
                expect(ARABIC('MMMDDDCCCLLLXXXVVVIII')).to.equal(4998);
                expect(ARABIC('DMCDLCXLVXIV')).to.equal(999);
                expect(ARABIC('MDCLXVI')).to.equal(1666);
                expect(ARABIC('IVXLCDM')).to.equal(334);
                expect(ARABIC(Utils.repeatString('M', 255))).to.equal(255000);
            });
            it('should accept lower-case strings', function () {
                expect(ARABIC('mdclxvi')).to.equal(1666);
                expect(ARABIC('mDcLxVi')).to.equal(1666);
                expect(ARABIC('MdClXvI')).to.equal(1666);
            });
            it('should trim white-space', function () {
                expect(ARABIC('  XLII  ')).to.equal(42);
            });
            it('should accept leading minus sign', function () {
                expect(ARABIC('-XLII')).to.equal(-42);
                expect(ARABIC('  -XLII  ')).to.equal(-42);
                expect(ARABIC('-' + Utils.repeatString('M', 254))).to.equal(-254000);
            });
            it('should throw #VALUE! error code for invalid parameter', function () {
                expect(ARABIC('abc')).to.equal(ErrorCode.VALUE);
                expect(ARABIC('I I')).to.equal(ErrorCode.VALUE);
                expect(ARABIC('- I')).to.equal(ErrorCode.VALUE);
                expect(ARABIC('I-')).to.equal(ErrorCode.VALUE);
                expect(ARABIC('42')).to.equal(ErrorCode.VALUE);
                expect(ARABIC(Utils.repeatString('M', 256))).to.equal(ErrorCode.VALUE);
                expect(ARABIC('-' + Utils.repeatString('M', 255))).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('BASE', function (BASE) {
            it('should convert any base to decimal', function () {
                expect(BASE(0, 2)).to.equal('0');
                expect(BASE(42, 2)).to.equal('101010');
                expect(BASE(42, 3)).to.equal('1120');
                expect(BASE(42, 4)).to.equal('222');
                expect(BASE(42, 5)).to.equal('132');
                expect(BASE(42, 6)).to.equal('110');
                expect(BASE(42, 7)).to.equal('60');
                expect(BASE(42, 8)).to.equal('52');
                expect(BASE(42, 10)).to.equal('42');
                expect(BASE(42, 15)).to.equal('2C');
                expect(BASE(42, 16)).to.equal('2A');
                expect(BASE(42, 22)).to.equal('1K');
                expect(BASE(1295, 36)).to.equal('ZZ');
                expect(BASE(Math.pow(2, 53) - 1, 2)).to.match(/^1{53}$/);
                expect(BASE(Math.pow(2, 53) - 1, 5)).to.equal('33421042423033203202431');
                expect(BASE(Math.pow(2, 53) - 1, 9)).to.equal('47664754584305344');
                expect(BASE(Math.pow(2, 53) - 1, 17)).to.equal('F7DED8C9E1F8E');
                expect(BASE(Math.pow(2, 53) - 1, 32)).to.equal('7VVVVVVVVVV');
                expect(BASE(Math.pow(2, 53) - 1, 36)).to.equal('2GOSA7PA2GV');
            });
            it('should expand result with zeros', function () {
                expect(BASE(3, 2, 0)).to.equal('11');
                expect(BASE(3, 2, 1)).to.equal('11');
                expect(BASE(3, 2, 2)).to.equal('11');
                expect(BASE(3, 2, 4)).to.equal('0011');
                expect(BASE(3, 2, 10)).to.equal('0000000011');
                expect(BASE(3, 2, 255)).to.match(/^0{253}11$/);
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(BASE(-1, 2)).to.equal(ErrorCode.NUM);
                expect(BASE(-1, Math.pow(2, 53))).to.equal(ErrorCode.NUM);
                expect(BASE(1, 1)).to.equal(ErrorCode.NUM);
                expect(BASE(1, 37)).to.equal(ErrorCode.NUM);
                expect(BASE(1, 2, -1)).to.equal(ErrorCode.NUM);
                expect(BASE(1, 2, 256)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('BIN2DEC', function (BIN2DEC) {
            it('should convert binary to decimal', function () {
                expect(BIN2DEC('')).to.equal(0);
                expect(BIN2DEC('0')).to.equal(0);
                expect(BIN2DEC('101010')).to.equal(42);
                expect(BIN2DEC('0111111111')).to.equal(511);
                expect(BIN2DEC('1000000000')).to.equal(-512);
                expect(BIN2DEC('1111111111')).to.equal(-1);
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(BIN2DEC('2')).to.equal(ErrorCode.NUM);
                expect(BIN2DEC('abc')).to.equal(ErrorCode.NUM);
                expect(BIN2DEC('10000000000')).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('BIN2OCT', function (BIN2OCT) {
            it('should convert binary to octal', function () {
                expect(BIN2OCT('')).to.equal('0');
                expect(BIN2OCT('0')).to.equal('0');
                expect(BIN2OCT('100010')).to.equal('42');
                expect(BIN2OCT('0111111111')).to.equal('777');
                expect(BIN2OCT('1000000000')).to.equal('7777777000');
                expect(BIN2OCT('1111111111')).to.equal('7777777777');
            });
            it('should expand the result with leading zeros', function () {
                expect(BIN2OCT('100010', 2)).to.equal('42');
                expect(BIN2OCT('100010', 4)).to.equal('0042');
                expect(BIN2OCT('100010', 10)).to.equal('0000000042');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(BIN2OCT('abc')).to.equal(ErrorCode.NUM);
                expect(BIN2OCT('10000000000')).to.equal(ErrorCode.NUM);
                expect(BIN2OCT('100010', 1)).to.equal(ErrorCode.NUM);
                expect(BIN2OCT('100010', 11)).to.equal(ErrorCode.NUM);
                expect(BIN2OCT('0', 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('BIN2HEX', function (BIN2HEX) {
            it('should convert binary to hexadecimal', function () {
                expect(BIN2HEX('')).to.equal('0');
                expect(BIN2HEX('0')).to.equal('0');
                expect(BIN2HEX('1000010')).to.equal('42');
                expect(BIN2HEX('0111111111')).to.equal('1FF');
                expect(BIN2HEX('1000000000')).to.equal('FFFFFFFE00');
                expect(BIN2HEX('1111111111')).to.equal('FFFFFFFFFF');
            });
            it('should expand the result with leading zeros', function () {
                expect(BIN2HEX('1000010', 2)).to.equal('42');
                expect(BIN2HEX('1000010', 4)).to.equal('0042');
                expect(BIN2HEX('1000010', 10)).to.equal('0000000042');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(BIN2HEX('efg')).to.equal(ErrorCode.NUM);
                expect(BIN2HEX('10000000000')).to.equal(ErrorCode.NUM);
                expect(BIN2HEX('1000010', 1)).to.equal(ErrorCode.NUM);
                expect(BIN2HEX('1000010', 11)).to.equal(ErrorCode.NUM);
                expect(BIN2HEX('0', 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('COLOR', function (COLOR) {
            it('should return RGB color values', function () {
                expect(COLOR(0, 0, 0)).to.equal(0);
                expect(COLOR(1, 2, 3)).to.equal(0x00010203);
                expect(COLOR(252, 253, 254)).to.equal(0x00fcfdfe);
            });
            it('should return RGBA color values', function () {
                expect(COLOR(0, 0, 0, 0)).to.equal(0);
                expect(COLOR(1, 2, 3, 4)).to.equal(0x04010203);
                expect(COLOR(252, 253, 254, 255)).to.equal(0xfffcfdfe);
            });
            it('should throw error code for invalid arguments', function () {
                expect(COLOR(-1, 0, 0, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(256, 0, 0, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, -1, 0, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, 256, 0, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, 0, -1, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, 0, 256, 0)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, 0, 0, -1)).to.equal(ErrorCode.VALUE);
                expect(COLOR(0, 0, 0, 256)).to.equal(ErrorCode.VALUE);
            });
        });

        moduleTester.testFunction('CONVERT', function (CONVERT) {
            it('should return correct results for length', function () {
                expect(CONVERT(1, 'm', 'm')).to.equal(1);
                expect(CONVERT(1, 'mi', 'm')).to.equal(1609.344);
                expect(CONVERT(1, 'Nmi', 'm')).to.equal(1852);
                expect(CONVERT(1, 'in', 'm')).to.equal(0.0254);
                expect(CONVERT(1, 'ft', 'm')).to.equal(0.3048);
                expect(CONVERT(1, 'yd', 'm')).to.equal(0.9144);
                expect(CONVERT(1, 'ang', 'm')).to.equal(1E-10);
                expect(CONVERT(1, 'Pica', 'm')).to.equal(0.003175 / 9);
                expect(CONVERT(1, 'ell', 'm')).to.equal(1.143);
                expect(CONVERT(1, 'parsec', 'm')).to.equal(3.08567758128155E+16);
                expect(CONVERT(1, 'pc', 'm')).to.equal(3.08567758128155E+16);
                expect(CONVERT(1, 'ly', 'm')).to.equal(9.4607304725808E+15);
                expect(CONVERT(1, 'survey_mi', 'm')).to.equal(1609.34721869444);
            });
            it('should return correct results for area', function () {
                expect(CONVERT(1, 'm2', 'm2')).to.equal(1);
                expect(CONVERT(1, 'mi2', 'm2')).to.equal(2589988.110336);
                expect(CONVERT(1, 'Nmi2', 'm2')).to.equal(3429904);
                expect(CONVERT(1, 'in2', 'm2')).to.equal(0.00064516);
                expect(CONVERT(1, 'ft2', 'm2')).to.equal(0.09290304);
                expect(CONVERT(1, 'yd2', 'm2')).to.almostEqual(0.83612736);
                expect(CONVERT(1, 'ang2', 'm2')).to.equal(1E-20);
                expect(CONVERT(1, 'Pica2', 'm2')).to.equal(1.24452160493827E-07);
                expect(CONVERT(1, 'ly2', 'm2')).to.equal(8.95054210748189E+31);
            });
            it('should return correct results for volume', function () {
                expect(CONVERT(1, 'm3', 'l')).to.equal(1000);
                expect(CONVERT(1, 'mi3', 'l')).to.equal(4.16818182544058E+12);
                expect(CONVERT(1, 'Nmi3', 'l')).to.equal(6.352182208E+12);
                expect(CONVERT(1, 'in3', 'l')).to.equal(0.016387064);
                expect(CONVERT(1, 'ft3', 'l')).to.almostEqual(28.316846592);
                expect(CONVERT(1, 'yd3', 'l')).to.almostEqual(764.554857984);
                expect(CONVERT(1, 'ang3', 'l')).to.equal(1E-27);
                expect(CONVERT(1, 'Pica3', 'l')).to.equal(4.39039566186557E-08);
                expect(CONVERT(1, 'ly3', 'l')).to.equal(8.46786664623715E+50);
            });
            it('should return correct results for temperatures', function () {
                expect(CONVERT(1000, 'g', 'kg')).to.equal(1);
                expect(CONVERT(1, 'kg', 'g')).to.equal(1000);
                expect(CONVERT(1, 'km', 'm')).to.equal(1000);
                expect(CONVERT(1000, 'm', 'km')).to.equal(1);
                expect(CONVERT(1000000, 'm2', 'km2')).to.equal(1);
                expect(CONVERT(1000000000, 'm3', 'km3')).to.equal(1);
                expect(CONVERT(1, 'lbm', 'kg')).to.be.closeTo(0.45359237, 1e-6);
                expect(CONVERT(0, 'C', 'K')).to.equal(273.15);
                expect(CONVERT(0, 'K', 'C')).to.equal(-273.15);
                expect(CONVERT(68, 'F', 'K')).to.equal(293.15);
                expect(CONVERT(68, 'F', 'C')).to.equal(20);
                expect(CONVERT(100, 'ft', 'm')).to.equal(30.48);
                expect(CONVERT(30.48, 'ft', 'm')).to.equal(9.290304);
            });
            it('should throw #N/A error for invalid parameters', function () {
                expect(CONVERT(1, 'ly', 'm2')).to.equal(ErrorCode.NA);
                expect(CONVERT(68, 'f', 'c')).to.equal(ErrorCode.NA);
                expect(CONVERT(2.5, 'ft', 'sec')).to.equal(ErrorCode.NA);
            });
        });

        moduleTester.testFunction('DEC2BIN', function (DEC2BIN) {
            it('should convert decimal to binary', function () {
                expect(DEC2BIN(0)).to.equal('0');
                expect(DEC2BIN(42)).to.equal('101010');
                expect(DEC2BIN(511)).to.equal('111111111');
                expect(DEC2BIN(-512)).to.equal('1000000000');
                expect(DEC2BIN(-1)).to.equal('1111111111');
            });
            it('should expand the result with leading zeros', function () {
                expect(DEC2BIN(3, 2)).to.equal('11');
                expect(DEC2BIN(3, 4)).to.equal('0011');
                expect(DEC2BIN(3, 10)).to.equal('0000000011');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(DEC2BIN(512)).to.equal(ErrorCode.NUM);
                expect(DEC2BIN(-513)).to.equal(ErrorCode.NUM);
                expect(DEC2BIN(3, 1)).to.equal(ErrorCode.NUM);
                expect(DEC2BIN(3, 11)).to.equal(ErrorCode.NUM);
                expect(DEC2BIN(0, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DEC2HEX', function (DEC2HEX) {
            it('should convert decimal to binary', function () {
                expect(DEC2HEX(0)).to.equal('0');
                expect(DEC2HEX(42)).to.equal('2A');
                expect(DEC2HEX(0x7FFFFFFFFF)).to.equal('7FFFFFFFFF');
                expect(DEC2HEX(-0x8000000000)).to.equal('8000000000');
                expect(DEC2HEX(-1)).to.equal('FFFFFFFFFF');
            });
            it('should expand the result with leading zeros', function () {
                expect(DEC2HEX(42, 2)).to.equal('2A');
                expect(DEC2HEX(42, 4)).to.equal('002A');
                expect(DEC2HEX(42, 10)).to.equal('000000002A');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(DEC2HEX(0x8000000000)).to.equal(ErrorCode.NUM);
                expect(DEC2HEX(-0x8000000001)).to.equal(ErrorCode.NUM);
                expect(DEC2HEX(42, 1)).to.equal(ErrorCode.NUM);
                expect(DEC2HEX(42, 11)).to.equal(ErrorCode.NUM);
                expect(DEC2HEX(0, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DEC2OCT', function (DEC2OCT) {
            it('should convert decimal to binary', function () {
                expect(DEC2OCT(0)).to.equal('0');
                expect(DEC2OCT(42)).to.equal('52');
                expect(DEC2OCT(0x1FFFFFFF)).to.equal('3777777777');
                expect(DEC2OCT(-0x20000000)).to.equal('4000000000');
                expect(DEC2OCT(-1)).to.equal('7777777777');
            });
            it('should expand the result with leading zeros', function () {
                expect(DEC2OCT(42, 2)).to.equal('52');
                expect(DEC2OCT(42, 4)).to.equal('0052');
                expect(DEC2OCT(42, 10)).to.equal('0000000052');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(DEC2OCT(0x20000000)).to.equal(ErrorCode.NUM);
                expect(DEC2OCT(-0x20000001)).to.equal(ErrorCode.NUM);
                expect(DEC2OCT(42, 1)).to.equal(ErrorCode.NUM);
                expect(DEC2OCT(42, 11)).to.equal(ErrorCode.NUM);
                expect(DEC2OCT(0, 0)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('DECIMAL', function (DECIMAL) {
            it('should convert a value to the decimal value', function () {
                expect(DECIMAL(1, 36)).to.equal(1);
                expect(DECIMAL('FF', 16)).to.equal(255);
                expect(DECIMAL('111', 2)).to.equal(7);
                expect(DECIMAL('zap', 36)).to.equal(45745);
                expect(DECIMAL(Utils.repeatString('f', 198), 36)).to.almostEqual(6.02446334927868e+307);
            });
            it('should throw #NUM! for invalid parameters', function () {
                expect(DECIMAL(1, 37)).to.equal(ErrorCode.NUM);
                expect(DECIMAL(-1, 36)).to.equal(ErrorCode.NUM);
                expect(DECIMAL(Utils.repeatString('f', 199), 36)).to.equal(ErrorCode.NUM);
                expect(DECIMAL('123h', 8)).to.equal(ErrorCode.NUM);
                expect(DECIMAL('ff', 15)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('HEX2BIN', function (HEX2BIN) {
            it('should convert hexadecimal to binary', function () {
                expect(HEX2BIN('')).to.equal('0');
                expect(HEX2BIN('0')).to.equal('0');
                expect(HEX2BIN('2A')).to.equal('101010');
                expect(HEX2BIN('02a')).to.equal('101010');
                expect(HEX2BIN('1FF')).to.equal('111111111');
                expect(HEX2BIN('FFFFFFFE00')).to.equal('1000000000');
                expect(HEX2BIN('FFFFFFFFFF')).to.equal('1111111111');
            });
            it('should expand the result with leading zeros', function () {
                expect(HEX2BIN('3', 2)).to.equal('11');
                expect(HEX2BIN('3', 4)).to.equal('0011');
                expect(HEX2BIN('3', 10)).to.equal('0000000011');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(HEX2BIN('efg')).to.equal(ErrorCode.NUM);
                expect(HEX2BIN('200')).to.equal(ErrorCode.NUM);
                expect(HEX2BIN('FFFFFFFDFF')).to.equal(ErrorCode.NUM);
                expect(HEX2BIN('3', 1)).to.equal(ErrorCode.NUM);
                expect(HEX2BIN('3', 11)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('HEX2DEC', function (HEX2DEC) {
            it('should convert hexadecimal to decimal', function () {
                expect(HEX2DEC('')).to.equal(0);
                expect(HEX2DEC('0')).to.equal(0);
                expect(HEX2DEC('2A')).to.equal(42);
                expect(HEX2DEC('02a')).to.equal(42);
                expect(HEX2DEC('7FFFFFFFFF')).to.equal(0x7fffffffff);
                expect(HEX2DEC('8000000000')).to.equal(-0x8000000000);
                expect(HEX2DEC('FFFFFFFFFF')).to.equal(-1);
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(HEX2DEC('efg')).to.equal(ErrorCode.NUM);
                expect(HEX2DEC('10000000000')).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('HEX2OCT', function (HEX2OCT) {
            it('should convert hexadecimal to octal', function () {
                expect(HEX2OCT('')).to.equal('0');
                expect(HEX2OCT('0')).to.equal('0');
                expect(HEX2OCT('2A')).to.equal('52');
                expect(HEX2OCT('02a')).to.equal('52');
                expect(HEX2OCT('1FFFFFFF')).to.equal('3777777777');
                expect(HEX2OCT('FFE0000000')).to.equal('4000000000');
                expect(HEX2OCT('FFFFFFFFFF')).to.equal('7777777777');
            });
            it('should expand the result with leading zeros', function () {
                expect(HEX2OCT('22', 2)).to.equal('42');
                expect(HEX2OCT('22', 4)).to.equal('0042');
                expect(HEX2OCT('22', 10)).to.equal('0000000042');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(HEX2OCT('efg')).to.equal(ErrorCode.NUM);
                expect(HEX2OCT('20000000')).to.equal(ErrorCode.NUM);
                expect(HEX2OCT('FFDFFFFFFF')).to.equal(ErrorCode.NUM);
                expect(HEX2OCT('22', 1)).to.equal(ErrorCode.NUM);
                expect(HEX2OCT('22', 11)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('OCT2BIN', function (OCT2BIN) {
            it('should convert octal to binary', function () {
                expect(OCT2BIN('')).to.equal('0');
                expect(OCT2BIN('0')).to.equal('0');
                expect(OCT2BIN('42')).to.equal('100010');
                expect(OCT2BIN('042')).to.equal('100010');
                expect(OCT2BIN('777')).to.equal('111111111');
                expect(OCT2BIN('7777777000')).to.equal('1000000000');
                expect(OCT2BIN('7777777777')).to.equal('1111111111');
            });
            it('should expand the result with leading zeros', function () {
                expect(OCT2BIN('3', 2)).to.equal('11');
                expect(OCT2BIN('3', 4)).to.equal('0011');
                expect(OCT2BIN('3', 10)).to.equal('0000000011');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(OCT2BIN('678')).to.equal(ErrorCode.NUM);
                expect(OCT2BIN('1000')).to.equal(ErrorCode.NUM);
                expect(OCT2BIN('7777776777')).to.equal(ErrorCode.NUM);
                expect(OCT2BIN('3', 1)).to.equal(ErrorCode.NUM);
                expect(OCT2BIN('3', 11)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('OCT2DEC', function (OCT2DEC) {
            it('should convert octal to decimal', function () {
                expect(OCT2DEC('')).to.equal(0);
                expect(OCT2DEC('0')).to.equal(0);
                expect(OCT2DEC('42')).to.equal(34);
                expect(OCT2DEC('042')).to.equal(34);
                expect(OCT2DEC('3777777777')).to.equal(0x1FFFFFFF);
                expect(OCT2DEC('4000000000')).to.equal(-0x20000000);
                expect(OCT2DEC('7777777777')).to.equal(-1);
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(OCT2DEC('678')).to.equal(ErrorCode.NUM);
                expect(OCT2DEC('10000000000')).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('OCT2HEX', function (OCT2HEX) {
            it('should convert octal to hexadecimal', function () {
                expect(OCT2HEX('')).to.equal('0');
                expect(OCT2HEX('0')).to.equal('0');
                expect(OCT2HEX('42')).to.equal('22');
                expect(OCT2HEX('042')).to.equal('22');
                expect(OCT2HEX('3777777777')).to.equal('1FFFFFFF');
                expect(OCT2HEX('4000000000')).to.equal('FFE0000000');
                expect(OCT2HEX('7777777777')).to.equal('FFFFFFFFFF');
            });
            it('should expand the result with leading zeros', function () {
                expect(OCT2HEX('42', 2)).to.equal('22');
                expect(OCT2HEX('42', 4)).to.equal('0022');
                expect(OCT2HEX('42', 10)).to.equal('0000000022');
            });
            it('should throw #NUM! error code for invalid parameter', function () {
                expect(OCT2HEX('678')).to.equal(ErrorCode.NUM);
                expect(OCT2HEX('10000000000')).to.equal(ErrorCode.NUM);
                expect(OCT2HEX('42', 1)).to.equal(ErrorCode.NUM);
                expect(OCT2HEX('42', 11)).to.equal(ErrorCode.NUM);
            });
        });

        moduleTester.testFunction('ROMAN', function (ROMAN) {
            it('should convert decimal to roman', function () {
                expect(ROMAN(1)).to.equal('I');
                expect(ROMAN(2)).to.equal('II');
                expect(ROMAN(3)).to.equal('III');
                expect(ROMAN(4)).to.equal('IV');
                expect(ROMAN(499)).to.equal('CDXCIX');
                expect(ROMAN(499, 0)).to.equal('CDXCIX');
                expect(ROMAN(499, 1)).to.equal('LDVLIV');
                expect(ROMAN(499, 2)).to.equal('XDIX');
                expect(ROMAN(499, 3)).to.equal('VDIV');
                expect(ROMAN(499, 4)).to.equal('ID');
                expect(ROMAN(0)).to.equal('');
            });
            it('should throw #VALUE! error code for invalid parameter', function () {
                expect(ROMAN(-1)).to.equal(ErrorCode.VALUE);
                expect(ROMAN(4000)).to.equal(ErrorCode.VALUE);
                expect(ROMAN(1, -1)).to.equal(ErrorCode.VALUE);
                expect(ROMAN(1, 5)).to.equal(ErrorCode.VALUE);
            });
        });
    });

    // ========================================================================
});
