/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'io.ox/office/tk/locale/parser',
    'io.ox/office/spreadsheet/utils/errorcode',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/utils/mathutils',
    'io.ox/office/spreadsheet/model/formula/utils/scalar',
    'io.ox/office/spreadsheet/model/formula/utils/complex',
    'io.ox/office/spreadsheet/model/formula/utils/matrix',
    'io.ox/office/spreadsheet/model/formula/utils/dimension',
    'io.ox/office/spreadsheet/model/formula/utils/formulaerror',
    'io.ox/office/spreadsheet/model/formula/utils/cellref',
    'io.ox/office/spreadsheet/model/formula/utils/sheetref'
], function (Parser, ErrorCode, FormulaUtils, MathUtils, Scalar, Complex, Matrix, Dimension, FormulaError, CellRef, SheetRef) {

    'use strict';

    // static class FormulaUtils ==============================================

    describe('Spreadsheet module FormulaUtils', function () {

        it('should exist', function () {
            expect(FormulaUtils).to.be.an('object');
        });

        // helper classes -----------------------------------------------------

        describe('helper class MathUtils', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('Math', MathUtils);
            });
        });

        describe('helper class Scalar', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('Scalar', Scalar);
            });
        });

        describe('helper class Complex', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('Complex', Complex);
            });
        });

        describe('helper class Matrix', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('Matrix', Matrix);
            });
        });

        describe('helper class Dimension', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('Dimension', Dimension);
            });
        });

        describe('helper class FormulaError', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('FormulaError', FormulaError);
            });
        });

        describe('helper class CellRef', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('CellRef', CellRef);
            });
        });

        describe('helper class SheetRef', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('SheetRef', SheetRef);
            });
        });

        // constants ----------------------------------------------------------

        describe('constant "MAX_PARAM_COUNT"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_PARAM_COUNT').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_REF_LIST_SIZE"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_REF_LIST_SIZE').that.is.a('number').and.above(0);
            });
        });

        describe('constant "MAX_EVAL_TIME"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('MAX_EVAL_TIME').that.is.a('number').and.above(0);
            });
        });

        describe('constant "UNSUPPORTED_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('UNSUPPORTED_ERROR').that.is.an('object');
                expect(FormulaUtils.UNSUPPORTED_ERROR).to.have.a.property('key', 'unsupported');
            });
        });

        describe('constant "CIRCULAR_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('CIRCULAR_ERROR').that.is.an('object');
                expect(FormulaUtils.CIRCULAR_ERROR).to.have.a.property('key', 'circular');
            });
        });

        describe('constant "TIMEOUT_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('TIMEOUT_ERROR').that.is.an('object');
                expect(FormulaUtils.TIMEOUT_ERROR).to.have.a.property('key', 'timeout');
            });
        });

        describe('constant "INTERNAL_ERROR"', function () {
            it('should exist', function () {
                expect(FormulaUtils).to.have.a.property('INTERNAL_ERROR').that.is.an('object');
                expect(FormulaUtils.INTERNAL_ERROR).to.have.a.property('key', 'internal');
            });
        });

        // logger -------------------------------------------------------------

        describe('should support the logger interface"', function () {
            expect(FormulaUtils).itself.to.respondTo('log');
            expect(FormulaUtils).itself.to.respondTo('info');
            expect(FormulaUtils).itself.to.respondTo('warn');
            expect(FormulaUtils).itself.to.respondTo('error');
        });

        // static methods -----------------------------------------------------

        describe('method "isInternalError"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('isInternalError');
            });
            it('should return true for internal error codes', function () {
                expect(FormulaUtils.isInternalError(FormulaUtils.UNSUPPORTED_ERROR)).to.equal(true);
                expect(FormulaUtils.isInternalError(FormulaUtils.CIRCULAR_ERROR)).to.equal(true);
                expect(FormulaUtils.isInternalError(FormulaUtils.INTERNAL_ERROR)).to.equal(true);
            });
            it('should return false for other values', function () {
                expect(FormulaUtils.isInternalError(null)).to.equal(false);
                expect(FormulaUtils.isInternalError(42)).to.equal(false);
                expect(FormulaUtils.isInternalError(ErrorCode.VALUE)).to.equal(false);
            });
        });

        describe('method "throwInternal"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwInternal');
            });
            it('should always throw', function () {
                expect(FormulaUtils.throwInternal).to.throw(ErrorCode).that.equals(FormulaUtils.INTERNAL_ERROR);
            });
        });

        describe('method "throwErrorCode"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('throwErrorCode');
            });
            it('should return scalar values', function () {
                expect(FormulaUtils.throwErrorCode(42)).to.equal(42);
                expect(FormulaUtils.throwErrorCode('abc')).to.equal('abc');
                expect(FormulaUtils.throwErrorCode(true)).to.equal(true);
                expect(FormulaUtils.throwErrorCode(new Date())).to.be.a('date');
                expect(FormulaUtils.throwErrorCode(null)).to.equal(null);
            });
            it('should throw error codes', function () {
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.NULL)).to.throw(ErrorCode).that.equals(ErrorCode.NULL);
                expect(FormulaUtils.throwErrorCode.bind(null, ErrorCode.VALUE)).to.throw(ErrorCode).that.equals(ErrorCode.VALUE);
            });
        });

        describe('method "ensureMatrixDim"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('ensureMatrixDim');
            });
            it('should do nothing for valid matrix sizes', function () {
                // simply expect that the function does not throw
                FormulaUtils.ensureMatrixDim(1, 1);
                FormulaUtils.ensureMatrixDim(Matrix.MAX_SIZE, 1);
                FormulaUtils.ensureMatrixDim(1, Matrix.MAX_SIZE);
                FormulaUtils.ensureMatrixDim(new Dimension(1, Matrix.MAX_SIZE));
            });
            it('should throw for invalid matrix sizes', function () {
                expect(FormulaUtils.ensureMatrixDim.bind(null, 0, 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixDim.bind(null, 1, 0)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixDim.bind(null, Matrix.MAX_SIZE + 1, 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixDim.bind(null, 1, Matrix.MAX_SIZE + 1)).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
                expect(FormulaUtils.ensureMatrixDim.bind(null, new Dimension(1, Matrix.MAX_SIZE + 1))).to.throw(ErrorCode).that.equals(FormulaUtils.UNSUPPORTED_ERROR);
            });
        });

        describe('method "combineParsedFormats"', function () {

            var stdFormat, percentFormat, currFormat, fracFormat, dateFormat, timeFormat, dateTimeFormat;
            before(function () {
                stdFormat = Parser.parseFormatCode('ooxml', 'op', 'General');
                percentFormat = Parser.parseFormatCode('ooxml', 'op', '0%');
                currFormat = Parser.parseFormatCode('ooxml', 'op', '0[$\u20ac]');
                fracFormat = Parser.parseFormatCode('ooxml', 'op', '# ?/?');
                dateFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD');
                timeFormat = Parser.parseFormatCode('ooxml', 'op', 'hh:mm:ss');
                dateTimeFormat = Parser.parseFormatCode('ooxml', 'op', 'YYYY-MM-DD hh:mm:ss');
            });

            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('combineParsedFormats');
            });
            it('should return null without if any format is null', function () {
                expect(FormulaUtils.combineParsedFormats(null, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(stdFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(percentFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(currFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(fracFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, null)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, stdFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, percentFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, currFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, fracFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(null, dateTimeFormat)).to.equal(null);
            });
            it('should skip standard category', function () {
                expect(FormulaUtils.combineParsedFormats(stdFormat, stdFormat)).to.equal(stdFormat);

                expect(FormulaUtils.combineParsedFormats(percentFormat, stdFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, stdFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, stdFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, stdFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, stdFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, stdFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(stdFormat, percentFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, currFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, fracFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(stdFormat, dateTimeFormat)).to.equal(dateTimeFormat);
            });
            it('should add different category commbinations', function () {
                expect(FormulaUtils.combineParsedFormats(percentFormat, percentFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, currFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, fracFormat)).to.equal(percentFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(percentFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(currFormat, percentFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, currFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, fracFormat)).to.equal(currFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(currFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(fracFormat, percentFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, currFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, fracFormat)).to.equal(fracFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, dateFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, timeFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(fracFormat, dateTimeFormat)).to.equal(dateTimeFormat);

                expect(FormulaUtils.combineParsedFormats(dateFormat, percentFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, currFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, fracFormat)).to.equal(dateFormat);
                expect(FormulaUtils.combineParsedFormats(dateFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateFormat, dateTimeFormat)).to.equal(null);

                expect(FormulaUtils.combineParsedFormats(timeFormat, percentFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, currFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, fracFormat)).to.equal(timeFormat);
                expect(FormulaUtils.combineParsedFormats(timeFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(timeFormat, dateTimeFormat)).to.equal(null);

                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, percentFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, currFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, fracFormat)).to.equal(dateTimeFormat);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, dateFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, timeFormat)).to.equal(null);
                expect(FormulaUtils.combineParsedFormats(dateTimeFormat, dateTimeFormat)).to.equal(null);
            });
        });

        describe('method "logTokens"', function () {
            it('should exist', function () {
                expect(FormulaUtils).itself.to.respondTo('logTokens');
            });
        });
    });

    // ========================================================================
});
