/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper',
    'io.ox/office/spreadsheet/utils/operationcontext',
    'io.ox/office/spreadsheet/model/formula/deps/dependencymanager'
], function (AppHelper, SheetHelper, OperationContext, DependencyManager) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;

    // class Compiler =========================================================

    describe('Spreadsheet class DependencyManager', function () {

        it('should exist', function () {
            expect(DependencyManager).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: 'B6', contents: [[0]] },
            { name: 'changeCells', sheet: 0, start: 'A9', contents: [[{ v: 0, f: 'B6*2' }], [{ v: 1, f: 'A9+1' }], [{ v: 1, f: 'B6+A10' }], [{ v: 2, f: 'SUM(A9:A11)' }]] },
            { name: 'changeCells', sheet: 0, start: 'A100', contents: [[1, 2], [3, 4]] },
            { name: 'insertCFRule', sheet: 0, id: '0', ranges: 'A100:B101', type: 'equal', value1: 'Sheet2!A100', priority: 1, attrs: { character: { bold: true } } },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' }
        ];

        // initialize test document
        var docModel = null, depManager = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            depManager = docModel.getDependencyManager();
        });

        function waitForEvent(source, type) {
            var promise = docModel.waitForEvent(source, type, 1000);
            promise.always(function () { expect(promise.state()).to.equal('resolved'); });
            return promise;
        }

        function getSheetModel(sheet) {
            return docModel.getSheetModel(sheet);
        }

        function getCellCollection(sheet) {
            return getSheetModel(sheet).getCellCollection();
        }

        function getCellModel(sheet, address) {
            return getCellCollection(sheet).getCellModel(a(address));
        }

        function expectCellValue(sheet, address, value) {
            var cellModel = getCellModel(sheet, address);
            expect(cellModel).to.have.a.property('v', value);
            return cellModel;
        }

            // changes the value of a single cell
        function changeCell(sheet, address, value) {
            var json = _.extend({ name: 'changeCells', sheet: sheet, start: a(address).toJSON(), contents: [{ c: [{ v: value }] }] });
            getCellCollection(sheet).applyChangeCellsOperation(new OperationContext(docModel, json));
        }

        // indirect tests -----------------------------------------------------

        it('should be contained in a document', function () {
            expect(depManager).to.be.an.instanceof(DependencyManager);
        });

        describe('method "recalcFormulas"', function () {
            it('should wait for initial update cycle', function (done) {
                var promise = depManager.recalcFormulas({ volatile: false, timeout: 1000 });
                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');
                    done();
                });
            });
        });

        describe('background update task', function () {

            it('should update formula results after changing a cell', function (done) {
                changeCell(0, 'B6', 1);
                waitForEvent(depManager, 'recalc:end').done(function () {
                    expectCellValue(0, 'A9', 2);
                    expectCellValue(0, 'A10', 3);
                    expectCellValue(0, 'A11', 4);
                    expectCellValue(0, 'A12', 9);
                    done();
                });
            });

            it('should update formatting rules after changing a cell', function (done) {
                changeCell(1, 'A100', 1);
                waitForEvent(getSheetModel(0), 'refresh:ranges').done(function (event, ranges) {
                    expect(ranges).to.stringifyTo('A100:B101');
                    done();
                });
            });
        });
    });

    // ========================================================================
});
