/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/presentation/model/model'
], function (AppHelper, DOM, Position, PresentationModel) {

    'use strict';

    // class PresentationModel ================================================

    describe('Presentation class PresentationModel', function () {

        // private helpers ----------------------------------------------------

        var slideDefaultName = 'slide_1',
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            masterId_1 = 'master1',
            backgroundColor = { type: 'rgb', value: 'FFCC00' },
            secondDrawingText = 'Hello Second Drawing',
            pasteText = '123',
            clipboardOperations = [{ name: 'insertText', start: [0, 0], text: pasteText }],
            pasteDrawingText1 = 'Hello Second Drawing123',
            pasteDrawingText2 = 'Hello 123Second Drawing123';

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } } },
            { name: 'insertMasterSlide', id: masterId_1 },
            { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
            { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
            { name: 'insertSlide', start: [0], target: layoutId_1, attrs: { fill: { type: 'solid', color: { type: 'rgb', value: 'FF0000' } } } },
            { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1' } } },
            { name: 'insertParagraph', start: [0, 0, 0] },
            { name: 'insertText', text: 'Hello World', start: [0, 0, 0, 0] },
            { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2' } } },
            { name: 'insertParagraph', start: [0, 1, 0] },
            { name: 'insertText', text: secondDrawingText, start: [0, 1, 0, 0] }
        ];

        var model = null;
        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
            model.getClipboardOperations = function () { return clipboardOperations; };
            model.hasInternalClipboard = function () { return true; };
            model.isClipboardOriginText = function () { return true; };
            model.checkSetClipboardPasteInProgress = function () { return false; };
        });

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(PresentationModel).to.be.a('function');
        });

        // public methods -----------------------------------------------------

        describe('method "isStandardSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isStandardSlideId');
            });
            it('should return whether the specified ID is the ID of a normal (standard) slide', function () {
                expect(model.isStandardSlideId(layoutId_1)).to.equal(false);
                expect(model.isStandardSlideId(masterId_1)).to.equal(false);
                expect(model.isStandardSlideId(slideDefaultName)).to.equal(true);
            });
        });

        describe('method "isLayoutSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isLayoutSlideId');
            });
            it('should return whether the specified ID is the ID of a layout slide', function () {
                expect(model.isLayoutSlideId(layoutId_1)).to.equal(true);
                expect(model.isLayoutSlideId(masterId_1)).to.equal(false);
                expect(model.isLayoutSlideId(slideDefaultName)).to.equal(false);
            });
        });

        describe('method "isMasterSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isMasterSlideId');
            });
            it('should return whether the specified ID is the ID of a master slide', function () {
                expect(model.isMasterSlideId(layoutId_1)).to.equal(false);
                expect(model.isMasterSlideId(masterId_1)).to.equal(true);
                expect(model.isMasterSlideId(slideDefaultName)).to.equal(false);
            });
        });

        describe('method "getStandardSlideCount"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getStandardSlideCount');
            });
            it('should return the number of normal (standard) slides', function () {
                expect(model.getStandardSlideCount()).to.equal(1);
            });
        });

        describe('method "getMasterSlideCount"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getMasterSlideCount');
            });
            it('should return the number of master and layout slides', function () {
                expect(model.getMasterSlideCount()).to.equal(3);
            });
        });

        describe('method "isMasterView"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('isMasterView');
            });
            it('should return whether the master/layout view is activated', function () {
                expect(model.isMasterView()).to.equal(false);
            });
        });

        describe('method "getActiveSlideId"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlideId');
            });
            it('should return the ID of the active slide', function () {
                expect(model.getActiveSlideId()).to.equal(slideDefaultName);
            });
        });

        describe('method "getSlideById"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getSlideById');
            });
            it('should return a slide', function () {
                expect(model.getSlideById('notExistent')).to.equal(null);
                expect(model.getSlideById(slideDefaultName)).to.be.an.instanceof($);
                expect(model.getSlideById(slideDefaultName)).to.have.property('hasClass');
                expect(model.getSlideById(slideDefaultName).hasClass('slide')).to.equal(true);
            });

        });

        describe('method "getActiveSlidePosition"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getActiveSlidePosition');
            });
            it('should return the logical position of the active slide', function () {
                expect(model.getActiveSlidePosition()).to.be.an('array');
                expect(model.getActiveSlidePosition().length).to.equal(1);
                expect(model.getActiveSlidePosition()[0]).to.equal(0);
            });
        });

        describe('method "resetBackground"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('resetBackground');
            });
            it('should remove the slide background', function () {
                expect(model.getSlideById(slideDefaultName).data('attributes').fill.type).to.equal('solid');
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.be.a('string').and.not.to.be.empty;
                model.resetBackground();
                expect(model.getSlideById(slideDefaultName).data('attributes').fill).to.be.undefined;
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.equal('');
            });
        });

        describe('method "setBackgroundColor"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setBackgroundColor');
            });
            it('should set the slide background color', function () {
                expect(model.getSlideById(slideDefaultName).data('attributes').fill).to.be.undefined;
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.equal('');
                model.setBackgroundColor(backgroundColor);
                expect(model.getSlideById(slideDefaultName).data('attributes').fill.type).to.equal('solid');
                expect(model.getSlideById(slideDefaultName).css('background-color')).to.be.a('string').and.not.to.be.empty;
            });
        });

        describe('method "getAllFamilySlideAttributes"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getAllFamilySlideAttributes');
            });
            it('should get the slide attributes without specifying an ID', function () {
                expect(model.getAllFamilySlideAttributes().fill.type).to.equal('solid');
            });
            it('should get the slide attributes with specifying an ID', function () {
                expect(model.getAllFamilySlideAttributes(slideDefaultName).fill.type).to.equal('solid');
            });
        });

        describe('method "getVerticalAlignmentMode"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getVerticalAlignmentMode');
            });
            it('should get the vertical alignment mode of the text inside the selected drawing', function () {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                expect(model.getSelection().getSelectedDrawing().length).to.equal(1);
                expect(model.getSelection().getSelectedDrawing().text()).to.equal(secondDrawingText);
                expect(_.last(model.getSelection().getStartPosition())).equal(1);
                expect(_.last(model.getSelection().getEndPosition())).equal(2);
                expect(model.getVerticalAlignmentMode()).to.equal('top');
            });
        });

        describe('method "setVerticalAlignmentMode"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('setVerticalAlignmentMode');
            });
            it('should set the vertical alignment mode of the text inside the selected drawing', function () {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                expect(model.getSelection().getSelectedDrawing().text()).to.equal(secondDrawingText);
                expect(_.last(model.getSelection().getStartPosition())).equal(1);
                expect(_.last(model.getSelection().getEndPosition())).equal(2);
                model.setVerticalAlignmentMode('centered');
                expect(model.getVerticalAlignmentMode()).to.equal('centered');
                model.setVerticalAlignmentMode('bottom');
                expect(model.getVerticalAlignmentMode()).to.equal('bottom');
                model.setVerticalAlignmentMode('top');
                expect(model.getVerticalAlignmentMode()).to.equal('top');
            });
        });

        describe('method "pasteInternalClipboard"', function () {
            var // a drawing node
                drawing = null,
                // the delete promise
                clipboardPromise = null;

            it('should exist', function () {
                expect(model).to.respondTo('pasteInternalClipboard');
            });
            it('should paste some text into the text frame that is selected', function (done) {
                model.getSelection().setTextSelection([0, 1], [0, 2]);
                drawing = model.getSelection().getSelectedDrawing();
                expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(secondDrawingText);
                clipboardPromise = model.pasteInternalClipboard();
                clipboardPromise.always(function () {
                    model.getSelection().setTextSelection([0, 1], [0, 2]);
                    expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(pasteDrawingText1);
                    done();
                });
            });
            it('should paste some text into the text frame that contains the selection', function (done) {
                model.getSelection().setTextSelection([0, 1, 0, 6]);
                clipboardPromise = model.pasteInternalClipboard();
                clipboardPromise.always(function () {
                    expect(drawing.find(DOM.PARAGRAPH_NODE_SELECTOR).first().text()).to.equal(pasteDrawingText2);
                    done();
                });
            });
            it('should paste some text into a new created text frame, if the browser selection is in the clipboard', function (done) {
                var followingDrawings = drawing.nextAll('div.drawing').length;
                expect(drawing.nextAll('div.drawing').length).to.equal(followingDrawings);
                expect(model.getSelection().getClipboardNode().css('display')).to.equal('none');
                model.getSelection().setTextSelection([0, 0]);
                expect(model.getSelection().getClipboardNode().css('display')).to.equal('block');
                clipboardPromise = model.pasteInternalClipboard();
                clipboardPromise.always(function () {
                    expect(drawing.nextAll('div.drawing').length).to.equal(followingDrawings + 1);
                    expect(drawing.nextAll('div.drawing').text()).to.equal(pasteText);
                    done();
                });
            });
        });

        describe('method "getDefaultTextForTextFrame"', function () {

            var // the affected drawing, paragraph and text span nodes
                drawing = null, paragraph = null,
                // the delete promise
                delPromise = null,
                // the length of the default text
                defaultTextLength = 0;

            it('should exist', function () {
                expect(model).to.respondTo('getDefaultTextForTextFrame');
            });
            it('should return some text for the place holder drawing', function () {
                var defaultText = null;
                model.getSelection().setTextSelection([0, 1, 0, 0]); // setting cursor into drawing
                drawing = model.getSelection().getAnyDrawingSelection();
                defaultText = model.getDefaultTextForTextFrame(drawing);
                expect(defaultText).to.be.a.string;
                defaultTextLength = defaultText.length;
                expect(defaultTextLength).to.be.above(0);
            });
            it('should insert the default template text into an empty text frame', function (done) {
                model.getSelection().setTextSelection([0, 1, 0, 0]); // setting cursor into drawing
                model.getSelection().selectAll(); //

                drawing = model.getSelection().getAnyDrawingSelection();
                paragraph = drawing.find('div.p');

                expect(drawing.length).to.equal(1);
                expect(paragraph.length).to.equal(1);
                expect(_.last(model.getSelection().getStartPosition())).to.equal(0);
                expect(_.last(model.getSelection().getEndPosition())).to.be.above(0);

                delPromise = model.deleteSelected({ deleteKey: true });
                delPromise.always(function () {

                    var // the text span inside the paragraph
                        textspan = null;

                    expect(delPromise.state()).to.equal('resolved');

                    expect(_.last(model.getSelection().getStartPosition())).to.equal(0);
                    expect(_.last(model.getSelection().getEndPosition())).to.equal(0);

                    // paragraph has no 'official' length
                    expect(Position.getParagraphNodeLength(paragraph)).to.equal(0);
                    // and there is really no content inside the one and only text span
                    textspan = paragraph.children('span');
                    expect(textspan.length).to.equal(1);
                    expect(textspan.text().length).to.equal(0);
                    expect(DOM.isTextFrameTemplateTextSpan(textspan)).to.equal(false);

                    // selecting the drawing, no longer the text itself
                    model.getSelection().setTextSelection([0, 1], [0, 2]);

                    textspan = paragraph.children('span');
                    expect(textspan.length).to.equal(1);
                    expect(textspan.text().length).to.equal(defaultTextLength);
                    expect(DOM.isTextFrameTemplateTextSpan(textspan)).to.equal(true);

                    // but paragraph has still no 'official' length
                    expect(Position.getParagraphNodeLength(paragraph)).to.equal(0);

                    // selecting again the text in the (empty) text frame
                    model.getSelection().setTextSelection([0, 1, 0, 0]);

                    textspan = paragraph.children('span');
                    expect(textspan.length).to.equal(1);
                    expect(textspan.text().length).to.equal(0);
                    expect(DOM.isTextFrameTemplateTextSpan(textspan)).to.equal(false);

                    // setting the cursor into the other drawing on this slide
                    model.getSelection().setTextSelection([0, 0, 0, 0]);

                    textspan = paragraph.children('span');
                    expect(textspan.length).to.equal(1);
                    expect(textspan.text().length).to.equal(defaultTextLength);
                    expect(DOM.isTextFrameTemplateTextSpan(textspan)).to.equal(true);

                    done();
                });

            });
        });

    });

    // ========================================================================

    describe('Presentation class PresentationModel', function () {

        // private helpers ----------------------------------------------------

        var model = null,
            layoutId_1 = 'layout1',
            layoutId_2 = 'layout2',
            layoutId_3 = 'layout3',
            layoutId_4 = 'layout4',

            masterId_1 = 'master1',
            masterId_2 = 'master2',
            masterId_3 = 'master3',
            //activeSlide = null,

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: {
                        page: { width: 33866, height: 19050, orientation: 'landscape' }
                    }
                },
                { name: 'insertMasterSlide', id: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_2, target: masterId_1 },
                { name: 'insertLayoutSlide', id: layoutId_3, target: masterId_1 },

                { name: 'insertMasterSlide', id: masterId_2 },
                { name: 'insertLayoutSlide', id: layoutId_4, target: masterId_2 },

                { name: 'insertMasterSlide', id: masterId_3 },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertSlide', start: [1], target: layoutId_1 }

            ];

        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (app) {
            model = app.getModel();
        });

        // ========================================================================

        describe('method "getNextValidSlideIndex"', function () {
            it('should exist', function () {
                expect(model).to.respondTo('getNextValidSlideIndex');
            });
            it('should return the next valid slide according to the parameters', function () {

                ////// Master //////
                // next valid (master) upwards -> -1 because there is none before
                expect(model.getNextValidSlideIndex(masterId_1, { upwards: true, masterIsValidSlide: true })).to.equal(-1);
                expect(model.getNextValidSlideIndex(masterId_1, { upwards: false, masterIsValidSlide: true })).to.equal(4);

                expect(model.getNextValidSlideIndex(masterId_2, { upwards: true, masterIsValidSlide: true })).to.equal(0);
                expect(model.getNextValidSlideIndex(masterId_2, { upwards: false, masterIsValidSlide: true })).to.equal(6);

                expect(model.getNextValidSlideIndex(masterId_3, { upwards: true, masterIsValidSlide: true })).to.equal(4);
                // next valid (master) downwards -> -> -1 because there is not a next master
                expect(model.getNextValidSlideIndex(masterId_3, { upwards: false, masterIsValidSlide: true })).to.equal(-1);

                ////// Layout //////
                // next valid (layout) upwards -> -1 because there is no layout slide before
                expect(model.getNextValidSlideIndex(layoutId_1, { upwards: true, masterIsValidSlide: false })).to.equal(-1);
                expect(model.getNextValidSlideIndex(layoutId_1, { upwards: false, masterIsValidSlide: false })).to.equal(2);

                expect(model.getNextValidSlideIndex(layoutId_4, { upwards: true, masterIsValidSlide: false })).to.equal(3);
                // next valid (layout) upwards -> -1 because there is not a next layout slide
                expect(model.getNextValidSlideIndex(layoutId_4, { upwards: false, masterIsValidSlide: false })).to.equal(-1);

                ////// Standard //////
                expect(model.getNextValidSlideIndex('slide_1', { upwards: true, masterIsValidSlide: false })).to.equal(-1);
                expect(model.getNextValidSlideIndex('slide_1', { upwards: false, masterIsValidSlide: false })).to.equal(1);

                expect(model.getNextValidSlideIndex('slide_2', { upwards: true, masterIsValidSlide: false })).to.equal(0);
                expect(model.getNextValidSlideIndex('slide_2', { upwards: false, masterIsValidSlide: false })).to.equal(-1);
            });
        });
    });
});
