/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/utils/operationcontext'
], function (OperationError, OperationContext) {

    'use strict';

    // class OperationContext =================================================

    describe('EditFramework class OperationContext', function () {

        it('should exist', function () {
            expect(OperationContext).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        var // a fake document model
            docModel = {},
            // the test operation
            operation = { name: 'test', bool0: false, bool1: true, num: 2.5, nan: Number.NaN, str: 'a', empty: '', obj: { i: 1 }, arr: [3.5] },
            // the tested operation context
            context = new OperationContext(docModel, operation, false),
            // a default object value
            defObj = { a: 42 },
            // a default array value
            defArr = [42];

        // constructor --------------------------------------------------------

        describe('constructor', function () {
            it('should create a context', function () {
                expect(context).to.be.an.instanceof(OperationContext);
                expect(context).to.have.a.property('docModel', docModel);
                expect(context).to.have.a.property('operation', operation);
                expect(context).to.have.a.property('external', false);
                expect(new OperationContext(docModel, operation, true)).to.have.a.property('external', true);
            });
        });

        // public methods -----------------------------------------------------

        describe('method "error"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('error');
            });
            it('should throw', function () {
                expect(context.error.bind(context, 'bad')).to.throw(OperationError, 'bad');
            });
        });

        describe('method "ensure"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('ensure');
            });
            function ensure(cond) { return context.ensure.bind(context, cond, 'bad'); }
            it('should do nothing on truthy condition', function () {
                expect(ensure(true)).to.not.throw;
                expect(ensure(1)).to.not.throw;
                expect(ensure('a')).to.not.throw;
                expect(ensure({})).to.not.throw;
                expect(ensure([])).to.not.throw;
            });
            it('should throw on falsy condition', function () {
                expect(ensure(false)).to.throw(OperationError, 'bad');
                expect(ensure(0)).to.throw(OperationError, 'bad');
                expect(ensure('')).to.throw(OperationError, 'bad');
                expect(ensure(null)).to.throw(OperationError, 'bad');
            });
        });

        describe('method "has"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('has');
            });
            it('should return true for existing properties', function () {
                expect(context.has('bool0')).to.equal(true);
                expect(context.has('bool1')).to.equal(true);
                expect(context.has('num')).to.equal(true);
                expect(context.has('nan')).to.equal(true);
                expect(context.has('str')).to.equal(true);
                expect(context.has('empty')).to.equal(true);
                expect(context.has('obj')).to.equal(true);
                expect(context.has('arr')).to.equal(true);
            });
            it('should return false for missing properties', function () {
                expect(context.has('wrong')).to.equal(false);
            });
        });

        describe('method "get"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('get');
            });
            function get(name) { return context.get.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.get('bool0')).to.equal(operation.bool0);
                expect(context.get('bool1')).to.equal(operation.bool1);
                expect(context.get('num')).to.equal(operation.num);
                expect(context.get('nan')).to.be.nan;
                expect(context.get('str')).to.equal(operation.str);
                expect(context.get('empty')).to.equal(operation.empty);
                expect(context.get('obj')).to.equal(operation.obj);
                expect(context.get('arr')).to.equal(operation.arr);
            });
            it('should throw on missing properties', function () {
                expect(get('wrong')).to.throw(OperationError);
            });
        });

        describe('method "getOpt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOpt');
            });
            it('should return existing properties', function () {
                expect(context.getOpt('bool0', true)).to.equal(operation.bool0);
                expect(context.getOpt('bool1', true)).to.equal(operation.bool1);
                expect(context.getOpt('num', 42)).to.equal(operation.num);
                expect(context.getOpt('nan', 42)).to.be.nan;
                expect(context.getOpt('str', '42')).to.equal(operation.str);
                expect(context.getOpt('empty', '42')).to.equal(operation.empty);
                expect(context.getOpt('obj', defObj)).to.equal(operation.obj);
                expect(context.getOpt('arr', defArr)).to.equal(operation.arr);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOpt('wrong', 42)).to.equal(42);
            });
        });

        describe('method "getBool"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getBool');
            });
            function getBool(name) { return context.getBool.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getBool('bool0')).to.equal(operation.bool0);
                expect(context.getBool('bool1')).to.equal(operation.bool1);
            });
            it('should throw on missing properties', function () {
                expect(getBool('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getBool('num')).to.throw(OperationError);
                expect(getBool('nan')).to.throw(OperationError);
                expect(getBool('str')).to.throw(OperationError);
                expect(getBool('empty')).to.throw(OperationError);
                expect(getBool('obj')).to.throw(OperationError);
                expect(getBool('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptBool"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptBool');
            });
            function getOptBool(name, def) { return context.getOptBool.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptBool('bool0', true)).to.equal(operation.bool0);
                expect(context.getOptBool('bool1', true)).to.equal(operation.bool1);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptBool('wrong', true)).to.equal(true);
                expect(context.getOptBool('wrong', false)).to.equal(false);
                expect(context.getOptBool('wrong')).to.equal(false);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptBool('num')).to.throw(OperationError);
                expect(getOptBool('nan')).to.throw(OperationError);
                expect(getOptBool('str')).to.throw(OperationError);
                expect(getOptBool('empty')).to.throw(OperationError);
                expect(getOptBool('obj')).to.throw(OperationError);
                expect(getOptBool('arr')).to.throw(OperationError);
            });
        });

        describe('method "getNum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getNum');
            });
            function getNum(name) { return context.getNum.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getNum('num')).to.equal(operation.num);
            });
            it('should throw on missing properties', function () {
                expect(getNum('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getNum('bool0')).to.throw(OperationError);
                expect(getNum('bool1')).to.throw(OperationError);
                expect(getNum('nan')).to.throw(OperationError);
                expect(getNum('str')).to.throw(OperationError);
                expect(getNum('empty')).to.throw(OperationError);
                expect(getNum('obj')).to.throw(OperationError);
                expect(getNum('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptNum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptNum');
            });
            function getOptNum(name, def) { return context.getOptNum.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptNum('num', 42)).to.equal(operation.num);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptNum('wrong', 42)).to.equal(42);
                expect(context.getOptNum('wrong')).to.equal(0);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptNum('bool0')).to.throw(OperationError);
                expect(getOptNum('bool1')).to.throw(OperationError);
                expect(getOptNum('nan')).to.throw(OperationError);
                expect(getOptNum('str')).to.throw(OperationError);
                expect(getOptNum('empty')).to.throw(OperationError);
                expect(getOptNum('obj')).to.throw(OperationError);
                expect(getOptNum('arr')).to.throw(OperationError);
            });
        });

        describe('method "getInt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getInt');
            });
            function getInt(name) { return context.getInt.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getInt('num')).to.equal(Math.floor(operation.num));
            });
            it('should throw on missing properties', function () {
                expect(getInt('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getInt('bool0')).to.throw(OperationError);
                expect(getInt('bool1')).to.throw(OperationError);
                expect(getInt('nan')).to.throw(OperationError);
                expect(getInt('str')).to.throw(OperationError);
                expect(getInt('empty')).to.throw(OperationError);
                expect(getInt('obj')).to.throw(OperationError);
                expect(getInt('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptInt"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptInt');
            });
            function getOptInt(name, def) { return context.getOptInt.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptInt('num', 42)).to.equal(Math.floor(operation.num));
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptInt('wrong', 42)).to.equal(42);
                expect(context.getOptInt('wrong', 42.5)).to.equal(42);
                expect(context.getOptInt('wrong')).to.equal(0);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptInt('bool0')).to.throw(OperationError);
                expect(getOptInt('bool1')).to.throw(OperationError);
                expect(getOptInt('nan')).to.throw(OperationError);
                expect(getOptInt('str')).to.throw(OperationError);
                expect(getOptInt('empty')).to.throw(OperationError);
                expect(getOptInt('obj')).to.throw(OperationError);
                expect(getOptInt('arr')).to.throw(OperationError);
            });
        });

        describe('method "getStr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getStr');
            });
            function getStr(name) { return context.getStr.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getStr('str')).to.equal(operation.str);
                expect(context.getStr('str', true)).to.equal(operation.str);
            });
            it('should throw on missing properties', function () {
                expect(getStr('wrong')).to.throw(OperationError);
            });
            it('should handle empty strings', function () {
                expect(getStr('empty')).to.throw(OperationError);
                expect(context.getStr('empty', true)).to.equal('');
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getStr('bool0')).to.throw(OperationError);
                expect(getStr('bool1')).to.throw(OperationError);
                expect(getStr('num')).to.throw(OperationError);
                expect(getStr('nan')).to.throw(OperationError);
                expect(getStr('obj')).to.throw(OperationError);
                expect(getStr('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptStr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptStr');
            });
            function getOptStr(name, def) { return context.getOptStr.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptStr('str', '42')).to.equal(operation.str);
                expect(context.getOptStr('str', '42', true)).to.equal(operation.str);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptStr('wrong', '42')).to.equal('42');
                expect(context.getOptStr('wrong')).to.equal('');
            });
            it('should handle empty strings', function () {
                expect(getOptStr('empty')).to.throw(OperationError);
                expect(getOptStr('empty', '42')).to.throw(OperationError);
                expect(context.getOptStr('empty', true)).to.equal('');
                expect(context.getOptStr('empty', '42', true)).to.equal('');
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptStr('bool0')).to.throw(OperationError);
                expect(getOptStr('bool1')).to.throw(OperationError);
                expect(getOptStr('num')).to.throw(OperationError);
                expect(getOptStr('nan')).to.throw(OperationError);
                expect(getOptStr('obj')).to.throw(OperationError);
                expect(getOptStr('arr')).to.throw(OperationError);
            });
        });

        describe('method "getEnum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getEnum');
            });
            function getEnum(name) { return context.getEnum.bind(context, name, /.*/); }
            it('should return existing properties', function () {
                expect(context.getEnum('str', /a/)).to.equal(operation.str);
                expect(context.getEnum('str', /^(a|b)$/)).to.equal(operation.str);
            });
            it('should throw on missing properties', function () {
                expect(getEnum('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getEnum('bool0')).to.throw(OperationError);
                expect(getEnum('bool1')).to.throw(OperationError);
                expect(getEnum('num')).to.throw(OperationError);
                expect(getEnum('nan')).to.throw(OperationError);
                expect(getEnum('empty')).to.throw(OperationError);
                expect(getEnum('obj')).to.throw(OperationError);
                expect(getEnum('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptEnum"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptEnum');
            });
            function getOptEnum(name, valid, def) { return context.getOptEnum.bind(context, name, valid, def); }
            it('should return existing properties', function () {
                expect(context.getOptEnum('str', /a/, '42')).to.equal(operation.str);
                expect(context.getOptEnum('str', /^(a|b)$/, '42')).to.equal(operation.str);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptEnum('wrong', /a/, '42')).to.equal('42');
                expect(context.getOptEnum('wrong', /a/)).to.equal('');
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptEnum('bool0', /a/)).to.throw(OperationError);
                expect(getOptEnum('bool1', /a/)).to.throw(OperationError);
                expect(getOptEnum('num', /a/)).to.throw(OperationError);
                expect(getOptEnum('nan', /a/)).to.throw(OperationError);
                expect(getOptEnum('empty', /a/)).to.throw(OperationError);
                expect(getOptEnum('empty', /.*/)).to.throw(OperationError);
                expect(getOptEnum('obj', /a/)).to.throw(OperationError);
                expect(getOptEnum('arr', /a/)).to.throw(OperationError);
            });
            it('should throw on non-matching value', function () {
                expect(getOptEnum('str', /c/)).to.throw(OperationError);
            });
        });

        describe('method "getObj"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getObj');
            });
            function getObj(name) { return context.getObj.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getObj('obj')).to.equal(operation.obj);
            });
            it('should throw on missing properties', function () {
                expect(getObj('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getObj('bool0')).to.throw(OperationError);
                expect(getObj('bool1')).to.throw(OperationError);
                expect(getObj('num')).to.throw(OperationError);
                expect(getObj('nan')).to.throw(OperationError);
                expect(getObj('str')).to.throw(OperationError);
                expect(getObj('empty')).to.throw(OperationError);
                expect(getObj('arr')).to.throw(OperationError);
            });
        });

        describe('method "getOptObj"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptObj');
            });
            function getOptObj(name, def) { return context.getOptObj.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptObj('obj', defObj)).to.equal(operation.obj);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptObj('wrong', defObj)).to.equal(defObj);
                expect(context.getOptObj('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptObj('bool0')).to.throw(OperationError);
                expect(getOptObj('bool1')).to.throw(OperationError);
                expect(getOptObj('num')).to.throw(OperationError);
                expect(getOptObj('nan')).to.throw(OperationError);
                expect(getOptObj('str')).to.throw(OperationError);
                expect(getOptObj('empty')).to.throw(OperationError);
                expect(getOptObj('arr')).to.throw(OperationError);
            });
        });

        describe('method "getArr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getArr');
            });
            function getArr(name) { return context.getArr.bind(context, name); }
            it('should return existing properties', function () {
                expect(context.getArr('arr')).to.equal(operation.arr);
            });
            it('should throw on missing properties', function () {
                expect(getArr('wrong')).to.throw(OperationError);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getArr('bool0')).to.throw(OperationError);
                expect(getArr('bool1')).to.throw(OperationError);
                expect(getArr('num')).to.throw(OperationError);
                expect(getArr('nan')).to.throw(OperationError);
                expect(getArr('str')).to.throw(OperationError);
                expect(getArr('empty')).to.throw(OperationError);
                expect(getArr('obj')).to.throw(OperationError);
            });
        });

        describe('method "getOptArr"', function () {
            it('should exist', function () {
                expect(OperationContext).to.respondTo('getOptArr');
            });
            function getOptArr(name, def) { return context.getOptArr.bind(context, name, def); }
            it('should return existing properties', function () {
                expect(context.getOptArr('arr', defArr)).to.equal(operation.arr);
            });
            it('should return default value for missing properties', function () {
                expect(context.getOptArr('wrong', defArr)).to.equal(defArr);
                expect(context.getOptArr('wrong')).to.equal(null);
            });
            it('should throw on existing properties with wrong type', function () {
                expect(getOptArr('bool0')).to.throw(OperationError);
                expect(getOptArr('bool1')).to.throw(OperationError);
                expect(getOptArr('num')).to.throw(OperationError);
                expect(getOptArr('nan')).to.throw(OperationError);
                expect(getOptArr('str')).to.throw(OperationError);
                expect(getOptArr('empty')).to.throw(OperationError);
                expect(getOptArr('obj')).to.throw(OperationError);
            });
        });
    });

    // ========================================================================
});
