/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/tk/utils/deferredutils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/object/timermixin',
    'io.ox/office/editframework/app/seleniumwrapper',
    'io.ox/office/textframework/utils/position'
], function (AppHelper, DeferredUtils, BaseObject, TimerMixin, SeleniumWrapper, Position) {

    'use strict';

    // class SeleniumWrapper ==================================================

    describe('EditFramework class SeleniumWrapper', function () {

        it('should exist', function () {
            expect(SeleniumWrapper).to.be.an('object');
        });

        // static methods -----------------------------------------------------

        describe('method "getAllFilesInFolder"', function () {
            it('should exist', function () {
                expect(SeleniumWrapper).itself.to.respondTo('getAllFilesInFolder');
            });
        });

        describe('method "getAllSubfoldersInFolder"', function () {
            it('should exist', function () {
                expect(SeleniumWrapper).itself.to.respondTo('getAllSubfoldersInFolder');
            });
        });

        describe('method "oxLaunch"', function () {
            it('should exist', function () {
                expect(SeleniumWrapper).itself.to.respondTo('oxLaunch');
            });
        });

        describe('method "clearTrashFolder"', function () {
            it('should exist', function () {
                expect(SeleniumWrapper).itself.to.respondTo('clearTrashFolder');
            });
        });

        describe('method "setSettingsStates"', function () {
            it('should exist', function () {
                expect(SeleniumWrapper).itself.to.respondTo('setSettingsStates');
            });
        });

        // static methods for OX Text =========================================

        (function () {

            var OPERATIONS = [
                { name: 'setDocumentAttributes',
                    attrs: {
                        document:  { defaultTabStop: 1270, zoom: { value: 100 } },
                        page:      { width: 21590, height: 27940, marginLeft: 2540, marginTop: 2540, marginRight: 2540, marginBottom: 2540, marginHeader: 1248, marginFooter: 1248 },
                        character: { fontName: 'Arial', fontSize: 11, language: 'en-US', languageEa: 'en-US', languageBidi: 'ar-SA' },
                        paragraph: { lineHeight: { type: 'percent', value: 115 }, marginBottom: 352 }
                    }
                },
                { name: 'insertStyleSheet', type: 'paragraph', styleId: 'Heading1', styleName: 'heading 1',
                    attrs: {
                        character: { bold: true, fontName: 'Times New Roman', fontSize: 14, color: { transformations: [{ type: 'shade', value: 74902 }], type: 'scheme', value: 'accent1' } },
                        paragraph: { marginTop: 846, outlineLevel: 0, nextStyleId: 'Normal' }
                    },
                    parent: 'Normal',
                    uiPriority: 9
                },
                { name: 'insertParagraph', start: [0] },
                { name: 'insertText', text: 'Hello World', start: [0, 0] }
            ];

            var transactionIDCounter = 0;

            // initialize test application
            var docModel = null, node = null;
            AppHelper.createTextApp('ooxml', OPERATIONS).done(function (app) {
                docModel = app.getModel();
                node = docModel.getNode();
            });

            function testWrappedFn(fnName, options) {
                var def = $.Deferred();

                var transactionID = 'tr' + String(transactionIDCounter);
                transactionIDCounter++;

                SeleniumWrapper[fnName](docModel.getApp(), transactionID, options);

                var interval = window.setInterval(function () {
                    var res = SeleniumWrapper.get(this);
                    if (res) {
                        if (res.state) {
                            def.resolve(res.result);
                        } else {
                            def.reject(res.result);
                        }
                        window.clearInterval(interval);
                    }
                }.bind(transactionID), 10);

                return def;
            }

            describe('method "applyOperations"', function () {
                it('should exist', function () {
                    expect(SeleniumWrapper).itself.to.respondTo('applyOperations');
                });

                it('should apply document operations', function (done) {

                    var ops = [
                        { name: 'splitParagraph', start: [0, 11] },
                        { name: 'insertText', start: [1, 0], text: 'lorem ipsum' }
                    ];

                    testWrappedFn('applyOperations', { ops: ops }).always(function () {
                        var secondParagraph = $(Position.getParagraphElement(node, [1]));
                        expect(secondParagraph.text()).to.equal('lorem ipsum');
                        done();
                    });
                });
            });

            describe('method "getAttributesFromTextApp"', function () {
                it('should exist', function () {
                    expect(SeleniumWrapper).itself.to.respondTo('getAttributesFromTextApp');
                });

                it('should return the correct attributes', function (done) {
                    docModel.getSelection().setTextSelection([1, 0], [1, 11]);
                    docModel.setAttributes('paragraph', { styleId: 'Heading1' }, { clear: true });

                    testWrappedFn('getAttributesFromTextApp', { family: 'character' }).always(function (attrs) {
                        expect(attrs.bold).to.equal(true);
                        done();
                    });
                });
            });

            describe('method "setTextSelection"', function () {
                it('should exist', function () {
                    expect(SeleniumWrapper).itself.to.respondTo('setTextSelection');
                });

                it('should change the selection in the document', function (done) {
                    var positionOld = [0, 2];
                    docModel.getSelection().setTextSelection(positionOld);
                    expect(docModel.getSelection().getStartPosition()).to.deep.equal(positionOld);

                    var start = [1, 5];
                    var end = [1, 6];
                    testWrappedFn('setTextSelection', { start: start, end: end }).always(function () {
                        expect(docModel.getSelection().getStartPosition()).to.deep.equal(start);
                        expect(docModel.getSelection().getEndPosition()).to.deep.equal(end);
                        done();
                    });
                });
            });

            describe('method "getRunningJobs"', function () {
                it('should exist', function () {
                    expect(SeleniumWrapper).itself.to.respondTo('getRunningJobs');
                });

                it('should log deferred objects', function (done) {
                    var infoString = '123-test-infostring';
                    var testdef = DeferredUtils.createDeferred(docModel.getApp(), infoString);

                    testWrappedFn('getRunningJobs', {}).always(function (jobs1) {
                        var index1 = _.findIndex(jobs1.deferred, function (el) {
                            return el.key === infoString;
                        });
                        expect(index1 >= 0).to.be.true;

                        testdef.resolve();

                        testWrappedFn('getRunningJobs', {}).always(function (jobs2) {
                            var index2 = _.findIndex(jobs2.deferred, function (el) {
                                return el.key === infoString;
                            });
                            expect(index2 < 0).to.be.true;
                            done();
                        });
                    });
                });

                it('should log timers', function (done) {

                    var infoString = '123-test-infostring';

                    function TimerClass() {
                        BaseObject.call(this);
                        TimerMixin.call(this);
                    }

                    var obj = new TimerClass();
                    var timer = obj.executeDelayed(function () {}, { delay: 20, infoString: infoString });

                    testWrappedFn('getRunningJobs', {}).always(function (jobs1) {
                        var index1 = _.findIndex(jobs1.timer, function (el) {
                            return el.key === infoString;
                        });
                        expect(index1 >= 0).to.be.true;

                        window.setTimeout(function () {

                            expect(timer.state()).to.equal('resolved');

                            testWrappedFn('getRunningJobs', {}).always(function (jobs2) {
                                var index2 = _.findIndex(jobs2.timer, function (el) {
                                    return el.key === infoString;
                                });
                                expect(index2 < 0).to.be.true;
                                done();
                            });
                        }, 20);
                    });
                });
            });
        }());

        // static methods for OX Spreadsheet ==================================

        (function () {
            //TODO: write tests for NOCE branch

            it('SeleniumWrapper.getAttributesFromSpreadsheetApp should exist', function () {
                expect(SeleniumWrapper.getAttributesFromSpreadsheetApp).to.be.a('function');
            });

            it('SeleniumWrapper.getActiveCellEntry should exist', function () {
                expect(SeleniumWrapper.getActiveCellEntry).to.be.a('function');
            });

            it('SeleniumWrapper.getSheetName should exist', function () {
                expect(SeleniumWrapper.getSheetName).to.be.a('function');
            });
        }());
    });

    // ========================================================================
});
