/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([], function () {

    'use strict';

    // Chai extensions ========================================================

    describe('Chai extensions', function () {

        var EPSILON = Math.pow(2, -51);
        var $testDiv = $('<div>').addClass('test');

        // properties ---------------------------------------------------------

        describe('property "jqNode"', function () {
            it('should pass for single-element collections', function () {
                expect($('<div>')).to.be.a.jqNode;
                expect($('<div>text</div>').contents()).to.be.a.jqNode;
            });
            it('should fail for empty collections', function () {
                expect($('<div>').children()).to.not.be.a.jqNode;
            });
            it('should fail for multi-element collections', function () {
                expect($('<div></div><div></div>')).to.not.be.a.jqNode;
            });
        });

        describe('property "jqElementNode"', function () {
            it('should pass for single-element collections', function () {
                expect($('<div>')).to.be.a.jqElementNode;
            });
            it('should fail for empty collections', function () {
                expect($('<div>').children()).to.not.be.a.jqElementNode;
            });
            it('should fail for non-element nodes', function () {
                expect($('<div>text</div>').contents()).to.not.be.a.jqElementNode;
            });
            it('should fail for multi-element collections', function () {
                expect($('<div></div><div></div>')).to.not.be.a.jqElementNode;
            });
        });

        describe('property "jqTextNode"', function () {
            it('should pass for single-element collections', function () {
                expect($('<div>text</div>').contents()).to.be.a.jqTextNode;
            });
            it('should fail for empty collections', function () {
                expect($('<div>').children()).to.not.be.a.jqTextNode;
            });
            it('should fail for non-text nodes', function () {
                expect($('<div>')).to.not.be.a.jqTextNode;
            });
            it('should fail for multi-element collections', function () {
                expect($('<div>text<div></div>text</div>')).to.not.be.a.jqTextNode;
            });
        });

        describe('property "almostZero"', function () {
            it('should pass for numbers almost zero', function () {
                expect(0).to.be.almostZero;
                expect(EPSILON / 2).to.be.almostZero;
                expect(EPSILON).to.be.almostZero;
                expect(-EPSILON / 2).to.be.almostZero;
                expect(-EPSILON).to.be.almostZero;
            });
            it('should fail for numbers not almost zero', function () {
                expect(EPSILON * 1.5).to.not.be.almostZero;
                expect(-EPSILON * 1.5).to.not.be.almostZero;
            });
        });

        describe('property "finite"', function () {
            it('should pass for finite numbers', function () {
                expect(0).to.be.finite;
                expect(1).to.be.finite;
                expect(1e307).to.be.finite;
                expect(-1).to.be.finite;
                expect(-1e307).to.be.finite;
            });
            it('should fail for infinite numbers', function () {
                expect(1 / 0).to.not.be.finite;
                expect(Number.POSITIVE_INFINITY).to.not.be.finite;
                expect(Number.NEGATIVE_INFINITY).to.not.be.finite;
            });
            it('should fail for NaN', function () {
                expect(Number.NaN).to.not.be.finite;
            });
        });

        describe('property "NaN"', function () {
            it('should pass for NaN', function () {
                expect(Math.log(-1)).to.be.NaN;
                expect(Number.NaN).to.be.NaN;
            });
            it('should fail for finite numbers', function () {
                expect(0).to.not.be.NaN;
                expect(1).to.not.be.NaN;
                expect(1e307).to.not.be.NaN;
                expect(-1).to.not.be.NaN;
                expect(-1e307).to.not.be.NaN;
            });
            it('should fail for infinite numbers', function () {
                expect(1 / 0).to.not.be.NaN;
                expect(Number.POSITIVE_INFINITY).to.not.be.NaN;
                expect(Number.NEGATIVE_INFINITY).to.not.be.NaN;
            });
        });

        // methods ------------------------------------------------------------

        describe('method "almostEqual"', function () {
            it('should pass for numbers almost equal to the expected number', function () {
                expect(1).to.almostEqual(1);
                expect(1 + EPSILON / 2).to.almostEqual(1);
                expect(1 - EPSILON / 2).to.almostEqual(1);
                expect(1 + EPSILON).to.almostEqual(1);
                expect(1 - EPSILON).to.almostEqual(1);
                expect(-1).to.almostEqual(-1);
                expect(-1 + EPSILON / 2).to.almostEqual(-1);
                expect(-1 - EPSILON / 2).to.almostEqual(-1);
                expect(-1 + EPSILON).to.almostEqual(-1);
                expect(-1 - EPSILON).to.almostEqual(-1);
                expect(1024).to.almostEqual(1024);
                expect(1024 + EPSILON * 512).to.almostEqual(1024);
                expect(1024 - EPSILON * 512).to.almostEqual(1024);
                expect(1024 + EPSILON * 1024).to.almostEqual(1024);
                expect(1024 - EPSILON * 1024).to.almostEqual(1024);
                expect(-1024).to.almostEqual(-1024);
                expect(-1024 + EPSILON * 512).to.almostEqual(-1024);
                expect(-1024 - EPSILON * 512).to.almostEqual(-1024);
                expect(-1024 + EPSILON * 1024).to.almostEqual(-1024);
                expect(-1024 - EPSILON * 1024).to.almostEqual(-1024);
            });
            it('should fail for numbers not close to the expected numbers', function () {
                expect(-1).to.not.almostEqual(1);
                expect(1 + EPSILON * 1.5).to.not.almostEqual(1);
                expect(1 - EPSILON * 1.5).to.not.almostEqual(1);
                expect(1).to.not.almostEqual(-1);
                expect(-1 + EPSILON * 1.5).to.not.almostEqual(-1);
                expect(-1 - EPSILON * 1.5).to.not.almostEqual(-1);
                expect(-1024).to.not.almostEqual(1024);
                expect(1024 + EPSILON * 1536).to.not.almostEqual(1024);
                expect(1024 - EPSILON * 1536).to.not.almostEqual(1024);
                expect(1024).to.not.almostEqual(-1024);
                expect(-1024 + EPSILON * 1536).to.not.almostEqual(-1024);
                expect(-1024 - EPSILON * 1536).to.not.almostEqual(-1024);
            });
        });

        describe('method "stringifyTo"', function () {
            it('should pass', function () {
                expect(1).to.stringifyTo('1');
                expect('a').to.stringifyTo('a');
                expect(true).to.stringifyTo('true');
                expect(null).to.stringifyTo('null');
                expect(/abc/i).to.stringifyTo('/abc/i');
                expect([1, 2, 3]).to.stringifyTo('1,2,3');
                expect({ toString: function () { return '42'; } }).to.stringifyTo('42');
            });
        });

        describe('method "stringifiesTo"', function () {
            it('should pass', function () {
                expect({ a: 42 }).to.have.a.property('a').that.stringifiesTo('42');
            });
        });

        describe('method "className"', function () {
            it('should have given class', function () {
                expect($testDiv).to.have.className('test');
                expect($testDiv).not.to.have.className('.test');
            });
        });
    });

    // ========================================================================
});
