/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var fs = require('fs');
    var _ = require('underscore');
    var Utils = require('../utils/gruntutils')(grunt);
    var PNGImage = require('node-png').PNG;

    // configuration ==========================================================

    // Root source directory.
    var SRC_ROOT = 'resource/icons';

    // Path to the icons definition file.
    var ICONS_JSON_PATH = SRC_ROOT + '/icons.json';

    // Source directory of all images.
    var IMAGES_ROOT = SRC_ROOT + '/images';

    // Target directory name for all generated files.
    var APPS_ICONS_DIR = 'apps/io.ox/office/tk/icons';

    // Root destination directory for all generated files.
    var DEST_ROOT = 'build/' + APPS_ICONS_DIR;

    // Path to the generated LESS file with a LESS mix-in class for defining bitmap URLs.
    var DEFINITIONS_LESS_PATH = DEST_ROOT + '/definitions.less';

    // Path to the generated LESS file with icon class names and positions.
    var DOCS_ICONS_LESS_PATH = DEST_ROOT + '/docs-icons.less';

    // Base height of all icons, for scaling factor 1.
    var ICON_HEIGHT = 16;

    // Base width of an icon set, for scaling factor 1.
    var ICONSET_WIDTH = 720;

    // Definitions for all supported icon set pixel resolutions.
    var ICONSET_RESOLUTIONS = { desktop: 1, retina: 2 };

    // Definition for all colors of the target icon sets.
    var ICONSET_COLORS = { black: '#333333', white: '#ffffff', blue: '#3c73aa' };

    // helper functions =======================================================

    /**
     * Returns the file name of an icon set bitmap file.
     *
     * @param {String} type
     *  The type of the icon set ('desktop', or 'retina').
     *
     * @param {String} color
     *  The color name to be inserted into the file name.
     *
     * @returns {String}
     *  The file name of the icon set bitmap file, without path, but with file
     *  extension).
     */
    function getIconSetFileName(type, color) {
        return 'docs-icons-' + type + '-' + color + '.png';
    }

    // task configuration =====================================================

    // convert icon set resolution configuration to array
    ICONSET_RESOLUTIONS = Object.keys(ICONSET_RESOLUTIONS).map(function (resName) {
        return { name: resName, scale: ICONSET_RESOLUTIONS[resName] };
    });

    // convert icon set color configuration to array
    ICONSET_COLORS = Object.keys(ICONSET_COLORS).map(function (colorName) {
        var matches = /^#([0-9a-f]{2})([0-9a-f]{2})([0-9a-f]{2})$/i.exec(ICONSET_COLORS[colorName]);
        Utils.ensureArrayNE(matches, 'Invalid color definition in task "iconsets" (expected CSS style "#RRGGBB" but got "' + ICONSET_COLORS[colorName] + '").');
        return { name: colorName, color: [parseInt(matches[1], 16), parseInt(matches[2], 16), parseInt(matches[3], 16)] };
    });

    // the paths of all generated bitmap files
    var destPaths = _.flatten(ICONSET_RESOLUTIONS.map(function (resolutionData) {
        var resName = resolutionData.name;
        return ICONSET_COLORS.map(function (colorData) {
            return DEST_ROOT + '/' + getIconSetFileName(resName, colorData.name);
        });
    }));

    // specify source/destination mapping for 'newer' plug-in (every destination file depends on all source files)
    grunt.config('iconsets', {
        all: {
            files: [DEFINITIONS_LESS_PATH, DOCS_ICONS_LESS_PATH].concat(destPaths).map(function (destPath) {
                return { src: [ICONS_JSON_PATH, IMAGES_ROOT + '/**/*.png'], dest: destPath };
            })
        }
    });

    // class CountDown ========================================================

    /**
     * Simple helper class that will invoke the registered callback functions,
     * after the end of a count-down has been reached.
     *
     * @constructor
     *
     * @param {Number} count
     *  The count-down size.
     */
    function CountDown(count) {
        this._steps = count;
        this._callbacks = [];
    }

    /**
     * Registers a callback function that will be executed after the end of the
     * count-down has been reached.
     *
     * @param {Function} callback
     *  The callback function to be registered.
     */
    CountDown.prototype.done = function (callback) {
        this._callbacks.push(callback.bind(this));
    };

    /**
     * Moves the count-down one step closer to the end. After this method has
     * been invoked the number of times specified in the constructor of this
     * instance, all registered callback functions will be invoked.
     */
    CountDown.prototype.step = function () {
        this._steps -= 1;
        if (this._steps === 0) {
            process.nextTick(function () {
                this._callbacks.forEach(function (callback) {
                    callback();
                });
            }.bind(this));
        }
    };

    // task implementation ====================================================

    grunt.registerTask('iconsets', 'Assembles the icon sets from single image files.', function () {

        // conversion runs asynchronously
        var resolveTask = this.async();

        // ensure existing destination directory
        grunt.file.mkdir(DEST_ROOT);

        // read and validate the icon definitions
        var definitions = grunt.file.readJSON(ICONS_JSON_PATH);
        Utils.ensureArrayNE(definitions, 'Invalid icon definitions, expecting JSON array.');

        // the path to the bitmap files, with a unique prefix for the bitmap URLs needed for cache busting
        var ICONSET_URL_PATH = 'v=' + grunt.config('pkg.version') + '.' + grunt.template.date(new Date(), 'yyyymmdd.HHMMss') + '/' + APPS_ICONS_DIR;

        // the contents of the resulting LESS file with the LESS mix-in class to define bitmap URLs
        var defLessLines = [
            '.docs-icons-url(@color-id) {',
            '    i[class^="docs-"] {',
            '        background-image: ~"url(' + ICONSET_URL_PATH + '/' + getIconSetFileName('desktop', '@{color-id}') + ')";',
            '        &.retina { background-image: ~"url(' + ICONSET_URL_PATH + '/' + getIconSetFileName('retina', '@{color-id}') + ')"; }',
            '    }',
            '}'
        ];

        // the contents of the resulting LESS file with icon definitions
        var iconLessLines = [];

        // create count-down triggers
        var imageCountDown = new CountDown(definitions.length * ICONSET_RESOLUTIONS.length);
        var iconSetCountDown = new CountDown(ICONSET_RESOLUTIONS.length * ICONSET_COLORS.length);

        // load all source images, store them in the entries of the 'definitions' array
        definitions.forEach(function (def) {

            // validate array element
            Utils.ensureObject(def, 'Invalid icon definition, expecting objects as array elements.');
            Utils.ensureStringNE(def.path, 'Invalid icon definition, missing path to source image.');

            // validate class property (may be a plain string)
            def.class = Utils.makeArray(def.class);
            Utils.ensureArrayNE(def.class, 'Invalid icon definition, expecting string or array as "class" property.');
            def.class.forEach(function (cls) { Utils.ensureStringNE(cls, 'Invalid icon definition, expecting strings in "class" property.'); });

            // validate locale property (may be a plain string)
            if ('locale' in def) {
                def.locale = Utils.makeArray(def.locale);
                Utils.ensureArrayNE(def.locale, 'Invalid icon definition, expecting string or array as "locale" property.');
                def.locale.forEach(function (lc) { Utils.ensureStringNE(lc, 'Invalid icon definition, expecting strings in "locale" property.'); });
            }

            // the source PNG images, mapped by icon set name ('desktop', 'retina')
            def.images = {};

            // the base size of the icon, for scaling factor 1 (width will be determined dynamically, see below)
            def.width = null;
            def.height = ICON_HEIGHT;

            // load the images for all scale factors, store them in the 'images' map
            ICONSET_RESOLUTIONS.forEach(function (resolutionData) {

                // current icon set name, scaling factor, and expected height for the scaling factor
                var resName = resolutionData.name;
                var resScale = resolutionData.scale;
                var height = def.height * resScale;

                // read the source image file into a binary buffer
                var path = IMAGES_ROOT + '/' + def.path + '_' + height + '.png';
                var buffer = grunt.file.read(path, { encoding: null });

                // parse the source image file
                var image = new PNGImage();
                image.parse(buffer, function (err) {
                    Utils.ensure(!err, 'Invalid image at ' + path + '.');

                    // validate the image height
                    Utils.ensure(image.height === height, 'Wrong image height in file ' + path + ' (expected ' + height + 'px but got ' + image.height + 'px).');

                    // store width for first imported image, check width of images with other scaling factor
                    if (!_.isNumber(def.width)) {
                        def.width = image.width / resScale;
                    } else {
                        Utils.ensure(image.width === def.width * resScale, 'Wrong image width in file ' + path + ' (expected ' + (def.width * resScale) + 'px but got ' + image.width + 'px).');
                    }

                    // store the image object, continue after all images have been loaded
                    def.images[resName] = image;
                    imageCountDown.step();
                });
            });
        });

        // wait for all source images to be loaded
        imageCountDown.done(function () {

            // the height of the icon occupied in the icon set (one pixel padding above and below)
            var height = ICON_HEIGHT + 2;

            // initialize the LESS file containing the icon definitions
            iconLessLines.push(
                '.io-ox-office-main i[class^="docs-"] {',
                'width: ' + height + 'px;', // default width, may be overridden by some icons
                'height: ' + height + 'px;'
            );

            // calculate the positions of all icons in the final icon set bitmaps
            var offsetx = 0, offsety = 0;
            definitions.forEach(function (def) {

                // the width of the icon occupied in the icon set (one pixel padding left and right)
                var width = def.width + 2;

                // start a new line in the icon set on demand
                if (offsetx + width > ICONSET_WIDTH) {
                    offsetx = 0;
                    offsety += height;
                }

                // store the position in the 'def' array element
                def.x = offsetx;
                def.y = offsety;

                // generate the text line for the current icon in the LESS file
                var line = def.class.map(function (cls) { return '&.docs-' + cls; }).join(', ');
                if (def.locale) { line += ' { ' + def.locale.map(function (lc) { return '&.lc-' + lc; }).join(', '); }
                line += ' { background-position: ' + -offsetx + 'px ' + -offsety + 'px;';
                if (def.width !== ICON_HEIGHT) { line += ' width: ' + (def.width + 2) + 'px;'; }
                line += ' }';
                if (def.locale) { line += ' }'; }
                iconLessLines.push(line);

                // go to next offset position
                offsetx += width;
            });

            // the resulting height of an icon set, for scaling factor 1
            var ICONSET_HEIGHT = offsety + height;

            // finalize the text lines of the LESS file
            iconLessLines.push(
                'background-size: ' + ICONSET_WIDTH + 'px ' + ICONSET_HEIGHT + 'px;',
                '}'
            );

            // create and write an icon set for each combination of scaling factor and output color
            ICONSET_RESOLUTIONS.forEach(function (resolutionData) {

                // current icon set name, and scaling factor
                var resName = resolutionData.name;
                var resScale = resolutionData.scale;

                // process all target colors
                ICONSET_COLORS.forEach(function (colorData) {

                    // new image data is not clean out-of-the-box, explicitly fill with zeros
                    var image = new PNGImage({ width: ICONSET_WIDTH * resScale, height: ICONSET_HEIGHT * resScale });
                    image.data.fill(0);

                    // insert all source images into the icon set
                    definitions.forEach(function (def) {
                        def.images[resName].bitblt(image, 0, 0, def.width * resScale, def.height * resScale, (def.x + 1) * resScale, (def.y + 1) * resScale);
                    });

                    // redye all pixels in the image set
                    var color = colorData.color;
                    for (var d = image.data, i = 0, l = d.length; i < l; i += 4) {
                        for (var j = 0; j < 3; j += 1) { d[i + j] = color[j]; }
                    }

                    // create the output stream for the icon set
                    var path = DEST_ROOT + '/' + getIconSetFileName(resName, colorData.name);
                    var strm = fs.createWriteStream(path);
                    strm.on('error', function () { Utils.ensure(false, 'Cannot create icon set file ' + path + '.'); });

                    // stream the image to the output file
                    strm = image.pack().pipe(strm);
                    strm.on('error', function () { Utils.ensure(false, 'Cannot write image file ' + path + '.'); });

                    // continue after all icon sets have been written
                    strm.on('finish', function () {
                        Utils.logFileCreated(path);
                        iconSetCountDown.step();
                    });
                });
            });
        });

        // finalize and write the LESS files (as last step, after generating the icon sets)
        iconSetCountDown.done(function () {
            Utils.writeLines(DEFINITIONS_LESS_PATH, defLessLines);
            Utils.writeLines(DOCS_ICONS_LESS_PATH, iconLessLines);
            grunt.log.ok('Icon sets created successfully.');
            resolveTask();
        });
    });
};
