/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/tk/utils/debugutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/tracking'
], function (Config, Utils, Tracking) {

    'use strict';

    var STORAGE_KEY = 'office:debug:timer-height';
    var DEBUG_TIMERS = Config.getDebugUrlFlag('office:debug-timers');
    var TEST_ENV = Config.AUTOTEST || Config.UNITTEST;

    var deferredLogs = [];
    var timerLogs = [];

    var debugPanel = null;
    var timerPanel = null;
    var deferredPanel = null;

    // initialize the GUI elements
    if (TEST_ENV && DEBUG_TIMERS) {
        (function () {

            var height = localStorage.getItem(STORAGE_KEY) || 50;

            function updateHeight(newHeight) {
                height = newHeight;
                localStorage.setItem(STORAGE_KEY, height);
                debugPanel.css('height', height + 'px');
                $('#io-ox-core').css('bottom', height + 'px');
                $(window).trigger('resize');
            }

            debugPanel = $('<div style="-webkit-user-select: text; user-select: text; position: fixed; left: 0; bottom: 0; width: 100%; height: 200px; z-index: 5000;">');
            timerPanel = $('<div style="white-space: pre; background: #3c73aa; color: white; padding: 10px; float: left; border-right: 1px solid white; height: 100%; width: 50%; overflow-y: scroll; overflow-x: auto;">');
            deferredPanel = $('<div style="white-space: pre; background: #3c73aa; color: white; padding: 10px; height: 100%; width: 50%; overflow-y: scroll; overflow-x: auto;">');

            $('body').append(debugPanel);
            debugPanel.append(timerPanel, deferredPanel);
            updateHeight(height);

            var resizerNode = $('<div class="resizer" tabindex="-1" style="top: -5px;position: absolute; width: 100%; cursor: row-resize; height: 10px;">');
            debugPanel.append(resizerNode);

            Tracking.enableTracking(resizerNode);

            resizerNode.on('tracking:move', function (evt) {
                updateHeight(Utils.minMax(30, Math.round(window.innerHeight / 2), height - evt.moveY));
            });
        }());
    }

    // static class DebugUtils ================================================

    var DebugUtils = {};

    // static methods ---------------------------------------------------------

    DebugUtils.logQueue = TEST_ENV ? function (queue, deferred) {

        var text = {};
        var panel = null;
        var logs = null;

        if (deferred) {
            deferredLogs = [];
            panel = deferredPanel;
            logs = deferredLogs;
            text['Open deferreds: ' + _.size(queue)] = true;
        } else {
            timerLogs = [];
            panel = timerPanel;
            logs = timerLogs;
            text['Open timers: ' + _.size(queue)] = true;
        }

        _.each(queue, function (entry) {
            var infoString;
            var windowID;
            var priority = entry.priority ? entry.priority : 'default-proc';
            if (entry.consoleString) {
                infoString = entry.consoleString;
            } else {
                infoString = entry.id + '\t';
                if (!entry.app || !entry.app.getWindow) {
                    infoString += '-------';
                    //Testcode for checking if new timers or deferreds without app are created
//                    if (!entry.infoString ||
//                        (entry.infoString.indexOf('BasePopup') === -1 &&
//                        entry.infoString.indexOf('Group') === -1 &&
//                        entry.infoString.indexOf('ListMenu') === -1 &&
//                        entry.infoString.indexOf('Dialogs') === -1 &&
//                        entry.infoString.indexOf('BaseMenu') === -1)) {
//                        window.console.log('NO APP: ' + entry.infoString);
//                    }
                } else {
                    windowID = entry.app.getWindow().nodes.outer.attr('id');
                    infoString += windowID;
                }
                infoString +=  '\t' + entry.infoString;
                infoString +=  '\t' + priority;
                entry.consoleString = infoString;
            }
            text[infoString] = true;
            if (!entry.log) {
                entry.log = {
                    windowID: windowID,
                    key: entry.infoString,
                    priority: priority
                };
            }
            logs.push(entry.log);
        });

        if (DEBUG_TIMERS) {
            panel.text(Object.keys(text).slice(0, 50).join('\n'));
        }
    } : _.noop;

    DebugUtils.getRunningJobs = function () {
        return {
            deferred: deferredLogs,
            timer: timerLogs
        };
    };

    // exports ================================================================

    return DebugUtils;

});
