/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/utils', [
    'io.ox/core/event',
    'io.ox/core/api/user',
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils/logger',
    'static/3rd.party/purify.min.js',
    'less!io.ox/office/tk/style'
], function (Events, UserAPI, Config, Logger, DOMPurify) {

    'use strict';

    var // the global element used to store DOM elements without showing them
        hiddenStorageNode = $('<div id="io-ox-office-temp" class="noI18n">').prependTo('body'),

        // a helper node to workaround selection problems in IE (bug 28515, bug 28711)
        focusHelperNode = $('<div class="focus-helper" contenteditable="true" tabindex="-1">'),

        // a helper node to workaround focus behavior on touch devices
        hiddenFocusNode = $('<div class="focus-target" tabindex="-1">'),

        // a global timer that can be used for performance debugging
        globalTimer = null,

        // a global counter that can be used for performance debugging
        globalCounter = null,

        // whether the times of several following global timers shall be accumulated
        accumulateTime = false,

        // the accumulated time of several following global timers
        accumulatedTime = null,

        // the number of accumulated values
        accumulatedCounter = null,

        // the last cached keydown event on the document
        cachedKeyEvent = null,

        // logging protocol for Selenium events
        seleniumProtocol = [];

    // static class Utils =====================================================

    var // the exported Utils class
        Utils = {};

    // static extensions ------------------------------------------------------

    Events.extend(Utils);
    Logger.extend(Utils);

    // constants --------------------------------------------------------------

    /**
     * Distance between the number 1 and the smallest number greater than 1
     * (the value of 2^-52). Provided as non-standard property Number.EPSILON
     * by Chrome and Firefox, but not supported by IE.
     *
     * @constant
     */
    Utils.EPSILON = Math.pow(2, -52);

    /**
     * The smallest positive normalized floating-point number (browsers may
     * support smaller numbers though, which are stored with a denormalized
     * mantissa).
     *
     * @constant
     */
    Utils.MIN_NUMBER = Math.pow(2, -1022);

    /**
     * The Unicode character for a horizontal ellipsis.
     *
     * @constant
     */
    Utils.ELLIPSIS_CHAR = '\u2026';

    /**
     * A unique object used as return value in callback functions of iteration
     * loops to break the iteration process immediately.
     *
     * @constant
     */
    Utils.BREAK = {};

    /**
     * A function that does nothing and returns its calling context (the symbol
     * this).
     *
     * @constant
     */
    Utils.NOOP = function () { return this; };

    /**
     * Number of colors in the color scheme used for highlighting of specific
     * document contents.
     *
     * @constant
     */
    Utils.SCHEME_COLOR_COUNT = 8;

    /**
     * A Boolean flag specifying whether the current display is a retina
     * display.
     *
     * @constant
     */
    Utils.RETINA = _.device('retina');

    /**
     * A Boolean flag specifying whether the current device is a touch device.
     *
     * @constant
     */
    Utils.TOUCH = Utils.TOUCHDEVICE = _.device('touch'); // bug 39937: this is the 'official' touch detection!

    /**
     * A Boolean flag specifying whether the current device is small (smart
     * phones).
     *
     * @constant
     */
    Utils.SMALL_DEVICE = _.device('smartphone');

    /**
     * A Boolean flag specifying whether the current device is small (smart
     * phones) or medium sized (tablets with a width of less than 1024px).
     *
     * @constant
     */
    Utils.COMPACT_DEVICE = Utils.SMALL_DEVICE || _.device('tablet');

    /**
     * A Boolean flag specifying whether the current device is an IOS Device.
     *
     * @constant
     */
    Utils.IOS = !!(Utils.TOUCHDEVICE && _.browser.iOS && _.browser.Safari);

    /**
     * A Boolean flag specifying whether the current device is Android and the
     * browser is Chrome.
     *
     * @constant
     */
    Utils.CHROME_ON_ANDROID = !!(_.browser.Android && _.browser.Chrome);

    /**
     * A Boolean flag specifying whether the current (really) browser supports the
     * clipboard API.
     *
     * @constant
     */
    Utils.CLIPBOARD_API_SUPPORTED = !Utils.CHROME_ON_ANDROID && !_.browser.Edge;

    /**
     * Determines the performance level of the system using some system and
     * browser specific values.
     *
     * @returns {Number}
     *  A performance level, as percentage between 10 and 100, where the value
     *  10 means poor performance, and the value 100 best performance.
     */
    Utils.PERFORMANCE_LEVEL = (function () {

        var // default factor for unknown devices/browsers
            DEFAULT_FACTOR = 0.8,
            // the resulting factor
            resultLevel = 100;

        function getAspectFactor(aspects) {
            var aspectFactor = _.find(aspects, function (factor, key) { return _.device(key); });
            return _.isNumber(aspectFactor) ? aspectFactor : DEFAULT_FACTOR;
        }

        // reduce performance level by device type
        resultLevel *= getAspectFactor({ desktop: 1, tablet: 0.6, smartphone: 0.5 });
        // reduce performance level by browser type
        resultLevel *= getAspectFactor({ chrome: 1, safari: 1, firefox: 0.9, ie: 0.6 });

        // work-around for lousy Safari/Windows
        if (_.device('safari') && _.device('windows')) { resultLevel *= 0.25; }

        // ensure value is at least 10%
        resultLevel = Math.max(10, resultLevel);
        Utils.log('Utils: platform performance level: ' + (Math.round(resultLevel * 10) / 10) + '%');
        return resultLevel;
    }());

    /**
     * A Boolean flag specifying whether the current device is considered to be
     * slow.
     *
     * @constant
     */
    Utils.SLOW_DEVICE = Utils.PERFORMANCE_LEVEL <= 50;

    /**
     * The maximum explicit size of single DOM nodes, in pixels. The size is
     * limited by browsers; by using larger sizes the elements may collapse to
     * zero size.
     * - IE limits this to 1,533,917 (2^31/1400) in both directions.
     * - Firefox limits this to 17,895,696 (0x1111110) in both directions.
     * - Chrome limits this to 33,554,428 (0x1FFFFFC) in both directions.
     *
     * @constant
     */
    Utils.MAX_NODE_SIZE = _.browser.IE ? 1.5e6 : _.browser.WebKit ? 33.5e6 : 17.8e6;

    /**
     * The maximum width of a container node that can be reached by inserting
     * multiple child nodes, in pixels. The width is limited by browsers; by
     * inserting more nodes the width of the container node becomes incorrect
     * or collapses to zero.
     * - IE limits this to 10,737,418 (2^31/200) pixels.
     * - Firefox and Chrome do not allow to extend the container node beyond
     *   the explicit size limits of a single node (see comment for the
     *   Utils.MAX_NODE_SIZE constant above).
     *
     * @constant
     */
    Utils.MAX_CONTAINER_WIDTH = _.browser.IE ? 10.7e6 : Utils.MAX_NODE_SIZE;

    /**
     * The maximum height of a container node that can be reached by inserting
     * multiple child nodes, in pixels. The height is limited by browsers; by
     * inserting more nodes the height of the container node becomes incorrect
     * or collapses to zero.
     * - IE limits this to 21,474,836 (2^31/100) pixels.
     * - Firefox and Chrome do not allow to extend the container node beyond
     *   the explicit size limits of a single node (see comment for the
     *   Utils.MAX_NODE_SIZE constant above).
     *
     * @constant
     */
    Utils.MAX_CONTAINER_HEIGHT = _.browser.IE ? 21.4e6 : Utils.MAX_NODE_SIZE;

    /**
     * max 600ms for detection of double touch tap & double mouse click
     *
     * @constant
     */
    Utils.DBL_CLICK_TIMEOUT = 600;

    // browser helpers --------------------------------------------------------

    /**
     * if the browser (really) supports the clipboard API
     * and the assigned event has clipboard data, the clipboard data is returned
     *
     * @param {JQuery-Event}
     *
     * @return {Object} clipboardData from the assigned event,
     * but only if the browser support the clipboard API
     */
    Utils.getClipboardData = function (event) {
        var // the clipboard event data
            clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;

        if (clipboardData && Utils.CLIPBOARD_API_SUPPORTED) {
            return clipboardData;
        }
    };

    /**
     * Checks that a specific browser at least has the necessary major version.
     *
     * @param {String}
     *  The name of the browser, e.g. Chrome, Firefox, IE, Safari
     *
     * @param {Number}
     *  The minimal major version of browser.
     *
     * @returns {Boolean}
     *  Returns TRUE if the browser name and the minimal major version matches,
     *  otherwise FALSE.
     */
    Utils.isBrowserAndMinVersion = function (browser, neededMajorVersion) {
        var version = _.browser[browser];

        return _.isNumber(version) && (version > (neededMajorVersion - 1));
    };

    // generic JS object helpers ----------------------------------------------

    /**
     * Returns a new object containing a single property with the specified key
     * and value.
     *
     * @param {String} key
     *  The name of the property to be inserted into the returned object.
     *
     * @param {Any} value
     *  The value of the property to be inserted into the returned object.
     *
     * @returns {Object}
     *  A new object with a single property.
     */
    Utils.makeSimpleObject = function (key, value) {
        var object = {};
        object[key] = value;
        return object;
    };

    /**
     * Creates a new object with the specified property names, and the return
     * values of a callback function.
     *
     * @param {Array<String>} keys
     *  The property names for the resulting object. Despite of a string array,
     *  this property can be any other array-like object that can be iterated
     *  over; especially a string to create an object with single-character
     *  properties.
     *
     * @param {Function} callback
     *  The callback function invoked for all property names. Receives the
     *  following parameters:
     *  (1) {String} key
     *      The name of the new object property.
     *  The return value of the function will be used as value for the new
     *  object property, except for the undefined return value that will not be
     *  added to the new object.
     *
     * @param {Object} [context]
     *  The context bound to the callback function.
     *
     * @returns {Object}
     *  A new object with all properties generated by the callback function.
     */
    Utils.makeObject = function (keys, callback, context) {
        return _.reduce(keys, function (result, key) {
            var value = callback.call(context, key);
            if (value !== undefined) { result[key] = value; }
            return result;
        }, {});
    };

    /**
     * Returns a new object containing properties with the specified keys and
     * the same value.
     *
     * @param {Array<String>} keys
     *  The property names for the resulting set. Despite of a string array,
     *  this property can be any other array-like object that can be iterated
     *  over; especially a string to create a set with single-character
     *  properties.
     *
     * @param {Any} [value=true]
     *  The value to be set for all properties. If omitted, the boolean value
     *  true will be used.
     *
     * @returns {Object}
     *  A new set object with the specified properties.
     */
    Utils.makeSet = function (keys, value) {
        if (value === undefined) { value = true; }
        return _.reduce(keys, function (result, key) {
            result[key] = value;
            return result;
        }, {});
    };

    /**
     * Returns whether the passed object contains the specified property.
     *
     * @param {Object} object
     *  The object to be checked for its properties.
     *
     * @param {String|RegExp} propertyName
     *  The name of the property to be checked. If set to a regular expression,
     *  at least one of the object's property names has to match it.
     *
     * @returns {Boolean}
     *  Whether the specified property exists in the object.
     */
    Utils.hasProperty = function (object, propertyName) {
        return _.isString(propertyName) ? _.has(object, propertyName) :
            (_.isRegExp(propertyName) && _.any(object, function (value, name) { return propertyName.test(name); }));
    };

    /**
     * Returns whether the passed object contains any of the specified
     * properties.
     *
     * @param {Object} object
     *  The object to be checked for its properties.
     *
     * @param {Array<String>} propertyNames
     *  The names of all properties to be checked.
     *
     * @returns {Boolean}
     *  Whether at least one of the specified properties exists in the object.
     */
    Utils.hasAnyProperty = function (object, propertyNames) {
        return _.any(propertyNames, Object.prototype.hasOwnProperty, object);
    };

    /**
     * Returns whether the passed object contains all of the specified
     * properties.
     *
     * @param {Object} object
     *  The object to be checked for its properties.
     *
     * @param {String[]} propertyNames
     *  The names of all properties to be checked.
     *
     * @returns {Boolean}
     *  Whether all of the specified properties exists in the object.
     */
    Utils.hasAllProperties = function (object, propertyNames) {
        return _.all(propertyNames, Object.prototype.hasOwnProperty, object);
    };

    /**
     * Returns whether the specified properties of the passed objects are
     * equal, while ignoring all other properties. If a property is missing in
     * both objects, it is considered to be equal. Uses a custom comparator or
     * the Underscore method _.isEqual() to compare the property values.
     *
     * @param {Object} object1
     *  The first object to be compared to the other.
     *
     * @param {Object} object2
     *  The second object to be compared to the other.
     *
     * @param {String[]} propertyNames
     *  The names of all properties of the objects that will be compared.
     *
     * @param {Function} [comparator=_.isEqual]
     *  A binary predicate function that returns true if the passed property
     *  values are considered being equal. Will be called, if both objects
     *  passed to this method contain a specific property, and receives the
     *  property values from both objects.
     *
     * @returns {Boolean}
     *  Whether all specified properties are equal in both objects.
     */
    Utils.hasEqualProperties = function (object1, object2, propertyNames, comparator) {

        // default to the _isEqual() method to compare property values
        comparator = _.isFunction(comparator) ? comparator : _.isEqual;

        // process all specified properties
        return _.all(propertyNames, function (propName) {
            var hasProp1 = _.has(object1, propName),
                hasProp2 = _.has(object2, propName);
            return (hasProp1 === hasProp2) && (!hasProp1 || comparator(object1[propName], object2[propName]));
        });
    };

    /**
     * Creates a new object with the same properties as contained in the passed
     * object, and assigns the result of the passed converter callback function
     * to the properties.
     *
     * @param {Object} object
     *  The object whose properties will be transformed.
     *
     * @param {Function} converter
     *  The converter callback function invoked for all properties of the
     *  passed object. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current object property.
     *  (2) {String} name
     *      The name of the current object property.
     *  (3) {Object} object
     *      The original object passed to this method.
     *  The return value of the function will be used as value for the current
     *  property in the resulting object.
     *
     * @param {Object} [context]
     *  The context bound to the callback function.
     *
     * @returns {Object}
     *  A new object with all properties of the object passed to this method,
     *  containing the return values of the converter callback function.
     */
    Utils.mapProperties = function (object, converter, context) {
        var result = {};
        _.each(object, function (propValue, propName) {
            result[propName] = converter.call(context, propValue, propName, object);
        });
        return result;
    };

    /**
     * Adds a property with a specific value to all objects in the passed list.
     * An arbitrary number of property names and values can be passed as pairs
     * of parameters.
     *
     * @param {Object|Array} list
     *  An iterable list that MUST contain objects as elements.
     *
     * @param {String} name1
     *  The name of the first property to be added.
     *
     * @param {Any} value1
     *  The value of the first property to be added.
     *
     * @param {String} [name2]
     *  The name of the second property to be added.
     *
     * @param {Any} [value2]
     *  The value of the second property to be added.
     *
     * @returns {Object|Array}
     *  The list passed to this method, for convenience.
     */
    Utils.addProperty = function (list) {
        _.each(list, function (object) {
            for (var i = 1, l = this.length; i < l; i += 2) {
                object[this[i]] = this[i + 1];
            }
        }, arguments);
        return list;
    };

    /**
     * Deletes a property from all objects in the passed list.
     *
     * @param {Object|Array} list
     *  An iterable list that MUST contain objects as elements.
     *
     * @param {String} name1
     *  The name of the first property to be deleted.
     *
     * @param {String} [name2]
     *  The name of the second property to be deleted.
     *
     * @returns {Object|Array}
     *  The list passed to this method, for convenience.
     */
    Utils.deleteProperty = function (list) {
        _.each(list, function (object) {
            for (var i = 1, l = this.length; i < l; i += 1) {
                delete object[this[i]];
            }
        }, arguments);
        return list;
    };

    /**
     * Compares that the passed arrays contain the exact same elements. The
     * elements will be compared with the equality operator '===', in contrast
     * to the method _.isEqual() which compares complex array elements (objects
     * and arrays) deeply.
     *
     * @param {Number[]} array1
     *  The first array that will be compared to the second array.
     *
     * @param {Number[]} array2
     *  The second array that will be compared to the first array.
     *
     * @returns {Boolean}
     *  Whether the arrays contain the exact same elements, without comparing
     *  the elements deeply.
     */
    Utils.equalArrays = function (array1, array2) {
        return (array1 === array2) || ((array1.length === array2.length) && _.all(array1, function (element, index) { return element === array2[index]; }));
    };

    /**
     * Compares two pairs of values. Defines a natural order for sorting with
     * the following behavior (let A and B be the pairs): If the first element
     * of A is less than the first element of B; or if both first elements are
     * equal, and the second element of A is less than the second element of B,
     * then A is considered less than B.
     *
     * @param {Number} a1
     *  The first element of the first pair.
     *
     * @param {Number} a2
     *  The second element of the first pair.
     *
     * @param {Number} b1
     *  The first element of the second pair.
     *
     * @param {Number} b2
     *  The second element of the second pair.
     *
     * @param {Function} [comparator]
     *  A custom comparator function for the passed values. Defaults to a
     *  function that returns the difference of two numbers using the built-in
     *  minus operator.
     *
     * @returns {Number}
     *  - A negative number, if the first pair is less than the second pair.
     *  - A positive number, if the first pair is greater than the second pair.
     *  - Zero, if both pairs are equal.
     */
    Utils.comparePairs = function (a1, a2, b1, b2, comparator) {
        if (typeof comparator !== 'function') { comparator = function (n1, n2) { return n1 - n2; }; }
        var diff = comparator(a1, b1);
        return (diff === 0) ? comparator(a2, b2) : diff;
    };

    /**
     * Compares the two passed numeric arrays lexicographically (starting with
     * the first element of both arrays, and visiting the following elements
     * until a pair of elements is different).
     *
     * @param {Number[]} array1
     *  The first array that will be compared to the second array.
     *
     * @param {Number[]} array2
     *  The second array that will be compared to the first array.
     *
     * @param {Number} [maxLength]
     *  If specified, compares the specified number of leading elements in the
     *  passed arrays. Otherwise, the entire arrays will be compared.
     *
     * returns {Number}
     *  A negative value, if array1 is lexicographically less than array2; a
     *  positive value, if array1 is lexicographically greater than array2; or
     *  zero, if both arrays are equal.
     */
    Utils.compareNumberArrays = function (array1, array2, maxLength) {

        var // number of elements in the arrays to be compared
            length1 = _.isNumber(maxLength) ? Math.min(array1.length, maxLength) : array1.length,
            length2 = _.isNumber(maxLength) ? Math.min(array2.length, maxLength) : array2.length,
            // minimum length of both arrays
            length = Math.min(length1, length2),
            // loop index
            index = 0;

        // compare all array elements
        for (index = 0; index < length; index += 1) {
            if (array1[index] !== array2[index]) {
                return array1[index] - array2[index];
            }
        }

        // all compared elements are equal, compare array lengths (restricted to the passed maximum length)
        return length1 - length2;
    };

    /**
     * Removes all occurrences of the specified value from the array in-place.
     * Uses a custom comparator or the Underscore method _.isEqual() to compare
     * the property values.
     *
     * @param {Array} array
     *  (in/out) The array to be shortened.
     *
     * @param {Any} value
     *  The value to be removed from the array.
     *
     * @param {Function} [comparator=_.isEqual]
     *  A binary predicate function that returns true if the two passed values
     *  are considered being equal. Will be called to compare the passed value
     *  against the elements of the array.
     *
     * @returns {Number}
     *  The number of elements removed from the array.
     */
    Utils.spliceValue = function (array, value, comparator) {

        var // the old array length
            length = array.length;

        // default to the _isEqual() method to compare values
        comparator = _.isFunction(comparator) ? comparator : _.isEqual;

        // remove all elements matching the passed value
        for (var index = length - 1; index >= 0; index -= 1) {
            if (comparator(value, array[index])) {
                array.splice(index, 1);
            }
        }

        return length - array.length;
    };

    /**
     * Calls the passed callback function for all elements of the passed array.
     *
     * @param {Array} array
     *  The array whose elements will be iterated, or another array-like object
     *  providing a 'length' property and element access via bracket operator.
     *
     * @param {Function} callback
     *  The callback function invoked for all elements in the array. Receives
     *  the following parameters:
     *  (1) {Any} element
     *      The current array element.
     *  (2) {Number} index
     *      The index of the array element.
     *  (3) {Array} array
     *      The array passed to this method.
     *  If the callback function returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.begin]
     *      If specified, this method only visits array elements with an index
     *      greater than or equal to this value only. If omitted, this method
     *      visits all elements from the beginning of the array.
     *  @param {Number} [options.end]
     *      If specified, this method only visits array elements with an index
     *      less than this value only (half-open interval). If omitted, this
     *      method visits all elements to the end of the array.
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the elements will be visited in reversed order. In
     *      reverse mode, the callback function may remove the array element
     *      currently visited, or elements that are following the visited
     *      element, from the array in-place.
     *  @param {Object} [options.context]
     *      The context the callback function will be bound to.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    Utils.iterateArray = function (array, callback, options) {

        var begin = Utils.getIntegerOption(options, 'begin', 0),
            end = Utils.getIntegerOption(options, 'end', array.length),
            context = Utils.getOption(options, 'context'),
            index = 0;

        if (Utils.getBooleanOption(options, 'reverse', false)) {
            for (index = end - 1; index >= begin; index -= 1) {
                if (callback.call(context, array[index], index, array) === Utils.BREAK) {
                    return Utils.BREAK;
                }
            }
        } else {
            for (index = begin; index < end; index += 1) {
                if (callback.call(context, array[index], index, array) === Utils.BREAK) {
                    return Utils.BREAK;
                }
            }
        }
    };

    /**
     * Returns the array index of the first element in the passed array, that
     * passes a truth test.
     *
     * @param {Array} array
     *  The array to be searched for a matching element, or another array-like
     *  object providing a 'length' property and element access via bracket
     *  operator.
     *
     * @param {Function} predicate
     *  The callback function implementing the truth test. Receives the array
     *  element, the element index, and the complete array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.begin]
     *      If specified, this method searches for array elements with an index
     *      greater than or equal to this value only. If omitted, this method
     *      takes all elements from the beginning of the array into account.
     *  @param {Number} [options.end]
     *      If specified, this method searches for array elements with an index
     *      less than this value only (half-open interval). If omitted, this
     *      method takes all elements to the end of the array into account.
     *  @param {Boolean} [options.sorted=false]
     *      If set to true, the elements in the passed array are assumed to be
     *      sorted in a way that no element that passes the truth test precedes
     *      an element that does not pass (all matching elements are following
     *      all non-matching elements). Internally, a binary search algorithm
     *      will be used to speed up the search process.
     *  @param {Object} [options.context]
     *      The context the callback function will be bound to.
     *
     * @returns {Number}
     *  The array index of the first matching element in the array; or -1 if no
     *  element was found.
     */
    Utils.findFirstIndex = function (array, predicate, options) {

        var index = 0,
            begin = Utils.getIntegerOption(options, 'begin', 0),
            end = Utils.getIntegerOption(options, 'end', array.length),
            context = Utils.getOption(options, 'context');

        if (Utils.getBooleanOption(options, 'sorted', false)) {
            // fast binary search
            while (begin < end) {
                index = Math.floor((begin + end) / 2);
                if (predicate.call(context, array[index], index, array)) {
                    end = index;
                } else {
                    begin = index + 1;
                }
            }
            if (begin < array.length) { return begin; }
        } else {
            // linear search
            for (index = begin; index < end; index += 1) {
                if (predicate.call(context, array[index], index, array)) {
                    return index;
                }
            }
        }

        return -1;
    };

    /**
     * Returns the array index of the last element in the passed array, that
     * passes a truth test.
     *
     * @param {Array} array
     *  The array to be searched for a matching element, or another array-like
     *  object providing a 'length' property and element access via bracket
     *  operator.
     *
     * @param {Function} predicate
     *  The callback function implementing the truth test. Receives the array
     *  element, the element index, and the complete array.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.begin]
     *      If specified, this method searches for array elements with an index
     *      greater than or equal to this value only. If omitted, this method
     *      takes all elements from the beginning of the array into account.
     *  @param {Number} [options.end]
     *      If specified, this method searches for array elements with an index
     *      less than this value only (half-open interval). If omitted, this
     *      method takes all elements to the end of the array into account.
     *  @param {Boolean} [options.sorted=false]
     *      If set to true, the elements in the passed array are assumed to be
     *      sorted in a way that no element that passes the truth test follows
     *      an element that does not pass (all non-matching elements are
     *      following all matching elements). Internally, a binary search
     *      algorithm will be used to speed up the search process. By default,
     *      without the 'sorted' flag, this method iterates backwards through
     *      the array, and returns the first array element that passes the
     *      truth test.
     *  @param {Object} [options.context]
     *      The context the callback function will be bound to.
     *
     * @returns {Number}
     *  The array index of the last matching element in the array; or -1 if no
     *  element was found.
     */
    Utils.findLastIndex = function (array, predicate, options) {

        var index = 0,
            begin = Utils.getIntegerOption(options, 'end', array.length) - 1,
            end = Utils.getIntegerOption(options, 'begin', 0) - 1,
            context = Utils.getOption(options, 'context');

        if (Utils.getBooleanOption(options, 'sorted', false)) {
            // fast binary search
            while (begin > end) {
                index = Math.ceil((begin + end) / 2);
                if (predicate.call(context, array[index], index, array)) {
                    end = index;
                } else {
                    begin = index - 1;
                }
            }
            if (begin < array.length) { return begin; }
        } else {
            // linear search
            for (index = begin; index > end; index -= 1) {
                if (predicate.call(context, array[index], index, array)) {
                    return index;
                }
            }
        }

        return -1;
    };

    /**
     * Finds the first element in the passed array, that passed a truth test.
     *
     * @param {Array} array
     *  The array to be searched for a matching element, or another array-like
     *  object providing a 'length' property and element access via bracket
     *  operator.
     *
     * @param {Function} predicate
     *  The callback function implementing the truth test. Receives the array
     *  element, the element index, and the complete array.
     *
     * @param {Object} [options]
     *  Optional parameters. See method Utils.findFirstIndex() for details.
     *
     * @returns {Any}
     *  The last matching element in the array; or undefined, if no element
     *  passes the truth test.
     */
    Utils.findFirst = function (array, predicate, options) {
        var index = Utils.findFirstIndex(array, predicate, options);
        return (index >= 0) ? array[index] : undefined;
    };

    /**
     * Finds the last element in the passed array, that passed a truth test.
     *
     * @param {Array} array
     *  The array to be searched for a matching element, or another array-like
     *  object providing a 'length' property and element access via bracket
     *  operator.
     *
     * @param {Function} predicate
     *  The callback function implementing the truth test. Receives the array
     *  element, the element index, and the complete array.
     *
     * @param {Object} [options]
     *  Optional parameters. See method Utils.findLastIndex() for details.
     *
     * @returns {Any}
     *  The last matching element in the array; or undefined, if no element
     *  passes the truth test.
     */
    Utils.findLast = function (array, predicate, options) {
        var index = Utils.findLastIndex(array, predicate, options);
        return (index >= 0) ? array[index] : undefined;
    };

    /**
     * A method the works similarly to JSON.stringify(), but normalizes the
     * order of object properties.
     *
     * Example: JSON.stringify({a:1,b:2}) may return another string than
     * JSON.stringify({b:2,a:1}) although the objects are equal, as the order
     * of JavaScript object properties is unspecified. This method will return
     * the same key for both objects though.
     *
     * @param {Any} value
     *  The JSON value to be stringified. MUST be a JSON compatible value (no
     *  function, no regular expression, not undefined). Arrays and objects
     *  must consist of JSON compatible values too, and must not contain cyclic
     *  references. Arrays must not be sparse.
     *
     * @returns {String}
     *  The stringified JSON value.
     */
    Utils.stringifyJSON = function stringifyJSON(value) {

        // stringify array elements recursively
        if (_.isArray(value)) {
            return '[' + value.reduce(function (result, element) {
                return result + (result ? ',' : '') + stringifyJSON(element);
            }, '') + ']';
        }

        // sort and stringify object properties
        if (_.isObject(value)) {
            return '{' + _.keys(value).sort().reduce(function (result, key) {
                return result + (result ? ',' : '') + JSON.stringify(key) + ':' + stringifyJSON(value[key]);
            }, '') + '}';
        }

        // stringify everything else directly
        return JSON.stringify(value);
    };

    /**
     * Returns whether the passed value is a jQuery promise object (including
     * deferred objects).
     *
     * @param {Any} value
     *  Any value to be tested.
     *
     * @returns {Boolean}
     *  Whether the passed value is a jQuery promise object.
     */
    Utils.isPromise = function (value) {
        return _.isObject(value) && _.isFunction(value.promise) && _.isFunction(value.then);
    };

    /**
     * Invokes the passed asynchronous callback functions as a promise chain.
     *
     * @param {Function|Array<Function>|Null} [...]
     *  The asynchronous callback functions to be invoked. Every parameter of
     *  this method can be a single callback function, an array of callback
     *  functions, or null or undefined (will be skipped silently). Each
     *  callback function MUST return a promise.
     *
     * @returns {jQuery.Promise}
     *  A promise representing the chained callback functions.
     */
    Utils.invokeChainedAsync = function () {
        return _.reduce(arguments, function (promise1, arg) {
            return arg ? _.getArray(arg).reduce(function (promise2, callback) {
                return promise2 ? promise2.then(callback) : callback();
            }, promise1) : promise1;
        }, null) || $.when();
    };

    // calculation, conversion, string manipulation ---------------------------

    /**
     * Returns whether the Boolean representation of the passed values are
     * equal (whether both values are 'truthy', or both values are 'falsy').
     *
     * @param {Any} value1
     *  The first value.
     *
     * @param {Any} value2
     *  The second value.
     *
     * @returns {Boolean}
     *  Whether the Boolean representation of the passed values are equal.
     */
    Utils.boolEq = function (value1, value2) {
        return !value1 === !value2;
    };

    /**
     * Returns whether the Boolean representation of the passed values are
     * different (whether exactly one of the passed values is 'truthy', and the
     * other value is 'falsy').
     *
     * @param {Any} value1
     *  The first value.
     *
     * @param {Any} value2
     *  The second value.
     *
     * @returns {Boolean}
     *  Whether the Boolean representation of the passed values are different.
     */
    Utils.boolXor = function (value1, value2) {
        return !value1 !== !value2;
    };

    /**
     * Restricts the passed value to the specified numeric range.
     *
     * @param {Number} value
     *  The value to be restricted to the given range.
     *
     * @param {Number} min
     *  The lower border of the range.
     *
     * @param {Number} max
     *  The upper border of the range.
     *
     * @returns {Number}
     *  The passed value, if inside the given range, otherwise either the lower
     *  or upper border.
     */
    Utils.minMax = function (value, min, max) {
        return Math.min(Math.max(value, min), max);
    };

    /**
     * Fix for JavaScript's weird behavior of the modulus operator with
     * negative numbers.
     *
     * @param {Number} number1
     *  The dividend.
     *
     * @param {Number} number2
     *  The divisor.
     *
     * @returns {Number}
     *  The remainder of the division. The result will have the same sign as
     *  the divisor. Example: While -5%3 results in -2, Utils.mod(-5,3) results
     *  in 1 as expected.
     */
    Utils.mod = function (number1, number2) {
        return (number1 % number2 + number2) % number2;
    };

    /**
     * Calculates the decimal mantissa and exponent of the passed number.
     *
     * @param {Number} number
     *  The number to be split into mantissa and exponent.
     *
     * @returns {Object}
     *  The mantissa (property 'mant') in the interval +-[1;10), and the
     *  exponent (property 'exp') of the passed number, so that mant*10^exp is
     *  equal to the passed number. If the passed number is zero, the mantissa
     *  will be set to 0, and the exponent will be set to NEGATIVE_INFINITY.
     */
    Utils.normalizeNumber = function (number) {

        // special handling for zero
        if (number === 0) { return { mant: 0, exp: Number.NEGATIVE_INFINITY }; }

        var exp = Math.floor(Math.log(Math.abs(number)) / Math.LN10),
            mant = number / Math.pow(10, exp);

        // due to rounding errors, the absolute value of the mantissa may become less than 1
        if (Math.abs(mant) < 1) { mant /= Math.abs(mant); }

        // due to rounding errors, the mantissa may become +-10 instead +-1, e.g. for the number 1000
        if (Math.abs(mant) >= 10) { mant /= 10; exp += 1; }

        return { mant: mant, exp: exp };
    };

    /**
     * calculates the mantissa of assigned number and exponent
     *
     * @param {Number} number
     * @param {Number} exponent
     *
     * @returns {Number}
     */
    Utils.mant = function (number, exp) {
        return number / Math.pow(10, exp);
    };

    /**
     * Rounds the passed floating-point number to the nearest multiple of the
     * specified precision.
     *
     * @param {Number} number
     *  The floating-point number to be rounded.
     *
     * @param {Number} precision
     *  The precision used to round the number. The number 1 will round to
     *  integers (exactly like the Math.round() method). Must be positive. If
     *  less than 1, must be the inverse of an integer to prevent further
     *  internal rounding errors.
     *
     * @returns {Number}
     *  The rounded number.
     */
    Utils.round = function (number, precision) {
        // Multiplication with small number may result in rounding errors (e.g.,
        // 227*0.1 results in 22.700000000000003), division by inverse number
        // works sometimes (e.g. 227/(1/0.1) results in 22.7), rounding the
        // inverse before division finally should work in all(?) cases, but
        // restricts valid precisions to inverses of integer numbers.
        number = Math.round((precision < 1) ? (number * Math.round(1 / precision)) : (number / precision));
        return (precision < 1) ? (number / Math.round(1 / precision)) : (number * precision);
    };

    /**
     * Rounds the passed floating-point number down to the nearest multiple of
     * the specified precision.
     *
     * @param {Number} number
     *  The floating-point number to be rounded.
     *
     * @param {Number} precision
     *  The precision used to round the number. The number 1 will round down to
     *  integers (exactly like the Math.floor() method). Must be positive. If
     *  less than 1, must be the inverse of an integer to prevent further
     *  internal rounding errors.
     *
     * @returns {Number}
     *  The rounded number.
     */
    Utils.roundDown = function (number, precision) {
        // see comment in Utils.round()
        number = Math.floor((precision < 1) ? (number * Math.round(1 / precision)) : (number / precision));
        return (precision < 1) ? (number / Math.round(1 / precision)) : (number * precision);
    };

    /**
     * Rounds the passed floating-point number up to the nearest multiple of
     * the specified precision.
     *
     * @param {Number} number
     *  The floating-point number to be rounded.
     *
     * @param {Number} precision
     *  The precision used to round the number. The number 1 will round up to
     *  integers (exactly like the Math.ceil() method). Must be positive. If
     *  less than 1, must be the inverse of an integer to prevent further
     *  internal rounding errors.
     *
     * @returns {Number}
     *  The rounded number.
     */
    Utils.roundUp = function (number, precision) {
        // see comment in Utils.round()
        number = Math.ceil((precision < 1) ? (number * Math.round(1 / precision)) : (number / precision));
        return (precision < 1) ? (number / Math.round(1 / precision)) : (number * precision);
    };

    /**
     * Rounds the passed floating-point number to the specified number of
     * significant digits, independent from the number of digits before and
     * after the decimal point.
     *
     * @param {Number} number
     *  The number to be rounded.
     *
     * @param {Number} digits
     *  The number of significant digits. Must be positive.
     *
     * @returns {Number}
     *  The rounded number.
     */
    Utils.roundSignificantDigits = function (number, digits) {
        var intDigits = Math.floor(Math.log(Math.abs(number)) / Math.LN10) + 1,
            exp = digits - intDigits,
            pow10 = (exp < 0) ? Math.pow(10, -exp) : Math.pow(10, exp);
        return (exp < 0) ? (Math.round(number / pow10) * pow10) : (Math.round(number * pow10) / pow10);
    };

    /**
     * Iterates through a range of numbers, without creating a temporary array.
     * Can be used as replacement for the for-loop to work-around the JSHint
     * warning that prevents to create local functions inside plain loops.
     *
     * @param {Number} begin
     *  The iteration will be started with this value.
     *
     * @param {Number} end
     *  The iteration will be stopped before this value will be reached
     *  (half-open range).
     *
     * @param {Function} iterator
     *  The iterator function that will be called for every value. Receives the
     *  current value as first parameter. If the iterator returns the
     *  Utils.BREAK object, the iteration process will be stopped immediately.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Object} [options.context]
     *      If specified, the iterator will be called with this context (the
     *      symbol 'this' will be bound to the context inside the iterator
     *      function).
     *  @param {Number} [options.step=1]
     *      If specified, the current value will be increased or decreased by
     *      this amount.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    Utils.iterateRange = function (begin, end, iterator, options) {

        var // context for iterator function
            context = Utils.getOption(options, 'context'),
            // step value
            step = Utils.getNumberOption(options, 'step', 1),
            // the current value
            value = begin;

        while ((step > 0) ? (value < end) : (value > end)) {
            if (iterator.call(context, value) === Utils.BREAK) { return Utils.BREAK; }
            value += step;
        }
    };

    Utils.convertToEMValue = (function () {

        var // the conversion factors between pixels and other units
            FACTORS = {
                px: 1 / 16,
                pt: 1 / 12
            };

        function convertToEM(value, fromUnit) {
            var em = (value * (FACTORS[fromUnit] || 1));
            em = Math.round(em * 10000) / 10000;
            return em;
        }
        return convertToEM;
    }());

    Utils.convertToEM = function (value, fromUnit) {
        return Utils.convertToEMValue(value, fromUnit) + 'em';
    };

    /**
     * Converts a length value from an absolute CSS measurement unit into
     * another absolute CSS measurement unit.
     *
     * @param {Number} value
     *  The length value to convert, as floating-point number.
     *
     * @param {String} fromUnit
     *  The CSS measurement unit of the passed value, as string. Supported
     *  units are 'px' (pixels), 'pc' (picas), 'pt' (points), 'in' (inches),
     *  'cm' (centimeters), and 'mm' (millimeters).
     *
     * @param {String} toUnit
     *  The target measurement unit.
     *
     * @param {Number} [precision]
     *  If specified, the resulting length will be rounded to the nearest
     *  multiple of this value. Must be positive.
     *
     * @returns {Number}
     *  The length value converted to the target measurement unit, as
     *  floating-point number.
     */
    Utils.convertLength = (function () {

        var // the conversion factors between pixels and other units
            FACTORS = {
                px: 1,
                pc: 1 / 9,
                pt: 4 / 3,
                in: 96,
                cm: 96 / 2.54,
                mm: 96 / 25.4
            };

        function convertLength(value, fromUnit, toUnit, precision) {
            value *= (FACTORS[fromUnit] || 1) / (FACTORS[toUnit] || 1);
            return _.isFinite(precision) ? Utils.round(value, precision) : value;
        }

        return convertLength;
    }());

    /**
     * Converts a length value from an absolute CSS measurement unit into 1/100
     * of millimeters.
     *
     * @param {Number} value
     *  The length value to convert, as floating-point number.
     *
     * @param {String} fromUnit
     *  The CSS measurement unit of the passed value, as string. See method
     *  Utils.convertLength() for a list of supported units.
     *
     * @returns {Number}
     *  The length value converted to 1/100 of millimeters, as integer.
     */
    Utils.convertLengthToHmm = function (value, fromUnit) {
        return Math.round(Utils.convertLength(value, fromUnit, 'mm') * 100);
    };

    /**
     * Converts a length value from 1/100 of millimeters into an absolute CSS
     * measurement unit.
     *
     * @param {Number} value
     *  The length value in 1/100 of millimeters to convert, as integer.
     *
     * @param {String} toUnit
     *  The target measurement unit. See method Utils.convertLength() for a
     *  list of supported units.
     *
     * @param {Number} [precision]
     *  If specified, the resulting length will be rounded to the nearest
     *  multiple of this value. Must be positive.
     *
     * @returns {Number}
     *  The length value converted to the target measurement unit, as
     *  floating-point number.
     */
    Utils.convertHmmToLength = function (value, toUnit, precision) {
        return Utils.convertLength(value / 100, 'mm', toUnit, precision);
    };

    /**
     * Converts a CSS length value with measurement unit into a value of
     * another absolute CSS measurement unit.
     *
     * @param {String} valueAndUnit
     *  The value with its measurement unit to be converted, as string.
     *
     * @param {String} toUnit
     *  The target CSS measurement unit. See method Utils.convertLength() for
     *  a list of supported units.
     *
     * @param {Number} [precision]
     *  If specified, the resulting length will be rounded to the nearest
     *  multiple of this value. Must be positive.
     *
     * @returns {Number}
     *  The length value converted to the target measurement unit, as
     *  floating-point number.
     */
    Utils.convertCssLength = function (valueAndUnit, toUnit, precision) {
        var value = parseFloat(valueAndUnit);
        if (!_.isFinite(value)) {
            value = 0;
        }
        if (value && (valueAndUnit.length > 2)) {
            value = Utils.convertLength(value, valueAndUnit.substr(-2), toUnit, precision);
        }
        return value;
    };

    /**
     * Converts a CSS length value with measurement unit into 1/100 of
     * millimeters.
     *
     * @param {String} valueAndUnit
     *  The value with its measurement unit to be converted, as string.
     *
     * @returns {Number}
     *  The length value converted to 1/100 of millimeters, as integer.
     */
    Utils.convertCssLengthToHmm = function (valueAndUnit) {
        return Math.round(Utils.convertCssLength(valueAndUnit, 'mm') * 100);
    };

    /**
     * Converts a length value from 1/100 of millimeters into an absolute CSS
     * measurement unit and returns the CSS length with its unit as string.
     *
     * @param {Number} value
     *  The length value in 1/100 of millimeters to convert, as integer.
     *
     * @param {String} toUnit
     *  The target measurement unit. See method Utils.convertLength() for a
     *  list of supported units.
     *
     * @param {Number} [precision]
     *  If specified, the resulting length will be rounded to the nearest
     *  multiple of this value. Must be positive.
     *
     * @returns {String}
     *  The length value converted to the target measurement unit, followed by
     *  the unit name.
     */
    Utils.convertHmmToCssLength = function (value, toUnit, precision) {
        return Utils.convertHmmToLength(value, toUnit, precision) + toUnit;
    };

    /**
     * Returns whether the passed space-separated token list contains the
     * specified token.
     *
     * @param {String} list
     *  Space-separated list of tokens.
     *
     * @param {String} token
     *  The token to look up in the token list.
     *
     * @returns {Boolean}
     *  Whether the token is contained in the token list.
     */
    Utils.containsToken = function (list, token) {
        return _.contains(list.split(/\s+/), token);
    };

    /**
     * Inserts the specified token into a space-separated token list. The token
     * will not be inserted if it is already contained in the list.
     *
     * @param {String} list
     *  Space-separated list of tokens.
     *
     * @param {String} token
     *  The token to be inserted into the token list.
     *
     * @param {String} [nothing]
     *  If specified, the name of a token that represents a special 'nothing'
     *  or 'empty' state. If this token is contained in the passed token list,
     *  it will be removed.
     *
     * @returns {String}
     *  The new token list containing the specified token.
     */
    Utils.addToken = function (list, token, nothing) {
        var tokens = (list.length > 0) ? list.split(/\s+/) : [];
        if (_.isString(nothing)) {
            tokens = _.without(tokens, nothing);
        }
        if (!_.contains(tokens, token)) {
            tokens.push(token);
        }
        return tokens.join(' ');
    };

    /**
     * Removes the specified token from a space-separated token list.
     *
     * @param {String} list
     *  Space-separated list of tokens.
     *
     * @param {String} token
     *  The token to be removed from the token list.
     *
     * @param {String} [nothing]
     *  If specified, the name of a token that represents a special 'nothing'
     *  or 'empty' state. If the resulting token list is empty, this token will
     *  be inserted instead.
     *
     * @returns {String}
     *  The new token list without the specified token.
     */
    Utils.removeToken = function (list, token, nothing) {
        var tokens = (list.length > 0) ? _.without(list.split(/\s+/), token) : [];
        if (_.isString(nothing)) {
            if (tokens.length > 0) { tokens = _.without(tokens, nothing); }
            if (tokens.length === 0) { tokens.push(nothing); }
        }
        return tokens.join(' ');
    };

    /**
     * Inserts a token into or removes a token from the specified
     * space-separated token list.
     *
     * @param {String} list
     *  Space-separated list of tokens.
     *
     * @param {String} token
     *  The token to be inserted into or removed from the token list.
     *
     * @param {Boolean} state
     *  If set to true, the token will be inserted into the token list,
     *  otherwise removed from the token list.
     *
     * @param {String} [nothing]
     *  If specified, the name of a token that represents a special 'nothing'
     *  or 'empty' state.
     *
     * @returns {String}
     *  The new token list.
     */
    Utils.toggleToken = function (list, token, state, nothing) {
        return Utils[state ? 'addToken' : 'removeToken'](list, token, nothing);
    };

    /**
     * Returns the passed text where a substring has been replaced with another
     * string.
     *
     * @param {String} text
     *  The original text.
     *
     * @param {Number} start
     *  The start position of the substring to be replaced.
     *
     * @param {Number} end
     *  The end position of the substring to be replaced.
     *
     * @param {String} newText
     *  The new text to be inserted into the original text.
     *
     * @returns {String}
     *  The resulting text with the replaced substring.
     */
    Utils.replaceSubString = function (text, start, end, newText) {
        return text.substring(0, start) + newText + text.substring(end);
    };

    /**
     * Repeats the passed string.
     *
     * @param {String} text
     *  The text to be repeated.
     *
     * @param {Number} count
     *  The number of repetitions.
     *
     * @param {String} [sep='']
     *  A separator string that will be inserted between the repetitions of the
     *  passed string.
     *
     * @returns {String}
     *  The generated string.
     */
    Utils.repeatString = function (text, count, sep) {
        var result = '';
        sep = sep || '';
        text += sep;
        while (count > 0) {
            if (count & 1) { result += text; }
            text += text;
            count >>= 1;
        }
        // check existence; String.slice(0, -0) returns an empty string!
        return sep ? result.slice(0, -sep.length) : result;
    };

    /**
     * Trims non-printable characters and white-spaces at the beginning and end
     * of the passed string.
     *
     * @param {String} text
     *  The text to be trimmed.
     *
     * @returns {String}
     *  The trimmed text.
     */
    Utils.trimString = function (text) {
        return text.replace(/^[\x00-\x1f\x80-\x9f\s]+|[\x00-\x1f\x80-\x9f\s]+$/g, '');
    };

    /**
     * Replaces all non-printable characters (Unicode characters 0x00-0x1F and
     * 0x80-0x9F) in the passed string with space characters.
     *
     * @param {String} text
     *  The text to be cleaned from non-printable characters.
     *
     * @returns {String}
     *  The new text without non-printable characters.
     */
    Utils.cleanString = function (text) {
        return text.replace(/[\x00-\x1f\x80-\x9f]/g, ' ');
    };

    /**
     * Trims non-printable characters and white-spaces at the beginning and end
     * of the passed string, and replaces all embedded non-printable characters
     * (ASCII 0x00-0x1F) with space characters.
     *
     * @param {String} text
     *  The text to be trimmed and cleaned from non-printable characters.
     *
     * @returns {String}
     *  The new trimmed text without non-printable characters.
     */
    Utils.trimAndCleanString = function (text) {
        return Utils.cleanString(Utils.trimString(text));
    };

    /**
     * Returns the passed text with a capitalized first character.
     *
     * @param {String} text
     *  The text to be converted.
     *
     * @returns {String}
     *  The passed text with a capitalized first character.
     */
    Utils.capitalize = function (text) {
        return (text.length > 0) ? (text[0].toUpperCase() + text.slice(1)) : '';
    };

    /**
     * Returns the passed text with capitalized words.
     *
     * @param {String} text
     *  The text to be converted.
     *
     * @returns {String}
     *  The passed text with capitalized words.
     */
    Utils.capitalizeWords = function (text) {
        return _.map(text.split(' '), Utils.capitalize).join(' ');
    };

    /**
     * Returns the translation of the passed word for the passed translation
     * database. Additionally, the passed word will be adjusted further for GUI
     * display. All words will be capitalized, automatic line breaks before
     * numbers will be prevented by converting the preceding space characters
     * to NBSP characters.
     *
     * @param {String} word
     *  The original (English) word.
     *
     * @param {Object} database
     *  The locale translation database. The keys of this object are used as
     *  regular expressions to find a translation of the passed word. The
     *  values are translated strings that may contain replacement items such
     *  as '$1', '$2', etc. for the matching groups in the RE keys.
     *
     * @returns {String}
     *  The translated and adjusted word. If no translation is available, the
     *  passed word will be adjusted and returned.
     */
    Utils.translateWordFromDatabase = function (word, database) {

        // the translated style name
        var translatedName = null;

        // try to find a database entry for the base word
        _.any(database, function (value, key) {

            var // the matches for the pattern of the current entry
                matches = new RegExp('^' + key + '$', 'i').exec(word);

            // continue with next database entry, if key does not match
            if (!_.isArray(matches)) { return false; }

            // matching entry found, translate it and replace the placeholders
            translatedName = value;
            for (var index = matches.length - 1; index > 0; index -= 1) {
                translatedName = translatedName.replace(new RegExp('\\$' + index, 'g'), matches[index]);
            }

            // exit the _.any() loop
            return true;
        });

        // adjust the resulting style name for GUI display
        return translatedName || Utils.capitalizeWords(word);
    };

    /**
     * Generates a random UUID that complies to RFC 4122 (section 4.4).
     *
     * @returns {String}
     *  A random UUID that complies to RFC 4122 (section 4.4).
     */
    Utils.createUUID = (function () {

        // returns a random 16-bit hexadecimal number (4 digits)
        function random16() {
            var number = Math.floor(0x10000 * Math.random());
            return ('000' + number.toString(16)).substr(-4);
        }

        return function () {
            var result = random16() + random16() + '-' + random16() + '-';
            // first digit must be '4' (RFC 4122 version)
            result += '4' + random16().substr(1, 3) + '-';
            // first digit must have cleared bit 2, and set bit 3 (i.e. 8 to B)
            result += '8' + random16().substr(1, 3) + '-';
            result += random16() + random16() + random16();
            return result;
        };
    }());

    /**
     * Returns the lower-case extension of the passed file name.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     *  @param {Boolean} ignoreGuardExt
     *  Ignore/skip the Guard extension
     *
     * @returns {String}
     *  The lower-case extension of the passed file name.
     */
    Utils.getFileExtension = function (fileName, ignoreGuardExt) {
        var index = fileName.lastIndexOf('.');
        var ext = (index >= 0) ? fileName.substring(index + 1).toLowerCase() : '';
        if ((ignoreGuardExt === true) && (ext === 'pgp')) {
            var tmp = fileName.substring(0, index);
            ext = Utils.getFileExtension(tmp, false);
        }
        return ext;
    };

    /**
     * Returns the base name of the passed file name, without file extension.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     *  @param {Boolean} ignoreGuardExt
     *  Ignore/skip the Guard extension
     *
     * @returns {String}
     *  The base name of the passed file name. Character case will be
     *  preserved.
     */
    Utils.getFileBaseName = function (fileName, ignoreGuardExt) {
        // we need to check for the guard extension in any case
        var ext = Utils.getFileExtension(fileName, false);
        var base;
        if (ext.length > 0) {
            base = fileName.substring(0, fileName.length - ext.length - 1);
            if ((ignoreGuardExt === true) && (ext === 'pgp')) {
                // keep Guard ext just to make sure we don't skip multiple pgp extenssions
                base = Utils.getFileBaseName(base, false);
            }
        } else {
            base = fileName;
        }
        return base;
    };

    /**
     * Returns whether or not the file is Guard encrypted.
     *
     * @param {String} fileName
     *  The file name (case-insensitive).
     *
     * @returns {Boolean}
     *  whether or not the file is Guard encrypted.
     */
    Utils.isGuardEncrypted = function (fileName) {
        return Utils.getFileExtension(fileName, false) === 'pgp';
    };

    /**
     * Replaces HTML mark-up characters (angle brackets, ampersand, double
     * quotes, and apostrophes) in the passed text with the respective HTML
     * entities.
     *
     * @param {String} text
     *  The text containing special HTML mark-up characters.
     *
     * @returns {String}
     *  The passed text, with all special HTML mark-up characters replaced with
     *  the respective HTML entities.
     */
    Utils.escapeHTML = function (text) {
        return Utils.cleanString(text)
            .replace(/&/g, '&amp;')      // replace the literal ampersand (must be done first!)
            .replace(/</g, '&lt;')       // replace the left angle bracket (start of element tag)
            .replace(/>/g, '&gt;')       // replace the right angle bracket (end of element tag)
            .replace(/\//g, '&#47;')     // replace the slash character (end element marker)
            .replace(/"/g, '&quot;')     // replace the double quote character
            .replace(/'/g, '&#39;')      // replace the apostrophe (&apos; is not an HTML entity!)
            .replace(/\xa0/g, '&nbsp;'); // replace the non-breakable space
    };

    /**
     * Parses the passed HTML mark-up text, and removes insecure mark-up. In
     * detail, all scripting elements, and all elements that contain event
     * handlers called automatically by the browser will be removed.
     *
     * @param {String} markup
     *  The HTML mark-up to be parsed.
     *
     * @returns {jQuery}
     *  The parsed and sanitized mark-up, as jQuery collection.
     */
    Utils.parseAndSanitizeHTML = function (markup) {

        // replace NUL characters created by Firefox
        markup = markup.replace(/\x00/gi, '');

        // sanitize HTML, see: https://github.com/cure53/DOMPurify for all available options
        var sanitized = DOMPurify.sanitize(markup, { SAFE_FOR_JQUERY: true, ADD_TAGS: ['#comment'] });

        // $() is not able to handle DOM fragments, so use $.parseHTML() instead
        var result = $($.parseHTML(sanitized));

        // remove all 'position' CSS attributes to avoid clipboard content being visible
        result.find('*').addBack().css('position', '');

        return result;
    };

    /**
     * Converts the passed JSON object to a string for debugging purposes.
     *
     * @param {Object} object
     *  The object to be converted.
     *
     * @returns {String}
     *  A string representation of the object.
     */
    Utils.stringifyForDebug = function (object) {
        return JSON.stringify(object).replace(/"(\w+)":/g, '$1:').replace(/ /g, '\xb7');
    };

    /**
     * Converts the passed zero-based index to the identifier of a scheme color
     * defined in the LESS module tk/definitions.less.
     *
     * @param {Number} index
     *  An arbitrary zero-based index to be converted to a scheme color index.
     *
     * @returns {Number}
     *  The zero-based index of a scheme color associated to the passed index.
     */
    Utils.getSchemeColor = function (index) {
        return Utils.mod(index, Utils.SCHEME_COLOR_COUNT) + 1;
    };

    // options object ---------------------------------------------------------

    /**
     * Extracts a property value from the passed object. If the property does
     * not exist, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property.
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getOption = function (options, name, def) {
        return (_.isObject(options) && (name in options)) ? options[name] : def;
    };

    /**
     * Extracts a string property from the passed object. If the property does
     * not exist, or is not a string, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the string property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not a string. May be any value (not only strings).
     *
     * @param {Boolean} [nonEmpty=false]
     *  If set to true, only non-empty strings will be returned from the
     *  options object. Empty strings will be replaced with the specified
     *  default value.
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getStringOption = function (options, name, def, nonEmpty) {
        var value = Utils.getOption(options, name);
        return (_.isString(value) && (!nonEmpty || (value.length > 0))) ? value : def;
    };

    /**
     * Extracts a boolean property from the passed object. If the property does
     * not exist, or is not a boolean value, returns the specified default
     * value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the boolean property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not a boolean value. May be any value (not only booleans).
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getBooleanOption = function (options, name, def) {
        var value = Utils.getOption(options, name);
        return _.isBoolean(value) ? value : def;
    };

    /**
     * Extracts a floating-point property from the passed object. If the
     * property does not exist, or is not a number, returns the specified
     * default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the floating-point property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not a number. May be any value (not only numbers).
     *
     * @param {Number} [min]
     *  If specified and a number, set a lower bound for the returned value. Is
     *  not used, if neither the property nor the passed default are numbers.
     *
     * @param {Number} [max]
     *  If specified and a number, set an upper bound for the returned value.
     *  Is not used, if neither the property nor the passed default are
     *  numbers.
     *
     * @param {Number} [precision]
     *  If specified, the resulting number will be rounded to the nearest
     *  multiple of this value. Must be positive.
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getNumberOption = function (options, name, def, min, max, precision) {
        var value = Utils.getOption(options, name);
        value = (_.isNumber(value) && _.isFinite(value)) ? value : def;
        if (_.isNumber(value) && _.isFinite(value)) {
            if (_.isNumber(min) && _.isFinite(min) && (value < min)) { value = min; }
            if (_.isNumber(max) && _.isFinite(max) && (value > max)) { value = max; }
            return (_.isNumber(precision) && _.isFinite(precision)) ? Utils.round(value, precision) : value;
        }
        return value;
    };

    /**
     * Extracts an integer property from the passed object. If the property
     * does not exist, or is not a number, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the integer property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not a number. May be any value (not only numbers).
     *
     * @param {Number} [min]
     *  If specified and a number, set a lower bound for the returned value. Is
     *  not used, if neither the property nor the passed default are numbers.
     *
     * @param {Number} [max]
     *  If specified and a number, set an upper bound for the returned value.
     *  Is not used, if neither the property nor the passed default are
     *  numbers.
     *
     * @returns {Any}
     *  The value of the specified property, or the default value, rounded to
     *  an integer.
     */
    Utils.getIntegerOption = function (options, name, def, min, max) {
        return Utils.getNumberOption(options, name, def, min, max, 1);
    };

    /**
     * Extracts an object property from the passed object. If the property does
     * not exist, or is not an object, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not an object. May be any value (not only objects).
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getObjectOption = function (options, name, def) {
        var value = Utils.getOption(options, name);
        return (_.isObject(value) && !_.isFunction(value) && !_.isArray(value) && !_.isRegExp(value)) ? value : def;
    };

    /**
     * Extracts a function from the passed object. If the property does not
     * exist, or is not a function, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not an object. May be any value (not only functions).
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getFunctionOption = function (options, name, def) {
        var value = Utils.getOption(options, name);
        return _.isFunction(value) ? value : def;
    };

    /**
     * Extracts a array from the passed object. If the property does not exist,
     * or is not an array, returns the specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not an array. May be any value.
     *
     * @param {Boolean} [nonEmpty=false]
     *  If set to true, only non-empty arrays will be returned from the options
     *  object. Empty arrays will be replaced with the specified default value.
     *
     * @returns {Any}
     *  The value of the specified property, or the default value.
     */
    Utils.getArrayOption = function (options, name, def, nonEmpty) {
        var value = Utils.getOption(options, name);
        return (_.isArray(value) && (!nonEmpty || (value.length > 0))) ? value : def;
    };

    /**
     * Extracts a string property from the passed object consisting of
     * space-separated string tokens, and converts it to an array of string
     * tokens. If the property does not exist, or is not a string, returns the
     * specified default value.
     *
     * @param {Object|Undefined} options
     *  An object containing some properties. May be undefined.
     *
     * @param {String} name
     *  The name of the string list property to be returned.
     *
     * @param {Any} [def]
     *  The default value returned when the options parameter is not an object,
     *  or if it does not contain the specified property, or if the property is
     *  not a string. May be any value (not only strings).
     *
     * @param {Boolean} [nonEmpty=false]
     *  If set to true, only non-empty token lists will be returned from the
     *  options object. Empty token lists will be replaced with the specified
     *  default value.
     *
     * @returns {Any}
     *  The value of the specified property, as token list (array of strings),
     *  or the default value.
     */
    Utils.getTokenListOption = function (options, name, def, nonEmpty) {
        var value = Utils.getStringOption(options, name);
        if (_.isString(value)) { value = _.without(value.split(/\s+/), ''); }
        return (_.isArray(value) && (!nonEmpty || (value.length > 0))) ? value : def;
    };

    /**
     * Creates and returns a merged options map from the passed objects. Unlike
     * Underscore's extend() method, does not modify the passed objects, but
     * creates and returns a clone. Additionally, extends embedded plain JS
     * objects deeply instead of replacing them, for example, extending the
     * objects {a:{b:1}} and {a:{c:2}} will result in {a:{b:1,c:2}}.
     *
     * @param {Object} [...]
     *  One or more objects whose properties will be inserted into the
     *  resulting object.
     *
     * @returns {Object}
     *  A new object containing all properties of the passed objects.
     */
    Utils.extendOptions = function () {

        var // the resulting options
            result = {};

        function isPlainObject(value) {
            return _.isObject(value) && (value.constructor === Object);
        }

        function extend(options, extensions) {
            _.each(extensions, function (value, name) {
                if (isPlainObject(value)) {
                    // extension value is a plain object: ensure that the options map contains an embedded object
                    if (!isPlainObject(options[name])) {
                        options[name] = {};
                    }
                    extend(options[name], value);
                } else {
                    // extension value is not a plain object: clear old value, even if it was an object
                    options[name] = value;
                }
            });
        }

        // add all objects to the clone
        for (var index = 0; index < arguments.length; index += 1) {
            if (_.isObject(arguments[index])) {
                extend(result, arguments[index]);
            }
        }

        return result;
    };

    // generic DOM/CSS helpers ------------------------------------------------

    /**
     * A jQuery function selector that returns true if the DOM node bound to
     * the 'this' symbol is a text node. Can be used in all helper functions
     * that expect a jQuery selector including functions.
     */
    Utils.JQ_TEXTNODE_SELECTOR = function () { return this.nodeType === 3; };

    /**
     * Converts the passed object to a DOM node object.
     *
     * @param {Node|jQuery} node
     *  If the object is a DOM node object, returns it unmodified. If the
     *  object is a jQuery collection, returns its first node.
     *
     * @returns {Node}
     *  The DOM node object.
     */
    Utils.getDomNode = function (node) {
        return (node instanceof $) ? node.get(0) : node;
    };

    /**
     * Returns whether the passed node is a specific DOM element node.
     *
     * @param {Node|jQuery|Null|Undefined} node
     *  The DOM node to be checked. May be null or undefined.
     *
     * @param {String|Function|Node|jQuery} [selector]
     *  A jQuery selector that can be used to check the passed node for a
     *  specific type etc. The selector will be passed to the jQuery method
     *  jQuery.is(). If this selector is a function, it will be called with the
     *  DOM node bound to the symbol 'this'. See the jQuery API documentation
     *  at http://api.jquery.com/is for details.
     *
     * @returns {Boolean}
     *  Whether the passed node is an element node that matches the passed
     *  selector.
     */
    Utils.isElementNode = function (node, selector) {
        if (!node) { return false; }
        return (Utils.getDomNode(node).nodeType === 1) && (!selector || $(node).is(selector));
    };

    /**
     * Returns the lower-case name of a DOM node object.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose name will be returned. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @returns {String}
     *  The lower-case name of the DOM node object. If the node is a text node,
     *  the string '#text' will be returned.
     */
    Utils.getNodeName = function (node) {
        return Utils.getDomNode(node).nodeName.toLowerCase();
    };

    /**
     * Returns whether the passed outer node contains the passed inner node.
     *
     * @param {Node|jQuery} outerNode
     *  The outer DOM node. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @param {Node|jQuery} innerNode
     *  The inner DOM node. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.allowEqual=false]
     *      If set to true, and the passed outer node is the same as the passed
     *      inner node, this method will return true instead of false.
     *
     * @returns {Boolean}
     *  Whether the inner node is a descendant of the outer node.
     */
    Utils.containsNode = function (outerNode, innerNode, options) {

        // convert nodes to DOM node object
        outerNode = Utils.getDomNode(outerNode);
        innerNode = Utils.getDomNode(innerNode);

        // check equality according to passed option
        if ((outerNode === innerNode) && Utils.getBooleanOption(options, 'allowEqual', false)) {
            return true;
        }

        // IE does not support contains() method at document
        if (outerNode === document) {
            outerNode = document.body;
            if (innerNode === outerNode) {
                return true;
            }
        }

        // outer node must be an element; be sure that a node does not contain itself
        if ((outerNode.nodeType !== 1) || (outerNode === innerNode)) {
            return false;
        }

        // IE does not accept a DOM text node as parameter for the Node.contains() method
        if (innerNode.nodeType !== 1) {
            innerNode = innerNode.parentNode;
            // check if the parent of the inner text node is the outer node
            if (outerNode === innerNode) {
                return true;
            }
        }

        // Internet Explorer has no 'contains' at svg elements (26172)
        if (!_.isFunction(outerNode.contains)) {
            return $(innerNode).closest(outerNode).length > 0;
        }

        // use the native Node.contains() method
        return outerNode.contains(innerNode);
    };

    /**
     * Returns an integer attribute value of the passed element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose attribute will be returned. If this object is a
     *  jQuery collection, uses the first node it contains.
     *
     * @param {String} name
     *  The name of the element attribute.
     *
     * @param {Number} def
     *  A default value in case the element does not contain the specified
     *  attribute.
     *
     * @returns {Number}
     *  The attribute value parsed as integer, or the default value.
     */
    Utils.getElementAttributeAsInteger = function (node, name, def) {
        var attr = $(node).attr(name);
        return _.isString(attr) ? parseInt(attr, 10) : def;
    };

    /**
     * Returns the current value of the specified CSS length attribute, in
     * pixels.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose length attribute will be returned. If this object
     *  is a jQuery collection, uses the first node it contains.
     *
     * @param {String} name
     *  The name of the CSS attribute to be returned.
     *
     * @returns {Number}
     *  The length value, converted to pixels, rounded to integer.
     */
    Utils.getElementCssLength = function (node, name) {
        return Utils.convertCssLength($(node).css(name), 'px', 1);
    };

    /**
     * Returns whether the passed CSS border position is oriented vertically
     * (either 'top' or 'bottom').
     *
     * @param {String} position
     *  The CSS position, one of 'top', 'bottom', 'left', or 'right'.
     *
     * @returns {Boolean}
     *  Whether the passed position is either 'top' or 'bottom'.
     */
    Utils.isVerticalPosition = function (position) {
        return (position === 'top') || (position === 'bottom');
    };

    /**
     * Returns whether the passed CSS border position is the leading side
     * (either 'top' or 'left').
     *
     * @param {String} position
     *  The CSS position, one of 'top', 'bottom', 'left', or 'right'.
     *
     * @returns {Boolean}
     *  Whether the passed position is either 'top' or 'left'.
     */
    Utils.isLeadingPosition = function (position) {
        return (position === 'top') || (position === 'left');
    };

    /**
     * Sets a CSS formatting attribute with all browser-specific prefixes at
     * the passed element.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose CSS attribute will be changed. If this object is
     *  a jQuery collection, changes all contained nodes.
     *
     * @param {String} name
     *  The base name of the CSS attribute.
     *
     * @param {Any} value
     *  The new value of the CSS attribute.
     */
    Utils.setCssAttributeWithPrefixes = (function () {

        var // the prefix for the current browser
            prefix = _.browser.WebKit ? '-webkit-' : _.browser.Firefox ? '-moz-' : _.browser.IE ? '-ms-' : '';

        return function (node, name, value) {
            var props = Utils.makeSimpleObject(name, value);
            if (prefix) { props[prefix + name] = value; }
            $(node).css(props);
        };
    }());

    /**
     * Returns an integer indicating how the two passed nodes are located to
     * each other.
     *
     * @param {Node|jQuery} node1
     *  The first DOM node tested if it is located before the second node. If
     *  this object is a jQuery collection, uses the first node it contains.
     *
     * @param {Node|jQuery} node2
     *  The second DOM node. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @returns {Number}
     *  The value zero, if the nodes are equal, a negative number, if node1
     *  precedes or contains node2, or a positive number, if node2 precedes or
     *  contains node1.
     */
    Utils.compareNodes = function (node1, node2) {
        node1 = Utils.getDomNode(node1);
        node2 = Utils.getDomNode(node2);
        return (node1 === node2) ? 0 : ((node1.compareDocumentPosition(node2) & 4) === 4) ? -1 : 1;
    };

    /**
     * Iterates over all descendant DOM nodes of the specified element.
     *
     * @param {HTMLElement|jQuery} element
     *  A DOM element object whose descendant nodes will be iterated. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {Function} iterator
     *  The iterator function that will be called for every node. Receives the
     *  DOM node as first parameter. If the iterator returns the Utils.BREAK
     *  object, the iteration process will be stopped immediately. The iterator
     *  can remove the visited node descendants from the DOM. It is also
     *  allowed to remove any siblings of the visited node, as long as the
     *  visited node will not be removed itself.
     *
     * @param {Object} [context]
     *  If specified, the iterator will be called with this context (the symbol
     *  'this' will be bound to the context inside the iterator function).
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.children=false]
     *      If set to true, only direct child nodes will be visited.
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the descendant nodes are visited in reversed order.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    Utils.iterateDescendantNodes = function (element, iterator, context, options) {

        var // only child nodes
            childrenOnly = Utils.getBooleanOption(options, 'children', false),
            // iteration direction
            reverse = Utils.getBooleanOption(options, 'reverse', false),
            // visited node, and the next or previous sibling of the visited node
            childNode = null, nextSibling = null;

        // visit all child nodes
        element = Utils.getDomNode(element);
        for (childNode = reverse ? element.lastChild : element.firstChild; childNode; childNode = nextSibling) {

            // get next/previous sibling in case the iterator removes the node
            nextSibling = reverse ? childNode.previousSibling : childNode.nextSibling;

            // call iterator for child node; if it returns Utils.BREAK, exit loop and return too
            if (iterator.call(context, childNode) === Utils.BREAK) { return Utils.BREAK; }

            // iterate grand child nodes (only if the iterator did not remove the node from the DOM)
            if (Utils.containsNode(element, childNode)) {

                // refresh next sibling (iterator may have removed the old one, or may have inserted another one)
                nextSibling = reverse ? childNode.previousSibling : childNode.nextSibling;

                // if iterator for any descendant node returns Utils.BREAK, return too
                if (!childrenOnly && (childNode.nodeType === 1) && (Utils.iterateDescendantNodes(childNode, iterator, context, options) === Utils.BREAK)) {
                    return Utils.BREAK;
                }
            }
        }
    };

    /**
     * Iterates over selected descendant DOM nodes of the specified element.
     *
     * @param {HTMLElement|jQuery} element
     *  A DOM element object whose descendant nodes will be iterated. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} selector
     *  A jQuery selector that will be used to decide whether to call the
     *  iterator function for the current DOM node. The selector will be passed
     *  to the jQuery method jQuery.is() for each node. If this selector is a
     *  function, it will be called with the current DOM node bound to the
     *  symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details.
     *
     * @param {Function} iterator
     *  The iterator function that will be called for every matching node.
     *  Receives the DOM node as first parameter. If the iterator returns the
     *  Utils.BREAK object, the iteration process will be stopped immediately.
     *  The iterator can remove visited nodes from the DOM.
     *
     * @param {Object} [context]
     *  If specified, the iterator will be called with this context (the symbol
     *  'this' will be bound to the context inside the iterator function).
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  method Utils.iterateDescendantNodes().
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    Utils.iterateSelectedDescendantNodes = function (element, selector, iterator, context, options) {

        return Utils.iterateDescendantNodes(element, function (node) {
            if ($(node).is(selector) && (iterator.call(context, node) === Utils.BREAK)) {
                return Utils.BREAK;
            }
        }, context, options);
    };

    /**
     * Returns the first descendant DOM node in the specified element that
     * passes a truth test using the specified jQuery selector.
     *
     * @param {HTMLElement|jQuery} element
     *  A DOM element object whose descendant nodes will be searched. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} selector
     *  A jQuery selector that will be used to find the DOM node. The selector
     *  will be passed to the jQuery method jQuery.is() for each node. If this
     *  selector is a function, it will be called with the current DOM node
     *  bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  method Utils.iterateDescendantNodes().
     *
     * @returns {Node|Null}
     *  The first descendant DOM node that matches the passed selector. If no
     *  matching node has been found, returns null.
     */
    Utils.findDescendantNode = function (element, selector, options) {

        var // the node to be returned
            resultNode = null;

        // find the first node passing the iterator test
        Utils.iterateSelectedDescendantNodes(element, selector, function (node) {
            resultNode = node;
            return Utils.BREAK;
        }, undefined, options);

        return resultNode;
    };

    /**
     * Returns a child node of the passed node, that is at a specific index in
     * the array of all matching child nodes.
     *
     * @param {HTMLElement|jQuery} element
     *  A DOM element object whose child nodes will be visited. If this object
     *  is a jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} selector
     *  A jQuery selector that will be used to decide which child nodes are
     *  matching while searching to the specified index. The selector will be
     *  passed to the jQuery method jQuery.is() for each node. If this selector
     *  is a function, it will be called with the current DOM node bound to the
     *  symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details.
     *
     * @param {Number} index
     *  The zero-based index of the child node in the set of child nodes
     *  matching the selector that will be returned.
     *
     * @returns {Node|Null}
     *  The 'index'-th child node that matches the selector; or null, if the
     *  index is outside the valid range.
     */
    Utils.getSelectedChildNodeByIndex = function (element, selector, index) {

        var // the node to be returned
            resultNode = null;

        // find the 'index'-th matching child node
        Utils.iterateSelectedDescendantNodes(element, selector, function (node) {
            // node found: store and escape from loop
            if (index === 0) {
                resultNode = node;
                return Utils.BREAK;
            }
            index -= 1;
        }, undefined, { children: true });

        return resultNode;
    };

    /**
     * Finds the closest ancestor of the passed node that matches the specified
     * jQuery selector. If the specified node itself matches the selector, it
     * will be returned. In difference to the jQuery method jQuery.closest(),
     * the passed node selector can be a function.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The DOM root node that will not be left while searching for a matching
     *  node. If this object is a jQuery collection, uses the first node it
     *  contains.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose chain of ancestors will be searched for a matching
     *  node. Must be a descendant of the passed root node. If this object is a
     *  jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} selector
     *  A jQuery selector that will be used to find a matching DOM node. The
     *  selector will be passed to the jQuery method jQuery.is() for each node.
     *  If this selector is a function, it will be called with the current DOM
     *  node bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details.
     *
     * @returns {Node|Null}
     *  The passed node, or the first ancestor node that matches the passed
     *  selector, and is contained in or equal to the root node; or null, no
     *  node has been found.
     */
    Utils.findClosest = function (rootNode, node, selector) {

        rootNode = Utils.getDomNode(rootNode);
        node = Utils.getDomNode(node);

        do {
            if ($(node).is(selector)) { return node; }
            if (node === rootNode) { return null; }
            node = node.parentNode;
        } while (node);

        return null;
    };

    /**
     * Finds the farthest ancestor of the passed node that matches the
     * specified jQuery selector. If the specified node itself matches the
     * selector but none of its ancestors, the node will be returned itself. In
     * difference to most jQuery methods, the passed node selector can be a
     * function.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The DOM root node that will not be left while searching for matching
     *  nodes. If this object is a jQuery collection, uses the first node it
     *  contains.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose chain of ancestors will be searched for a matching
     *  node. Must be a descendant of the passed root node. If this object is a
     *  jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} selector
     *  A jQuery selector that will be used to find a matching DOM node. The
     *  selector will be passed to the jQuery method jQuery.is() for each node.
     *  If this selector is a function, it will be called with the current DOM
     *  node bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details.
     *
     * @returns {Node|Null}
     *  The passed node, or the farthest ancestor node that matches the passed
     *  selector, and is contained in or equal to the root node; or null, no
     *  node has been found.
     */
    Utils.findFarthest = function (rootNode, node, selector) {

        var // the last found matching node
            matchingNode = null;

        rootNode = Utils.getDomNode(rootNode);
        node = Utils.getDomNode(node);

        do {
            if ($(node).is(selector)) { matchingNode = node; }
            if (node === rootNode) { return matchingNode; }
            node = node.parentNode;
        } while (node);

        return null;
    };

    /**
     * Finds the closest previous sibling node of the passed node that matches
     * a jQuery selector.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose previous matching sibling will be returned. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} [selector]
     *  A jQuery selector that will be used to find a matching sibling. The
     *  selector will be passed to the jQuery method jQuery.is() for each node.
     *  If this selector is a function, it will be called with the current DOM
     *  node bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details. If omitted, this method returns
     *  the direct previous sibling of the passed node.
     *
     * @returns {Node|Null}
     *  The previous matching sibling of the passed node; or null, no previous
     *  sibling node has been found.
     */
    Utils.findPreviousSiblingNode = function (node, selector) {
        node = Utils.getDomNode(node).previousSibling;
        if (!_.isUndefined(selector)) {
            while (node && !$(node).is(selector)) {
                node = node.previousSibling;
            }
        }
        return node;
    };

    /**
     * Finds the closest next sibling node of the passed node that matches a
     * jQuery selector.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose next matching sibling will be returned. If this
     *  object is a jQuery collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} [selector]
     *  A jQuery selector that will be used to find a matching sibling. The
     *  selector will be passed to the jQuery method jQuery.is() for each node.
     *  If this selector is a function, it will be called with the current DOM
     *  node bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details. If omitted, this method returns
     *  the direct next sibling of the passed node.
     *
     * @returns {Node|Null}
     *  The next matching sibling of the passed node; or null, no next sibling
     *  node has been found.
     */
    Utils.findNextSiblingNode = function (node, selector) {
        node = Utils.getDomNode(node).nextSibling;
        if (!_.isUndefined(selector)) {
            while (node && !$(node).is(selector)) {
                node = node.nextSibling;
            }
        }
        return node;
    };

    /**
     * Finds a previous sibling node of the passed node in the DOM tree. If the
     * node is the first child of its parent, bubbles up the chain of parent
     * nodes and continues with the previous sibling of the first ancestor that
     * has more siblings. An optional jQuery selector can be specified to
     * search through the previous siblings until a matching node has been
     * found.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The DOM root node whose sub node tree will not be left when searching
     *  for the previous sibling node. If this object is a jQuery collection,
     *  uses the first node it contains.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose previous sibling will be returned. Must be a
     *  descendant of the passed root node. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @param {String|Function|Node|jQuery} [selector]
     *  A jQuery selector that will be used to find a DOM node. The selector
     *  will be passed to the jQuery method jQuery.is() for each node. If this
     *  selector is a function, it will be called with the current DOM node
     *  bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details. If omitted, this method returns
     *  the direct next sibling of the passed node.
     *
     * @param {String|Function|Node|jQuery} [skipSiblingsSelector]
     *  A jQuery selector that will be used to to identify DOM nodes whose
     *  siblings will be ignored completely, even if they match the specified
     *  selector. If omitted, this method visits all descendant nodes of all
     *  siblings while searching for a matching node.
     *
     * @returns {Node|Null}
     *  The previous sibling of the passed node in the DOM sub tree of the root
     *  node; or null, no node has been found.
     */
    Utils.findPreviousNode = function (rootNode, node, selector, skipSiblingsSelector) {

        // set 'node' to the previous sibling (of itself or one of its parents)
        function previousNode() {

            // go to previous sibling if existing
            if (node && node.previousSibling) {
                node = node.previousSibling;

                // start with deepest last descendant node in the previous sibling
                while (node && (_.isUndefined(skipSiblingsSelector) || !$(node).is(skipSiblingsSelector)) && node.lastChild) {
                    node = node.lastChild;
                }
            } else {
                // otherwise, go to parent, but do not leave the root node
                node = (node.parentNode === rootNode) ? null : node.parentNode;
            }
        }

        rootNode = Utils.getDomNode(rootNode);
        node = Utils.getDomNode(node);

        previousNode();
        if (!_.isUndefined(selector)) {
            while (node && !$(node).is(selector)) { previousNode(); }
        }

        return node;
    };

    /**
     * Finds a next sibling node of the passed node in the DOM tree. If the
     * node is the last child of its parent, bubbles up the chain of parent
     * nodes and continues with the next sibling of the first ancestor that has
     * more siblings. An optional jQuery selector can be specified to search
     * through the next siblings until a matching node has been found.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The DOM root node whose sub node tree will not be left when searching
     *  for the next sibling node. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @param {Node|jQuery} node
     *  The DOM node whose next sibling will be returned. Must be a descendant
     *  of the passed root node. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @param {String|Function|Node|jQuery} [selector]
     *  A jQuery selector that will be used to find a DOM node. The selector
     *  will be passed to the jQuery method jQuery.is() for each node. If this
     *  selector is a function, it will be called with the current DOM node
     *  bound to the symbol 'this'. See the jQuery API documentation at
     *  http://api.jquery.com/is for details. If omitted, this method returns
     *  the direct next sibling of the passed node.
     *
     * @param {String|Function|Node|jQuery} [skipSiblingsSelector]
     *  A jQuery selector that will be used to to identify DOM nodes whose
     *  siblings will be ignored completely, even if they match the specified
     *  selector. If omitted, this method visits all descendant nodes of all
     *  siblings while searching for a matching node.
     *
     * @returns {Node|Null}
     *  The next sibling of the passed node in the DOM sub tree of the root
     *  node; or null, no node has been found.
     */
    Utils.findNextNode = function (rootNode, node, selector, skipSiblingsSelector) {

        var // visit descendant nodes
            visitDescendants = false;

        // set 'node' to the next sibling (of itself or one of its parents)
        function nextNode() {

            // visit descendant nodes if specified
            if (visitDescendants && node && (_.isUndefined(skipSiblingsSelector) || !$(node).is(skipSiblingsSelector)) && node.firstChild) {
                node = node.firstChild;
                return;
            }

            // for next call: always visit descendant nodes of own or parent siblings
            visitDescendants = true;

            // find first node up the tree that has a next sibling
            while (node && !node.nextSibling) {
                // do not leave the root node
                node = (node.parentNode === rootNode) ? null : node.parentNode;
            }

            // go to that next sibling
            node = node && node.nextSibling;
        }

        rootNode = Utils.getDomNode(rootNode);
        node = Utils.getDomNode(node);

        nextNode();
        if (!_.isUndefined(selector)) {
            while (node && !$(node).is(selector)) { nextNode(); }
        }

        return node;
    };

    /**
     * Checking if a specified node is the last descendant of a specified root
     * node. The last descendant means, that the current node and all its
     * parents until the root node do not have a next sibling.
     *
     * @param {HTMLElement|jQuery} rootNode
     *  The DOM root node whose sub node tree will not be left when searching
     *  for the next sibling node. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @param {Node|jQuery} node
     *  The DOM node, that is checked, if it is the last descendant of the passed
     *  root node. Must be a descendant this root node. If this object is a
     *  jQuery collection, uses the first node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed node is the last descendant of the passed root node.
     */
    Utils.isLastDescendant = function (rootNode, node) {
        return Utils.findNextNode(rootNode, node) === null;
    };

    // positioning and scrolling ----------------------------------------------

    /**
     * Returns whether the passed rectangle contains a specific pixel.
     *
     * @param {Object} rectangle
     *  The position of the rectangle, in the numeric properties 'left', 'top',
     *  'width', and 'height'.
     *
     * @param {Object} offset
     *  The pixel position to be checked, in the numeric properties 'left', and
     *  'top'. The pixel is assumed to be of size 1x1.
     *
     * @returns {Boolean}
     *  Whether the passed pixel position is inside the rectangle.
     */
    Utils.rectangleContainsPixel = function (rectangle, offset) {
        return (rectangle.left <= offset.left) && (offset.left + 1 <= rectangle.left + rectangle.width) &&
            (rectangle.top <= offset.top) && (offset.top + 1 <= rectangle.top + rectangle.height);
    };

    /**
     * Returns whether the passed rectangles overlap each other.
     *
     * @param {Object} rectangle1
     *  The position of the first rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @param {Object} rectangle2
     *  The position of the second rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @returns {Boolean}
     *  Whether the passed rectangles overlap each other.
     */
    Utils.rectanglesOverlap = function (rectangle1, rectangle2) {
        return (rectangle1.left < rectangle2.left + rectangle2.width) &&
            (rectangle2.left < rectangle1.left + rectangle1.width) &&
            (rectangle1.top < rectangle2.top + rectangle2.height) &&
            (rectangle2.top < rectangle1.top + rectangle1.height);
    };

    /**
     * Returns the intersecting rectangle of the passed rectangles.
     *
     * @param {Object} rectangle1
     *  The position of the first rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @param {Object} rectangle2
     *  The position of the second rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @returns {Object|Null}
     *  The position of the rectangle covered by both passed rectangles, if
     *  existing; otherwise null. If one of the rectangles has zero width or
     *  height and is covered by the other rectangle, the resulting rectangle
     *  will have zero width or height too. Otherwise, the rectangles must
     *  overlap in order to return an existing intersection rectangle.
     */
    Utils.getIntersectionRectangle = function (rectangle1, rectangle2) {

        var // the left position of the intersection rectangle
            left = Math.max(rectangle1.left, rectangle2.left),
            // the top position of the intersection rectangle
            top = Math.max(rectangle1.top, rectangle2.top),
            // the right position of the intersection rectangle
            right = Math.min(rectangle1.left + rectangle1.width, rectangle2.left + rectangle2.width),
            // the bottom position of the intersection rectangle
            bottom = Math.min(rectangle1.top + rectangle1.height, rectangle2.top + rectangle2.height);

        if (((left < right) || ((left === right) && ((rectangle1.width === 0) || (rectangle2.width === 0)))) &&
            ((top < bottom) || ((top === bottom) && ((rectangle1.height === 0) || (rectangle2.height === 0))))
        ) {
            return { left: left, top: top, width: right - left, height: bottom - top };
        }

        return null;
    };

    /**
     * Returns the bounding rectangle of the passed rectangles (the smallest
     * rectangle that contains all passed rectangles).
     *
     * @param {Object|Array} [...]
     *  A single rectangle, or an array with rectangles. The number of
     *  parameters that can be passed to this method is not limited.
     *
     * @returns {Object|Null}
     *  The bounding rectangle containing all passed rectangles; or null, if no
     *  rectangle has been passed.
     */
    Utils.getBoundingRectangle = function () {

        var // the resulting bounding rectangle
            boundRectangle = null;

        _.each(arguments, function (rectangles) {
            _.chain(rectangles).getArray().each(function (rectangle) {
                var right = 0, bottom = 0;
                if (!boundRectangle) {
                    boundRectangle = _.copy(rectangle, true);
                } else {
                    right = Math.max(boundRectangle.left + boundRectangle.width, rectangle.left + rectangle.width);
                    bottom = Math.max(boundRectangle.top + boundRectangle.height, rectangle.top + rectangle.height);
                    boundRectangle.left = Math.min(boundRectangle.left, rectangle.left);
                    boundRectangle.top = Math.min(boundRectangle.top, rectangle.top);
                    boundRectangle.width = right - boundRectangle.left;
                    boundRectangle.height = bottom - boundRectangle.top;
                }
            });
        });

        return boundRectangle;
    };

    /**
     * Returns the difference of the passed rectangles (all rectangles
     * contained in the first rectangle, that are not contained in the second
     * rectangle).
     *
     * @param {Object} rectangle1
     *  The position of the first rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @param {Object} rectangle2
     *  The position of the second rectangle, in the numeric properties 'left',
     *  'top', 'width', and 'height'.
     *
     * @returns {Array}
     *  The positions of all rectangles that are contained in rectangle1 but
     *  not in rectangle2. May be an empty array, if rectangle2 completely
     *  covers rectangle1.
     */
    Utils.getRemainingRectangles = function (rectangle1, rectangle2) {

        var // the resulting rectangles
            rectangles = [];

        // check if the rectangles cover each other at all
        if (!Utils.rectanglesOverlap(rectangle1, rectangle2)) {
            rectangles.push(rectangle1);
            return rectangles;
        }

        var // the coordinates of rectangle1
            l1 = rectangle1.left, t1 = rectangle1.top,
            w1 = rectangle1.width, h1 = rectangle1.height,
            r1 = l1 + w1, b1 = t1 + h1,
            // the coordinates of rectangle2
            l2 = rectangle2.left, t2 = rectangle2.top,
            w2 = rectangle2.width, h2 = rectangle2.height,
            r2 = l2 + w2, b2 = t2 + h2;

        // if rectangle1 starts above rectangle2, extract the upper part of rectangle1
        if (t1 < t2) { rectangles.push({ left: l1, top: t1, width: w1, height: t2 - t1 }); }
        // if rectangle1 starts left of rectangle2, extract the left part of rectangle1
        if (l1 < l2) { rectangles.push({ left: l1, top: t2, width: l2 - l1, height: h2 }); }
        // if rectangle1 ends right of rectangle2, extract the right part of rectangle1
        if (r1 > r2) { rectangles.push({ left: r2, top: t2, width: r1 - r2, height: h2 }); }
        // if rectangle1 ends below rectangle2, extract the lower part of rectangle1
        if (b1 > b2) { rectangles.push({ left: l1, top: b2, width: w1, height: b1 - b2 }); }

        return rectangles;
    };

    // calculate size of system scroll bars
    (function () {

        var // dummy container used to calculate the scroll bar sizes
            node = $('<div style="width:100px;height:100px;overflow:scroll;">');

        $('body').append(node);
        Utils.SCROLLBAR_WIDTH = node.width() - node[0].clientWidth;
        Utils.SCROLLBAR_HEIGHT = node.height() - node[0].clientHeight;
        node.remove();
    }());

    /**
     * Returns whether the passed DOM element contains a visible vertical
     * scroll bar.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the DOM element contains a visible vertical scroll bar.
     */
    Utils.hasVerticalScrollBar = function (node) {
        return $(node).width() > Utils.getDomNode(node).clientWidth;
    };

    /**
     * Returns whether the passed DOM element contains a visible horizontal
     * scroll bar.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element. If this object is a jQuery collection, uses the first
     *  node it contains.
     *
     * @returns {Boolean}
     *  Whether the DOM element contains a visible horizontal scroll bar.
     */
    Utils.hasHorizontalScrollBar = function (node) {
        return $(node).height() > Utils.getDomNode(node).clientHeight;
    };

    /**
     * Returns the size of the entire document page.
     *
     * @returns {Object}
     *  An object with numeric properties representing the size of the document
     *  page, in pixels:
     *  - {Number} width
     *      The width of the document page.
     *  - {Number} height
     *      The height of the document page.
     */
    Utils.getPageSize = function () {

        var // the OX root node (<body> element does NOT cover the entire browser window)
            coreNode = $('#io-ox-core')[0];

        return { width: coreNode.offsetWidth, height: coreNode.offsetHeight };
    };

    /**
     * Returns the exact size of the passed node, as floating-point numbers.
     * Especially elements containing text may have a fractional width or
     * height, but browsers usually report node sizes rounded (down) to
     * integers when using properties like 'offsetWidth', 'scrollWidth', etc.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose size will be returned. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the size of the node, in
     *  floating-point pixels:
     *  - {Number} width
     *      The outer width of the node (including its borders).
     *  - {Number} height
     *      The outer height of the node (including its borders).
     */
    Utils.getExactNodeSize = function (node) {
        var boundingRect = Utils.getDomNode(node).getBoundingClientRect();
        return { width: boundingRect.width, height: boundingRect.height };
    };

    /**
     * Returns the size of the passed node, rounded up to the next integers.
     * See method Utils.getExactNodeSize() for more details about the exact
     * fractional size of a node.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose size will be returned. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the size of the node, in
     *  pixels:
     *  - {Number} width
     *      The outer width of the node (including its borders).
     *  - {Number} height
     *      The outer height of the node (including its borders).
     */
    Utils.getCeilNodeSize = function (node) {
        var size = Utils.getExactNodeSize(node);
        size.width = Math.ceil(size.width);
        size.height = Math.ceil(size.height);
        return size;
    };

    /**
     * Returns the size of the passed node, rounded down to the next integers.
     * See method Utils.getExactNodeSize() for more details about the exact
     * fractional size of a node.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose size will be returned. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the size of the node, in
     *  pixels:
     *  - {Number} width
     *      The outer width of the node (including its borders).
     *  - {Number} height
     *      The outer height of the node (including its borders).
     */
    Utils.getFloorNodeSize = function (node) {
        var size = Utils.getExactNodeSize(node);
        size.width = Math.floor(size.width);
        size.height = Math.floor(size.height);
        return size;
    };

    /**
     * Returns the position and size of the specified node inside the entire
     * document page. This includes the distances of all four borders of the
     * node to the borders of the document.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose position relative to the document page will be
     *  calculated. If this object is a jQuery collection, uses the first node
     *  it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the position and size of
     *  the node relative to the document page, in pixels:
     *  - {Number} left
     *      The distance of the left border of the node to the left border of
     *      the document page.
     *  - {Number} top
     *      The distance of the top border of the node to the top border of the
     *      document page.
     *  - {Number} right
     *      The distance of the right border of the node to the right border of
     *      the document page.
     *  - {Number} bottom
     *      The distance of the bottom border of the node to the bottom border
     *      of the document page.
     *  - {Number} width
     *      The outer width of the node (including its borders).
     *  - {Number} height
     *      The outer height of the node (including its borders).
     */
    Utils.getNodePositionInPage = function (node) {

        var // the passed node, as jQuery object
            $node = $(node),
            // the offset of the node, relative to the browser window
            position = $node.offset(),
            // the size of the entire document page
            pageSize = Utils.getPageSize();

        position.width = $node.outerWidth();
        position.height = $node.outerHeight();
        position.right = pageSize.width - position.left - position.width;
        position.bottom = pageSize.height - position.top - position.height;

        return position;
    };

    /**
     * Returns the position and size of the client area of the specified node
     * inside the entire document page. This includes the distances of all four
     * borders of the client area to the borders of the document.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM element whose client area position relative to the document
     *  page will be calculated. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the position and size of
     *  the client area of the node relative to the document page, in pixels:
     *  - {Number} left
     *      The distance of the left border of the node's client area to the
     *      left border of the document page.
     *  - {Number} top
     *      The distance of the top border of the node's client area to the top
     *      border of the document page.
     *  - {Number} right
     *      The distance of the right border of the node's client area to the
     *      right border of the document page.
     *  - {Number} bottom
     *      The distance of the bottom border of the node's client area to the
     *      bottom border of the document page.
     *  - {Number} width
     *      The width of the node's client area.
     *  - {Number} height
     *      The height of the node's client area.
     */
    Utils.getClientPositionInPage = function (node) {

        var // the passed node, as jQuery object
            $node = $(node),
            // the offset of the node, relative to the browser window
            position = $node.offset(),
            // the size of the entire document page
            pageSize = Utils.getPageSize();

        position.left += Utils.getElementCssLength($node, 'border-left-width');
        position.top += Utils.getElementCssLength($node, 'border-top-width');
        position.width = $node[0].clientWidth;
        position.height = $node[0].clientHeight;
        position.right = pageSize.width - position.left - position.width;
        position.bottom = pageSize.height - position.top - position.height;

        return position;
    };

    /**
     * Returns the position of the visible area of the passed container node.
     * This includes the size of the visible area without scroll bars (if
     * shown), and the distances of all four borders of the visible area to the
     * borders of the entire scroll area.
     *
     * @param {HTMLElement|jQuery} containerNode
     *  The DOM container element. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @returns {Object}
     *  An object with numeric properties representing the position and size of
     *  the visible area relative to the entire scroll area of the container
     *  node in pixels:
     *  - {Number} left
     *      The distance of the left border of the visible area to the left
     *      border of the entire scroll area.
     *  - {Number} top
     *      The distance of the top border of the visible area to the top
     *      border of the entire scroll area.
     *  - {Number} right
     *      The distance of the right border of the visible area (without
     *      scroll bar) to the right border of the entire scroll area.
     *  - {Number} bottom
     *      The distance of the bottom border of the visible area (without
     *      scroll bar) to the bottom border of the entire scroll area.
     *  - {Number} width
     *      The width of the visible area (without scroll bar).
     *  - {Number} height
     *      The height of the visible area (without scroll bar).
     */
    Utils.getVisibleAreaPosition = function (containerNode, options) {
        containerNode = Utils.getDomNode(containerNode);

        var clientHeight = containerNode.clientHeight;
        if (Utils.getBooleanOption(options, 'regardSoftkeyboard', false)) {
            //here comes the special code for softkeyboards
            //if the touch lands in the lower area we fake that its outsice of the client-rect,
            //so we scroll before the browser scroll the complete body
            clientHeight -= Utils.getSoftkeyboardHeight();
        }

        return {
            left: containerNode.scrollLeft,
            top: containerNode.scrollTop,
            right: containerNode.scrollWidth - containerNode.clientWidth - containerNode.scrollLeft,
            bottom: containerNode.scrollHeight - clientHeight - containerNode.scrollTop,
            width: containerNode.clientWidth,
            height: clientHeight
        };
    };

    /**
     * Returns the position and size of the specified document page rectangle
     * inside the passed DOM container node. This includes the size of the
     * rectangle as specified, and the distances of all four borders of the
     * rectangle to the borders of the visible area or entire scroll area of
     * the container node.
     *
     * @param {HTMLElement|jQuery} containerNode
     *  The container DOM element. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @param {Object} pageRect
     *  The document page rectangle whose position will be converted relatively
     *  to the position of the container node. Must provide the properties
     *  'left', 'top', 'width', and 'height' in pixels. The properties 'left'
     *  and 'top' are interpreted relatively to the entire document page.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.visibleArea=false]
     *      If set to true, calculates the distances of the rectangle to the
     *      visible area of the container node. Otherwise, calculates the
     *      distances of the rectangle to the entire scroll area of the
     *      container node.
     *
     * @returns {Object}
     *  An object with numeric properties representing the position and size of
     *  the page rectangle relative to the entire scroll area or visible area
     *  of the container node, in pixels:
     *  - {Number} left
     *      The distance of the left border of the rectangle to the left border
     *      of the container node.
     *  - {Number} top
     *      The distance of the top border of the rectangle to the top border
     *      of the container node.
     *  - {Number} right
     *      The distance of the right border of the rectangle to the right
     *      border of the container node.
     *  - {Number} bottom
     *      The distance of the bottom border of the rectangle to the bottom
     *      border of the container node.
     *  - {Number} width
     *      The width of the rectangle, as passed.
     *  - {Number} height
     *      The height of the rectangle, as passed.
     */
    Utils.getRectanglePositionInNode = function (containerNode, pageRect, options) {

        var // the offset of the container node, relative to the document body
            containerOffset = $(containerNode).offset(),
            // the width of the left and top border of the scrollable node, in pixels
            leftBorderWidth = Utils.getElementCssLength(containerNode, 'border-left-width'),
            topBorderWidth = Utils.getElementCssLength(containerNode, 'border-top-width'),
            // position and size of the visible area of the container node
            visiblePosition = Utils.getVisibleAreaPosition(containerNode, options),

            // the position and size, relative to the visible area of the container node
            position = {
                left: pageRect.left + leftBorderWidth - containerOffset.left,
                top: pageRect.top + topBorderWidth - containerOffset.top,
                width: pageRect.width,
                height: pageRect.height
            };

        // add right and bottom distance of child node to visible area
        position.right = visiblePosition.width - position.left - position.width;
        position.bottom = visiblePosition.height - position.top - position.height;

        // add distances to entire scroll area, if option 'visibleArea' is not set
        if (!Utils.getBooleanOption(options, 'visibleArea', false)) {
            _.each(['left', 'top', 'right', 'bottom'], function (border) {
                position[border] += visiblePosition[border];
            });
        }

        return position;
    };

    /**
     * Returns the position and size of the specified child node inside its
     * ancestor container node. This includes the outer size of the child node,
     * and the distances of all four borders of the child node to the borders
     * of the visible area or entire scroll area of the container node.
     *
     * @param {HTMLElement|jQuery} containerNode
     *  The DOM container element. If this object is a jQuery collection, uses
     *  the first node it contains.
     *
     * @param {HTMLElement|jQuery} childNode
     *  The DOM element whose dimensions will be calculated. Must be contained
     *  in the specified container element. If this object is a jQuery
     *  collection, uses the first node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.visibleArea=false]
     *      If set to true, calculates the distances of the child node to the
     *      visible area of the container node. Otherwise, calculates the
     *      distances of the child node to the entire scroll area of the
     *      container node.
     *
     * @returns {Object}
     *  An object with numeric properties representing the position and size of
     *  the child node relative to the entire scroll area or visible area of
     *  the container node, in pixels:
     *  - {Number} left
     *      The distance of the left border of the child node to the left
     *      border of the container node.
     *  - {Number} top
     *      The distance of the top border of the child node to the top border
     *      of the container node.
     *  - {Number} right
     *      The distance of the right border of the child node to the right
     *      border of the container node.
     *  - {Number} bottom
     *      The distance of the bottom border of the child node to the bottom
     *      border of the container node.
     *  - {Number} width
     *      The outer width of the child node (including its borders).
     *  - {Number} height
     *      The outer height of the child node (including its borders).
     */
    Utils.getChildNodePositionInNode = function (containerNode, childNode, options) {
        var pageRect = Utils.getNodePositionInPage(childNode);
        return Utils.getRectanglePositionInNode(containerNode, pageRect, options);
    };

    /**
     * Scrolls the passed page rectangle into the visible area of the specified
     * container node.
     *
     * @param {HTMLElement|jQuery} containerNode
     *  The DOM container element. If this object is a jQuery collection, uses
     *  the first node it contains. This method works independent from the
     *  current overflow mode of the container node.
     *
     * @param {Object} pageRect
     *  The document page rectangle that will be made visible by scrolling the
     *  container node. Must provide the properties 'left', 'top', 'width', and
     *  'height' in pixels. The properties 'left' and 'top' are interpreted
     *  relatively to the entire document page.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.padding=0]
     *      Minimum distance between the borders of the visible area in the
     *      container node, and the rectangle scrolled into the visible area.
     *  @param {Boolean} [options.forceToTop=false]
     *      Whether the specified pageRect shall be at the top border of the
     *      visible area of the scrolling node.
     */
    Utils.scrollToPageRectangle = function (containerNode, pageRect, options) {

        var // dimensions of the rectangle in the visible area of the scrollable node
            position = Utils.getRectanglePositionInNode(containerNode, pageRect, { visibleArea: true, regardSoftkeyboard: Utils.getBooleanOption(options, 'regardSoftkeyboard', false) }),
            // padding between scrolled element and border of visible area
            padding = Utils.getIntegerOption(options, 'padding', 0, 0);

        function updateScrollPosition(leadingChildOffset, trailingChildOffset, scrollAttributeName, opts) {

            var maxPadding = Utils.minMax((leadingChildOffset + trailingChildOffset) / 2, 0, padding),
                offset = Math.max(leadingChildOffset - Math.max(maxPadding - trailingChildOffset, 0), maxPadding);

            if (Utils.getBooleanOption(opts, 'forceToTop', false)) {
                Utils.getDomNode(containerNode)[scrollAttributeName] = leadingChildOffset + padding;
            } else {
                Utils.getDomNode(containerNode)[scrollAttributeName] -= (offset - leadingChildOffset);
            }
        }

        updateScrollPosition(position.left, position.right, 'scrollLeft');
        updateScrollPosition(position.top, position.bottom, 'scrollTop', { forceToTop: Utils.getBooleanOption(options, 'forceToTop', false) });
    };

    /**
     * Scrolls the passed child node into the visible area of the specified DOM
     * container node.
     *
     * @param {HTMLElement|jQuery} containerNode
     *  The DOM container element that contains the specified child node. If
     *  this object is a jQuery collection, uses the first node it contains.
     *
     * @param {HTMLElement|jQuery} childNode
     *  The DOM element that will be made visible by scrolling the specified
     *  container element. If this object is a jQuery collection, uses the
     *  first node it contains.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.padding=0]
     *      Minimum distance between the borders of the visible area and the
     *      child node.
     *  @param {Boolean} [options.forceToTop=false]
     *      Whether the specified node shall be at the top border of the
     *      visible area of the scrolling node.
     */
    Utils.scrollToChildNode = function (containerNode, childNode, options) {
        var pageRect = Utils.getNodePositionInPage(childNode);
        Utils.scrollToPageRectangle(containerNode, pageRect, options);
    };

    // hidden storage in DOM --------------------------------------------------

    /**
     * Inserts new DOM nodes into a global hidden storage container. The nodes
     * will not be visible but will be part of the living DOM, thus it will be
     * possible to access and modify the geometry of the nodes.
     *
     * @param {HTMLElement|jQuery} nodes
     *  The DOM node(s) to be inserted into the private storage container. If
     *  this object is a jQuery collection, inserts all contained DOM nodes
     *  into the container.
     *
     * @param {Boolean} [prepend=false]
     *  If set to true, the new nodes will be inserted before all other child
     *  nodes of the hidden storage container.
     */
    Utils.insertHiddenNodes = function (nodes, prepend) {
        if (prepend) {
            hiddenStorageNode.prepend(nodes);
        } else {
            hiddenStorageNode.append(nodes);
        }
    };

    /**
     * Returns the DOM nodes contained in the global hidden storage container.
     *
     * @param {String} selector
     *  The jQuery selector used to search for the contents in the global
     *  hidden storage node.
     *
     * @returns {jQuery}
     *  All matching DOM nodes, as jQuery collection.
     */
    Utils.findHiddenNodes = function (selector) {
        return hiddenStorageNode.find(selector);
    };

    // simulating function caretRangeFromPoint --------------------------------

    /**
     * Simulates the method document.caretRangeFromPoint() for all browsers.
     *
     * @param {Number} x
     *  The horizontal coordinate corresponding to the current window.
     *
     * @param {Number} y
     *  The vertical coordinate corresponding to the current window.
     *
     * @returns {TextRange}
     *  The range object specified by two coordinates.
     */
    Utils.caretRangeFromPoint = (function () {

        // poly-fill document.caretRangeFromPoint if not available
        if (!_.isFunction(document.caretRangeFromPoint)) {

            // FireFox supports a similar method 'caretPositionFromPoint'
            if (_.isFunction(document.caretPositionFromPoint)) {

                document.caretRangeFromPoint = function (x, y) {

                    var range = document.createRange(),
                        point = document.caretPositionFromPoint(x, y);

                    // Bug 1025815 in Mozilla: If the zoom factor is not 100%, the calculated offset is wrong.
                    // The node is always correct.

                    if (point.offsetNode) {
                        range.setStart(point.offsetNode, point.offset);
                        range.setEnd(point.offsetNode, point.offset);
                    }

                    return range;
                };

            } else if ((document.body || document.createElement('body')).createTextRange) {

                document.caretRangeFromPoint = function (x, y) {

                    var // try to create a text range at the specified location
                        range = document.body.createTextRange(),
                        // the element received from a specified point
                        elem = null;

                    try {
                        elem = document.elementFromPoint(x - 1, y - 1);
                        range = document.createRange();
                        range.setStartAfter(elem);
                        range._usedElementFromePoint_ = true; // marking the range for further evaluation
                        return range;
                    } catch (ex) {
                        return null;
                    }
                };
            }
        }

        return function (x, y) {
            return document.caretRangeFromPoint(x, y);
        };
    }());

    // focus and selection ----------------------------------------------------

    /**
     * Move focus to from given node to its next or previous focusable node. (keyboard accessibility)
     *
     * @param {jQuery} startNode
     *  Starting node.
     *
     * @param {jQuery} focusableNodes
     *  all focusable nodes
     *
     * @param {String} [direction='right']
     *  Available direction strings: 'right','left', 'up', 'down'.
     */
    Utils.moveNodeFocus = function (startNode, focusableNodes, direction) {

        var nodeToFocus = startNode,
            length = focusableNodes.length,
            index = $(focusableNodes).index(startNode);

        switch (direction || 'right') {
            case 'right':
            case 'down':
                if (index + 1 < length) {
                    nodeToFocus = focusableNodes[index + 1];
                }
                break;
            case 'left':
            case 'up':
                if (index > 0) {
                    nodeToFocus = focusableNodes[index - 1];
                }
                break;
        }

        if (!nodeToFocus) { return null; }

        Utils.setFocus(nodeToFocus);
        return nodeToFocus;
    };

    /**
     * Clears the current browser selection ranges.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.rescueFocus=true]
     *      Internet Explorer only: If set to true, the focus will be moved to
     *      an internal helper DOM element before the browser selection will
     *      actually be cleared. This workaround is needed to prevent JS errors
     *      when accessing the browser selection of a hidden element. This time
     *      consuming action can be skipped via this option, if it is sure,
     *      that the browser selection points to visible DOM elements.
     */
    Utils.clearBrowserSelection = function (options) {

        // Bug 28515, bug 28711: IE fails to clear the selection (and to modify it afterwards), if it
        // currently points to a DOM node that is not visible anymore. This happens e.g. after clicking
        // on a close button in a view pane that hides itself. Workaround is to move focus to an
        // editable DOM node which will cause IE to update the browser selection object. Using another
        // focusable node (e.g. the body element) is not sufficient. Use try/catch anyway to be notified
        // about other problems with the selection.

        // Bug 40167 (2015-Aug-19): Edge browser starts to trigger weird 'focusout' events when focusing
        // the focus helper node. New solution is to first try to clear the browser selection directly.
        // If this fails, and the browser is IE-ish, use the focus helper node in a second try.

        // helper function trying to clear the browser selection, any caught
        // exceptions will lead to call the passed callback
        function clearSelection(errorCallback) {
            try {
                if (!_.browser.IE || (window.getSelection().rangeCount > 0 && window.getSelection().getRangeAt(0).getClientRects.length > 0)) {
                    window.getSelection().removeAllRanges();
                }
            } catch (ex) {
                Utils.exception(ex);
                if (errorCallback) { errorCallback(); }
            }
        }

        // try to clear the browser selection directly
        clearSelection(function () {
            // clearing the selection has failed, do special magic for IE only
            if (_.browser.IE && Utils.getBooleanOption(options, 'rescueFocus', true)) {
                Utils.setFocus(focusHelperNode);
                clearSelection();
            }
        });
    };

    /**
     * this function sets foucs on a hidden node which is not editable.
     * On this ways the softkeyboard gets closed.
     */
    Utils.focusHiddenNode = function () {
        Utils.setFocus(hiddenFocusNode);
    };

    /**
     * Set the focus to the given element.
     * @param {HTMLElement|jQuery} element to set the focus.
     * @return {HTMLElement|jQuery} the given element.
     */
    Utils.setFocus = function (element) {
        if (element) {
            Utils.FocusLogger.log('type="setFocus"', 'target=', element, 'active=', Utils.getActiveElement(), 'id=', _.uniqueId());
            element.focus();
        }

        return element;
    };

    // event handling ---------------------------------------------------------

    /**
     * Attaches an event handler to the given jQuery object and assures
     * that the handler is the first one to be executed if more than
     * one handler is attached to the object.
     *
     * Note: $._data() is not a supported public interface; the actual
     * data structures may change incompatibly from jQuery version to version.
     *
     * @param {jQuery} object
     *  The jQuery object the handler function is attached.
     * @param {String} event
     *  The event type and optional namespaces, such as 'click' or 'keydown.myPlugin'.
     * @param {Function} func
     *  The handler function to execute when the event is triggered.
     * @param {TriggerObject} [observer]
     *  If assigned via its listenTo-function the object.on() is called,
     *  and if observer is destroyed, listener gets removed
     * @returns {Boolean}
     *  Returns true if handler has been added as first one.
     *
     */
    Utils.bindAsFirstHandler = function (object, event, func, observer) {

        object = (object instanceof $) ? object : $(object);

        // normally attach the event first
        if (observer) {
            observer.listenTo(object, event, func);
        } else {
            object.on(event, func);
        }

        if (!_.isFunction($._data)) { return false; }

        // move the handler function
        object.each(function () {
            var // all handlers for the event
                handlers = $._data(this, 'events')[event.split('.')[0]],
                // the handler that has just been inserted at the end of the list
                handler = _.isArray(handlers) ? handlers.pop() : null;

            // move the handler to the beginning of the list
            if (handler) {
                handlers.splice(0, 0, handler);
            }
        });

        return true;
    };

    // device screen and keyboard ---------------------------------------------

    /**
     * Returns the activeElement of the document object.
     * Microsoft Edge Browser throws an exception if you try to use this property when it's empty.
     *
     * @return {Node|Null}
     *  If the object is a DOM node object, returns it unmodified. If document.activeElement
     *  is empty, returns null.
     */
    Utils.getActiveElement = function () {
        var activeElement;

        try {
            activeElement = document.activeElement;
        } catch (e) {
            // console.log('ERROR: document.activeElement - ', e);
            activeElement = null;
        }

        return activeElement;
    };

    /**
     * Returns whether the display is in portrait mode currently (the current
     * screen width is less than the current screen height). On mobile devices,
     * the screen orientation may change over time according to the position of
     * the device.
     *
     * @returns {Boolean}
     *  Whether the display is in portrait mode currently.
     */
    Utils.isPortrait = function () {
        // iOS: check window.orientation (screen width and height is always constant)
        // Other: use screen width and height (meaning of window.orientation differs on Android devices)
        return (Utils.TOUCHDEVICE && _.browser.iOS) ? (Math.abs(window.orientation) !== 90) : (screen.height > screen.width);
    };

    /**
     * Returns whether the display is in landscape mode currently (the current
     * screen width is greater than the current screen height). On mobile
     * devices, the screen orientation may change over time according to the
     * position of the device.
     *
     * @returns {Boolean}
     *  Whether the display is in landscape mode currently.
     */
    Utils.isLandscape = function () {
        return !Utils.isPortrait();
    };

    /**
     * Returns the current effective screen width according to the orientation
     * of the device, in pixels.
     *
     * @returns {Number}
     *  The current effective screen width according to the orientation of the
     *  device, in pixels.
     */
    Utils.getScreenWidth = function () {
        // iOS: screen.width may return screen height depending on device orientation
        return Utils.isPortrait() ? Math.min(screen.width, screen.height) : Math.max(screen.width, screen.height);
    };

    /**
     * Returns the current effective screen height according to the orientation
     * of the device, in pixels.
     *
     * @returns {Number}
     *  The current effective screen height according to the orientation of the
     *  device, in pixels.
     */
    Utils.getScreenHeight = function () {
        // iOS: screen.height may return screen width depending on device orientation
        return Utils.isPortrait() ? Math.max(screen.width, screen.height) : Math.min(screen.width, screen.height);
    };

    /**
     * Returns the current effective width of the inner window area according
     * to the orientation of the device, in pixels.
     *
     * @returns {Number}
     *  The current effective width of the inner window area according to the
     *  orientation of the device, in pixels.
     */
    Utils.getInnerWindowWidth = function () {
        // iOS: window.innerWidth may return window height depending on device orientation
        return Utils.isPortrait() ? Math.min(window.innerWidth, window.innerHeight) : Math.max(window.innerWidth, window.innerHeight);
    };

    /**
     * Returns the current effective height of the inner window area according
     * to the orientation of the device, in pixels.
     *
     * @returns {Number}
     *  The current effective height of the inner window area according to the
     *  orientation of the device, in pixels.
     */
    Utils.getInnerWindowHeight = function () {
        // iOS: window.innerHeight may return window width depending on device orientation
        return Utils.isPortrait() ? Math.max(window.innerWidth, window.innerHeight) : Math.min(window.innerWidth, window.innerHeight);
    };

    /**
     * return true if a standard soft keyboard is opened.
     *
     * on Android by comparing screen-size with window-size
     * the code can calculate if a normal soft keyboard is openend
     * (if window-height is smaller than 70% of screen-height)
     *
     * on IOS there is checked the window-selection-type, if it is not "None",
     * there is a text-selection, so the keyboard is open
     *
     */
    Utils.isSoftKeyboardOpen = function () {
        if (!Utils.TOUCHDEVICE) { return false; }

        var selectionType = window.getSelection().type;
        return selectionType && selectionType !== 'None';
    };

    /**
     * return the estimated keyboard height for touch devices.
     * in portraitmode it is 1/3 of the screenheight and
     * in landscapemode it is a bit more than the half of the screenheight
     *
     * currently only for IOS!!!
     */
    Utils.getSoftkeyboardHeight = function () {
        // special value for iPhone in landscape mode
        var relKeyHeight = (Utils.isLandscape() && Utils.SMALL_DEVICE && Utils.IOS) ? (1.1 / 2) : (1 / 3);
        return Math.round(relKeyHeight * Utils.getScreenHeight());
    };

    /**
     * returns true if Device is Small Device or
     * Tablet is in nn Landscape Mode
     */
    Utils.shouldHideSoftKeyboard = function () {
        if (_.browser.iOS || _.browser.Android) {
            if (Utils.SMALL_DEVICE || Utils.isLandscape()) {
                var popupNode = $('body>.io-ox-office-main.popup-container');
                // when there is a popup-container and it's not a context menu
                // the softKeyboard should be hidden
                if (popupNode.length && !(popupNode.hasClass('context-menu'))) {
                    return true;
                }
            }
        }
        return false;
    };

    /**
     * Returns if a 'contextmenu' event was invoked by keyboard.
     *
     * @param {event} contextEvent
     *  A 'contextmenu' event in the handler where this function is used.

     * @returns {Boolean}
     *  Whether the 'contextmenu' event was triggered by keyboard.
     */
    Utils.isContextEventTriggeredByKeyboard = function (contextEvent) {

        // detect if the context event was invoked by keyboard -> when the last keydown event before the
        // context event matches the following conditions it's a keyboard event:
        // the event is not null and the last key was the 'context' key (93) or f10 (121) and the key event
        // was invoked max. 150 ms before this context event (so the key was pressed directly before the context event)
        return (cachedKeyEvent !== null) && (cachedKeyEvent.keyCode === 93 || cachedKeyEvent.keyCode === 121) && (contextEvent.timeStamp - cachedKeyEvent.timeStamp < 150);
    };

    // users --------------------------------------------------------------

    /**
     *  gives user Infos for gui and operations
     *
     * @param {Number} id
     *  users db-id
     *
     * @returns {Deferred}
     *  resolves with {Object} which has:
     *   {String}   displayName:    name of the user, can be used in gui
     *   {String}   operationName:  internal name of the user, can be used by operations
     *   {Boolean}  guest:          set on true if user is no normal appsuite user (invited guests by mail)
     *   {Number}   id:             users official db-id (same as assigned id)
     */
    Utils.getUserInfo = function (id) {
        return UserAPI.get({ id: id }).then(function (client) {
            //client has first_name, last_name, display_name, email1,sort_name, user_id
            var info = {};
            info.id = client.user_id;

            // same code as UserAPI.getName()
            info.displayName = _.noI18n(client.display_name || client.email1 || '');

            if (_.isUndefined(client.guest_created_by)) {
                //normal user

                var operationName = null;
                // first name + last name is the normal case
                // last name fallback1
                // first name fallback2
                if (client.first_name && client.last_name) {
                    operationName = _.noI18n(client.first_name + ' ' + client.last_name);
                } else if (client.last_name) {
                    operationName = _.noI18n(client.last_name);
                } else if (client.first_name) {
                    operationName = _.noI18n(client.first_name);
                } else {
                    operationName = info.displayName;
                }

                info.operationName = operationName;
                info.guest = false;
            } else {
                //guest user
                info.operationName = 'User ' + client.user_id;

                info.guest = true;
            }
            return info;
        });
    };

    // debugging --------------------------------------------------------------

    /**
     * Starting the global timer.
     *
     * @param {String} message
     *  A message that is displayed in the console, when the timer starts.
     *
     * @param {Boolean} addTime
     *  Whether the time of several following global timers shall be accumulated.
     */
    Utils.startGlobalTimer = Config.DEBUG ? function (message, addTime) {
        if (message) { Utils.log(message); }
        accumulateTime = addTime || false;
        if ((accumulateTime) && (!_.isNumber(accumulatedTime))) {
            accumulatedTime = 0;
            accumulatedCounter = 0;
        }
        globalTimer = _.now();
    } : $.noop;

    /**
     * Logging info from the global timer.
     *
     * @param {String} message
     *  A message that is displayed in the console, if the timer runs.
     */
    Utils.logGlobalTimer = Config.DEBUG ? function (message) {
        if (globalTimer) {
            Utils.log(message + ' (' + (_.now() - globalTimer) + 'ms)');
        }
    } : $.noop;

    /**
     * Stopping the global timer.
     *
     * @param {String} message
     *  A message that is displayed in the console, when the timer is stopped.
     *
     * @returns {Number|Null}
     *  The measured time in milliseconds, or null, if no timer is running.
     */
    Utils.stopGlobalTimer = Config.DEBUG ? function (message) {
        var diff = null;
        if (globalTimer) {
            diff = _.now() - globalTimer;
            if (accumulateTime) {
                accumulatedTime += diff;
                accumulatedCounter++;
            }
            Utils.log(message + ' (' + diff + 'ms)' + (accumulateTime ? ' Accumulated time: ' + accumulatedTime + 'ms. Run: ' + accumulatedCounter : ''));
            globalTimer = null;
        }
        return diff;
    } : $.noop;

    /**
     * Checking if the global timer is already running.
     *
     * @returns {Boolean}
     *  Whether the global timer is running.
     */
    Utils.isGlobalTimerRunning = function () {
        return globalTimer !== null;
    };

    /**
     * Logging info from starting the global counter.
     *
     * @param {String} message
     *  A message that is displayed in the console, if the counter runs.
     */
    Utils.startGlobalCounter = Config.DEBUG ? function (message) {
        Utils.log(message);
        globalCounter = 0;
    } : $.noop;

    /**
     * Logging info from incrementing the global counter.
     *
     * @param {String} message
     *  A message that is displayed in the console, if the counter is incremented.
     */
    Utils.increaseGlobalCounter = Config.DEBUG ? function (message) {
        if (globalCounter !== null) {
            globalCounter++;
            if (message) {
                Utils.log(message + ' : ' + globalCounter);
            }
        }
    } : $.noop;

    /**
     * Logging info from stopping the global counter.
     *
     * @param {String} message
     *  A message that is displayed in the console, if the global counter is stopped.
     */
    Utils.stopGlobalCounter = Config.DEBUG ? function (message) {
        if (globalCounter !== null) {
            Utils.log(message + ' : ' + globalCounter);
            globalCounter = null;
        }
    } : $.noop;

    /**
     * A simple check to detect whether the local storage in the browser is available.
     * Problem: In Safari the local browsing mode does not support the usage of the
     * local storage.
     *
     * @returns {Boolean}
     *  Whether the local storage can be used.
     */
    Utils.isLocalStorageSupported = function () {

        var testKey = '_localTestKey_';

        try {
            localStorage.setItem(testKey, '1');
            localStorage.removeItem(testKey);
            return true;
        } catch (ex) {
            return false;
        }
    };

    /**
     * Returns whether the given string ends with a given suffix.
     *
     * @param {String} str
     *  A specified string
     *
     * @param {String} suffix
     *  A specified suffix
     *
     * @returns {Boolean}
     *  Whether the given string ends with the specified suffix.
     */
    Utils.stringEndsWith = function (str, suffix) {
        return str && suffix && str.indexOf(suffix, str.length - suffix.length) > -1;
    };

    Utils.supportedChromeVersionOnAndroidForText = function () {
        // We know that M49 includes the fixes for return button
        // Bug 44262
        return _.browser.Android && Utils.isBrowserAndMinVersion('Chrome', 48);
    };

    Utils.supportedChromeVersionOnMacForText = function () {
        // Bug 48777
        return _.browser.MacOS && Utils.isBrowserAndMinVersion('Chrome', 53);
    };

    /**
    * Removes all contenteditables below the given node. It doesn't remove contenteditable = 'false'
    * from elements, because when an parent would get contenteditable= 'true' later, these explicitly
    * to 'false' set child elements would be contenteditable = 'true'.
    *
    * @param {jQuery} node
    *  A jQuery Node.
    */
    Utils.removeContentEditable = function (node) {
        $(node).find('[contenteditable=true]').each(function () {
            $(this).removeAttr('contenteditable');
        });
    };

    // console output ---------------------------------------------------------

    /**
     * Writes a log message to the hidden Seleniun log node, if debug mode is
     * enabled in the global configuration.
     *
     * @param {String} key
     *  A unique key for the functionality to be logged.
     *
     * @param {String} state
     *  The current state of functionality to be logged.
     */
    Utils.logSelenium = Config.DEBUG ? (function () {

        // console logger
        var logger = new Logger({ enable: 'office:log-selenium', prefix: 'SELENIUM' });
        // the global log node for Selenium tests
        var logNode = $('<ul id="io-ox-office-selenium">').prependTo('body');
        // global counter for log entries
        var counter = 0;

        return function (key, state) {
            counter += 1;
            var args = ['id=' + counter, 'time=' + _.now(), 'key=' + key, 'state=' + state];
            logger.info.apply(logger, args);
            var msg = args.join();
            logNode.append($('<li>').text(msg));
            seleniumProtocol.push(msg);
        };
    }()) : $.noop;

    Utils.shiftSeleniumLog = function () {
        return seleniumProtocol.splice(0, seleniumProtocol.length);
    };

    // global initialization ==================================================

    // insert the focus helper into the DOM
    Utils.insertHiddenNodes(focusHelperNode);
    Utils.insertHiddenNodes(hiddenFocusNode);

    // create global logger instances
    Utils.FocusLogger = new Logger({ enable: 'office:log-focus', prefix: 'FOCUS' });
    Utils.LayoutLogger = new Logger({ enable: 'office:log-layout', prefix: 'LAYOUT' });

    // focus tracker ----------------------------------------------------------

    // trigger normalized global 'change:focus' events
    (function () {

        var // the DOM node currently focused
            focusNode = null,
            // whether the last received event was a 'focusout' event
            pendingFocusOut = false,
            // last 'keydown' event received in this stack frame
            pendingKeyEvent = null;

        // update the current focus node, and trigger the global 'change:focus' event
        function updateFocus(newFocusNode) {
            var blurNode = focusNode;
            // ignore 'focusin' events on the document root (sent by Firefox when returning focus to browser tab)
            if ((document !== newFocusNode) && (focusNode !== newFocusNode)) {
                focusNode = newFocusNode;
                Utils.trigger('change:focus', focusNode, blurNode, pendingKeyEvent);
            }
        }

        // remember 'keydown' event object to be passed to the next 'change:focus' event
        function keyDownHandler(event) {
            pendingKeyEvent = event;
            cachedKeyEvent = event;
            _.defer(function () { pendingKeyEvent = null; });
        }

        // handle a new focused DOM element
        function focusInHandler(event) {
            Utils.FocusLogger.log('type="focusin"', 'target=', event.target, 'active=', Utils.getActiveElement());
            pendingFocusOut = false;
            updateFocus(event.target);
        }

        // handle a 'focusout' event, detect whether entire page (browser tab) has lost focus
        function focusOutHandler(event) {
            Utils.FocusLogger.log('type="focusout"', 'target=', event.target, 'active=', Utils.getActiveElement());
            // ignore 'focusout' event whose target does not match the current focus node
            if (event.target === focusNode) {
                pendingFocusOut = true;
                _.defer(function () {
                    if (pendingFocusOut) {
                        pendingFocusOut = false;
                        updateFocus(null);
                    }
                });
            }
        }

        // listen to 'keydown' events in the capturing phase, otherwise custom focus handlers
        // will be executed before the 'keydown' event bubbles up to the document node
        document.addEventListener('keydown', keyDownHandler, true);

        // listen to the global bubbling 'focusin' and 'focusout' events of jQuery
        $(document).on({ focusin: focusInHandler, focusout: focusOutHandler });

        // log focus events
        Utils.FocusLogger.logEvent(Utils, 'change:focus', 'focus', 'blur', 'keyboard');

        // initialize current focus node by calling the 'focusin' event handler directly
        focusInHandler({ target: Utils.getActiveElement() });
    }());

    // exports ================================================================

    return Utils;

});
