/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/baseobject', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/deferredutils'
], function (Utils, DeferredUtils) {

    'use strict';

    // a predicate function that matches everything
    var ALL_MATCHER = _.constant(true);

    // private global functions ===============================================

    /**
     * Returns a matcher predicate function for the passed event source object.
     */
    function createEventSourceMatcher(source) {

        // accept plain DOM elements, convert to jQuery collection
        if (source instanceof HTMLElement) { source = $(source); }

        // return the predicate function
        return function sourceMatcher(s) {
            // jQuery objects must be tested with is() method instead of equality operator
            return (s instanceof $) ? s.is(source) : (s === source);
        };
    }

    /**
     * Returns a matcher predicate function for the passed event types (null to
     * match all, or string list).
     */
    function createEventTypesMatcher(types) {

        // all types will match for non-string value
        if (!_.isString(types)) { return ALL_MATCHER; }

        // convert space-separated string to a flag set of single event types
        var typeSet = Utils.makeSet(types.split(/\s+/).filter(_.identity));

        // return the predicate function
        return function (t) { return t in typeSet; };
    }

    /**
     * Returns a matcher predicate function for the passed event listener (null
     * to match all, or function).
     */
    function createEventListenerMatcher(listener) {
        return listener ? function (l) { return l === listener; } : ALL_MATCHER;
    }

    /**
     * Unbinds the passed event type and/or listener (either parameter can be
     * null).
     */
    function unbindEventListener(listeners, source, type, listener) {

        // the matcher predicate for the passed event source object
        var sourceMatcher = createEventSourceMatcher(source);
        // a matcher predicate for an event type
        var typeMatcher = createEventTypesMatcher(type);
        // a matcher predicate for a listener callback function
        var listenerMatcher = createEventListenerMatcher(listener);

        // detach all matching event listeners for the passed object
        listeners.forEach(function (data) {
            if (sourceMatcher(data.source) && typeMatcher(data.type) && listenerMatcher(data.listener)) {
                source.off(data.type, data.listener);
            }
        });
    }

    /**
     * Registers and binds the passed event type and listener.
     */
    function bindEventListener(listeners, source, type, listener, once) {

        // once mode: create a version of the listener that unbinds itself on first invocation
        var listenerFunc = once ? function () {
            unbindEventListener(listeners, source, type, listenerFunc);
            return listener.apply(this, arguments);
        } : listener;

        // register and bind the listener
        listeners.push({ source: source, type: type, listener: listenerFunc });
        source.on(type, listenerFunc);
    }

    /**
     * Registers and binds the passed event types and listeners.
     */
    function bindEventListeners(listeners, source, type, listener, once) {

        // accept plain DOM elements, convert to jQuery collection
        if (source instanceof HTMLElement) { source = $(source); }

        // bug 36850: handle event maps and string/function parameters
        var eventMap = _.isObject(type) ? type : Utils.makeSimpleObject(type, listener);
        _.each(eventMap, function (evtListener, evtTypes) {
            _.each(evtTypes.split(/\s+/), function (evtType) {
                if (evtType.length > 0) {
                    bindEventListener(listeners, source, evtType, evtListener, once);
                }
            });
        });
    }

    /**
     * Returns whether the passed value is a pending abortable promise.
     *
     * @param {Any} value
     *  The value to be inspected.
     *
     * @returns {Boolean}
     *  Whether the passed value is a pending abortable promise.
     */
    function isPendingAbortablePromise(value) {
        return Utils.isPromise(value) && (value.state() === 'pending') && _.isFunction(value.abort);
    }

    /**
     * Adds an abort() method to the passed promise which invokes the specified
     * callback functions. The abort() method will accept a single optional
     * parameter 'cause', that if omitted defaults to the string 'abort'.
     *
     * @param {jQuery.Promise}
     *  A promise that will be extended with an abort() method.
     *
     * @param {Function} abort
     *  The implementation of the generated abort() method. Will be called with
     *  undefined context, and forwards the parameter 'cause' passed to the
     *  generated abort() method.
     *
     * @param {Function} [custom]
     *  An additional custom user-defined callback function that will be called
     *  before executing the specified 'abort' callback function. Will be
     *  called in the context of the promise, and forwards the parameter
     *  'cause' passed to the generated abort() method.
     */
    function createAbortMethod(promise, abort, custom) {

        // shortcut for resolved/rejected promises
        if (promise.state() !== 'pending') {
            promise.abort = Utils.NOOP;
            return;
        }

        // create the new abort() method
        promise.abort = function (cause) {
            // prevent recursive calls from callback functions
            if (promise.state() === 'pending') {
                if (_.isUndefined(cause)) { cause = 'abort'; }
                if (_.isFunction(custom)) { custom.call(this, cause); }
                abort(cause);
            }
            return this;
        };

        // replace the abort() method in resolved promise
        promise.always(function () { promise.abort = Utils.NOOP; });
    }

    /**
     * Overrides the then() method of the passed abortable promise with a
     * version that creates and returns a piped promise with an abort() method
     * which aborts the original abortable promise.
     *
     * @param {jQuery.Promise} abortablePromise
     *  The abortable promise whose then() method will be overridden.
     */
    function overrideThenMethod(abortablePromise) {

        // the original then() method of the passed promise
        var thenMethod = abortablePromise.then.bind(abortablePromise);
        // the original abort() method of the passed promise
        var abortMethod = abortablePromise.abort.bind(abortablePromise);

        // create the new then() method which returns an abortable promise
        abortablePromise.then = function (done, fail, notify, abort) {

            // the promise returned by the done or fail handler
            var resultPromise = null;

            // Returns a wrapper function for the passed callback function (the 'done' or 'fail' handlers passed
            // to the then() method of the source promise). Upon invocation, the result value of the callback will
            // be inspected, and if it is a pending abortable promise, it will be stored internally. If the
            // resulting chained promise will be aborted, the cached result promise will be aborted too.
            function callbackWrapper(callback) {
                return _.isFunction(callback) ? function () {
                    var result = callback.apply(this, arguments);
                    if (isPendingAbortablePromise(result)) { resultPromise = result; }
                    return result;
                } : callback;
            }

            // the new piped promise returned by the original then() method
            var pipedPromise = thenMethod(callbackWrapper(done), callbackWrapper(fail), notify);

            // add a custom abort() method, that aborts the passed promise
            createAbortMethod(pipedPromise, abortMethod, function (cause) {
                if (_.isFunction(abort)) { abort(cause); }
                if (resultPromise) { resultPromise.abort(cause); }
            });

            // override the then() method of the promise with an abortable version
            overrideThenMethod(pipedPromise);

            return pipedPromise;
        };

        // prevent using the deprecated pipe() method
        delete abortablePromise.pipe;
    }

    /**
     * Returns a new function wrapping the passed callback function, intended
     * to be used by an instance of BaseObject waiting for a promise.
     *
     * @param {BaseObject} object
     *  The instance of BaseObject that wants to wait for the passed promise,
     *  and invoke the passed callback function, unless it destroys itself.
     *
     * @param {Function} callback
     *  The callback function to be invoked by the passed object instance after
     *  waiting for a promise.
     *
     * @returns {Function}
     *  A new function that will invoke the passed callback function, unless
     *  the object instance is currently destroying itself, or has already been
     *  destroyed.
     */
    function wrapPromiseCallback(object, callback) {
        return function () {
            if (!object.destroying && !object.destroyed) {
                return callback.apply(object, arguments);
            }
        };
    }

    // class BaseObject =======================================================

    /**
     * An abstract base class for all classes that want to register destructor
     * code that will be executed when the public method BaseObject.destoy()
     * has been invoked.
     *
     * @constructor
     */
    function BaseObject() {

        // internal unique object identifier
        var uid = _.uniqueId('obj');

        // all event source objects and listeners
        var listeners = null;

        // all pending abortable promises
        var promises = null;

        // all destructor callbacks
        var destructors = null;

        // protected methods --------------------------------------------------

        /**
         * Registers a destructor callback function that will be invoked when
         * the method BaseObject.destroy() has been called.
         *
         * @internal
         *  Intended to be used by the internal implementations of sub classes.
         *  DO NOT CALL from external code!
         *
         * @param {Function} destructor
         *  A destructor callback function. The BaseObject.destroy() method
         *  will invoke all registered destructor callbacks in reverse order of
         *  registration. The callback will be invoked in the context of this
         *  instance.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.registerDestructor = function (destructor) {
            (destructors || (destructors = [])).unshift(destructor);
            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the globally unique identifier for this instance.
         *
         * @returns {String}
         *  The globally unique identifier for this instance.
         */
        this.getUid = function () {
            return uid;
        };

        /**
         * Registers an event listener at the specified event source object. On
         * destruction of this instance, all event listeners registered with
         * this method will be removed from the source object.
         *
         * @param {Events|jQuery|HTMLElement} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener (for example, jQuery objects,
         *  or any object extended with the core Events mix-in class).
         *  Additionally, a plain DOM element instance can be passed, which
         *  will be converted to a jQuery collection internally.
         *
         * @param {String|Object} type
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names. Alternatively, can be an object mapping event
         *  types to listener callback functions.
         *
         * @param {Function} [listener]
         *  The event listener that will be invoked for the events triggered by
         *  the event source object. This parameter will be ignored, if the
         *  parameter 'type' is an object mapping event types to listener
         *  callback functions.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.listenTo = function (source, type, listener) {

            // create the array of known listeners on demand
            listeners = listeners || [];

            // bind the listener to the event source
            bindEventListeners(listeners, source, type, listener, false);

            return this;
        };

        /**
         * Registers an event listener at the specified event source object,
         * that will be triggered exactly once, and will unregister itself
         * afterwards. On destruction of this instance, all event listeners
         * registered with this method will be removed from the source object.
         *
         * @param {Events|jQuery|HTMLElement} source
         *  The event source object. Can be any object that provides a method
         *  'on()' to register an event listener (for example, jQuery objects,
         *  or any object extended with the core Events mix-in class).
         *  Additionally, a plain DOM element instance can be passed, which
         *  will be converted to a jQuery collection internally.
         *
         * @param {String|Object} type
         *  The type of the event to listen to. Can be a space-separated list
         *  of event type names. Alternatively, can be an object mapping event
         *  types to listener callback functions.
         *
         * @param {Function} [listener]
         *  The event listener that will be invoked for the events triggered by
         *  the event source object. This parameter will be ignored, if the
         *  parameter 'type' is an object mapping event types to listener
         *  callback functions.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.listenOnceTo = function (source, type, listener) {

            // create the array of known listeners on demand
            listeners = listeners || [];

            // bind the listener to the event source
            bindEventListeners(listeners, source, type, listener, true);

            return this;
        };

        /**
         * Removes event listeners that have been registered for the passed
         * event source object using the method BaseObject.listenTo().
         *
         * @param {Events|jQuery|HTMLElement} source
         *  The event source object, as has been passed to the public method
         *  BaseObject.listenTo() before.
         *
         * @param {String|Object} [type]
         *  If specified, the type of the event to stop listening to. Can be a
         *  space-separated list of event type names. Alternatively, can be an
         *  object mapping event types to listener callback functions. If
         *  omitted, the event listeners for all registered event types will be
         *  removed (optionally filtered by a specific event listener function,
         *  see parameter 'listener').
         *
         * @param {Function} [listener]
         *  If specified, the event listener that will be removed for the event
         *  source object. If omitted, all event listeners of the event source
         *  will be removed (optionally filtered by specific event types, see
         *  parameter 'type'). This parameter will be ignored, if the parameter
         *  'type' is an object mapping event types to listener callback
         *  functions.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.stopListeningTo = function (source, type, listener) {

            // nothing to do without registered listeners
            if (!listeners) { return this; }

            if (_.isFunction(type) && _.isUndefined(listener)) {
                // 'type' may be omitted completely (instead of being set to null)
                unbindEventListener(listeners, source, null, type);
            } else if (_.isObject(type)) {
                // bug 36850: unbind all passed event types of an event map
                _.each(type, function (listener2, type2) {
                    unbindEventListener(listeners, source, type2, listener2);
                });
            } else {
                unbindEventListener(listeners, source, type, listener);
            }

            return this;
        };

        /**
         * Creates a promise for the passed deferred object representing code
         * running asynchronously. The promise will contain an additional
         * method abort() that (when called before the deferred object has been
         * resolved or rejected) invokes the specified callback function, and
         * rejects the passed deferred object. Additionally, the promise will
         * be stored internally as long as it is in pending state. When this
         * instance will be destroyed, all pending promises will be aborted
         * automatically with the value 'destroy'.
         *
         * @param {jQuery.Deferred} deferred
         *  The deferred object to create an abortable promise for.
         *
         * @param {Function} [callback]
         *  An optional callback function that will be invoked when the promise
         *  has been aborted, and the deferred object is still pending. Will be
         *  called in the context of this instance.
         *
         * @param {Number} [timeout]
         *  If specified and a positive number, the delay time in milliseconds
         *  after the promise returned by this method will be rejected with the
         *  value 'timeout', if the passed original deferred object is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  A promise for the passed deferred object, with an additional method
         *  abort().
         */
        this.createAbortablePromise = function (deferred, callback, timeout) {

            // the promise of the passed deferred object
            var promise = deferred.promise();

            // add a custom abort() method to the promise, rejecting the passed deferred object
            createAbortMethod(promise, deferred.reject.bind(deferred), callback ? callback.bind(this) : null);

            // override the then() method of the promise with an abortable version
            overrideThenMethod(promise);

            // do not process a deferred object that is not pending anymore, but create the abort() method
            if (promise.state() === 'pending') {

                // create the array of pending promises on demand, add the new promise
                (promises || (promises = [])).push(promise);

                // remove the promise from the array when it resolves
                promise.always(function () {
                    promises.splice(promises.indexOf(promise), 1);
                });

                // abort automatically after the specified timeout
                if (_.isNumber(timeout) && (timeout > 0)) {
                    var timer = window.setTimeout(function () {
                        if (promise.state() === 'pending') { promise.abort('timeout'); }
                    }, timeout);
                    promise.always(function () {
                        if (timer) { window.clearTimeout(timer); timer = null; }
                    });
                }
            }

            callback = null;
            return promise;
        };

        /**
         * Creates an abortable promise that has been resolved with the passed
         * result value.
         *
         * @param {Any} result
         *  The result value to be provided by the returned promise.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that has been resolved with the passed result
         *  value. The method abort() of the promise will do nothing. The
         *  method then() of the promise will return abortable promises too.
         */
        this.createResolvedPromise = function (result) {
            return this.createAbortablePromise($.Deferred().resolve(result));
        };

        /**
         * Creates an abortable promise that has been rejected with the passed
         * result value.
         *
         * @param {Any} result
         *  The result value to be provided by the returned promise.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that has been rejected with the passed result
         *  value. The method abort() of the promise will do nothing. The
         *  method then() of the promise will return abortable promises too.
         */
        this.createRejectedPromise = function (result) {
            return this.createAbortablePromise($.Deferred().reject(result));
        };

        /**
         * Creates a new pending deferred object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number|Null} [timeout]
         *      If specified and a positive number, the delay time in
         *      milliseconds after the deferred object returned by this method
         *      will be rejected automatically with the value 'timeout'.
         *  @param {BaseApplication) [options.app]
         *      app is needed for correct window-id
         *  @param {String} [options.infoString]
         *      If specified, this string is used for logging information about
         *      run times of asynchronous operations. This option is only
         *      evaluated, if debug mode is enabled.
         *  @param {String} [options.priority]
         *      If specified, this string is used for specifiy the priority of
         *      the Deferred/Timer. It used by selenium tests to see if all
         *      "jobs" are done.
         *
         * @returns {jQuery.Deferred}
         *  A new pending deferred object.
         */
        this.createDeferredObject = function (options) {

            var app = Utils.getOption(options, 'app');
            var timeout = Utils.getOption(options, 'timeout');
            var infoString = Utils.getStringOption(options, 'infoString');
            var priority = Utils.getStringOption(options, 'priority');

            var deferred = DeferredUtils.createDeferred(app, infoString, priority);
            this.createAbortablePromise(deferred, null, timeout);
            return deferred;
        };

        /**
         * Returns a promise that will be resolved when the passed event source
         * object triggers the specified event.
         *
         * @param {Events|jQuery|HTMLElement} source
         *  The event source object. See method BaseObject.listenTo() for more
         *  details about supported object types.
         *
         * @param {String} type
         *  The type of the event to wait for. Can be a space-separated list of
         *  event type names. In the latter case, the first event matching one
         *  of the specified types will resolve the promise.
         *
         * @param {Number} [timeout]
         *  If specified, a delay time in milliseconds. If the source object
         *  does not trigger the expected event before the time has elapsed,
         *  the promise will be rejected with the string 'timeout'.
         *
         * @returns {jQuery.Promise}
         *  An abortable promise that will be resolved with all parameters
         *  passed with the event (including the leading event object), or that
         *  will be rejected with the string 'timeout' after the specified
         *  delay time. The promise contains the additional method abort() that
         *  allows to stop listening to the pending event immediately.
         */
        this.waitForEvent = function (source, type, timeout) {

            // the deferred object representing the event state
            var deferred = DeferredUtils.createDeferred(this, 'BaseObject: waitForEvent');

            // the event listener callback function, resolves the deferred object with the event data
            function listener() { deferred.resolve.apply(deferred, arguments); }

            // listen to the specified event, and resolve the deferred object
            this.listenOnceTo(source, type, listener);

            // the abortable promise for the deferred object
            return this.createAbortablePromise(deferred, function () {
                this.stopListeningTo(source, type, listener);
            }, timeout);
        };

        /**
         * Registers a callback function at the specified promise that waits
         * for it to be resolved. When this instance will be destroyed before
         * the promise resolves, the callback function will not be invoked.
         *
         * @param {jQuery.Promise} promise
         *  The promise.
         *
         * @param {Function} callback
         *  The callback function to be invoked when the promise resolves.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.waitForSuccess = function (promise, callback) {
            // register a callback function that checks the destroy flags of this instance
            promise.done(wrapPromiseCallback(this, callback));
            return this;
        };

        /**
         * Registers a callback function at the specified promise that waits
         * for it to be rejected. When this instance will be destroyed before
         * the promise rejects, the callback function will not be invoked.
         *
         * @param {jQuery.Promise} promise
         *  The promise.
         *
         * @param {Function} callback
         *  The callback function to be invoked when the promise rejects.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.waitForFailure = function (promise, callback) {
            // register a callback function that checks the destroy flags of this instance
            promise.fail(wrapPromiseCallback(this, callback));
            return this;
        };

        /**
         * Registers a callback function at the specified promise that waits
         * for it to be resolved or rejected. When this instance will be
         * destroyed before the promise resolves or rejects, the callback
         * function will not be invoked.
         *
         * @param {jQuery.Promise} promise
         *  The promise.
         *
         * @param {Function} callback
         *  The callback function to be invoked when the promise resolves or
         *  rejects.
         *
         * @returns {BaseObject}
         *  A reference to this instance.
         */
        this.waitForAny = function (promise, callback) {
            // register a callback function that checks the destroy flags of this instance
            promise.always(wrapPromiseCallback(this, callback));
            return this;
        };

        /**
         * Destroys this object. Invokes all destructor callback functions that
         * have been registered for this instance in reverse order. Afterwards,
         * all public properties and methods of this instance will be deleted,
         * and a single property 'destroyed' will be inserted and set to the
         * value true.
         */
        this.destroy = function () {

            // set a flag that specifies that the destructor is currently running
            this.destroying = true;

            // Abort all asynchronous code to prevent JS errors from callbacks when the
            // promises resolve/reject normally. Each aborted promise removes itself from
            // the array, but a single abort() call inside the while-loop MAY abort other
            // dependent promises as well, therefore a while loop will be used that checks
            // the array length intentionally in each iteration.
            if (promises) {
                while (promises.length > 0) { promises[0].abort('destroy'); }
                promises = null;
            }

            // unregister all event listeners
            if (listeners) {
                listeners.forEach(function (data) {
                    // source object may have been destroyed already
                    if (data.source.off) { data.source.off(data.type, data.listener); }
                });
                listeners = null;
            }

            // invoke all destructor callbacks
            if (destructors) {
                destructors.forEach(function (destructor) { destructor.call(this); }, this);
                destructors = null;
            }

            // delete all public members, to detect any misuse after destruction
            _.each(this, function (member, name) { delete this[name]; }, this);
            this.destroyed = true;
            this.uid = uid;
        };

    } // class BaseObject

    // exports ================================================================

    return _.makeExtendable(BaseObject);

});
