/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/locale/formatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/tk/object/baseobject'
], function (Utils, DateUtils, LocaleData, Parser, BaseObject) {

    'use strict';

    // the exponent character
    var EXP = 'E';

    // different correction summands to workaround rounding errors caused by the limited
    // internal precision of floating-point numbers
    var EPSILON_1 = Utils.EPSILON * 0.75;
    var EPSILON_2 = Utils.EPSILON * 1.5;
    var EPSILON_4 = Utils.EPSILON * 3;
    var EPSILON_8 = Utils.EPSILON * 6;

    // zero characters, used for formatting numbers with leading or trailing zeros
    // (310 zeros are enough for all available floating point numbers, maximum exponent is 307)
    var ZEROS = Utils.repeatString('0000000000', 31);

    // values of roman numbers, mapped by roman digit
    var ROMAN_DIGIT_MAP = { M: 1000, D: 500, C: 100, L: 50, X: 10, V: 5, I: 1 };

    // digits of roman numbers, and their decimal values, sorted descending
    var ROMAN_DIGITS = _.chain(ROMAN_DIGIT_MAP)
            .map(function (value, char) { return { char: char, value: value }; })
            .sortBy(function (digit) { return -digit.value; })
            .value();

    // private static functions ===============================================

    /**
     * Formats the passed integral number with leading zero characters.
     *
     * @param {Number} number
     *  The number to be formatted.
     *
     * @param {Number} length
     *  The target length of the formatted number. If the resulting number is
     *  shorter, leading zero characters will be inserted; if it is already
     *  longer, it will NOT be truncated.
     *
     * @returns {String}
     *  The formatted number with leading zero characters as needed.
     */
    function formatInteger(number, length) {
        var text = String(Math.floor(Math.abs(number)));
        return (text.length < length) ? (ZEROS + text).substr(-length) : text;
    }

    /**
     * Returns the text representation of the year in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as year.
     *
     * @param {Number} length
     *  The length of the formatted year: 2 for short years, 4 for full years.
     *
     * @returns {String}
     *  The formatted year of the passed date.
     */
    function formatYear(date, length) {
        return (ZEROS + date.getUTCFullYear()).substr(-length);
    }

    /**
     * Returns the text representation of the month in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as month.
     *
     * @param {Number} length
     *  The length of the formatted month: 1 for short month number, 2 for
     *  month number with leading zero, 3 for abbreviated month name, 4 for
     *  full month name, or 5 for the first character of the month name.
     *
     * @returns {String}
     *  The formatted month of the passed date.
     */
    function formatMonth(date, length) {
        var month = date.getUTCMonth();
        switch (length) {
            case 1:
                return String(month + 1);
            case 3:
                return LocaleData.SHORT_MONTHS[month];
            case 4:
                return LocaleData.LONG_MONTHS[month];
            case 5:
                return LocaleData.LONG_MONTHS[month][0];
            default:
                return formatInteger(month + 1, 2);
        }
    }

    /**
     * Returns the text representation of the day in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as day.
     *
     * @param {Number} length
     *  The length of the formatted day: 1 for short day number, 2 for day
     *  number with leading zero, 3 for abbreviated name of week day, or 4
     *  for full name of week day.
     *
     * @returns {String}
     *  The formatted day of the passed date.
     */
    function formatDay(date, length) {
        switch (length) {
            case 1:
                return String(date.getUTCDate());
            case 3:
                return LocaleData.SHORT_WEEKDAYS[date.getUTCDay()];
            case 4:
                return LocaleData.LONG_WEEKDAYS[date.getUTCDay()];
            default:
                return formatInteger(date.getUTCDate(), 2);
        }
    }

    /**
     * Returns the text representation of the quarter in the passed date.
     *
     * @param {Date} date
     *  The UTC date to be formatted as quarter string.
     *
     * @param {Number} length
     *  The length of the formatted quarter: 1 for short quarter format e.g.
     *  'Q1', otherwise the long quarter format e.g. '1st quarter'.
     *
     * @returns {String}
     *  The formatted quarter of the passed date.
     */
    function formatQuarter(date, length) {
        var quarter = Math.floor(date.getUTCMonth() / 3);
        var quarters = (length === 1) ? LocaleData.SHORT_QUARTERS : LocaleData.LONG_QUARTERS;
        return quarters[quarter];
    }

    // class Formatter ========================================================

    /**
     * A formatter converts numbers and other values to strings, according to
     * specific format codes.
     *
     * @constructor
     *
     * @extends BaseObject
     *
     * @param {Object} [initOptions]
     *  Optional parameters:
     *  @param {String} [initOptions.dec=LocaleData.DEC]
     *      The decimal separator character. If omitted, the decimal separator
     *      of the current UI language will be used.
     *  @param {String} [initOptions.group=LocaleData.GROUP]
     *      The group separator (a.k.a. thousands separator) character. If
     *      omitted, the group separator of the current UI language will be
     *      used.
     *  @param {Date|Number} [initOptions.nullDate]
     *      The null date used by this formatter (i.e. the date that will be
     *      formatted for the serial number 0), as UTC date object or as number
     *      of milliseconds as returned by Date.UTC(). By default, the date
     *      1899-11-30 will be used as null date.
     *  @param {Boolean} [initOptions.leapYearBug=false]
     *      Whether to implement the faulty behavior that 1900 was a yeap year
     *      for compatibility with other spreadsheet applications, i.e. whether
     *      to treat 1900-02-29 as a valid date.
     *  @param {Number} [initOptions.centuryThreshold=30]
     *      An integer in the range 0 to 100 that specifies how to convert a
     *      number below 100 to a 4-digit year. See DateUtils.expandYear() for
     *      more details.
     *  @param {Boolean} [initOptions.negativeDates=false]
     *      Whether to support negative date values (for example, the number -1
     *      result in the day before the null date). If set to false or
     *      omitted, negative numbers cannot be converted to dates, and dates
     *      before the null date cannot be converted to numbers.
     *  @param {Number} [initOptions.generalLength=11]
     *      Maximum number of characters used to format a number with the
     *      'General' format code token, including the decimal separator and
     *      the complete exponent in scientific notation, but without the minus
     *      sign.
     */
    function Formatter(initOptions) {

        // self reference
        var self = this;

        // decimal separator
        var DEC = LocaleData.DEC;

        // group separator
        var GROUP = LocaleData.GROUP;

        // null date (corresponding to number zero)
        var nullDate = Date.UTC(1899, 11, 30);

        // whether to treat 1900-02-29 as a valid date
        var leapYearBug = false;

        // threshold for full-year conversion
        var centuryThreshold = 30;

        // whether to support negative date values
        var negativeDates = false;

        // number of characters for a 'General' format token
        var generalLength = 11;

        // parsed format code for long system date (used for LCID 0xF800)
        var sysDateFormat = Parser.parseFormatCode('ooxml', 'op', LocaleData.LONG_DATE);

        // parsed format code for long system date (used for LCID 0xF400)
        var sysTimeFormat = Parser.parseFormatCode('ooxml', 'op', LocaleData.LONG_TIME);

        // base constructor ---------------------------------------------------

        BaseObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Calculates various properties of a number needed for formatting to
         * the standard number format.
         *
         * @param {Number} number
         *  The original (signed) number to be processed.
         *
         * @param {Number} maxLength
         *  The maximum number of characters. See description of the method
         *  Formatter.formatStandardNumber() for details.
         *
         * @param {Object} [options]
         *  Optional parameters. See method Formatter.formatStandardNumber()
         *  for details.
         *
         * @returns {Object|Null}
         *  Either null, if the passed maximum length is too small to format
         *  the number at all; otherwise a result object with the following
         *  properties:
         *  - {Number} number
         *      The original (signed) number, as passed to this method.
         *  - {Number} maxLength
         *      The maximum number of characters, as passed to this method.
         *  - {Number} minExpLen
         *      The minimum number of digits for the exponent (the resolved
         *      value of the option 'expLength' passed to this method).
         *  - {Number} mant
         *      The absolute value of the mantissa for scientific notation. If
         *      the original number is zero, or very close to zero
         *      (denormalized mantissa), this property will be set to zero.
         *  - {Number} exp
         *      The exponent for scientific notation.
         *  - {Number} maxDecDigits
         *      The maximum number of significant digits available for decimal
         *      notation. May be much less than the maximum length, e.g. the
         *      number 0.00012 fits into 7 characters, but can show only 2
         *      significant digits. Can be negative to indicate how many digits
         *      are missing to be able to create an appropriately formatted
         *      number.
         *  - {Number} maxSciDigits
         *      The maximum number of significant digits available for
         *      scientific notation. Can be negative to indicate how many
         *      digits are missing to be able to create an appropriately
         *      formatted number.
         *  - {Number} expLength
         *      The number of digits required for the absolute value of the
         *      exponent (according to the exponent, and the value of the
         *      property 'minExpLen').
         */
        function getNumberInfo(number, maxLength, options) {

            // Initialize the resulting descriptor.
            var info = {
                number: number,
                maxLength: maxLength,
                minExpLen: Utils.getIntegerOption(options, 'expLength', 2, 1, 9),
                mant: 0,
                exp: 0,
                maxDecDigits: 0,
                maxSciDigits: 0,
                expLength: 0
            };

            // Fast exit, if the number is exactly or very close to zero (do not try to format denormalized numbers).
            if (Math.abs(number) < Utils.MIN_NUMBER) { return info; }

            // Start with the absolute value of the mantissa, and the exponent of the passed number.
            var normalized = Utils.normalizeNumber(number);
            info.mant = Math.abs(normalized.mant);
            info.exp = normalized.exp;

            // The native method Number.toFixed() will be used to round the mantissa to the desired number of
            // significant digits, and to convert the mantissa to a string. Add a small amount to the mantissa before
            // rounding is needed to workaround internal rounding errors due to the limited precision very close to
            // the rounding threshold. The number to be added to the mantissa before rounding needs to depend on the
            // mantissa itself, but will be kept below the internal resolution of double-precision numbers to prevent
            // visible modification of the number. In the range from 1 to 2, the internal resolution is 2^-52
            // (provided as Utils.EPSILON, see comments there). In the range from 2 to 4, the resolution is 2*EPSILON;
            // in the range from 4 to 8, the resolution is 4*EPSILON; and in the range from 8 to 10, it is 8*EPSILON.
            // To prevent any visible rounding errors, an amount of 75% of the actual epsilon will be added to the
            // number.
            //
            // Example: Calling (1.2345).toFixed(3) directly would result in '1.234', therefore adding an epsilon
            // value is needed to get the correct result '1.235'.
            //
            info.mant += (info.mant < 2) ? EPSILON_1 : (info.mant < 4) ? EPSILON_2 : (info.mant < 8) ? EPSILON_4 : EPSILON_8;

            // Find the maximum number of digits available for decimal notation.
            //
            // Case 1: The maximum number of characters is less than or equal to the exponent. In this case, the
            // integral part of the number cannot fit into the available space, therefore the number cannot be
            // formatted in decimal notation at all. Example: The number 1.23E+08 (exponent is 8) will be formatted
            // as 123000000 (9 digits). If the maximum length is less than or equal to 8, the number would not fit
            // into decimal notation.
            //
            // Case 2: Exponent is non-negative (the passed absolute number is equal to or greater than 1): The
            // number of available digits are equal to the passed maximum length.
            //
            // Case 3: Exponent is negative (the passed absolute number is inside 0 and 1): Available digits need
            // to be reduced due to leading zeros. Example: The number 1.23E-03 (exponent is -3) will be formatted
            // as '0.00123' (3 zero digits before the significant digits). Therefore, the number of available digits
            // must be reduced by the absolute value of the exponent.
            //
            info.maxDecDigits = (maxLength <= info.exp) ? (maxLength - info.exp) : (info.exp >= 0) ? maxLength : (maxLength + info.exp);

            // If the exponent increased by one is still less than the maximum number of characters, the formatted
            // number includes the decimal separator, thus the number of available digits needs to be reduced by
            // one. For negative exponents (case 3), this applies always.
            //
            if ((info.maxDecDigits > 0) && (info.exp + 1 < maxLength)) {
                info.maxDecDigits -= 1;
            }

            // The number of digits occupied by the absolute value of the exponent.
            //
            // Interestingly, the V8 engine is the slowest to calculate the expression
            //     Math.floor(Math.log(Math.abs(exp)) / Math.LN10) + 1
            //
            // Measured times for one evaluation of this expression on the same machine, as average of 1 million
            // calls with random integers between -500 and 500:
            // - V8 (Chrome 45): 238ns
            // - Chakra (Edge 20): 63ns
            // - SpiderMonkey (Firefox 41): 4ns (!)
            //
            info.expLength = Math.max(info.minExpLen, ((info.exp === 0) ? 0 : Math.floor(Math.log(Math.abs(info.exp)) / Math.LN10)) + 1);

            // Find the maximum number of digits available for scientific notation.
            //
            // The passed maximum length will be reduced by the number of characters used by the complete exponent
            // (including the 'E' character, and the following +/- sign character). If there are at least two
            // characters available, one of them is needed for the decimal separator.
            //
            info.maxSciDigits = maxLength - info.expLength - 2;
            if (info.maxSciDigits > 1) { info.maxSciDigits -= 1; }

            // Return the resulting properties.
            return info;
        }

        /**
         * Returns the passed formatted absolute number with the correct sign;
         * or returns null, if the string is too long to fit into the maximum
         * length (may happen when the conversion has rounded up a number to
         * the next power of 10).
         *
         * Example: The number 99.5 needs to be formatted to a maximum length
         * of two characters. The code that decides whether a number fits runs
         * before rounding, therefore the number is considered to be valid. It
         * will be rounded to the next number with two significant digits which
         * results in 100. Finally, this result is too large for the maximum
         * length, and this method has to return null.
         *
         * @param {Object} info
         *  The info structure for a floating-point number, as returned by the
         *  private method Formatter.getNumberInfo().
         *
         * @param {String} mant
         *  The resulting formatted absolute mantissa for scientific notation,
         *  or the entire formatted absolute number for decimal notation.
         *
         * @param {String} [exp]
         *  The resulting formatted exponent with sign for scientific notation.
         *  MUST be omitted for decimal notation.
         *
         * @returns {Object|Null}
         *  A result object with the finalized formatted number with correct
         *  sign, if the passed string fits into the maximum length; otherwise
         *  null. The result object will contain the following properties:
         *  - {String} text
         *      The finalized formatted number with correct sign, in either
         *      decimal, or scientific notation.
         *  - {String} mant
         *      The mantissa contained in the finalized formatted number for
         *      resulting scientific notation, or the entire formatted number
         *      for decimal notation, with correct sign. Example: If the
         *      formatted number will be '1.2e+30', this property will contain
         *      '-1.2'.
         *  - {String|Null} exp
         *      The exponent contained in the finalized formatted number with
         *      correct sign for resulting scientific notation, or null for
         *      decimal notation. Example: If the formatted number will be
         *      '1.2e+30', this property will contain '+30'.
         */
        function finalizeStandard(info, mant, exp) {

            // build the resulting formatted number (without sign for now)
            var text = mant;
            if (exp) { text += 'E' + exp; }

            // return null, if the formatted absolute (!) value does not fit into the maximum length
            if (info.maxLength < text.length) { return null; }

            // add the leading sign to the mantissa, and the final text
            if (info.number < 0) {
                mant = '-' + mant;
                text = '-' + text;
            }

            // return a result descriptor with the final properties
            return { text: text, mant: mant, exp: exp || null };
        }

        /**
         * Formats a number for the standard number format, according to the
         * passed settings.
         *
         * @param {Object} info
         *  The info structure for a floating-point number, as returned by the
         *  private method Formatter.getNumberInfo().
         *
         * @param {Function} finalizer
         *  A callback function used to convert the formatted absolute value of
         *  the number to the result object that will be returned by this
         *  method. Must support the function signature of the private method
         *  Formatter.finalizeStandard().
         *
         * @returns {Object|Null}
         *  A result object with the finalized formatted number; or null, if
         *  the number could not be formatted successfully. See description of
         *  the method finalizeStandard() for details.
         */
        function formatStandard(info, finalizer) {

            // Create local copies of the properties for temporary modifications below.
            var mant = info.mant;
            var exp = info.exp;
            var minExpLen = info.minExpLen;
            var decDigits = info.maxDecDigits;
            var sciDigits = info.maxSciDigits;
            var expLength = info.expLength;

            // Special edge case, if the number cannot be formatted according to the passed properties: If the
            // mantissa is at least 5, and the exponent is a negative power of 10, the number can be rounded so
            // that the length of the exponent shrinks by one character. The available scientific digits must be
            // exactly equal to zero which indicates that reducing the exponent by one digit makes space for the
            // required significant digit of the mantissa.
            //
            // Example: The formatted number '9E-100' would take 6 characters, but can be rounded to '1E-99' which
            // takes only 5 characters. The workaround is not needed for the exponent -1. Such numbers (e.g. 0.5)
            // will be formatted with decimal notation (see next edge case below).
            //
            if ((decDigits <= 0) && (sciDigits === 0) && (mant >= 5) && (((exp === -10) && (minExpLen <= 1)) || ((exp === -100) && (minExpLen <= 2)))) {
                mant = 1;
                exp += 1;
                sciDigits = 1;
                expLength -= 1;
            }

            // Special edge case: Allow numbers less than 1 to be formatted as single-digit integer ('0' or '1').
            //
            // Example: The number 0.00123 cannot be formatted into one or two characters with at least one
            // significant digit, but it can be formatted as '0'. Similarly, the number 0.987 does not fit, but
            // can be formatted as '1'.
            //
            if ((decDigits <= 0) && (sciDigits <= 0) && (exp < 0)) {
                mant = Math.round(Math.abs(info.number));
                exp = 0;
                decDigits = 1;
            }

            // Fast exit, if the number is exactly or very close to zero (do not try to format denormalized numbers).
            if (mant === 0) { return finalizer(info, '0'); }

            // Return null, if the number cannot be formatted according to the available digits.
            var maxDigits = Math.max(decDigits, sciDigits);
            if (maxDigits <= 0) { return null; }

            // Generate a rounded string representation of the mantissa to detect the number of significant digits.
            //
            // Divide the mantissa by 10 to move all significant digits into the fractional part. The number of digits
            // passed to Number.toFixed() needs to be restricted to 14 to prevent further rounding errors near the
            // precision limit of IEEE 754 double-precision numbers. In the end, this results in a string looking like
            // '0.nnnn000'.
            //
            var sigDigitsStr = (mant / 10).toFixed(Math.min(14, maxDigits));

            // If the original mantissa is very close to 10, the resulting string may become '1.00' due to rounding
            // of the mantissa. In this case, the exponent of the resulting formatted number will be increased by 1.
            //
            // Example: The number 9999.99 (mantissa is 9.99999, exponent is 3), rounded to two significant digits,
            // will result in the string '1.00'. Therefore, the new mantissa becomes 1, and the exponent becomes 4;
            // the number will be shown as '10000' or '1E+04'.
            //
            // Otherwise, simply remove the leading zero digit, the decimal point, and the trailing insignificant
            // zero digits from the string, to convert the string '0.nnnn000' to 'nnnn' where neither the first nor
            // the last digit is a zero.
            //
            if (sigDigitsStr[0] === '1') {
                exp += 1;
                sigDigitsStr = '1';
            } else {
                sigDigitsStr = sigDigitsStr.replace(/^0?\./, '').replace(/0+$/, '');
            }

            // Now, the generated string contains significant digits only. Check if the decimal notation can be used
            // to format the number. It will be preferred, if it can take all significant digits, or if it can take
            // at least as many digits as the scientific notation.
            //
            if ((decDigits >= sciDigits) || (sigDigitsStr.length <= decDigits)) {

                // Add trailing zeros to integral numbers. Trailing zeros are needed, if the number of significant
                // digits is less than or equal to the original (positive) exponent.  Example: The original number
                // 123400 (mantissa 1.234, exponent 5) results in the significant digits '1234' which is shorter than
                // the exponent. The resulting formatted number will contain 6 digits (one more than the exponent),
                // therefore two zero digits need to be appended.
                //
                if (sigDigitsStr.length <= exp) {
                    return finalizer(info, (sigDigitsStr + ZEROS).substr(0, exp + 1));
                }

                // Return the significant digits as they are, if their length matches the exponent increased by one.
                // Example: The number 1234 has exponent 3, and results in the significant digits string '1234' with
                // a length of 4 characters.
                //
                if (sigDigitsStr.length === exp + 1) {
                    return finalizer(info, sigDigitsStr);
                }

                // Insert the decimal separator into the significant digits, if the exponent is not negative.
                // Example: For the number 12.34 (exponent 1), the decimal separator needs to be inserted into the
                // significant digits string '1234'.
                //
                if (exp >= 0) {
                    return finalizer(info, sigDigitsStr.substr(0, exp + 1) + DEC + sigDigitsStr.substr(exp + 1));
                }

                // Insert a leading zero character and the decimal separator for an exponent of -1. Example: For the
                // number 0.1234 (exponent is -1), the string '0.' needs to be prepended to the significant digits
                // string '1234'.
                //
                if (exp === -1) {
                    return finalizer(info, '0' + DEC + sigDigitsStr);
                }

                // Exponent is less than -1: Insert zeros between the decimal separator and the significant digits.
                // Example: For the number 0.001234 (exponent -3), two zero digits need to be inserted after the
                // decimal separator.
                //
                return finalizer(info, '0' + DEC + ZEROS.substr(exp + 1) + sigDigitsStr);
            }

            // Generate the string representation of the absolute value of the exponent, and update the required
            // length of the exponent. The exponent length may increase e.g. when rounding the number 9.9E+99 (two
            // digits for the exponent) to 1E+100 (three digits for the exponent).
            //
            var expStr = String(Math.abs(exp));
            expLength = Math.max(minExpLen, expStr.length);

            // Create the scientific notation for the number. The mantissa will always be formatted with one integral
            // digit, followed by the significant digits in the fractional part.
            //
            if (sigDigitsStr.length > 1) { sigDigitsStr = sigDigitsStr[0] + DEC + sigDigitsStr.substr(1); }
            return finalizer(info, sigDigitsStr, ((exp < 0) ? '-' : '+') + ('00000000' + expStr).substr(-expLength));
        }

        /**
         * Returns the passed formatted absolute number with the correct sign;
         * or returns null, if the string is too long to fit into the maximum
         * length or pixel width. See method finalizeStandard() for more
         * details.
         *
         * @param {Object} info
         *  The info structure for a floating-point number, as returned by the
         *  private method Formatter.getNumberInfo(), with the additional
         *  properties 'font' and 'maxWidth'.
         *
         * @param {String} mant
         *  The resulting formatted absolute mantissa for scientific notation,
         *  or the entire formatted absolute number for decimal notation.
         *
         * @param {String} [exp]
         *  The resulting formatted exponent with sign for scientific notation.
         *  MUST be omitted for decimal notation.
         *
         * @returns {Object|Null}
         *  A result object with the finalized formatted number; or null, if
         *  the number could not be formatted successfully. See method
         *  Formatter.formatStandardNumberToWidth() for details.
         */
        function finalizeStandardToWidth(info, mant, exp) {

            // use the formatter of the base class to get a result with the current settings
            var result = finalizeStandard(info, mant, exp);
            if (!result) { return null; }

            // add the text portion information for richly formatted numbers
            result.portions = null;

            // determine and verify the pixel width of the complete text
            result.width = result.portions ? result.portions.reduce(function (width, portion) { return width + portion.width; }, 0) : info.font.getTextWidth(result.text);
            return (result.width <= info.maxWidth) ? result : null;
        }

        /**
         * Formats a number for the standard number format, according to the
         * passed settings including a maximum pixel width.
         *
         * @param {Object} info
         *  The info structure for a floating-point number, as returned by the
         *  private method Formatter.getNumberInfo(), with the additional
         *  properties 'font' and 'maxWidth'.
         *
         * @returns {Object|Null}
         *  A result object with the finalized formatted number; or null, if
         *  the number could not be formatted successfully. See method
         *  Formatter.formatStandardNumberToWidth() for details.
         */
        function formatStandardToWidth(info) {

            // Fast exit, if the number is exactly or very close to zero (do not try to format denormalized numbers).
            if (info.mant === 0) { return finalizeStandardToWidth(info, '0'); }

            // Get font measures for digits and other characters.
            var digitWidths = info.font.getDigitWidths();
            var minDigitWidth = Math.max(1, digitWidths.minWidth);
            var maxDigitWidth = Math.max(1, digitWidths.maxWidth);
            var decSepWidth = info.font.getTextWidth(DEC);

            // Get the maximum pixel width available for the formatted absolute value.
            var maxAbsWidth = info.maxWidth;
            if (info.number < 0) { maxAbsWidth -= info.font.getTextWidth('-'); }

            // Returns the number of digits fitting into the passed pixel width. Use the maximum width of a digit to
            // determine the available number of digits to be sure to not exceed the total pixel width.
            //
            function getDigitCount(availWidth, maxDigits) {
                return Math.min(maxDigits, Math.floor(availWidth / maxDigitWidth));
            }

            // Make a first guess about how many significant digits may fit for decimal notation, according to the
            // maximum pixel width. If the exponent is negative, the appropriate number of leading zero digits needs
            // to be removed from the available pixel width. If the number of digits indicates that the decimal
            // separator needs to be added, reduce the available space by its width, and recalculate the number of
            // available digits.
            //
            var maxDecDigits = info.maxDecDigits;
            var availDecWidth = maxAbsWidth + Math.min(0, info.exp) * Math.max(1, digitWidths[0]);
            info.maxDecDigits = getDigitCount(availDecWidth, maxDecDigits);
            if (info.exp + 1 < info.maxDecDigits) {
                info.maxDecDigits = getDigitCount(availDecWidth - decSepWidth, maxDecDigits);
            } else if (info.maxDecDigits <= info.exp) {
                info.maxDecDigits -= info.exp;
            }

            // Make a first guess about how many significant digits may fit for scientific notation, according to the
            // maximum pixel width. Remove the complete exponent from the available pixel width. If the number of
            // digits is greater than one, the decimal separator needs to be added. Reduce the available space by its
            // width in this case, and recalculate the number of available digits.
            //
            var maxSciDigits = info.maxSciDigits;
            var availSciWidth = maxAbsWidth - info.font.getTextWidth((info.exp < 0) ? 'E-' : 'E+') - (info.expLength * maxDigitWidth);
            info.maxSciDigits = getDigitCount(availSciWidth, maxSciDigits);
            if (info.maxSciDigits > 1) {
                info.maxSciDigits = getDigitCount(availSciWidth - decSepWidth, maxSciDigits);
            }

            // Try to format the number with the (possibly reduced) digit counts calculated for the pixel width. The
            // base class method StandardHelper.format() calls the method finalize() of this instance which results in
            // adding the property 'width' into the result object.
            //
            // If the number cannot be formatted into the available width (using the maximum digit width), or if the
            // widths of all digits of the used font are equal (as is the case for most fonts), do not try further.
            //
            var result = formatStandard(info, finalizeStandardToWidth);
            if (!result || (minDigitWidth === maxDigitWidth)) { return result; }

            // If the number has been formatted successfully, and there is still enough room for another digit, and
            // the number of available digits has been reduced before due to the pixel width, try again with one
            // more digit per step until the resulting text does not fit anymore, or no more digits can be added.
            //
            while (info.maxWidth - result.width >= minDigitWidth) {

                // Increase the appropriate digit count of decimal or scientific notation according to the current
                // result (at least by one, at most by the number of the widest digit that can fit into the remaining
                // space, to find the largest possible representation of the number). Exit the loop, if the number
                // cannot be enlarged anymore due to the maximum string length.
                //
                var incDigits = Math.max(1, Math.floor((info.maxWidth - result.width) / maxDigitWidth));
                if (result.exp) {
                    if (info.maxSciDigits === maxSciDigits) { break; }
                    info.maxSciDigits = Math.min(maxSciDigits, info.maxSciDigits + incDigits);
                } else {
                    if (info.maxDecDigits === maxDecDigits) { break; }
                    info.maxDecDigits = Math.min(maxDecDigits, info.maxDecDigits + incDigits);
                }

                // Format the number again with the new settings. Check the pixel width of the new formatted version
                // of the number. If the new number still fits into the width, store the new better result, and
                // continue searching for a better version.
                //
                var newResult = formatStandard(info, finalizeStandardToWidth);
                if (!newResult || (newResult.width > info.maxWidth)) { break; }
                result = newResult;
            }

            return result;
        }

        // public methods -----------------------------------------------------

        /**
         * Changes the configuration of this formatter.
         *
         * @param {Object} [options]
         *  All configuration properties to be changed. See constructor
         *  parameter 'initOptions' for details.
         *
         * @returns {Formatter}
         *  A reference to this instance.
         */
        this.configure = function (options) {

            // change the separator characters
            DEC = Utils.getStringOption(options, 'dec', DEC);
            GROUP = Utils.getStringOption(options, 'group', GROUP);

            // change the null date
            if (options && ('nullDate' in options)) {
                if (typeof options.nullDate === 'number') {
                    nullDate = options.nullDate;
                } else if (options.nullDate instanceof Date) {
                    nullDate = options.nullDate.getTime();
                }
            }

            // change the 1900-leap-year-bug flag
            leapYearBug = Utils.getBooleanOption(options, 'leapYearBug', leapYearBug);

            // change the century threshold
            if (options && _.isNumber(options.centuryThreshold)) {
                centuryThreshold = Utils.minMax(options.centuryThreshold, 0, 100);
            }

            // change the flag whether to support negative dates
            negativeDates = Utils.getBooleanOption(options, 'negativeDates', negativeDates);

            // number of characters for a 'General' format token
            generalLength = Utils.getIntegerOption(options, 'generalLength', generalLength, 5);

            return this;
        };

        /**
         * Returns whether the passed date object is valid to be processed by
         * the formatter. The year must be in the range from 0 to 9999, and
         * must not be located before the null date (see constructor option
         * 'nullDate' for details), unless such dates are supported by this
         * formatter (see constructor option 'negativeDates' for details).
         *
         * @param {Date} date
         *  The UTC date to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed date object is valid.
         */
        this.isValidDate = function (date) {

            // milliseconds from native null date (1970-01-01)
            var time = date.getTime();
            // the full (4-digit) year
            var year = date.getUTCFullYear();

            // check the year, check for negative dates
            return isFinite(time) && (negativeDates || (nullDate <= time)) && (year >= 0) && (year <= 9999);
        };

        /**
         * Returns a date object representing the passed floating-point number,
         * using the current null date of this formatter.
         *
         * @param {Number} number
         *  A floating-point number. The integral part represents the number of
         *  days elapsed since the null date of this formatter, the fractional
         *  part represents the time.
         *
         * @returns {Date|Null}
         *  An UTC date object representing the passed number; or null, if the
         *  passed number cannot be converted to a valid date (e.g. too large).
         */
        this.convertNumberToDate = function (number) {

            // get correct date/time for current null date
            var date = new Date(nullDate + Math.round(number * DateUtils.MSEC_PER_DAY));

            // return the date object if it is valid
            return this.isValidDate(date) ? date : null;
        };

        /**
         * Returns the floating-point number representing the passed date,
         * using the current null date of this formatter.
         *
         * @param {Date} date
         *  An UTC date object.
         *
         * @returns {Number|Null}
         *  The floating-point number representing the passed date; or null, if
         *  the passed date cannot be converted to a number.
         */
        this.convertDateToNumber = function (date) {

            // error, if date is invalid, or the year is not in the range 0 to 9999
            if (!this.isValidDate(date)) { return null; }

            // return the floating-point number
            return (date.getTime() - nullDate) / DateUtils.MSEC_PER_DAY;
        };

        /**
         * Converts a two-digit year to a four-digit year, according to the
         * century threshold value configured for this formatter.
         *
         * @param {Number} year
         *  A year number. MUST be non-negative. Values less than 100 will be
         *  converted to a year in the current or preceding century.
         *
         * @returns {Number}
         *  The resulting year number.
         */
        this.expandYear = function (year) {
            return DateUtils.expandYear(year, centuryThreshold);
        };

        /**
         * Tries to parse a date from the leading part of the passed string.
         *
         * @param {String} text
         *  The text potentially starting with a date.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      If set to true, the entire passed string must represent a valid
         *      date. No remaining 'garbage' text will be accepted.
         *
         * @returns {Object|Null}
         *  A result descriptor, if the passed string starts with a valid date
         *  (according to the passed options); otherwise null. See static
         *  method Parser.parseLeadingDate() for details.
         */
        this.parseLeadingDate = function (text, options) {

            // parse the passed text with the internal configuration
            var result = Parser.parseLeadingDate(text, {
                dec: DEC,
                centuryThreshold: centuryThreshold,
                complete: Utils.getBooleanOption(options, 'complete', false)
            });

            // check the resulting date according to the internal configuration (null date etc.)
            return (result && this.isValidDate(result.date)) ? result : null;
        };

        /**
         * Tries to parse a time from the leading part of the passed string.
         *
         * @param {String} text
         *  The text potentially starting with a time.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.complete=false]
         *      If set to true, the entire passed string must represent a valid
         *      time. No remaining 'garbage' text will be accepted.
         *
         * @returns {Object|Null}
         *  A result descriptor, if the passed string starts with a valid time
         *  (according to the passed options); otherwise null. See static
         *  method Parser.parseLeadingTime() for details.
         */
        this.parseLeadingTime = function (text, options) {

            // parse the passed text with the internal configuration
            return Parser.parseLeadingTime(text, {
                dec: DEC,
                multipleDays: true,
                milliseconds: true,
                complete: Utils.getBooleanOption(options, 'complete', false)
            });
        };

        /**
         * Converts the passed decimal number to a roman number string.
         *
         * @param {Number} number
         *  The number to be converted.
         *
         * @param {Number} [depth=0]
         *  The degree of simplification made for the decimal digits 4 and 9.
         *  MUST be a number in the range 0 to 4.
         *
         * @returns {String|Null}
         *  The roman number representation of the passed number (an empty
         *  string for the number zero); or null, if the passed number is not
         *  in the supported range.
         */
        this.formatRoman = function (number, depth) {

            number = Math.floor(number);
            if ((number < 0) || (number > 3999)) { return null; }

            depth = depth || 0;

            var result = '';
            for (var index = 0; (index < ROMAN_DIGITS.length) && (number > 0); index += 2) {

                // current decimal digit of the number to be processed
                var digit = Math.floor(number / ROMAN_DIGITS[index].value);

                // Special handling for digits 4 and 9: Put a smaller roman digit before a larger one,
                // for example IV which means 5 - 1 = 4; or XM which means 1000 - 10 = 990.
                if ((digit % 5) === 4) {

                    // get the nerest larger roman digit (e.g., 100 for 99, or 500 for 490)
                    var prevDigit = ROMAN_DIGITS[index - ((digit === 4) ? 1 : 2)];
                    // array index to the smaller roman digit (e.g., 50 for 99, or 100 for 490)
                    var nextIndex = index;
                    // try to find a smaller roman digit according to the passed depth
                    var remaining = Math.min(depth, ROMAN_DIGITS.length - index - 1);

                    // try to find a smaller roman digit so that the difference does not exceed the
                    // remaining number; but stop searching if the passed depth has been reached
                    while (remaining > 0) {
                        remaining -= 1;
                        if (number < prevDigit.value - ROMAN_DIGITS[nextIndex + 1].value) { break; }
                        nextIndex += 1;
                    }

                    // add the combination of roman digits to the result, and reduce the number
                    var nextDigit = ROMAN_DIGITS[nextIndex];
                    result += nextDigit.char + prevDigit.char;
                    number += nextDigit.value - prevDigit.value;

                } else {

                    // decimal digit is in range 5 to 8: add the roman digit for 5, 50, or 500
                    if (digit > 4) { result += ROMAN_DIGITS[index - 1].char; }

                    // repeat the digit for the current power of 10, and reduce the number
                    result += Utils.repeatString(ROMAN_DIGITS[index].char, digit % 5);
                    number %= ROMAN_DIGITS[index].value;
                }
            }
            return result;
        };

        /**
         * Tries to parse the passed text as a roman number.
         *
         * @param {String} text
         *  The text to be parsed as a roman number (case-insensitive).
         *
         * @returns {Number|Null}
         *  The decimal value of the roman number; or null, if the passed text
         *  contains any invalid characters.
         */
        this.parseRoman = function (text) {

            // the intermediate result
            var result = 0;
            // the largest roman digit found so far
            var max = 1;

            // process all roman digits in reversed order
            Utils.iterateArray(text.toUpperCase(), function (digit) {

                // convert digit to the decimal number, exit on invalid characters
                var value = ROMAN_DIGIT_MAP[digit];
                if (!value) {
                    result = null;
                    return Utils.BREAK;
                }

                // if the digit is less than the maximum digit found so far, subtract it from the result
                result += (value < max) ? -value : value;
                max = Math.max(max, value);

            }, { reverse: true });

            return result;
        };

        /**
         * Returns the display string of the passed number formatted with the
         * standard number format, fitting into the specified text length.
         *
         * @param {Number} number
         *  The number whose string representation in standard number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation, but without a leading minus sign.
         *  MUST be positive.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.expLength=2]
         *      The minimum number of characters for the exponent in scientific
         *      notation. If the resulting exponent is shorter, it will be
         *      filled with leading zero digits. MUST be a positive integer;
         *      must be less than or equal to 9.
         *
         * @returns {String|Null}
         *  The resulting display text of the number, formatted with the
         *  standard number format. If the passed maximum length is too small
         *  to take the passed number in either decimal or scientific notation,
         *  the null value will be returned instead.
         */
        this.formatStandardNumber = function (number, maxLength, options) {

            // build a descriptor for the passed number, immediately return null if that fails
            var numberInfo = getNumberInfo(number, maxLength, options);
            if (!numberInfo) { return null; }

            // format the number according to the settings passed to this method
            var result = formatStandard(numberInfo, finalizeStandard);
            return result ? result.text : null;
        };

        /**
         * Returns the display string of the passed number formatted with the
         * standard number format, fitting into the specified number of
         * characters, as well as into the specified pixel width.
         *
         * @param {Number} number
         *  The number whose string representation in standard number format
         *  will be returned.
         *
         * @param {Number} maxLength
         *  The maximum number of characters allowed for the absolute part of
         *  the passed number, including the decimal separator and the complete
         *  exponent in scientific notation.
         *
         * @param {Font} font
         *  The font settings influencing the text width.
         *
         * @param {Number} maxWidth
         *  The maximum width of the formatted number, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are also supported
         *  by the method Formatter.formatStandardNumber().
         *
         * @returns {Object|Null}
         *  A result object with the finalized formatted number with correct
         *  sign and maximum pixel width, if it has been formatted successfully
         *  to the specified pixel width; otherwise null. The result object
         *  will contain the following properties:
         *  - {String} text
         *      The finalized formatted number with correct sign.
         *  - {String} mant
         *      The mantissa contained in the finalized formatted number for
         *      resulting scientific notation, or the entire formatted number
         *      for decimal notation, with correct sign. Example: If the
         *      formatted number will be '1.2e+30', this property will contain
         *      '-1.2'.
         *  - {String|Null} exp
         *      The exponent contained in the finalized formatted number with
         *      correct sign for resulting scientific notation, or null for
         *      decimal notation. Example: If the formatted number will be
         *      '1.2e+30', this property will contain '+30'.
         *  - {Number} width
         *      The width of the formatted number with sign, in pixels.
         *  - {Array<Object>|Null} portions
         *      If the formatted number cannot be displayed as plain text, an
         *      array of descriptor objects for the single text portions of the
         *      formatted number. Each descriptor contains the following
         *      properties:
         *      - {String} portion.text
         *          The text contents of the text portion.
         *      - {Font} portion.font
         *          The font settings to be used for this text portion. May
         *          differ from the font passed to this method, e.g. for
         *          superscripted or subscripted text.
         *      - {Number} portion.width
         *          The width of this text portion, in pixels.
         */
        this.formatStandardNumberToWidth = function (number, maxLength, font, maxWidth, options) {

            // build a descriptor for the passed number, immediately return null if that fails
            var numberInfo = getNumberInfo(number, maxLength, options);
            if (!numberInfo) { return null; }
            numberInfo.font = font;
            numberInfo.maxWidth = maxWidth;

            // format the number according to the settings passed to this method
            return formatStandardToWidth(numberInfo);
        };

        /**
         * Formats the passed value according to the format code.
         *
         * @param {ParsedFormat} parsedFormat
         *  The parsed format code that will be used to format the passed value
         *  as string.
         *
         * @param {Number|String} value
         *  The value to be formatted.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.generalLength]
         *      Maximum number of characters used to format a number for the
         *      'General' format code token. If omitted, the default length
         *      specified in the constructor option will be used.
         *
         * @returns {String|Null}
         *  The formatted value; or null, if the passed format code is invalid,
         *  or if the value cannot be formatted with the format code (e.g., a
         *  number is too large to be formatted as date/time).
         */
        this.formatValue = function (parsedFormat, value, options) {

            /*
             * fills up the "text" number with whitespace to the length of the assigned code
             * iterate the code and the text and does:
             *      code: ? text: '0' -> ' '
             *      code: # text: '0' -> delete
             *      code: 0 text: ' ' -> '0'
             *      code: 0-9 -> break iteration
             *
             * @param {Boolean} [rightToLeft=false]
             *      if true the text is iterated from right to left (only used in significant part of the number)
             */
            function setPlaceHolders(text, code, rightToLeft) {

                if (_.isNumber(text)) {
                    text = String(text);
                }
                if (text.length > code.length) {
                    return text;
                }

                var i, counter, end;
                if (rightToLeft) {
                    text += Utils.repeatString(' ', code.length - text.length);
                    end = -1;
                    counter = -1;
                    i = text.length - 1;
                } else {
                    text = Utils.repeatString(' ', code.length - text.length) + text;
                    i = 0;
                    counter = 1;
                    end = text.length;
                }

                var arrT = [];
                for (var j = 0; j < text.length; j++) {
                    arrT.push(text[j]);
                }

                for (; i !== end; i += counter) {
                    var actT = arrT[i];
                    var actC = code[i];

                    if (actT !== '0' && actT !== ' ') {
                        break;
                    }
                    if (actC === '0') {
                        arrT[i] = '0';
                    } else {
                        if (actC === '?') {
                            arrT[i] = ' ';
                        } else if (actC === '#') {
                            arrT[i] = '';
                        }
                    }
                }
                return arrT.join('');
            }

            // Reduce a fraction by finding the Greatest Common Divisor and dividing by it.
            // see http://stackoverflow.com/questions/4652468/is-there-a-javascript-function-that-reduces-a-fraction
            function cancelFract(frac) {
                var num = frac.num;
                var preciseNum = frac.preciseNum;
                var deno = frac.deno;

                var gcd = function gcd(a, b) {
                    return b ? gcd(b, a % b) : a;
                };
                gcd = gcd(num, deno);

                frac.num = num / gcd;
                frac.preciseNum = preciseNum / gcd;
                frac.deno = deno / gcd;
            }

            // format of the integer part of the number
            function formatNumber() {
                var text = null;
                if (section.flags.float && section.flags.floatCode) {
                    var prepared = prepareSign(section.flags.floatCode);
                    if (prepared.round === 1) {
                        text = String(Math.ceil(number));
                    } else {
                        text = String(Math.floor(number));
                    }
                } else if (section.flags.fraction && section.flags.fractionCode) {
                    var closest = prepareFraction(section.flags.fractionCode);
                    if (!closest.num || !closest.fraction) {
                        return String(Math.round(number));
                    } else {
                        text = String(Math.floor(number));
                    }
                } else {
                    text = String(Math.round(number));
                }

                if (section.flags.group) {
                    var replacer = '$1' + GROUP + '$2';
                    var rgx = /(\d+)(\d{3})/;
                    while (rgx.test(text)) {
                        text = text.replace(rgx, replacer);
                    }
                }

                text = setPlaceHolders(text, section.flags.ungroupText);
                return text;
            }

            //prepares the format of the fraction part numerator & denominator
            //at the moment we only support until 3 signs for the denominator
            function prepareFraction(code) {
                var parts = code.split('/');

                var denomLen = parts[1].length;

                // reduce the denominator if it contains more than 3 signs
                if (denomLen > 3) {
                    parts[1] = parts[1].substring(0, 3);
                    denomLen = 3;
                }

                var fractStart = Math.pow(10, denomLen - 1);
                var fractEnd = fractStart * 10 - 1;

                var frac = section.isProperFraction ? number - Math.floor(number) : number;

                var closest = null;
                var near = Number.POSITIVE_INFINITY;
                var preciseNum = 0;
                var num = 0;

                if (frac === 0) {
                    closest = { num: 0, deno: fractEnd };
                } else if (isFinite(parts[1])) {
                    var deno = parseInt(parts[1], 10);
                    preciseNum = frac * deno;
                    // if preciseNum is smaller than "1" fraction gets ignored like "0"
                    num = preciseNum < 1 ? 0 : Math.round(preciseNum);
                    closest = { num: num, deno: deno, preciseNum: preciseNum };
                } else {
                    for (var i = fractStart; i <= fractEnd; i++) {
                        preciseNum = frac * i;
                        // if preciseNum is smaller than "1" fraction gets ignored like "0"
                        num = preciseNum < 1 ? 0 : Math.round(preciseNum);
                        if (num > i) {
                            break;
                        }
                        var compare = num / i;
                        var dif = Math.abs(frac - compare);
                        if (dif < near) {
                            closest = { num: num, deno: i, preciseNum: preciseNum };
                            near = dif;
                            if (dif === 0) { break; }
                        }
                    }
                }

                if (closest) {
                    if (parts[1].indexOf('?') >= 0) {
                        if (frac === 0) {
                            closest.num = 0;
                            closest.preciseNum = 0;
                        }
                        cancelFract(closest);
                    }

                    // preciseNum was not that preciseNum anymore after cancelFract(),
                    // so it is better to use a precision-rounded version of that
                    // fix for Bug 41434
                    if (closest.deno > 1 && Utils.round(closest.preciseNum, 0.00001) === closest.deno) {
                        closest.num = closest.deno;
                    }
                    closest.fraction = true;
                    closest.parts = parts;

                    if (closest.num === 0 || closest.num === closest.deno) {
                        closest.num = 0;
                        closest.deno = 0;
                    }
                } else {
                    closest = { fraction: false, num: frac };
                }
                return closest;
            }

            //formats of the fraction part numerator & denominator
            //at the moment we only support until 3 signs for the denominator
            function formatFraction(token) {
                var closest = prepareFraction(token.text);

                if (closest.fraction) {
                    var parts = closest.parts;
                    if (!closest.num) {
                        //zero & placeholders
                        return Utils.repeatString(' ', parts[0].length + parts[1].length + 1);
                    }
                    closest.num = setPlaceHolders(closest.num, parts[0]);
                    var deno = setPlaceHolders(closest.deno, parts[1]);

                    var denoWhiteSpace = deno.match(/^(\s*)(.*)/);
                    deno = denoWhiteSpace[2] + denoWhiteSpace[1];

                    closest.deno = deno;
                    return closest.num + '/' + closest.deno;
                }
                return DEC + closest.num;
            }

            //prepares the format of the significant part of the number incl. decimal-symbol
            function prepareSign(code) {
                var format = code.substring(1, code.length);
                var digits = format.length;

                var frac = number - parseInt(number, 10);
                var round = Utils.round(frac, Math.pow(10, -digits));
                return { round: round, format: format };
            }

            //formats the significant part of the number incl. decimal-symbol
            function formatSign(token) {
                var prepared = prepareSign(token.text);
                var format = prepared.format;
                var round = prepared.round;

                var result = String(round).substring(2);
                var res = setPlaceHolders(result, format, true);
                if (res.length === 0) { return ''; }
                if (res[0] === ' ') { return ' ' + res; }
                return DEC + res;
            }

            function formatScience(token) {
                var res = setPlaceHolders(Math.abs(exp), token.text);
                return EXP + ((exp < 0) ? '-' : '+') + res;
            }

            // the section of the format code to be used for the passed value
            var section = parsedFormat.getSection(value);
            // the absolute value of the passed number, or zero for strings
            var number = _.isNumber(value) ? Math.abs(value) : 0;
            // the passed number, converted to a date
            var date = _.isNumber(value) ? this.convertNumberToDate(value) : null;
            // the passed string, or empty string for numbers
            var string = _.isString(value) ? value : '';
            // number of characters for a 'General' format token
            var genLength = Utils.getIntegerOption(options, 'generalLength', generalLength, 5);
            // the formatted string representation
            var formatted = '';

            // bail out if the passed format code is invalid, or cannot be used to
            // format the value, or if the passed number is not finite
            if (!section || !isFinite(number)) { return null; }

            // special behaviour for a few section LCIDs
            switch (section.lcid) {
                case 0xf400: section = sysTimeFormat.numberSections[0]; break;
                case 0xf800: section = sysDateFormat.numberSections[0]; break;
            }

            // bail out if the passed format code contains date/time tokens,
            // but no valid date is available
            if ((section.flags.date || section.flags.time) && !date) { return null; }

            // prepare percentage formats (number may become infinite)
            if (section.flags.percent > 0) {
                number *= Math.pow(100, section.flags.percent);
                if (!isFinite(number)) { return null; }
            }

            // prepare scientific formats
            var exp = 0;
            if (section.flags.scientific && (number !== 0)) {
                var normalized = Utils.normalizeNumber(number);
                exp = normalized.exp;
                number = normalized.mant;
            }

            // initialize token formatters (called with token as context)
            var tokenFormatters = {
                general: function () { return self.formatStandardNumber(number, genLength); },
                nr: formatNumber,
                frac: formatFraction,
                sign: formatSign,
                scien: formatScience,
                year: function (token) { return formatYear(date, token.length); },
                month: function (token) { return formatMonth(date, token.length); },
                day: function (token) { return formatDay(date, token.length); },
                hour: function (token) { return formatInteger(section.flags.ampm ? (((date.getUTCHours() + 11) % 12) + 1) : date.getUTCHours(), token.length); },
                minute: function (token) { return formatInteger(date.getUTCMinutes(), token.length); },
                second: function (token) { return formatInteger(date.getUTCSeconds(), token.length); },
                hours: function (token) { return formatInteger(Math.floor(number * 24), token.length); },
                minutes: function (token) { return formatInteger(Math.floor(number * 1440), token.length); },
                seconds: function (token) { return formatInteger(Math.round(number * 86400), token.length); },
                ampm: function (token) { return ((date.getUTCHours() < 12) ? 'am' : 'pm').substr(0, token.length); },
                lit: function (token) { return token.text; },
                fill: function () { return ''; }, // bug 36439: ignore fill tokens silently for now
                text: function () { return string; },
                quarter: function (token) { return formatQuarter(date, token.length); },
                weeknumber: function () { return moment.utc(date).week(); }
            };

            var numberValue = null;
            // process all format code tokens
            section.tokens.forEach(function (token, index) {
                if ('rightNumbersCount' in token) {
                    if (numberValue === null) {
                        numberValue = formatNumber(token, index);
                    }

                    if (numberValue.length > token.rightNumbersCount) {
                        formatted += numberValue.slice(0, numberValue.length - token.rightNumbersCount);
                        numberValue = numberValue.slice(-token.rightNumbersCount);
                    }
                } else {
                    if ('fractionText' in token) {
                        formatted += token.fractionText;
                    } else {
                        formatted += tokenFormatters[token.type](token, index);
                    }
                }
            });

            // automatically add a minus sign for code sections covering positive and negative numbers
            if (section.autoMinus && _.isNumber(value) && (value < 0) && !(section.flags.date || section.flags.time)) {
                formatted = '-' + formatted;
            }

            return formatted;
        };

        // initialization -----------------------------------------------------

        // initialize with options apssed to constructor
        this.configure(initOptions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = initOptions = null;
        });

    } // class Formatter

    // exports ================================================================

    return BaseObject.extend({ constructor: Formatter });

});
