/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/dialogs', [
    'io.ox/core/tk/dialogs',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/utils/driveutils',
    'io.ox/office/tk/utils/deferredutils',
    'gettext!io.ox/office/tk/main'
], function (CoreDialogs, Utils, Forms, KeyCodes, DriveUtils, DeferredUtils, gt) {

    'use strict';

    var OK_LABEL = /*#. button label in a OK/Cancel dialog box */ gt('OK');
    var CANCEL_LABEL = /*#. button label in a OK/Cancel dialog box */ gt('Cancel');
    var YES_LABEL = /*#. button label in a Yes/No dialog */ gt('Yes');
    var NO_LABEL = /*#. button label in a Yes/No dialog */ gt('No');

    // static class Dialogs ===================================================

    /**
     * Provides different types of generic dialogs. Collects all standard
     * dialog classes defined in the toolkit in a single map for convenience.
     */
    var Dialogs = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns the root node of the modal dialog that is currently open. In
     * case multiple dialogs are open (stacked), the root node of the top-most
     * dialog will be returned.
     *
     * @returns {jQuery}
     *  The root node of the modal dialog that is currently open, as jQuery
     *  object. If no modal dialog is currently open, an empty jQuery
     *  collection will be returned instead.
     */
    Dialogs.getActiveModalDialogNode = function () {
        return $('.io-ox-dialog-popup').last();
    };

    /**
     * Returns whether a modal dialog is currently open.
     *
     * @returns {Boolean}
     *  Whether a modal dialog is currently open.
     */
    Dialogs.isModalDialogOpen = function () {
        return Dialogs.getActiveModalDialogNode().length > 0;
    };

    /**
     * Returns a dialog width suitable for the current device type, and for the
     * specified desired width.
     *
     * @param {Number} width
     *  The desired dialog width.
     *
     * @returns {Number}
     *  The effective width of the dialog, in pixels. Will not be greater than
     *  95% of the current screen width. On small devices, the dialog width
     *  returned by this method will be fixed to 95% of the screen width,
     *  regardless of the passed width.
     */
    Dialogs.getBestDialogWidth = function (width) {
        var maxWidth = Math.round(window.innerWidth * 0.95);
        return Utils.SMALL_DEVICE ? maxWidth : Math.min(width, maxWidth);
    };

    // class ModalDialog ======================================================

    /**
     * Creates an empty modal dialog.
     *
     * @constructor
     *
     * @extends CoreDialogs.ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [initOptions.title]
     *      The title of the dialog window that will be shown in a larger font.
     *  @param {Number} [initOptions.width=500]
     *      The width of the dialog, in pixels.
     *  @param {String} [initOptions.okLabel=gt('OK')]
     *      The label of the primary button that triggers the default action of
     *      the dialog. If omitted, the button will be created with the
     *      translated default label 'OK'.
     *  @param {String} [initOptions.cancelLabel=gt('Cancel')]
     *      The label of the cancel button that closes the dialog without
     *      performing any action. If omitted, the button will be created with
     *      the translated default label 'Cancel'.
     *  @param {Function} [initOptions.validator]
     *      A callback function that will be invoked before closing the dialog.
     *      Will be used to decide whether the current state of the dialog is
     *      valid in terms of passing the result to the user by resolving the
     *      promise returned by the public method show(). The callback function
     *      receives the action identifier associated to the clicked button. It
     *      must return a boolean value specifying whether the dialog state is
     *      valid; or a promise that will be resolved or rejected later. If the
     *      function returns true, or a promise that will be resolved, the show
     *      promise will be resolved too. Otherwise, the show promise will be
     *      rejected with the value 'invalid'. The callback function will NOT
     *      be invoked when canceling the dialog. In this case, the show
     *      promise will be rejected immediately with the value 'cancel'.
     *  @param {Boolean} [initOptions.async=false]
     *      Whether the dialog will be opened in asynchronous mode.
     *  @param {JQuery} [initOptions.container]
     *      The container node for the modal dialog, otherwise the container is $('#io-ox-core')
     */
    Dialogs.ModalDialog = _.makeExtendable(CoreDialogs.ModalDialog).extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // the title text
        var title = Utils.getStringOption(initOptions, 'title');

        // the text label of the OK button
        var okLabel = Utils.getStringOption(initOptions, 'okLabel', OK_LABEL);

        // the text label of the Cancel button
        var cancelLabel = Utils.getStringOption(initOptions, 'cancelLabel', CANCEL_LABEL);

        // the validator invoked before closing the dialog
        var validator = Utils.getFunctionOption(initOptions, 'validator', _.constant(true));

        // base constructor ---------------------------------------------------

        var dialogOptions = {
            width: Dialogs.getBestDialogWidth(Utils.getIntegerOption(initOptions, 'width', 500)),
            async: Utils.getBooleanOption(initOptions, 'async', false),
            enter: inputEnterHandler,
            focus: false
        };
        var container = Utils.getObjectOption(initOptions, 'container', null);
        if (container) {
            dialogOptions.container = container;
        }
        CoreDialogs.ModalDialog.call(this, dialogOptions);

        // private methods ----------------------------------------------------

        /**
         * Handler that triggers the primary button of the dialog, after the
         * ENTER key has been pressed in a text input control.
         */
        function inputEnterHandler() {

            // the primary button bound to the ENTER key
            var okButton = self.getOkButton();
            // whether all criteria are met to trigger the default action
            var triggerAction = (okButton.length > 0) && (okButton.attr('disabled') !== 'disabled');

            if (triggerAction) { self.invoke('ok'); }
            return !triggerAction;
        }

        /**
         * Handles global key events in the entire dialog.
         */
        function keyDownHandler(event) {
            // safety check: ignore key events triggered during clean-up ('self' already reset to null)
            if (self && (event.keyCode === KeyCodes.ENTER)) {
                inputEnterHandler();
                return false;
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the button element from the footer of this dialog bound to
         * the specified action.
         *
         * @param {String} action
         *  The identifier of the action the button is bound to.
         *
         * @returns {jQuery}
         *  The specified button element, as jQuery object.
         */
        this.getButton = function (action) {
            return this.getFooter().find('.btn[data-action="' + action + '"]');
        };

        /**
         * Returns the OK button of this dialog, as jQuery object.
         *
         * @returns {jQuery}
         *  The button element of the OK button.
         */
        this.getOkButton = function () {
            return this.getButton('ok');
        };

        /**
         * Returns the Cancel button of this dialog, as jQuery object.
         *
         * @returns {jQuery}
         *  The button element of the Cancel button.
         */
        this.getCancelButton = function () {
            return this.getButton('cancel');
        };

        /**
         * Enables or disables the button element from the footer of this
         * dialog bound to the specified action.
         *
         * @param {String} action
         *  The identifier of the action the button is bound to.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableButton = function (action, state) {
            Forms.enableBSButton(this.getButton(action), state);
            return this;
        };

        /**
         * Enables or disables the OK button element of this dialog.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableOkButton = function (state) {
            return this.enableButton('ok', state);
        };

        /**
         * Enables or disables the Cancel button element of this dialog.
         *
         * @param {Boolean} state
         *  Whether to enable (true) or disable (false) the button.
         *
         * @returns {ModalDialog}
         *  A reference to this instance.
         */
        this.enableCancelButton = function (state) {
            return this.enableButton('cancel', state);
        };

        /**
         * Shows this dialog.
         *
         * @attention
         *  The dialog can only be shown once. It will destroy itself after it
         *  has been closed.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked immediately after showing
         *  the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with the action identifier, if any
         *  action button (OK button, alternative buttons) has been pressed; or
         *  rejected, if the Cancel button has been pressed. The action
         *  identifier of the OK button is the string 'ok'.
         */
        this.show = _.wrap(this.show, function (show, callback) {

            // work-around the weird behavior of the core dialog in asynchronous mode:
            // - the promise returned by show() will not resolve/reject when triggering an action
            // - the generic 'action' event does not send the action identifier
            // solution is to process all dialog events and look for an action button to resolve/reject the own promise
            var def = DeferredUtils.createDeferred(null, 'Dialogs: show');

            // event handler for all events triggered by this dialog, resolves/rejects
            // the own deferred object after activating an action button
            function actionHandler(event, action) {

                // bug 32244: handle explicit "dialog.close()" calls
                if (action === 'close') {
                    self.close = $.noop; // prevent recursive calls from promise callbacks
                    def.reject(action, self);
                    return;
                }

                // check that a button exists for the action identifier
                if (self.getButton(action).length === 0) { return; }

                // reject the promise on Cancel
                if (action === 'cancel') {
                    def.reject(action);
                    return;
                }

                // invoke the validator callback function
                var result = validator.call(self, action);
                if (Utils.isPromise(result)) {
                    result.done(function () { def.resolve(action); }).fail(function () { def.reject('invalid'); });
                } else if (result === true) {
                    def.resolve(action);
                } else {
                    def.reject('invalid');
                }
            }

            // Clicking on the overlay node moves focus to document body. Add handlers
            // for the TAB key and ESCAPE key to be able to return to the dialog, or
            // to close it even without having the focus inside the dialog.
            function globalKeyDownHandler(event) {
                if (Utils.getActiveElement() === document.body) {
                    if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                        Forms.grabFocus(self.getPopup());
                        event.preventDefault();
                    } else if (KeyCodes.matchKeyCode(event, 'ESCAPE')) {
                        self.close();
                    }
                }
            }

            // Bug 41397: When focus leaves a node with browser selection (e.g. a clipboard
            // node, or a content-editable node), the browser selection MUST be destroyed.
            // Otherwise, all keyboard events bubbling to the document body will cause to
            // focus that previous element having the browser selection regardless whether
            // it is actually focused. This may cause for example to be able to edit a
            // document while a modal dialog is open.
            Utils.clearBrowserSelection();

            // register event handlers for the dialog lifetime
            this.on('triggered', actionHandler);
            $(document).on('keydown', globalKeyDownHandler);

            // clean-up after closing the dialog
            def.always(function () {
                self.off('triggered', actionHandler);
                $(document).off('keydown', globalKeyDownHandler);
                self = null;
            });

            // show the dialog, but ignore the promise it returns
            show.call(this, callback);

            // return the own promise that will always be resolved/rejected
            // (dialog remains still open in asynchronous mode)
            return def.promise();
        });

        // initialization -----------------------------------------------------

        // add own CSS marker class
        this.getPopup().addClass('io-ox-office-dialog');
        Forms.addDeviceMarkers(this.getPopup());

        // add title
        if (_.isString(title)) {
            this.header($('<h4>', { id: 'dialog-title' }).text(title));
        }

        // override all 'addSomeButton()' methods: set 'dataaction' parameter to 'action', add tabIndex option
        ['addButton', 'addDangerButton', 'addSuccessButton', 'addWarningButton', 'addPrimaryButton', 'addAlternativeButton'].forEach(function (methodName) {
            this[methodName] = _.wrap(this[methodName], function (baseMethod, action, label, options) {
                return baseMethod.call(this, action, label, action, _.extend({ tabIndex: 0 }, options));
            });
        }, this);

        // add the OK button
        this.addPrimaryButton('ok', okLabel);

        // add the Cancel button
        this.addButton('cancel', cancelLabel);

        // listen to key events for additional keyboard handling
        this.getBody().on('keydown', keyDownHandler);

    } }); // class ModalDialog

    // class ModalTabDialog ===================================================

    /**
     * Creates an empty modal dialog that contains a tab bar and tab panels.
     *
     * Additionally to the events triggered by the base class ModalDialog, an
     * instance of this class triggers the following events:
     * - 'tab:hide'
     *      Before the active tab panel will be deactivated. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     * - 'tab:show'
     *      After a tab panel has been activated. Event handlers receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {String} tabId
     *          The identifier of the active tab panel.
     *      (3) {jQuery} tabPanel
     *          The root node of the active tab panel.
     *      (4) {jQuery} tabButton
     *          The button element of the active tab button.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog.
     */
    Dialogs.ModalTabDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // the container for the tab buttons
        var tabBarNode = $('<ul class="nav nav-tabs" role="tablist">');

        // the container for the tab panel elements
        var tabPanelNode = $('<div class="tab-content" style="margin-top:13px;">');

        // maps tab panels by user identifiers
        var tabPanelsMap = {};

        // all tab panel identifiers in insertion order
        var tabPanelIds = [];

        // the identifier of the visible tab panel
        var activeId = null;

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Sets the browser focus into the button element of the active tab
         * panel.
         */
        function focusActiveTabButton() {
            Utils.setFocus(tabBarNode.find('li.active a').first());
        }

        /**
         * Handles keyboard events for browser focus navigation.
         */
        function keyDownHandler(event) {

            // index of the active tab panel
            var activeIndex = _.indexOf(tabPanelIds, activeId);
            // identifier of the new tab panel to be activated
            var newTabId = null;

            switch (event.keyCode) {
                case KeyCodes.RIGHT_ARROW:
                    newTabId = tabPanelIds[activeIndex + 1];
                    break;
                case KeyCodes.LEFT_ARROW:
                    newTabId = tabPanelIds[activeIndex - 1];
                    break;
                case KeyCodes.SPACE:
                    newTabId = $(event.target).attr('data-user-id');
                    break;
            }

            if (newTabId) {
                // focus active tab button first to allow the active tab panel
                // to finally focus a control inside the panel
                focusActiveTabButton();
                self.activateTabPanel(newTabId);
                return false;
            }

            //we need to call preventDefault() because of focus travelling back to the document when this dialog has no input-fields
            event.preventDefault();
        }

        // methods ------------------------------------------------------------

        /**
         * Creates a new tab panel container element associated to the
         * specified identifier.
         *
         * @param {String} tabId
         *  The identifier of the tab panel. The identifiers of all tab panels
         *  created for this dialog must be different.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.label]
         *      The text label to be shown in the tab button activating the new
         *      tab panel. If omitted, the tab button will not contain a text
         *      label.
         *  @param {String} [options.icon]
         *      The CSS class name of the icon to be shown in the tab button
         *      activating the new tab panel. If omitted, the tab button will
         *      not contain an icon. Must be an icon provided by the Font
         *      Awesome icon set.
         *
         * @returns {jQuery}
         *  The container element of the specified tab panel, as jQuery object.
         */
        this.addTabPanel = function (tabId, options) {

            // the unique DOM identifier for the tab button
            var buttonId = _.uniqueId('tab');
            // the unique DOM identifier for the tab panel container
            var panelId = _.uniqueId('panel');

            // the button element in the tab bar
            var tabButton = $('<a href="#' + panelId + '" id="' + buttonId + '" tabindex="0" role="tab" data-toggle="tab" data-user-id="' + tabId + '" aria-selected="false" aria-controls="' + panelId + '">');
            // the container element for the tab panel
            var tabPanel = $('<div class="tab-pane row-fluid" id="' + panelId + '" role="tabpanel" data-user-id="' + tabId + '" aria-labelledby="' + buttonId + '">');

            // set button caption
            tabButton.append(Forms.createCaptionMarkup(options));

            // insert the tab button and tab panel
            tabBarNode.append($('<li>').append(tabButton));
            tabPanelNode.append(tabPanel);
            tabPanelsMap[tabId] = tabPanel;
            tabPanelIds.push(tabId);

            // return the tab panel for further initialization
            return tabPanel;
        };

        /**
         * Returns the tab panel container element associated to the specified
         * identifier.
         *
         * @param {String} tabId
         *  The identifier of the tab panel, as passed to the method
         *  ModalTabDialog.addTabPanel().
         *
         * @returns {jQuery}
         *  The container element of the specified tab panel. If the passed
         *  identifier is invalid, returns an empty jQuery collection.
         */
        this.getTabPanel = function (tabId) {
            return (tabId in tabPanelsMap) ? tabPanelsMap[tabId] : $();
        };

        /**
         * Returns the identifier of the tab panel that is currently visible.
         *
         * @returns {String|Null}
         *  The identifier of the tab panel that is currently visible, or null
         *  if no panel is visible yet.
         */
        this.getActiveTabId = function () {
            return activeId;
        };

        /**
         * Activates (displays) the specified tab panel.
         *
         * @param {String} tabId
         *  The identifier of the tab panel, as passed to the method
         *  ModalTabDialog.addTabPanel().
         *
         * @returns {ModalTabDialog}
         *  A reference to this instance.
         */
        this.activateTabPanel = function (tabId) {
            tabBarNode.find('[data-user-id="' + tabId + '"]').tab('show');
            return this;
        };

        // initialization -----------------------------------------------------

        // insert the nodes into the dialog body
        this.append(tabBarNode, tabPanelNode);

        // initially focus the active tab button
        this.on('show', focusActiveTabButton);

        // trigger 'tab:hide' event before the active tab panel will be hidden
        tabBarNode.on('show.bs.tab', function (event) {
            var tabButton = $(event.relatedTarget).closest('a'),
                tabId = tabButton.attr('data-user-id');
            if (tabButton.length > 0) {
                tabButton.attr('aria-selected', false);
                self.trigger('tab:hide', tabId, tabPanelsMap[tabId], tabButton);
            }
        });

        // trigger 'tab:show' event after a tab panel has been shown
        tabBarNode.on('shown.bs.tab', function (event) {
            var tabButton = $(event.target).closest('a');
            activeId = tabButton.attr('data-user-id');
            tabButton.attr('aria-selected', true);
            self.trigger('tab:show', activeId, tabPanelsMap[activeId], tabButton);
        });

        // handle keyboard events for focus navigation
        tabBarNode.on('keydown', keyDownHandler);

        // destroy members
        this.on('close', function () {
            tabBarNode.remove();
            tabPanelNode.remove();
            self = tabBarNode = tabPanelNode = tabPanelsMap = null;
        });

    } }); // class ModalTabDialog

    // class ModalQueryDialog =================================================

    /**
     * Creates a simple dialog with a message text and an OK button.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {String} message
     *  The message text shown in the dialog body element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. The option 'cancelLabel' will be ignored (the Cancel
     *  button of the dialog will always be hidden).
     */
    Dialogs.ModalMessageDialog = Dialogs.ModalDialog.extend({ constructor: function (message, initOptions) {

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, Utils.extendOptions({ width: 400 }, initOptions));

        // initialization -----------------------------------------------------

        this.append($('<label>').text(message));

        // remove the Cancel button
        this.getCancelButton().hide();

        // focus the OK button when showing the dialog
        this.on('show', function () { Utils.setFocus(this.getOkButton()); });

    } }); // class ModalMessageDialog

    // class ModalQueryDialog =================================================

    /**
     * Creates a simple dialog with a message text and Yes and No buttons.
     * Optionally, the dialog may contain an additional Cancel button next to
     * the Yes and No buttons.
     *
     * By default, the promise returned by the public method show() of an
     * instance of this dialog will be resolved with the boolean value true
     * when pressing the Yes button, and rejected when pressing the No button.
     *
     * If the dialog has been opened with an additional Cancel button (see
     * constructor option 'cancel' below), the promise returned by the public
     * method show() will behave differently: When pressing the Yes button, the
     * promise will be resolved with the boolean value true. When pressing the
     * No button, the promise will be resolved with the boolean value false.
     * Otherwise (dialog canceled), the promise will be rejected.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {String} message
     *  The message text shown in the dialog body element.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. The default value of the option 'okLabel' will be changed
     *  to the translated string 'Yes', and the default value of the option
     *  'cancelLabel' will be changed to the translated string 'No'.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.cancel=false]
     *      Whether to show an additional Cancel button next to the Yes and No
     *      buttons. See comments above about the behavioral changes for this
     *      case.
     */
    Dialogs.ModalQueryDialog = Dialogs.ModalDialog.extend({ constructor: function (message, initOptions) {

        // whether to show an additional Cancel button
        var cancelMode = Utils.getBooleanOption(initOptions, 'cancel', false);

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, Utils.extendOptions({
            width: 400
        }, initOptions, {
            okLabel: YES_LABEL,
            cancelLabel: cancelMode ? null : NO_LABEL
        }));

        // public methods -----------------------------------------------------

        /**
         * Shows this dialog.
         *
         * @attention
         *  The dialog can only be shown once. It will destroy itself after it
         *  has been closed.
         *
         * @param {Function} [callback]
         *  A callback function that will be invoked immediately after showing
         *  the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved with a boolean value. See the
         *  description of the class constructor for details.
         */
        this.show = _.wrap(this.show, function (show, callback) {
            return show.call(this, callback).then(function (action) { return action === 'ok'; });
        });

        // initialization -----------------------------------------------------

        this.append($('<label>').text(message));

        // add an own No button in Yes/No/Cancel mode
        if (cancelMode) {
            this.addButton('no', NO_LABEL);
            // there is no better way to position a button
            this.getButton('no').insertBefore(this.getOkButton());
        }

        // focus the OK button when showing the dialog
        this.on('show', function () { Utils.setFocus(this.getOkButton()); });

    } }); // class ModalQueryDialog

    // class ModalInputDialog =================================================

    /**
     * Creates a simple text input dialog.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.inputLabel]
     *      A leading label for the input field.
     *  @param {String} [initOptions.value='']
     *      The initial value of the text field.
     *  @param {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {Boolean} [initOptions.allowEmpty=false]
     *      If set to true, empty text will be considered valid. Otherwise, the
     *      primary button will be disabled as long as the text input field is
     *      empty.
     *  @param {Boolean} [initOptions.maxLength]
     *      If specified, the maximum number of characters that can be entered.
     */
    Dialogs.ModalInputDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        // self reference
        var self = this;

        // maximum length allowed in the text field
        var maxLength = Utils.getIntegerOption(initOptions, 'maxLength', null, 1);

        // whether to allow empty text
        var allowEmpty = Utils.getBooleanOption(initOptions, 'allowEmpty', false);

        // the label text
        var inputLabel = Utils.getStringOption(initOptions, 'inputLabel', '');

        // the text label element
        var labelNode = inputLabel ? $('<label>').text(inputLabel) : $();

        // the text field control
        var inputNode = $('<input type="text" class="form-control" tabindex="0">').attr({
            placeholder: Utils.getStringOption(initOptions, 'placeholder', ''),
            value: Utils.getStringOption(initOptions, 'value', '')
        });

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, initOptions);

        // private functions --------------------------------------------------

        /**
         * Enables or disables the primary button according to the contents of
         * the text field.
         */
        function updateOkButton() {
            self.enableOkButton(allowEmpty || (inputNode.val().length > 0));
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the text input element, as jQuery object.
         *
         * @returns {jQuery}
         *  The text input element, as jQuery object.
         */
        this.getInputNode = function () {
            return inputNode;
        };

        /**
         * Returns the text that is contained in the text input element.
         *
         * @returns {String}
         *  The text that is contained in the text input element.
         */
        this.getText = function () {
            if (_.isNumber(maxLength)) {
                return inputNode.val().substr(0, maxLength);
            }
            return inputNode.val();
        };

        /**
         * Changes the text contained in the text input element.
         *
         * @param {String} text
         *  The text to be inserted into the text input element.
         *
         * @returns {ModalInputDialog}
         *  A reference to this instance.
         */
        this.setText = function (text) {
            inputNode.val(text);
            return this;
        };

        // initialization -----------------------------------------------------

        // add the text field
        this.append(labelNode, inputNode);

        // initialize maximum length
        if (_.isNumber(maxLength)) {
            inputNode.val(inputNode.val().substr(0, maxLength));
            inputNode.attr('maxlength', maxLength);
        }

        // enable/disable the primary button
        inputNode.on('input', updateOkButton);
        updateOkButton();

        // focus and select the text in the input field
        this.on('show', function () {
            // us 102078426: dialogs on small devices should come up w/o keyboard,
            // so the focus should not be in an input element

            // Bug 46886
            // Also for IPad because since IOS 9, body is not scrolled when code triggers focus on input element
            if (Utils.SMALL_DEVICE || Utils.IOS) {
                Utils.setFocus(self.getOkButton());
            } else {
                Utils.setFocus(inputNode).select();
            }
        });

    } }); // class ModalInputDialog

    // class ModalDriveDialog =================================================

    /**
     * A custom dialog for Drive Utils
     * DriveUtils are assigned as "driveUtil", you can take FolderPicker,
     * FlatFolderPicker & FilePicker
     *
     * @extends ModalDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.preselect]
     *      Drive-File-Id to preselect a folder
     *  @param {String} [initOptions.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {Object} [initOptions.driveUtil]
     *      the assigned DriveUtil from the DriveUtils class
     *      can be: FolderPicker, FlatFolderPicker & FilePicker
     */
    Dialogs.ModalDriveDialog = Dialogs.ModalDialog.extend({ constructor: function (initOptions) {

        var self = this;

        var driveUtil = Utils.getOption(initOptions, 'driveUtil');

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, initOptions);

        // public methods -----------------------------------------------------

        /**
         * @returns {Object}
         *  returns the current file-selection of the DriveUtil
         *  file can have folder_id (parentId), id (file-id) & filename,
         *  but all are optional
         */
        this.getSelectedFile = function () {
            return driveUtil.getSelectedFile();
        };

        // initialization -----------------------------------------------------

        this.getPopup().find('.modal-body').append(driveUtil.getNode());

        driveUtil.on('change', function () {
            self.trigger('change');
        });

    } }); // class ModalDriveDialog

    // class ModalFolderPicker ================================================

    /**
     * A dialog with DriveUtils' FolderPicker
     *
     * @constructor
     *
     * @extends ModalDriveDialog
     *
     * @param {Object} [initOptions]
     *  Optional parameters that control the appearance and behavior of the
     *  dialog. Supports all options also supported by the base class
     *  ModalDriveDialog. Additionally, the following options are supported:
     *  @param {String} [initOptions.preselect]
     *      Drive-File-Id to preselect a folder
     */
    Dialogs.ModalFolderPicker = Dialogs.ModalDriveDialog.extend({ constructor: function (initOptions) {

        var driveUtil = new DriveUtils.FolderPicker(initOptions);

        // base constructor ---------------------------------------------------

        Dialogs.ModalDriveDialog.call(this, Utils.extendOptions({
            driveUtil: driveUtil
        }, initOptions));

        // public methods -----------------------------------------------------

        /**
         * @returns {String}
         *  The current folder identifier.
         */
        this.getSelectedFolderId = function () {
            var selectedFile = this.getSelectedFile();
            if (selectedFile) {
                return selectedFile.folder_id;
            }
        };

    } }); // class ModalFolderPicker

    // exports ================================================================

    return Dialogs;

});
