/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/control/checkbox', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/control/button'
], function (Utils, Forms, Button) {

    'use strict';

    // class CheckBox =========================================================

    /**
     * Creates a container element used to hold a toggle button, styled like a
     * regular check box.
     *
     * @constructor
     *
     * @extends Button
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options of the Button base class,
     *  except for the 'toggle' option which will always be set to true, and
     *  'highlight' (check boxes will never be highlighted). Additionally, the
     *  following options are supported:
     *  @param {Boolean} [initOptions.boxed=false]
     *      If set to true, the icons to visualize the unchecked and checked
     *      state will contain rectangular boxes. Otherwise, unchecked state is
     *      simply empty space, and checked state is a simple check mark.
     *  @param {Boolean} [options.ambiguous=false]
     *      If set to true, the ambiguous state (any value but Booleans) will
     *      be represented by a specific distinct icon. Otherwise, the icon for
     *      the unchecked state will be used.
     */
    function CheckBox(initOptions) {

        var // self reference
            self = this,

            // options for the check icon
            checkOptions = {
                design: Utils.getBooleanOption(initOptions, 'boxed', false) ? 'boxed' : 'check',
                ambiguous: Utils.getBooleanOption(initOptions, 'ambiguous', false)
            };

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({
            attributes: { role: 'checkbox' } // default ARIA role
        }, initOptions, {
            toggle: true,
            highlight: _.constant(false) // never highlight the check box
        }));

        // private methods ----------------------------------------------------

        /**
         * The update handler for this button.
         */
        function updateHandler(state) {
            Forms.checkButtonNodes(self.getButtonNode(), state, checkOptions);
        }

        // initialization -----------------------------------------------------

        // add special marker classes used to adjust formatting
        this.getNode().addClass('check-box');
        Forms.checkButtonNodes(this.getButtonNode(), false, checkOptions);

        // register an update handler that updates the check mark
        this.registerUpdateHandler(updateHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = null;
        });

    } // class CheckBox

    // exports ================================================================

    // derive this class from class Button
    return Button.extend({ constructor: CheckBox });

});
