/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/basemap', [
    'io.ox/office/tk/utils/iteratorutils'
], function (IteratorUtils) {

    'use strict';

    /**
     * Implementation helper for iterator methods of the class BaseMap.
     *
     * @param {Function} func
     *  The iterator implementation, e.g. _.each or _.some.
     *
     * @param {BaseMap} map
     *  The map to be iterated.
     *
     * @param {Function} callback
     *  The callback function.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    function implIterate(func, map, callback, context) {
        return func(map._map, function (value, key) {
            return callback.call(context, value, key, map);
        });
    }

    // class BaseMap ==========================================================

    /**
     * An internal base class used as implementation helper for prototype based
     * container classes that represent an associative map. This class provides
     * the protected property '_map' (plain JavaScript object) and various
     * iterator methods.
     *
     * @constructor
     *
     * @param {Object|BaseMap} [source]
     *  A source map to clone elements from. Can be a plain JavaScript object,
     *  or another instance of this class. If omitted, this constructor creates
     *  an empty map.
     *
     * @param {Function} [generator]
     *  A generator callback function that will be used to deeply clone the
     *  elements of the passed source map. Receives the following parameters:
     *  (1) {Any} value
     *      The element of the source map to be cloned.
     *  (2) {String} key
     *      The key of the map element.
     *  The return value of this function will be inserted as new element into
     *  this map. If omitted, a flat copy will be created (the elements of the
     *  source map will be copied by reference).
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     */
    var BaseMap = _.makeExtendable(function (source, generator, context) {

        // create a clone of the passed source map, or an empty object
        BaseMap.prototype._init.call(this, source, generator, context);

    }); // class BaseMap

    // protected methods ------------------------------------------------------

    /**
     * Initializes or resets the internal data storage for the map elements.
     *
     * @param {Object|BaseMap} [source]
     *  A source map to clone elements from. Can be a plain JavaScript object,
     *  or another instance of this class. If omitted, this constructor creates
     *  an empty map.
     *
     * @param {Function} [generator]
     *  A generator callback function that will be used to deeply clone the
     *  elements of the passed source map. Receives the following parameters:
     *  (1) {Any} value
     *      The element of the source map to be cloned.
     *  (2) {String} key
     *      The key of the map element.
     *  The return value of this function will be inserted as new element into
     *  this map. If omitted, a flat copy will be created (the elements of the
     *  source map will be copied by reference).
     *
     * @param {Object} [context]
     *  The calling context for the generator callback function.
     *
     * @returns {BaseMap}
     *  A reference to this instance.
     */
    BaseMap.prototype._init = function (source, generator, context) {
        this._map = source ? (function () {
            source = (source instanceof BaseMap) ? source._map : source;
            return generator ? _.mapObject(source, generator, context) : _.clone(source);
        }()) : {};
        return this;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this map is empty.
     *
     * @returns {Boolean}
     *  Whether this map is empty.
     */
    BaseMap.prototype.empty = function () {
        // fastest way to detect an empty object
        for (var key in this._map) { return false; } // eslint-disable-line no-unused-vars
        return true;
    };

    /**
     * Returns whether this map contains exactly one element.
     *
     * @returns {Boolean}
     *  Whether this map contains exactly one element.
     */
    BaseMap.prototype.single = function () {
        // avoid helpers with O(n) complexity, such as _.size() or Object.keys()
        var count = 0;
        for (var key in this._map) { // eslint-disable-line no-unused-vars
            count += 1;
            if (count > 1) { break; }
        }
        return count === 1;
    };

    /**
     * Removes all elements from this map.
     *
     * @returns {BaseMap}
     *  A reference to this instance.
     */
    BaseMap.prototype.clear = function () {
        this._map = {};
        return this;
    };

    /**
     * Returns whether this map contains an element with the specified key.
     *
     * @param {String} key
     *  The key to be checked.
     *
     * @returns {Boolean}
     *  Whether this map contains an element with the specified key.
     */
    BaseMap.prototype.has = function (key) {
        return key in this._map;
    };

    /**
     * Returns the map element with the specified key.
     *
     * @param {String} key
     *  The key of the map element to be returned.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this map does not contain an
     *  element with the specified key. If omitted, undefined will be returned.
     *
     * @returns {Any}
     *  The map element with the specified key; or the specified default value,
     *  if the map does not contain such an element.
     */
    BaseMap.prototype.get = function (key, def) {
        return (key in this._map) ? this._map[key] : def;
    };

    /**
     * Returns a single unspecified element from this map.
     *
     * @param {type} [def]
     *  The default value that will be returned if this map is empty. If
     *  omitted, undefined will be returned.
     *
     * @returns {Any}
     *  An unspecified map element; or the specified default value, if the map
     *  is empty. Repeated invocations of this method will always return the
     *  same value unless the map has been modified.
     */
    BaseMap.prototype.getAny = function (def) {
        for (var key in this._map) { return this._map[key]; }
        return def;
    };

    /**
     * Returns the keys of all elements contained in this map as array.
     *
     * @returns {Array<String>}
     *  The keys of all elements contained in this map, in no specific order.
     */
    BaseMap.prototype.keys = function () {
        return Object.keys(this._map);
    };

    /**
     * Returns all elements contained in this map as array.
     *
     * @returns {Array<Any>}
     *  The elements contained in this map, in no specific order.
     */
    BaseMap.prototype.values = function () {
        return _.values(this._map);
    };

    /**
     * Inserts an element into this map. An old existing map element will be
     * overwritten.
     *
     * @param {String} key
     *  The key for the map element to be inserted.
     *
     * @param {Any} value
     *  The map element to be inserted.
     *
     * @returns {Any}
     *  The map element passed to this method, for convenience.
     */
    BaseMap.prototype.insert = function (key, value) {
        return (this._map[key] = value);
    };

    /**
     * Removes an element from this map.
     *
     * @param {String} key
     *  The key of the map element to be removed.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this map does not contain an
     *  element with the specified key. If omitted, undefined will be returned.
     *
     * @returns {Any}
     *  The map element that has been removed; or the specified default value,
     *  if the map does not contain such an element.
     */
    BaseMap.prototype.remove = function (key, def) {
        var value = BaseMap.prototype.get.call(this, key, def);
        delete this._map[key];
        return value;
    };

    /**
     * Returns an iterator that visits all elements in this map. The map
     * elements will be visited in no specific order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the map has been visited completely. No more map
     *      elements are available; this result object does not contain any
     *      other properties!
     *  - {Any} value
     *      The map element currently visited. This property will be omitted,
     *      if the iterator is done (see property 'done').
     *  - {String} key
     *      The key of the map element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     */
    BaseMap.prototype.iterator = function () {
        return IteratorUtils.createObjectIterator(this._map);
    };

    /**
     * Returns an iterator that visits the keys of all elements in this map.
     * The map elements will be visited in no specific order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the map has been visited completely. No more map
     *      elements are available; this result object does not contain any
     *      other properties!
     *  - {String} value
     *      The key of the map element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     *  - {Any} element
     *      The map element currently visited. This property will be omitted,
     *      if the iterator is done (see property 'done').
     */
    BaseMap.prototype.keyIterator = function () {
        var iterator = IteratorUtils.createObjectIterator(this._map);
        return IteratorUtils.createTransformIterator(iterator, function (value, result) {
            return { value: result.key, element: value };
        });
    };

    /**
     * Invokes the passed callback function for all elements in this map. The
     * map elements will be visited in no specific order.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each map element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {BaseMap}
     *  A reference to this instance.
     */
    BaseMap.prototype.forEach = function (callback, context) {
        implIterate(_.each, this, callback, context);
        return this;
    };

    /**
     * Invokes the passed callback function for all keys in this map. The map
     * keys will be visited in no specific order. This additional method next
     * to BaseMap.forEach() may be useful when a callback function wants to
     * process the map keys only, and thus can be passed directly as parameter
     * to this method without having to create an anonymous function with an
     * unused first parameter.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each map key. Receives
     *  the following parameters:
     *  (1) {String} key
     *      The key of the current map element.
     *  (2) {Any} value
     *      The current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {BaseMap}
     *  A reference to this instance.
     */
    BaseMap.prototype.forEachKey = function (callback, context) {
        _.each(this._map, function (value, key) {
            callback.call(context, key, value, this);
        }, this);
        return this;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for at least one element in this map. The map elements will be visited
     * in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a truthy value. After that, iteration will
     *  be stopped immediately, the remaining map elements will not be visited
     *  anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for at
     *  least one element in this map.
     */
    BaseMap.prototype.some = function (predicate, context) {
        return implIterate(_.some, this, predicate, context);
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for all elements in this map. The map elements will be visited in no
     * specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a falsy value. After that, iteration will
     *  be stopped immediately, the remaining map elements will not be visited
     *  anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for all
     *  elements in this map.
     */
    BaseMap.prototype.every = function (predicate, context) {
        return implIterate(_.every, this, predicate, context);
    };

    /**
     * Invokes the passed callback function for all elements in this map, and
     * returns the return values as plain array. The map elements will be
     * visited in no specific order.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *  The return value of the callback function will be inserted into the
     *  resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The return values of the callback function, in no specific order.
     */
    BaseMap.prototype.map = function (callback, context) {
        return implIterate(_.map, this, callback, context);
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * map, and returns a plain array with all elements resulting in a truthy
     * return value. The map elements will be visited in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {BaseMap} map
     *      A reference to this map instance.
     *  If the return value is truthy, the map element will be inserted into
     *  the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The map elements that passed the truth test, in no specific order.
     */
    BaseMap.prototype.filter = function (predicate, context) {
        return implIterate(_.filter, this, predicate, context);
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * map, and returns a plain array with all elements resulting in a falsy
     * return value. The map elements will be visited in no specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  If the return value is falsy, the map element will be inserted into the
     *  resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The map elements that do not pass the truth test, in no specific order.
     */
    BaseMap.prototype.reject = function (predicate, context) {
        return implIterate(_.reject, this, predicate, context);
    };

    /**
     * Invokes the passed callback function for all elements in this map, and
     * reduces the values to a single result value. The map elements will be
     * visited in no specific order.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the map is empty, this value becomes the final result
     *  value immediately).
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element in this
     *  map. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The return value of the previous invocation of this callback
     *      function (or the initial value on first invocation).
     *  (2) {Any} value
     *      The value of the current map element.
     *  (3) {String} key
     *      The key of the current map element.
     *  The return value of the callback function will become the new result
     *  value that will be passed as 'prevValue' with the next map element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback fucntion).
     */
    BaseMap.prototype.reduce = function (initial, callback, context) {
        return _.reduce(this._map, function (accu, value, key) {
            return callback.call(context, accu, value, key, this);
        }, initial, this);
    };

    /**
     * Returns the values of a specific property from all object elements in
     * this map. The map elements will be visited in no specific order.
     *
     * @param {String} name
     *  The name of an element property.
     *
     * @returns {Array<Any>}
     *  The values of the specified property from all object elements in this
     *  map, in no specific order.
     */
    BaseMap.prototype.pluck = function (name) {
        return _.pluck(this._map, name);
    };

    /**
     * Returns the first element in this map that passes a truth test according
     * to the specified predicate. The map elements will be visited in no
     * specific order.
     *
     * @param {Function} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this map until it returns a truthy value. After that, the element
     *  will be returned immediately. Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Any}
     *  The first map elements that passes the truth test; or undefined, if no
     *  map element passes the truth test.
     */
    BaseMap.prototype.find = function (predicate, context) {
        return implIterate(_.find, this, predicate, context);
    };

    // static methods ---------------------------------------------------------

    /**
     * Invokes the passed callback function for all elements in the passed map.
     * The map elements will be visited in no specific order.
     *
     * @param {Object|BaseMap} map
     *  The map to be iterated. Can be a plain JavaScript object, or an
     *  instance of the class BaseMap or any of its sub-classes.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each map element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The current map element.
     *  (2) {String} key
     *      The key of the current map element.
     *  (3) {Object|BaseMap} map
     *      A reference to the map currently iterated.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    BaseMap.forEach = function (map, callback, context) {
        var obj = (map instanceof BaseMap) ? map._map : map;
        for (var key in obj) { callback.call(context, obj[key], key, map); }
    };

    /**
     * Invokes the passed callback function for the keys of all elements in the
     * passed map. The map elements will be visited in no specific order.
     *
     * @param {Object|BaseMap} map
     *  The map to be iterated. Can be a plain JavaScript object, or an
     *  instance of the class BaseMap or any of its sub-classes.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for the key of each map
     *  element. Receives the following parameters:
     *  (1) {String} key
     *      The key of the current map element.
     *  (2) {Any} value
     *      The current map element.
     *  (3) {Object|BaseMap} map
     *      A reference to the map currently iterated.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    BaseMap.forEachKey = function (map, callback, context) {
        var obj = (map instanceof BaseMap) ? map._map : map;
        for (var key in obj) { callback.call(context, key, obj[key], map); }
    };

    // exports ================================================================

    return BaseMap;

});
