/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/textframework/view/textdialogs', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/editframework/view/editdialogs',
    'io.ox/office/textframework/view/dialog/pagesettingsdialog'
], function (Utils, Forms, EditDialogs, PageSettingsDialog) {

    'use strict';

    // static class TextDialogs ===============================================

    /**
     * Additional classes defining specialized modal dialogs for the text editor
     * application.
     *
     * @extends EditDialogs
     */
    var TextDialogs = _.extend({}, EditDialogs, {
        PageSettingsDialog: PageSettingsDialog
    });

    // class InputValidateDialog ===============================================

    /**
     *
     */
    TextDialogs.InputValidateDialog = EditDialogs.ModalInputDialog.extend({ constructor: function (docView, initOptions) {

        var self = this;
        var warningNode = $('<div style="margin:5px 0 13px;font-size:13px;">');
        var validator = Utils.getFunctionOption(initOptions, 'validator');

        // base constructor ---------------------------------------------------

        EditDialogs.ModalInputDialog.call(this, Utils.extendOptions(initOptions, { validator: baseValidator }));

        // private methods ----------------------------------------------------

        /**
         * Invokes the validator passed to the constructor, updates the warning
         * label shown in this dialog, and returns the warning text.
         *
         * @returns {String|Null}
         *  The warning text returned by the validator; or null if the input
         *  text is valid.
         */
        function updateWarning() {
            var warningText = validator ? validator(self.getText()) : null;
            if (warningText) {
                warningNode.empty().append(
                    Forms.createIconMarkup('fa-warning', { style: 'margin-left:3px;color:#eb0;' }),
                    $('<span style="margin-left:5px;">').text(warningText)
                );
            } else {
                warningNode.text('\xa0');
            }
            return warningText;
        }

        /**
         * Validator passed to the base class that will be invoked after
         * pressing the OK button, and that must return a boolean value.
         */
        function baseValidator() {

            // call passed validator which is expected to return a warning text
            var warningText = updateWarning();
            if (!warningText) { return true; }

            // show the warning text as warning alert
            docView.yell({ type: 'warning', message: warningText });
            return false;
        }

        // initialization -----------------------------------------------------

        // add the warning label to the dialog body
        this.append(warningNode);

        // close dialog when losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        // update the warning label after every change of the text input field
        this.getInputNode().on('input', updateWarning);
        updateWarning();

    } }); // class InputValidateDialog

    // exports ================================================================

    return TextDialogs;

});
