/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/selection/multiselectionmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/tk/utils'
], function (DrawingFrame, DOM, Position, Utils) {

    'use strict';

    // class MultiSelectionMixin ==============================================

    /**
     * A mix-in class for the multi selection handling of an application.
     *
     * @constructor
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function MultiSelectionMixin(app) {

        var // self reference
            self = this,

            // whether the application supports multiple selections
            isMultiSelectionApp = true,

            // the container for the objects describing a selection with the
            // properties 'startPosition' and 'endPosition'. The selections are
            // sorted corresponding to the start position value.  Furthermore
            // the selection contains the property 'drawing', that contains the
            //  jQuerified drawing node.
            multipleSelections = [];

        // private methods ----------------------------------------------------

        /**
         * Adding one selection object to the container of all selections. The logical
         * position are sorted.
         *
         * @param {Object} selection
         *  The object describing one selection. This contains the logical start
         *  and the logical end position of the selection. Furthermore it contains
         *  the property 'drawing', that contains the jQuerified drawing node.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        function addOneDrawingSelectionIntoMultiSelection(selection) {
            multipleSelections.push(selection);
            multipleSelections.sort(function (sel1, sel2) {
                return Utils.compareNumberArrays(sel1.startPosition, sel2.startPosition);
            });
            self.setValidMultiSelectionValues(multipleSelections);
            return multipleSelections.length;
        }

        /**
         * Removing one selection object from the container of all selections. The logical
         * start position is used to find the selection, that needs to be removed.
         *
         * @param {Number[]} startPosition
         *  The logical start position of the selection that shall be removed.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        function removeOneDrawingSelectionFromMultiSelection(startPosition) {

            multipleSelections = _.reject(multipleSelections, function (oneSelection) {
                return _.isEqual(oneSelection.startPosition, startPosition);
            });

            self.setValidMultiSelectionValues(multipleSelections);
            return multipleSelections.length;
        }

        /**
         * Check if the current selection is a drawing selection and another drawing
         * than the specified drawing is selected.
         *
         * @param {Node|jQuery} drawing
         *  A drawing node.
         *
         * @returns {Boolean}
         *  Whether the current selection is a drawing selection and another drawing
         *  than the specified drawing is selected.
         */
        function anotherDrawingSelected(drawing) {
            return self.isAnyDrawingSelection() && Utils.getDomNode(drawing) !== Utils.getDomNode(self.getAnyDrawingSelection());
        }

        // public methods -----------------------------------------------------

        /**
         * Returns whether this application supports multiple selections.
         *
         * @returns {Boolean}
         *  Whether the application supports multiple selections.
         */
        this.isMultiSelectionSupported = function () {
            return isMultiSelectionApp;
        };

        /**
         * Returns whether several drawings are selected.
         *
         * @returns {Boolean}
         *  Whether several drawings are selected.
         */
        this.isMultiSelection = function () {
            return multipleSelections.length > 0;
        };

        /**
         * Returns whether several drawings are selected and none of these
         * drawings is of type 'table'.
         *
         * @returns {Boolean}
         *  Whether several drawings are selected and none of these drawings
         *  is of type 'table'.
         */
        this.isMultiSelectionWithoutTable = function () {
            return self.isMultiSelection() && !self.isAtLeastOneTableInMultiSelection();
        };

        /**
         * Returns the container of selections. This container stores objects
         * containing the properties 'startPosition' and 'endPosition'. The values
         * are the logical positions, that describe one selection.
         *
         * @returns {any[]}
         *  The container for the objects that describe one selection by its
         *  logical start and its logical end position.
         */
        this.getMultiSelection = function () {
            return multipleSelections;
        };

        /**
         * Returns a collector with all drawing nodes in the selection. If this
         * is no multiple drawing selection, an empty array is returned.
         *
         * @returns {Node[]|jQuery[]|[]}
         *  An array with all drawing nodes in the selection. If this
         *  is no multiple drawing selection, an empty array is returned.
         */
        this.getAllSelectedDrawingNodes = function () {

            var // the container for all drawing nodes
                allDrawingNodes = [];

            _.each(multipleSelections, function (drawingSelection) {
                allDrawingNodes.push(self.getDrawingNodeFromMultiSelection(drawingSelection));
            });

            return allDrawingNodes;
        };

        /**
         * Returns, whether in a multi drawing selection at least one text frame
         * shape is seleted.
         *
         * @returns {Boolean}
         *  Whether in a multi drawing selection at least one text frame shape
         *  is seleted.
         */
        this.isAtLeastOneTextFrameInMultiSelection = function () {
            return self.isMultiSelectionSupported() && self.isMultiSelection() && _.isObject(_.find(multipleSelections, function (selection) {
                return DrawingFrame.isTextFrameShapeDrawingFrame(self.getDrawingNodeFromMultiSelection(selection));
            }));
        };

        /**
         * Returns, whether in a multi drawing selection at least one table drawing
         * is seleted.
         *
         * @returns {Boolean}
         *  Whether in a multi drawing selection at least one table drawing is seleted.
         */
        this.isAtLeastOneTableInMultiSelection = function () {
            return self.isMultiSelectionSupported() && self.isMultiSelection() && _.isObject(_.find(multipleSelections, function (selection) {
                return DrawingFrame.isTableDrawingFrame(self.getDrawingNodeFromMultiSelection(selection));
            }));
        };

        /**
         * Returns, whether in a multi drawing selection at least one drawing of type 'group' is selected.
         *
         * @returns {Boolean}
         *  Whether in a multi drawing selection at least one drawing of type 'group' is selected.
         */
        this.isAtLeastOneGroupInMultiSelection = function () {
            return self.isMultiSelectionSupported() && self.isMultiSelection() && _.isObject(_.find(multipleSelections, function (selection) {
                return DrawingFrame.isGroupDrawingFrame(self.getDrawingNodeFromMultiSelection(selection));
            }));
        };

        /**
         * Returns all selection objects of drawing nodes of type 'group' from a multiple selection.
         *
         * @returns {Object[]}
         *  An object with all selection objects from a multiple drawing selection that describe
         *  drawings of type 'group'.
         */
        this.getAllDrawingGroupsFromMultiSelection = function () {
            return _.filter(multipleSelections, function (selection) {
                return DrawingFrame.isGroupDrawingFrame(self.getDrawingNodeFromMultiSelection(selection));
            });
        };

        /**
         * Returns all selection objects of drawing nodes that are not of type 'group' from a multiple selection.
         *
         * @returns {Object[]}
         *  An object with all selection objects from a multiple drawing selection that do not describe
         *  drawings of type 'group'.
         */
        this.getAllNoneDrawingGroupsFromMultiSelection = function () {
            return _.filter(multipleSelections, function (selection) {
                return !DrawingFrame.isGroupDrawingFrame(self.getDrawingNodeFromMultiSelection(selection));
            });
        };

        /**
         * Returns from a specified selection object of a multiple drawing selection
         * the drawing node. If no selection object exists or the drawing node cannot
         * be found, null is returned.
         *
         * @param {Object} selection
         *  The selection object from a multiple drawing selection.
         *
         * @returns {jQuery|Null}
         *  The drawing node belonging to a multiple drawing selection.
         */
        this.getDrawingNodeFromMultiSelection = function (selection) {
            return selection && selection.drawing ? selection.drawing : null;
        };

        /**
         * Returns from a specified selection object of a multiple drawing selection
         * the logical start position. If no selection object exists or the start
         * position cannot be found, null is returned.
         *
         * @param {Object} selection
         *  The selection object from a multiple drawing selection.
         *
         * @returns {Number[]|Null}
         *  The logical start position belonging to a multiple drawing selection.
         */
        this.getStartPositionFromMultiSelection = function (selection) {
            return selection && selection.startPosition ? selection.startPosition : null;
        };

        /**
         * Returns from a specified selection object of a multiple drawing selection
         * the logical end position. If no selection object exists or the start
         * position cannot be found, null is returned.
         *
         * @param {Object} selection
         *  The selection object from a multiple drawing selection.
         *
         * @returns {Number[]|Null}
         *  The logical end position belonging to a multiple drawing selection.
         */
        this.getEndPositionFromMultiSelection = function (selection) {
            return selection && selection.endPosition ? selection.endPosition : null;
        };

        /**
         * Clearing the container for the multiple selections and removing the
         * selection boxes around every drawing.
         */
        this.clearMultiSelection = function () {

            // remove drawing selection boxes
            _.each(multipleSelections, function (oneSelection) {
                DrawingFrame.clearSelection(oneSelection.drawing);
            });

            // empty collector
            multipleSelections = [];

            // informing the listeners
            self.trigger('change');
        };

        /**
         * Selecting all drawings of the active slide. This function is triggered
         * by ctrl-A, if this is a 'slide selection'.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.selectAllDrawingsOnSlide = function () {

            var // the slide position
                slidePos = [_.first(self.getStartPosition())],
                // the slide node
                slide = Position.getParagraphElement(self.getRootNode(), slidePos),
                // the container for all drawings on the slide
                allDrawings = $(slide).children(DOM.ABSOLUTE_DRAWING_SELECTOR);

            if (allDrawings.length > 0) {

                // clearing an existing multi drawing selection
                self.clearAllDrawingSelections();

                _.each(allDrawings, function (oneDrawing) {

                    var // the start position of the drawing
                        startPos = Position.getOxoPosition(slide, oneDrawing),
                        // the end position of the drawing
                        endPos = Position.increaseLastIndex(startPos);

                    self.drawDrawingSelection(app, oneDrawing);
                    addOneDrawingSelectionIntoMultiSelection({ drawing: $(oneDrawing), startPosition: slidePos.concat(startPos), endPosition: slidePos.concat(endPos) });
                });

            }

            // bring clipboard node in sync with the currently selected drawings
            self.updateClipboardNode();
            // focus clipboard node and set the browser selection to it's contents
            self.setFocusIntoClipboardNode({ specialTouchHandling: true });
            self.setBrowserSelectionToContents(self.getClipboardNode(), { preserveFocus: true });

            return allDrawings.length;
        };

        /**
         * Creating a multi drawing selection with the specified drawings nodes.
         * All drawings must be located on the same (and the active) slide.
         *
         * Important: The slide containing the specified drawing nodes must
         *            be activated!
         *
         * @param {jQuery|Node[]} allDrawings
         *  A container with all drawings nodes that shall be selected.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.setMultiDrawingSelection = function (allDrawings) {

            var // the slide position
                slidePos = null,
                // the slide node
                slide = null;

            if (allDrawings.length > 0) {

                slide = $(allDrawings[0]).closest(DOM.SLIDE_SELECTOR);
                slidePos = app.getModel().getSlidePositionById(app.getModel().getSlideId(slide));

                // clearing an existing multi drawing selection
                self.clearAllDrawingSelections();

                _.each(allDrawings, function (oneDrawing) {

                    var // the start position of the drawing
                        startPos = Position.getOxoPosition(slide, oneDrawing),
                        // the end position of the drawing
                        endPos = Position.increaseLastIndex(startPos);

                    self.drawDrawingSelection(app, oneDrawing);
                    addOneDrawingSelectionIntoMultiSelection({ drawing: $(oneDrawing), startPosition: slidePos.concat(startPos), endPosition: slidePos.concat(endPos) });
                });

                // informing the listeners
                self.trigger('change');
            }

            return allDrawings ? allDrawings.length : 0;
        };

        /**
         * Creating a multi drawing selection with the specified logical drawing positions.
         * All drawings must be located on the same (and the active) slide.
         *
         * Important: The slide containing the specified drawing positions must
         *            be activated!
         *
         * @param {Number[][]} allPositions
         *  A container with all logical positions of the drawings nodes that shall be selected.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.setMultiDrawingSelectionByPosition = function (allPositions) {

            var // the slide position
                slidePos = null,
                // the slide node
                slide = null;

            if (allPositions && allPositions.length > 0) {

                slidePos = [_.first(allPositions[0])];
                slide = Position.getParagraphElement(self.getRootNode(), slidePos);

                // clearing an existing multi drawing selection
                self.clearAllDrawingSelections();

                _.each(allPositions, function (position) {

                    var // the start position of the drawing
                        oneDrawing = Position.getDOMPosition(slide, [_.last(position)], true),
                        // the end position of the drawing
                        endPos = Position.increaseLastIndex(position);

                    if (oneDrawing && oneDrawing.node) {
                        oneDrawing = $(oneDrawing.node);
                        self.drawDrawingSelection(app, oneDrawing);
                        addOneDrawingSelectionIntoMultiSelection({ drawing: oneDrawing, startPosition: position, endPosition: endPos });
                    }
                });

                // informing the listeners
                self.trigger('change');
            }

            return allPositions ? allPositions.length : 0;
        };

        /**
         * If this is a drawing selection (one drawing is selected), this function creates
         * a container that can be used like the container with drawing selections from a
         * multiple drawing selection. This makes is superfluous to distinguish between
         * single drawing selections and multiple drawing selections.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.anyDrawing=false]
         *      Whether the an additional text frame drawing is in the result.
         *
         * @returns {Obejct[]}
         *  An array containing a selection object for single drawing selections. If this
         *  is no single drawing selection, the container is empty.
         */
        this.generateMultiSelectionObjectFromDrawingSelection = function (options) {
            var anyDrawing = Utils.getBooleanOption(options, 'anyDrawing', false);
            return self.isDrawingSelection() || (anyDrawing && self.isAdditionalTextframeSelection()) ? [{ drawing: anyDrawing ? self.getAnyDrawingSelection() : self.getSelectedDrawing(), startPosition: self.getStartPosition(), endPosition: self.getEndPosition() }] : [];
        };

        /**
         * Checking, whether the selection of a specified drawing needs an update of the
         * multiple drawing selection.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing element node, that might require an update of multiple
         *  drawing selection.
         *
         * @returns {Boolean}
         *  Whether the selection of the specified drawing needs to trigger an update of the
         *  multiple drawing selection.
         */
        this.checkMutipleSelection = function (drawingNode) {
            // another drawing already selected -> creating, expanding or removing a multiple drawing selection
            return self.isMultiSelection() || anotherDrawingSelected(drawingNode);
        };

        /**
         * Adding one specified drawing node into the multiple drawing selection. This function
         * includes the drawing of the artificial drawing selection, the creation of the new
         * drawing selection object and the addition of this new selection object into the
         * multiple drawing container.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing node that needs to be added into the multiple drawing selection.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.addOneDrawingIntoMultipleSelection = function (drawingNode) {

            var // the model
                model = app.getModel(),
                // the active slide node
                slide = model.getSlideById(model.getActiveSlideId()),
                // the logical position of the active slide
                slidePos = model.getActiveSlidePosition(),
                // the logical start position of a drawing
                startPos = Position.getOxoPosition(slide, drawingNode),
                // the logical end position of a drawing
                endPos = Position.increaseLastIndex(startPos);

            self.drawDrawingSelection(app, drawingNode);
            return addOneDrawingSelectionIntoMultiSelection({ drawing: $(drawingNode), startPosition: slidePos.concat(startPos), endPosition: slidePos.concat(endPos) });
        };

        /**
         * Removing one specified drawing node from the multiple drawing selection. This function
         * includes the removal of the artificial drawing selection, the calculation of the logical
         * position of the drawing and the removal of the selection object of this drawing from the
         * multiple drawing container.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing node that needs to be removed from the multiple drawing selection.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.removeOneDrawingFromMultipleSelection = function (drawingNode) {

            var // the model
                model = app.getModel(),
                // the active slide node
                slide = model.getSlideById(model.getActiveSlideId()),
                // the logical position of the active slide
                slidePos = model.getActiveSlidePosition(),
                // the logical start position of a drawing
                startPos = Position.getOxoPosition(slide, drawingNode);

            DrawingFrame.clearSelection(drawingNode);
            return removeOneDrawingSelectionFromMultiSelection(slidePos.concat(startPos));
        };

        /**
         * Handling the multiple drawing selection with respect to the specified
         * drawing node. It is possible, that the drawing is already selected, or that
         * it is not selected. The multiple drawing selection needs to be created,
         * expanded, shrinked or removed.
         *
         * @param {HTMLElement|jQuery} drawingNode
         *  The drawing node that triggers an update of the multiple drawing selection.
         */
        this.handleMultipleSelection = function (drawingNode) {

            var // the jQueryfied drawing node
                $drawingNode = $(drawingNode),
                // the currently selected drawing (or additional text frame drawing)
                selectedDrawing = self.getAnyDrawingSelection(),
                // the logical start and end position of a drawing
                startPos = null, endPos = null,
                // whether the clipboard node should be focused and the browser selection set to it's contents
                focusClipboard = true;

            // handling the group, not a grouped drawing node
            if (DrawingFrame.isGroupedDrawingFrame(drawingNode)) {
                $drawingNode = DrawingFrame.getGroupNode(drawingNode, { farthest: true, rootNode: $drawingNode.closest(DOM.SLIDE_SELECTOR) });
            }

            if (self.isMultiSelection()) {

                // checking, if the specified drawing is already in the selection
                if ($drawingNode.hasClass('selected')) {

                    self.removeOneDrawingFromMultipleSelection($drawingNode);

                    if (self.getMultiSelection().length === 1) {

                        // saving the selection values of the remaining selected drawing
                        startPos = _.clone(multipleSelections[0].startPosition);
                        endPos = _.clone(multipleSelections[0].endPosition);

                        // clearing multiple selection, setting single drawing selection
                        self.clearMultiSelection();

                        self.setTextSelection(startPos, endPos);

                        focusClipboard = false;
                    }

                } else {
                    self.addOneDrawingIntoMultipleSelection($drawingNode);
                }

            } else {

                // checking, if the specified drawing is already in the selection
                if ($drawingNode.hasClass('selected')) {
                    // the one drawing is already selected
                    // -> removing the selection
                    self.clearAllDrawingSelections();

                    // if the drawing was not a text frame, the selection needs to be set into the slide
                    if (!self.isAdditionalTextframeSelection()) {
                        if (app.getModel().useSlideMode()) { self.setSlideSelection({ triggerChange: false }); }
                    }

                } else {

                    // expanding the selection and creating a multiple selection

                    // removing all kind of drawing selections
                    self.clearAllDrawingSelections();

                    // filling two drawings into multiple selection
                    self.addOneDrawingIntoMultipleSelection(selectedDrawing);
                    self.addOneDrawingIntoMultipleSelection($drawingNode);
                }
            }

            // bring clipboard node in sync with the currently selected drawings
            self.updateClipboardNode();
            // focus clipboard node and set the browser selection to it's contents
            if (focusClipboard) {
                self.setFocusIntoClipboardNode();
                self.setBrowserSelectionToContents(self.getClipboardNode(), { preserveFocus: true });
            }

            // informing the listeners about the new selection
            self.trigger('change');
        };

        /**
         * The start handler for the selection box. If the user clicked into the 'div.slide' node
         * or into the 'app-content-root node', the complete selection must be removed.
         * For the selection box, there is a filter set (PresentationModel.MULTI_SELECTION_BOX_FILTER),
         * so that only events on these nodes cause this function to be executed.
         */
        this.selectionStartHandler = function () {

            // removing any selection, if the user clicked on 'div.slide' or in the 'app-content-root' node.
            // -> setting the 'slide selection' ([3,0] for the slide at position [3]).
            // A trigger of 'change' is not required in start handler, because it can be done in the
            // stop handler.
            if (app.getModel().useSlideMode()) { self.setSlideSelection({ triggerChange: false }); }

            // additionally clear a possible multi selection in the slide pane
            app.getModel().trigger('slidepane:clearselection');
        };

        /**
         * Selecting all drawings that are located inside a specified rectangle.
         *
         * @param {Object} box
         *  The object describing the rectangle with the properties top, left, width and height.
         *
         * @returns {Number}
         *  The number of selected drawings.
         */
        this.selectAllDrawingsInSelectionBox = function (box) {

            // selecting all drawings on the active slide, that are completely inside
            // the selection box.

            var // the model
                model = app.getModel(),
                // the active slide node
                slide = model.getSlideById(model.getActiveSlideId()),
                // the container for all drawings on the slide
                allDrawings = slide.children(DOM.ABSOLUTE_DRAWING_SELECTOR),
                // the logical position of the active slide
                slidePos = model.getActiveSlidePosition(),
                // a collector for all selected drawings
                allSelectedDrawings = [],
                // the content root node, the base for the position calculation
                contentRootNode = app.getView().getContentRootNode(),
                // the position of the content root node
                rootNodePos = contentRootNode.offset(),
                // the horizontal scroll shift
                scrollLeft = contentRootNode.scrollLeft(),
                // the vertical scroll shift
                scrollTop = contentRootNode.scrollTop(),
                // whether the trigger for the new selection is required
                doTrigger = true,
                // the logical start and end position of a drawing
                startPos = null, endPos = null,
                // whether the clipboard node should be focused and the browser selection set to it's contents
                focusClipboard = true;

            // a helper function to check, if the drawing is completely
            // inside the specified selection box.
            function nodeIsInsideSelectionBox(drawing) {

                var // the position object
                    pos = drawing.getBoundingClientRect(),
                    // the left position of the drawing
                    drawingLeft = pos.left + scrollLeft - rootNodePos.left,
                    // the top position of the drawing
                    drawingTop = pos.top + scrollTop - rootNodePos.top,
                    // the width of the drawing
                    width = pos.width,
                    // the height of the drawing
                    height = pos.height;

                return (box.left <= drawingLeft) &&
                       (drawingLeft + width <= box.left + box.width) &&
                       (box.top <= drawingTop) &&
                       (drawingTop + height <= box.top + box.height);
            }

            // Setting the focus into the clipboard node.
            // This step is necessary to set the focus correctly (into the clip-board), so that a
            // following keydown (like Ctrl-A) will be handled correctly.
            self.setFocusIntoClipboardNode({ specialTouchHandling: true });

            if (allDrawings.length > 0) {

                _.each(allDrawings, function (oneDrawing) {
                    if (nodeIsInsideSelectionBox(oneDrawing)) {
                        allSelectedDrawings.push(oneDrawing);
                    }
                });

                if (allSelectedDrawings.length > 1) {
                    _.each(allSelectedDrawings, function (oneDrawing) {
                        self.addOneDrawingIntoMultipleSelection(oneDrawing);
                    });
                } else if (allSelectedDrawings.length === 1) {

                    startPos = Position.getOxoPosition(slide, allSelectedDrawings[0]);
                    endPos = Position.increaseLastIndex(startPos);

                    self.setTextSelection(slidePos.concat(startPos), slidePos.concat(endPos));
                    doTrigger = false;
                    focusClipboard = false;

                } else {
                    // no drawing in selection box -> removing selections from drawings required.
                    // The new selection was already set in 'self.selectionStartHandler()' (but not triggered)
                    doTrigger = true;
                }

            } else {
                doTrigger = false;
                focusClipboard = false;
            }

            // bring clipboard node in sync with the currently selected drawings
            self.updateClipboardNode();
            // focus clipboard node and set the browser selection to it's contents
            if (focusClipboard) {
                self.setFocusIntoClipboardNode({ specialTouchHandling: true });
                self.setBrowserSelectionToContents(self.getClipboardNode(), { preserveFocus: true });
            }

            // informing the listeners about the new selection
            if (doTrigger) { self.trigger('change'); }

            return allSelectedDrawings.length;
        };

        /**
         * Getting an array containing all logical start or end positions of the
         * multiple selection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.start=true]
         *      Whether the logical start or the end positions shall be returned
         *      in the array.
         *
         * @returns {Number[][]}
         *  An array with all logical start or end positions.
         */
        this.getArrayOfLogicalPositions = function (options) {

            var // whether the list of start or end positions shall be returned
                useStart = Utils.getBooleanOption(options, 'start', true),
                // the container with all logical positions
                allPositions = [];

            _.each(multipleSelections, function (oneSelection) {
                allPositions.push(useStart ? _.clone(oneSelection.startPosition) : _.clone(oneSelection.endPosition));
            });

            return allPositions;
        };

        /**
         * Getting an array containing all drawing nodes of the multiple selection.
         * The drawing nodes are sorted corresponding to their logical positions.
         *
         * @returns {jQuery[]}
         *  An array with all jQuerified drawing nodes.
         */
        this.getArrayOfSelectedDrawingNodes = function () {

            var // the container with all jQuerified drawing nodes
                allNodes = [];

            _.each(multipleSelections, function (oneSelection) {
                allNodes.push(oneSelection.drawing);
            });

            return allNodes;
        };

        /**
         * Getting the first drawing node of a multiple selection.
         *
         * @returns {jQuery}
         *  The first drawing node of a multiple selection.
         */
        this.getFirstSelectedDrawingNode = function () {
            return multipleSelections[0].drawing;
        };

        /**
         * Getting a string containing all logical start or end positions of the
         * multiple selection.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.start=true]
         *      Whether the logical start or the end positions shall be converted
         *      into a string.
         *
         * @returns {String}
         *  A string with all logical start or end positions.
         */
        this.getListOfPositions = function (options) {

            var // whether the list of start or end positions shall be returned
                useStart = Utils.getBooleanOption(options, 'start', true),
                // the string with all logical positions
                posString = '';

            _.each(multipleSelections, function (oneSelection) {
                var pos = useStart ? oneSelection.startPosition : oneSelection.endPosition;
                posString += '(' + pos.join(',') + '),';

            });

            return posString.substring(0, posString.length - 1);
        };

        /**
         * Whether the drawings in the multi selection can be grouped.
         *
         * @returns {Boolean}
         *  Returns whether the drawings in the multi selection can be grouped.
         */
        this.isGroupableSelection = function () {

            if (!(self.isMultiSelectionSupported() && self.isMultiSelection())) { return false; }

            if (!app.getModel().useSlideMode()) { return true; }

            return !app.getModel().isAtLeastOneNotGroupableDrawingInSelection();
        };

        /**
         * Whether the selection contains a drawing of type 'group', so that it is possible to
         * 'ungroup' the drawing.
         *
         * @returns {Boolean}
         *  Returns, whether the selection contains a drawing of type 'group', so that it is
         *  possible to 'ungroup' the drawing.
         */
        this.isUngroupableSelection = function () {
            return self.isAnyDrawingGroupSelection() || self.isAtLeastOneGroupInMultiSelection();
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // self = null;
        });

        isMultiSelectionApp = app.isMultiSelectionApplication();

    } // class MultiSelectionMixin

    // constants --------------------------------------------------------------

    // export =================================================================

    return MultiSelectionMixin;

});
