/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/tableoperationmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/components/table/table',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/config',
    'io.ox/office/textframework/utils/operations'
], function (Utils, DrawingFrame, AttributeUtils, Table, DOM, Position, Config, Operations) {

    'use strict';

    // mix-in class TableOperationMixin ======================================

    /**
     * A mix-in class for the document model class providing the table
     * operation handling used in a presentation and text document.
     *
     * @constructor
     *
     * @param {EditApplication} app
     *  The application instance.
     */
    function TableOperationMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Removing all cell contents.
         *
         * @param {HTMLElement|jQuery} cellNode
         *  The cell node, whose content will be replaced by an implicit paragraph.
         */
        function implClearCell(cellNode) {

            var // the container for the content nodes
                container = DOM.getCellContentNode(cellNode),
                // the last paragraph in the cell
                paragraph = null;

            paragraph = DOM.createParagraphNode();
            container.empty().append(paragraph);
            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            self.repairEmptyTextNodes(paragraph);

            // validate the paragraph (add the dummy node)
            self.validateParagraphNode(paragraph);
            // and NOT formatting the implicit paragraph -> leads to 'jumping' of table after inserting row or column
            // paragraphStyles.updateElementFormatting(paragraph);
        }

        /**
         * Handler for inserting a new table.
         *
         * @param {Object} operation
         *  The insertTable operation.
         *
         * @returns {Boolean}
         *  Whether the table has been inserted successfully.
         */
        function implInsertTable(operation) {

            var // the new table
                table = $('<table>').attr('role', 'grid').append($('<colgroup>')),
                // insert the table into the DOM tree
                inserted,
                // new implicit paragraph behind the table
                newParagraph = null,
                // number of rows in the table (only for operation.sizeExceeded)
                tableRows,
                // number of columns in the table (only for operation.sizeExceeded)
                tableColumns,
                // number of cells in the table (only for operation.sizeExceeded)
                tableCells,
                // element from which pagebreaks calculus is started downwards
                currentElement,
                // specifies which part of the table exceeds the size
                overflowPart,
                // the specific value which exceeds the size
                overflowValue,
                // the maximal allowed value for the part
                maxValue,
                // undo operation for handled operation
                undoOperation = {};

            if (operation.start.length > 2) {
                // if its paragraph creation inside of table
                currentElement = Position.getContentNodeElement(self.getNode(), operation.start.slice(0, 1));
            } else {
                currentElement = Position.getContentNodeElement(self.getNode(), operation.start);
            }

            if (operation.target) {
                inserted = self.insertContentNode(_.clone(operation.start), table, operation.target);
            } else {
                inserted = self.insertContentNode(_.clone(operation.start), table);
            }

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // generate undo/redo operations
            if (self.getUndoManager().isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                self.getUndoManager().addUndo(undoOperation, operation);
            }

            // Special replacement setting for a table, that cannot be displayed because of its size.
            if (_.isObject(operation.sizeExceeded)) {

                tableRows = Utils.getIntegerOption(operation.sizeExceeded, 'rows', 0);
                tableColumns = Utils.getIntegerOption(operation.sizeExceeded, 'columns', 0);
                tableCells = tableRows * tableColumns;

                if ((tableRows > 0) && (tableColumns > 0)) {

                    if (tableRows > Config.MAX_TABLE_ROWS) {
                        overflowPart = 'rows';
                        overflowValue = tableRows;
                        maxValue = Config.MAX_TABLE_ROWS;
                    } else if (tableColumns > Config.MAX_TABLE_COLUMNS) {
                        overflowPart = 'cols';
                        overflowValue = tableColumns;
                        maxValue = Config.MAX_TABLE_COLUMNS;
                    } else if (tableCells > Config.MAX_TABLE_CELLS) {
                        overflowPart = 'cols';
                        overflowValue = tableCells;
                        maxValue = Config.tableCells;
                    }
                }
                DOM.makeExceededSizeTable(table, overflowPart, overflowValue, maxValue);
                table.data('attributes', operation.attrs);  // setting attributes from operation directly to data without using 'setElementAttributes'

            // apply the passed table attributes
            } else if (_.isObject(operation.attrs) && _.isObject(operation.attrs.table)) {

                // We don't want operations to create operations themself, so we can't
                // call 'checkForLateralTableStyle' here.

                if (self.getBlockKeyboardEvent()  && _.browser.Firefox) { table.data('internalClipboard', true); }  // Fix for task 29401
                self.getTableStyles().setElementAttributes(table, operation.attrs);
            }

            if (operation.target) {
                // trigger header/footer content update on other elements of same type, if change was made inside header/footer
                if (DOM.isMarginalNode(table) || DOM.getMarginalTargetNode(self.getNode(), table).length) {
                    self.updateEditingHeaderFooterDebounced(DOM.getMarginalTargetNode(self.getNode(), table));
                }
                self.setBlockOnInsertPageBreaks(true);
            }

            // call pagebreaks re-render after inserting table
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(table));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            // adding an implicit paragraph behind the table (table must not be the last node in the document)
            if (table.next().length === 0) {
                newParagraph = DOM.createImplicitParagraphNode();
                self.validateParagraphNode(newParagraph);
                // insert the new paragraph behind the existing table node
                table.after(newParagraph);
                self.implParagraphChanged(newParagraph);
                // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                // newly, also in other browsers (new jQuery version?)
                self.repairEmptyTextNodes(newParagraph);
            } else if (DOM.isImplicitParagraphNode(table.next())) {
                if (DOM.isTableInTableNode(table)) {
                    table.next().css('height', 0);  // hiding the paragraph behind the table inside another table
                }
            }

            if (self.isUndoRedoRunning()  && _.browser.Firefox) { table.data('undoRedoRunning', true); }  // Fix for task 30477

            return true;
        }

        /**
         * The hander function that inserts row(s) into a table in the DOM.
         *
         * @param {Number[]} start
         *  The logical position for inserting the column.
         *
         * @param {Number} [count=1]
         *  The number of rows to be inserted.
         *
         * @param {Boolean} insertDefaultCells
         *  Whether default cells shall be inserted in the new rows.
         *
         * @param {Number} [referenceRow]
         *  Whether a specified reference row can be used for the new row.
         *
         * @param {Object} [attrs]
         *  An object with the row attributes.
         *
         * @param {String} [target]
         *  The target corresponding to the logical position.
         *
         * @returns {Boolean}
         *  Whether the table column has been inserted successfully.
         */
        function implInsertRows(start, count, insertDefaultCells, referenceRow, attrs, target) {

            var localPosition = _.copy(start, true),
                useReferenceRow = _.isNumber(referenceRow),
                newRow = null,
                currentElement,
                rootNode = self.getRootNode(target),
                table = null; // the table node

            currentElement = Position.getContentNodeElement(self.getNode(), start.slice(0, 1));

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return false;
            }

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            var tablePos = _.copy(localPosition, true);
            tablePos.pop();

            if (self.useSlideMode()) {
                table = Position.getDOMPosition(rootNode, tablePos, true).node; // returns the drawing that contains the table node
                if (DrawingFrame.isTableDrawingFrame(table)) {
                    table = $(table).find('table'); // getting the table node inside the table drawing
                }
            } else {
                table = Position.getDOMPosition(rootNode, tablePos).node;
            }

            var tableRowDomPos = Position.getDOMPosition(rootNode, localPosition),
                tableRowNode = null,
                row = null,
                cellsInserted = false;

            if (!table) { return false; } // not a valid table position specified

            if (self.isGUITriggeredOperation()) {  // called from 'insertRow' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            if (tableRowDomPos) {
                tableRowNode = tableRowDomPos.node;
            }

            if (useReferenceRow) {

                if (self.isGUITriggeredOperation()) {
                    // Performance: Called from 'insertRow' -> simple table do not need new formatting.
                    // Tables with conditional styles only need to be updated below the referenceRow
                    // (and the row itself, too, for example southEast cell may have changed)

                    // Bugfix 44230: introduced a check to prevent wrong values for '.data('reducedTableFormatting')' in updateRowFormatting(), when inserting
                    // rows very quickly via button, so that more than one rows are inserted before the debounced updateRowFormatting() is executed
                    if (_.isNumber($(table).data('reducedTableFormatting'))) {
                        $(table).data('reducedTableFormatting', Math.min(referenceRow, $(table).data('reducedTableFormatting')));
                    } else {
                        $(table).data('reducedTableFormatting', referenceRow);
                    }
                }

                row = DOM.getTableRows(table).eq(referenceRow).clone(true);

                // clear the cell contents in the cloned row
                row.children('td').each(function () {
                    implClearCell(this);
                });

                cellsInserted = true;

            } else if (insertDefaultCells) {

                var columnCount = Table.getColumnCount(table),
                    // prototype elements for row, cell, and paragraph
                    paragraph = DOM.createParagraphNode(), // no implicit paragraph
                    cell = DOM.createTableCellNode(paragraph);

                // insert empty text node into the paragraph
                self.validateParagraphNode(paragraph);

                row = $('<tr>').attr('role', 'row');

                // clone the cells in the row element
                _.times(columnCount, function () { row.append(cell.clone(true).attr('role', 'gridcell')); });

                cellsInserted = true;

            } else {
                row = $('<tr>').attr('role', 'row');
            }

            _.times(count, function () {
                newRow = row.clone(true);

                // apply the passed attributes
                if (_.isObject(attrs)) { self.getTableRowStyles().setElementAttributes(newRow, attrs); }

                if (tableRowNode) {
                    // insert the new row before the existing row at the specified position
                    $(tableRowNode).before(newRow);
                } else {
                    // append the new row to the table
                    $(table).append(newRow);
                }

                // in Internet Explorer it is necessary to add new empty text nodes in rows again
                // newly, also in other browsers (new jQuery version?)
                self.repairEmptyTextNodes(newRow);

            });

            // recalculating the attributes of the table cells
            if (cellsInserted && self.requiresElementFormattingUpdate()) { self.implTableChanged(table); }

            // Setting cursor
            if (insertDefaultCells || useReferenceRow) {
                localPosition.push(0);
                localPosition.push(0);
                localPosition.push(0);

                self.setLastOperationEnd(localPosition);
            }

            // if operation is not targeted, render page breaks
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            self.setQuitFromPageBreaks(true); // quit from any currently running pagebreak calculus - performance optimization
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(table));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            return true;
        }

        /**
         * The hander function that inserts the column into a table in the DOM.
         *
         * @param {Number[]} start
         *  The logical position for inserting the column.
         *
         * @param {Number} gripPosition
         *  The index of the new column inside the table grid.
         *
         * @param {String} [insertMode]
         *  The insert mode determines whether the column is inserted behind or before
         *  an existing column. Currently only the value 'behind' is evaluated. In all
         *  other cases, the new column is inserted before an existing column.
         *
         * @param {String} [target]
         *  The target corresponding to the logical position.
         *
         * @returns {Boolean}
         *  Whether the table column has been inserted successfully.
         */
        function implInsertColumn(start, gridPosition, insertMode, target) {

            var localPosition = _.copy(start, true),
                rootNode = self.getRootNode(target),
                table = null,
                allRows = null,
                // whether the presentation model needs to be used
                isSlideMode = self.useSlideMode();

            if (!Position.isPositionInTable(rootNode, localPosition)) {
                return false;
            }

            table = isSlideMode ? Position.getDOMPosition(rootNode, localPosition, true).node : Position.getDOMPosition(rootNode, localPosition).node;
            allRows = DOM.getTableRows(table);

            if (isSlideMode && DrawingFrame.isTableDrawingFrame(table)) { table = $(table).find('table'); }

            if (self.isGUITriggeredOperation()) {  // called from 'insertColumn' -> temporary branding for table
                $(table).data('gui', 'insert');
            }

            _.each(allRows, function (row) {

                var cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                    cellNode = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).eq(cellPosition),
                    expCellAttributes = AttributeUtils.getExplicitAttributes(cellNode, { family: 'cell' }),
                    isMergedCell = expCellAttributes && _.isNumber(expCellAttributes.gridSpan) && expCellAttributes.gridSpan > 1,
                    // gridPos = null,
                    cellClone = null;
                    // increaseCellLength = false;

                  // -> this process simulates the PowerPoint behavior.
                  // -> merged table cells are increased in length, no new table cell is inserted
//                if (isSlideMode) {
//
//                    if (isMergedCell) {
//
//                        gridPos = Table.getGridPositionFromCellPosition(row, cellPosition);
//
//                        if ((insertMode === 'behind' && gridPos.end > gridPosition) || (insertMode !== 'behind' && gridPos.start < gridPosition)) {
//                            increaseCellLength = true;
//                        }
//                    }
//
//                    if (increaseCellLength) {
//                        // apply the passed table cell attributes
//                        self.getTableCellStyles().setElementAttributes(cellNode, { cell: { gridSpan: (expCellAttributes.gridSpan + 1) } });
//
//                    } else {
//                        // inserting a new cell
//                        cellClone = cellNode.clone(true);
//                        implClearCell(cellClone);
//
//                        if (insertMode === 'behind') {
//                            cellClone.insertAfter($(row).children().get(cellPosition));
//                        } else {
//                            cellClone.insertBefore($(row).children().get(cellPosition));
//                        }
//
//                        if (isMergedCell) {
//                            self.getTableCellStyles().setElementAttributes(cellClone, { cell: { gridSpan: 1 } });
//                        }
//                    }
//
//                } else {

                // this behavior is the OX Text behavior, that client and filter also support in OX Presentation

                cellClone = cellNode.clone(true);

                implClearCell(cellClone);

                if (insertMode === 'behind') {
                    cellClone.insertAfter($(row).children().get(cellPosition));
                } else {
                    cellClone.insertBefore($(row).children().get(cellPosition));
                }

                if (isMergedCell) {
                    self.getTableCellStyles().setElementAttributes(cellClone, { cell: { gridSpan: 1 } });
                }

//                }

                if (cellClone) {
                    // in Internet Explorer it is necessary to add new empty text nodes in columns again
                    // newly, also in other browsers (new jQuery version?)
                    self.repairEmptyTextNodes(cellClone);
                }

            });

            // recalculating the attributes of the table cells
            if (self.requiresElementFormattingUpdate()) {
                self.implTableChanged(table);
            }

            // Setting cursor to first position in table
            localPosition.push(0);
            localPosition.push(gridPosition);
            localPosition.push(0);
            localPosition.push(0);

            self.setLastOperationEnd(localPosition);

            return true;
        }

        /**
         * Splitting a table at a specified logical position.
         *
         * @param {Number[]} position
         *  The logical position at which the table shall be splitted.
         *
         * @param {String[]} target
         *  ID of target root node for operation
         *
         * @returns {Boolean}
         *  Whether the table has been splitted successfully.
         */
        function implSplitTable(position, target) {
            var
                positionRowToSplit = _.last(position),
                // the position of the 'old' table
                tablePosition = position.slice(0, -1),
                // root node
                rootNode = self.getRootNode(target),
                // the 'old' table node
                tableNode = Position.getContentNodeElement(rootNode, tablePosition),
                // the position of the 'new' table
                //newTablePosition = Position.increaseLastIndex(tablePosition),
                newTableNode = $(tableNode).clone(true),
                // collection of first level table rows, excluding rows containing page break
                tableNodeRows = DOM.getTableRows(tableNode),
                // collection of first level new table rows, excluding rows containing page break
                newTableNodeRows = DOM.getTableRows(newTableNode);

            // if we clone table with manual page break, it should not be inherited in new table
            if (newTableNode.hasClass('manual-page-break')) {
                newTableNode.find(DOM.MANUAL_PAGE_BREAK_SELECTOR).removeClass('manual-page-break');
                newTableNode.removeClass('manual-page-break');
            }
            // remove ending rows from old table
            $(tableNodeRows[positionRowToSplit]).nextAll('tr').addBack().remove();
            // remove begining rows from new table
            $(newTableNodeRows[positionRowToSplit]).prevAll('tr').remove();
            newTableNode.insertAfter($(tableNode));

            self.implTableChanged(tableNode);
            self.implTableChanged(newTableNode);

            self.repairEmptyTextNodes(newTableNode, { allNodes: true }); // IE fix for empty table impl paragraphs after split, #34632

            // new cursor position at merge position
            self.setLastOperationEnd(_.clone(Position.getFirstPositionInParagraph(rootNode, position)));

            // run page break render if operation is not targeted
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            self.insertPageBreaks(tableNode, DrawingFrame.getClosestTextFrameDrawingNode(tableNode));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            return true;
        }

        /**
         * Merging two tables.
         *
         * @param {Object} operation - Contains name of the operation, start - the logical position
         *  of table from where merging should start, and optionally target ID of root node
         *
         * @returns {Boolean}
         *  Whether the tables are merged successfully.
         */
        function implMergeTable(operation) {
            var
                // the position of the 'old' table
                tablePosition = operation.start,
                // currently active root node
                rootNode = self.getRootNode(operation.target),
                // the 'old' table node
                tableNode = Position.getContentNodeElement(rootNode, tablePosition),
                // new table node, next to old table
                nextTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: true }),
                // cached collection of rows for next table
                nextTableNodeRows = DOM.getTableRows(nextTableNode),
                // position of next table
                nextTablePosition = _.clone(tablePosition),
                // information for table split, if undo is called:
                // position of last row inside table
                rowPosInTable,
                // helper for stored value of row position in iteration
                rowPosition,
                // helper for stored value of cell position in iteration
                cellPosition,
                // logical position of table and its last row
                tableAndRowPosition,
                // the undo manager object
                undoManager = self.getUndoManager(),
                //generator of operations
                generator = undoManager.isUndoEnabled() ? self.createOperationsGenerator() : null,
                // created operation
                newOperation = null;

            nextTablePosition[nextTablePosition.length - 1] += 1;
            // calculate last row position from current table, used for undo table split operation
            rowPosInTable = Position.getLastRowIndexInTable(rootNode, tablePosition);
            if (rowPosInTable > -1) {
                // increased by 1 at the end, because split starts from above of passed row
                rowPosInTable += 1;
            } else { // invalid value
                rowPosInTable = 0;
            }
            tableAndRowPosition = Position.appendNewIndex(tablePosition, rowPosInTable);

            if (DOM.isTableNode(nextTableNode)) {
                if (generator) {
                    newOperation = { start: tableAndRowPosition };
                    self.extendPropertiesWithTarget(newOperation, operation.target);
                    generator.generateOperation(Operations.TABLE_SPLIT, newOperation);
                    generator.generateSetAttributesOperation(nextTableNode, { start: nextTablePosition }); // generate undo operation to set attributes to table
                    //generate set attributes for rows and cells
                    _.each(nextTableNodeRows, function (row, index) {
                        rowPosition = _.clone(nextTablePosition);
                        rowPosition.push(index);
                        generator.generateSetAttributesOperation(row, { start: rowPosition });
                        _.each(row.cells, function (cell, index) {
                            cellPosition = _.clone(rowPosition);
                            cellPosition.push(index);
                            generator.generateSetAttributesOperation(cell, { start: cellPosition });
                        });
                    });
                    undoManager.addUndo(generator.getOperations(), operation);
                }

                $(tableNode).find('> tbody').last().append($(nextTableNodeRows));
                $(nextTableNode).remove();

                self.implTableChanged(tableNode);

                // new cursor position at merge position
                self.setLastOperationEnd(_.clone(Position.getFirstPositionInParagraph(rootNode, tablePosition)));

                // run page break render if operation is not targeted
                if (operation.target) {
                    self.setBlockOnInsertPageBreaks(true);
                }

                self.insertPageBreaks(tableNode, DrawingFrame.getClosestTextFrameDrawingNode(tableNode));
                if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

                return true;
            } else {
                Utils.error('implMergeTable: Element following the table is not a table.');
                return false;
            }
        }

        /**
         * The hander function that merges table cells in the DOM.
         *
         * @param {Number[]} start
         *  The logical position of the row.
         *
         * @param {Number} count
         *  The number of cells to be merged.
         *
         * @param {String} [target]
         *  The target corresponding to the logical position.
         *
         * @returns {Boolean}
         *  Whether the cells were merged successfully.
         */
        function implMergeCell(start, count, target) {

            var rowPosition = _.copy(start, true),
                localStartCol = rowPosition.pop(),
                localEndCol = localStartCol + count,
                rootNode = self.getRootNode(target),
                // Counting the colSpan off all cells in the range
                row = Position.getDOMPosition(rootNode, rowPosition).node,
                allSelectedCells = $(row).children().slice(localStartCol, localEndCol + 1),
                colSpanSum = Table.getColSpanSum(allSelectedCells),
                // Shifting the content of all following cells to the first cell
                targetCell = $(row).children().slice(localStartCol, localStartCol + 1),
                sourceCells = $(row).children().slice(localStartCol + 1, localEndCol + 1);

            Table.shiftCellContent(targetCell, sourceCells);

            sourceCells.remove();

            // apply the passed table cell attributes
            self.getTableCellStyles().setElementAttributes(targetCell, { cell: { gridSpan: colSpanSum } });

            return true;
        }

        /**
         * The hander function that inserting table cells in the DOM.
         *
         * @param {Number[]} start
         *  The logical start position.
         *
         * @param {Number} count
         *  The number of cells to be inserted.
         *
         * @param {Object} [attrs]
         *  Some optional table cell attributes.
         *
         * @param {String} [target]
         *  The target corresponding to the logical position.
         *
         * @returns {Boolean}
         *  Whether the cells were inserted successfully.
         */
        function implInsertCells(start, count, attrs, target) {

            var localPosition = _.clone(start),
                tableNode,
                tableCellDomPos = null,
                tableCellNode = null,
                paragraph = null,
                cell = null,
                row = null,
                rootNode = self.getRootNode(target),
                selector = self.useSlideMode() ? DOM.TABLE_NODE_AND_DRAWING_SELECTOR : DOM.TABLE_NODE_SELECTOR;

            tableNode = Position.getLastNodeFromPositionByNodeName(rootNode, start, selector);

            if (!tableNode) {
                return;
            }

            if (DrawingFrame.isTableDrawingFrame(tableNode)) {
                // the table is a drawing, but the table node is required
                tableNode = $(tableNode).find('table');
            }

            if (!_.isNumber(count)) {
                count = 1; // setting default for number of rows
            }

            tableCellDomPos = Position.getDOMPosition(rootNode, localPosition);

            if (tableCellDomPos) {
                tableCellNode = tableCellDomPos.node;
            }

            // prototype elements for row, cell, and paragraph
            paragraph = DOM.createImplicitParagraphNode();  // must be implicit because of undo-operation

            // insert empty text node into the paragraph
            self.validateParagraphNode(paragraph);

            cell = DOM.createTableCellNode(paragraph);

            // apply the passed table cell attributes
            if (_.isObject(attrs)) {
                self.getTableCellStyles().setElementAttributes(cell, attrs);
            }

            if (tableCellNode) {
                _.times(count, function () { $(tableCellNode).before(cell.clone(true)); });
                row = $(tableCellNode).parent();
            } else {
                var rowPos = localPosition.slice(0, -1);
                row = Position.getDOMPosition(rootNode, rowPos).node;
                _.times(count, function () { $(row).append(cell.clone(true)); });
            }

            // setting cursor to first paragraph in the cell
            localPosition.push(0);
            localPosition.push(0);
            self.setLastOperationEnd(localPosition);

            // in Internet Explorer it is necessary to add new empty text nodes in rows again
            // newly, also in other browsers (new jQuery version?)
            self.repairEmptyTextNodes(row);

            // recalculating the attributes of the table cells
            if (self.requiresElementFormattingUpdate()) {
                self.implTableChanged(tableNode);
            }

            return true;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserting a table into the document.
         * The undo manager returns the return value of the callback function.
         * For inserting a table it is important, that there is always a
         * paragraph between the new table and the neighboring table. This is
         * important for merging of tables, where the removal of this paragraph
         * leads to an automatic merge.
         *
         * @param {Object} size
         *  An object containing the properties 'width' and 'height' for the
         *  column and row count of the table.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved if the dialog has been closed with
         *  the default action; or rejected, if the dialog has been canceled.
         *  If no dialog is shown, the promise is resolved immediately.
         */
        this.insertTable = function (size) {

            // undo manager returns the return value of the callback function
            return self.getUndoManager().enterUndoGroup(function () {

                var // cursor position used to split the paragraph
                    startPosition = null,
                    // paragraph to be split for the new table, and its position
                    paragraph = null, position = null,
                    // text offset in paragraph, first and last text position in paragraph
                    offset = 0, startOffset = 0, endOffset = 0,
                    // table attributes
                    attributes = { table: { tableGrid: [], width: 'auto' } },
                    // table row and paragraph attributes
                    rowAttributes = null, rowOperationAttrs = null, paraOperationAttrs = {},
                    // default table style
                    tableStyleId = self.getDefaultUITableStylesheet(),
                    // a new implicit paragraph
                    newParagraph = null,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // whether an additional paragraph is required behind or before the table
                    insertAdditionalParagraph = false,
                    // the logical position of the additional paragraph
                    paraPosition = null,
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // container root node of table
                    rootNode = self.getCurrentRootNode(),
                    // properties to be passed to generator for insert table operation
                    generatorProperties = {},
                    // the selection object
                    selection = self.getSelection();

                // #49674 - odf's default table style has no borders, so we overwrite it with 'TableGridOx'
                //  'TableGrid' can be already sent with document, and it also has no border data
                if (app.isODF() && tableStyleId === '_default') {
                    tableStyleId = self.getDefaultLateralTableODFDefinition().styleId;
                }

                function doInsertTable() {

                    startPosition = selection.getStartPosition();
                    position = startPosition.slice(0, -1);
                    paragraph = Position.getParagraphElement(rootNode, position);
                    if (!paragraph) { return; }

                    if (!DOM.isImplicitParagraphNode(paragraph)) {
                        // split paragraph, if the cursor is between two characters,
                        // or if the paragraph is the very last in the container node
                        offset = _.last(startPosition);
                        startOffset = Position.getFirstTextNodePositionInParagraph(paragraph);
                        endOffset = Position.getLastTextNodePositionInParagraph(paragraph);

                        if ((!paragraph.nextSibling) && (offset === endOffset)) {
                            if (DOM.isCellContentNode(paragraph.parentNode)) {
                                insertAdditionalParagraph = true; // inserting a valid paragraph behind the table in a table cell
                                position = Position.increaseLastIndex(position);
                                paraPosition = _.clone(position);
                                paraPosition = Position.increaseLastIndex(paraPosition);
                            } else {
                                // create a new empty implicit paragraph behind the table (if not inside another table)
                                newParagraph = DOM.createImplicitParagraphNode();
                                self.validateParagraphNode(newParagraph);
                                $(paragraph).after(newParagraph);
                                self.implParagraphChanged(newParagraph);
                                position = Position.increaseLastIndex(position);
                                // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
                                // newly, also in other browsers (new jQuery version?)
                                self.repairEmptyTextNodes(newParagraph);
                            }
                        } else if ((startOffset < offset) && (offset < endOffset)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if ((!paragraph.nextSibling) && (offset > 0)) {
                            self.splitParagraph(startPosition);
                            position = Position.increaseLastIndex(position);
                        } else if (offset === endOffset) {
                            // cursor at the end of the paragraph: insert before next content node
                            position = Position.increaseLastIndex(position);
                            // additionally adding an empty paragraph behind the table, if the following
                            // node is a table. Therefore there is always one paragraph between two tables.
                            // If this paragraph is removed, this leads to a merge of the tables (if this
                            // is possible).
                            if (paragraph.nextSibling && DOM.isTableNode(paragraph.nextSibling)) {
                                insertAdditionalParagraph = true;
                                paraPosition = _.clone(position);
                                paraPosition = Position.increaseLastIndex(paraPosition);
                            }
                        }
                    } else {
                        // if this is an explicit paragraph and the previous node is a table,
                        // there must be a new paragraph before the new table.
                        if (paragraph.previousSibling && DOM.isTableNode(paragraph.previousSibling)) {
                            insertAdditionalParagraph = true;
                            paraPosition = _.clone(position);
                        }
                    }

                    // prepare table column widths (values are relative to each other)
                    _(size.width).times(function () { attributes.table.tableGrid.push(1000); });

                    // set default table style
                    if (_.isString(tableStyleId)) {

                        // insert a pending table style if needed
                        Table.checkForLateralTableStyle(generator, self, tableStyleId);

                        // add table style name to attributes
                        attributes.styleId = tableStyleId;

                        // default: tables do not have last row, last column and vertical bands
                        attributes.table.exclude = ['lastRow', 'lastCol', 'bandsVert'];
                    }

                    // modifying the attributes, if changeTracking is activated
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        attributes.changes = { inserted: self.getChangeTrack().getChangeTrackInfo() };
                        rowAttributes = { changes: { inserted: self.getChangeTrack().getChangeTrackInfo() } };
                    }

                    // insert the table, and add empty rows

                    generatorProperties = { start: _.clone(position), attrs: attributes };
                    self.extendPropertiesWithTarget(generatorProperties, target);
                    generator.generateOperation(Operations.TABLE_INSERT, generatorProperties);

                    rowOperationAttrs = { start: Position.appendNewIndex(position, 0), count: size.height, insertDefaultCells: true };
                    self.extendPropertiesWithTarget(rowOperationAttrs, target);
                    if (rowAttributes) { rowOperationAttrs.attrs = rowAttributes; }
                    generator.generateOperation(Operations.ROWS_INSERT, rowOperationAttrs);

                    // also adding a paragraph behind the table, if there is a following table
                    if (insertAdditionalParagraph) {
                        // modifying the attributes, if changeTracking is activated
                        if (self.getChangeTrack().isActiveChangeTracking()) {
                            paraOperationAttrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo() };
                        }
                        generatorProperties =  _.isEmpty(paraOperationAttrs) ? { start: paraPosition } : { start: paraPosition, attrs: paraOperationAttrs };
                        self.extendPropertiesWithTarget(generatorProperties, target);
                        generator.generateOperation(Operations.PARA_INSERT, generatorProperties);
                    }

                    // apply all collected operations
                    self.applyOperations(generator);

                    // set the cursor to first paragraph in first table cell
                    selection.setTextSelection(position.concat([0, 0, 0, 0]));
                }

                if (selection.hasRange()) {
                    return self.deleteSelected()
                    .done(function () {
                        doInsertTable();
                    });
                }

                doInsertTable();
                return $.when();

            }, this); // enterUndoGroup()

        };

        /**
         * Inserting a table row into a table in the document.
         */
        this.insertRow = function () {

            if (!this.isRowAddable()) {
                app.getView().rejectEditTextAttempt('tablesizerow'); // checking table size (26809)
                return;
            }

            self.getUndoManager().enterUndoGroup(function () {

                var // inserting only one row
                    count = 1,
                    // whether default cells shall be inserted into the new row
                    insertDefaultCells = false,
                    // the current logical position of the selection
                    position = self.getSelection().getEndPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // current root container node for element
                    rootNode = self.getCurrentRootNode(target),
                    // the row number from the logical position
                    referenceRow = null,
                    // the html dom node of the row
                    rowNode = null,
                    // the logical position of the row
                    rowPos = Position.getLastPositionFromPositionByNodeName(rootNode, position, 'tr'),
                    // the row attributes
                    rowAttrs = {},
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // created operation
                    newOperation = null;

                if (rowPos !== null) {

                    rowNode = Position.getTableRowElement(rootNode, rowPos);

                    referenceRow = _.last(rowPos);

                    rowPos[rowPos.length - 1] += 1;

                    // modifying the attributes, if changeTracking is activated
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        rowAttrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null };
                    } else if (DOM.isChangeTrackNode(rowNode)) {
                        // change tracking is not active, but reference row has change track attribute -> this attribute needs to be removed
                        rowAttrs.changes = { inserted: null, removed: null, modified: null };
                    }

                    newOperation = { start: rowPos, count: count, insertDefaultCells: insertDefaultCells, referenceRow: referenceRow, attrs: rowAttrs };
                    self.extendPropertiesWithTarget(newOperation, target);
                    generator.generateOperation(Operations.ROWS_INSERT, newOperation);

                    // cells with list in the first paragraph, should automatically receive this list information, too
                    if (rowNode) {
                        // checking the cell contents in the cloned row
                        $(rowNode).children('td').each(function () {
                            var // the container for the content nodes
                                container = DOM.getCellContentNode(this),
                                // the first paragraph in the cell determines the attributes for the new cell
                                paragraph = container[0].firstChild,
                                // the paragraph attributes
                                paraAttributes = null,
                                // the logical position of the new paragraph
                                paraPos = _.clone(rowPos);

                            if (DOM.isParagraphNode(paragraph)) {
                                paraAttributes = AttributeUtils.getExplicitAttributes(paragraph);

                                if (!_.isEmpty(paraAttributes)) {
                                    paraPos.push($(this).prevAll().length);
                                    paraPos.push(0);
                                    // generating an operation for setting paragraph attributes at the paragraphs in the new row
                                    newOperation = { start: paraPos, attrs: paraAttributes };
                                    self.extendPropertiesWithTarget(newOperation, target);
                                    generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                                }
                            }
                        });
                    }

                    self.setGUITriggeredOperation(true);
                    // apply all collected operations
                    this.applyOperations(generator);
                    self.setGUITriggeredOperation(false);
                }

                // setting the cursor position
                self.getSelection().setTextSelection(self.getLastOperationEnd());

            }, this); // enterUndoGroup()
        };

        /**
         * Inserting a table column into a table in the document.
         */
        this.insertColumn = function () {

            if (!this.isColumnAddable()) {
                app.getView().rejectEditTextAttempt('tablesizecolumn'); // checking table size (26809)
                return;
            }

            self.getUndoManager().enterUndoGroup(function () {

                var // the selection object
                    selection = self.getSelection(),
                    // the selector for searching the table position
                    selector = self.useSlideMode ? DOM.TABLE_NODE_AND_DRAWING_SELECTOR : DOM.TABLE_NODE_SELECTOR,
                    // the logical end position of the selection
                    position = selection.getEndPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    rootNode = self.getCurrentRootNode(target),
                    cellPosition = Position.getColumnIndexInRow(rootNode, position),
                    tablePos = Position.getLastPositionFromPositionByNodeName(rootNode, position, selector),
                    rowNode = Position.getLastNodeFromPositionByNodeName(rootNode, position, 'tr'),
                    insertMode = 'behind',
                    gridPosition = Table.getGridPositionFromCellPosition(rowNode, cellPosition).end, // inserting behind the end of the grid
                    tableGrid = Table.getTableGridWithNewColumn(rootNode, tablePos, gridPosition, insertMode),
                    // table node element
                    tablePoint = Position.getDOMPosition(rootNode, tablePos, true),
                    // all rows in the table
                    allRows = DOM.getTableRows(tablePoint.node),
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    // the attributes assigned to the cell(s)
                    cellAttrs = null,
                    // created operation
                    newOperation = null;

                newOperation = { start: tablePos, tableGrid: tableGrid, gridPosition: gridPosition, insertMode: insertMode };
                self.extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.COLUMN_INSERT, newOperation);

                // Setting new table grid attribute to table
                newOperation = { attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                self.extendPropertiesWithTarget(newOperation, target);
                generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);

                // Iterating over all rows to keep paragraph attributes of first paragraph in each new cell
                // -> these paragraphs must not be explicit
                if (allRows) {
                    allRows.each(function (i, row) {
                        var // the current logical cell position
                            cellPosition = Table.getCellPositionFromGridPosition(row, gridPosition),
                            // the current cell node
                            cellClone = $(row).children(DOM.TABLE_CELLNODE_SELECTOR).slice(cellPosition, cellPosition + 1),
                            // the container for the content nodes
                            container = null,
                            // the first paragraph in the cell determines the attributes for the new cell
                            paragraph = null,
                            // whether the current or the following cell is used to find paragraph attributes
                            currentCellUsed = true,
                            // the paragraph attributes
                            paraAttributes = null,
                            // the position of the cell inside its row
                            cellNumber = null,
                            // the logical position of the row
                            rowPos = Position.getOxoPosition(rootNode, row, 0),
                            // the logical position of the new cell
                            cellPos = _.clone(rowPos),
                            // the logical position of the new paragraph
                            paraPos = _.clone(rowPos);

                        // using the current cell only, if it is the last cell in the row. Otherwise the attributes from the
                        // following cell need to be examined.
                        if (cellClone.next().length > 0) {
                            cellClone = cellClone.next();
                            currentCellUsed = false;
                        }

                        container = DOM.getCellContentNode(cellClone);
                        paragraph = container[0].firstChild;

                        if (DOM.isParagraphNode(paragraph)) {
                            paraAttributes = AttributeUtils.getExplicitAttributes(paragraph);

                            if (!_.isEmpty(paraAttributes)) {
                                cellNumber = $(cellClone).prevAll().length;
                                if (currentCellUsed) { cellNumber++; }
                                paraPos.push(cellNumber);
                                paraPos.push(0);
                                // generating an operation for setting paragraph attributes in the new row
                                // -> no implicit paragraph in these cells
                                newOperation = { start: paraPos, attrs: paraAttributes };
                                self.extendPropertiesWithTarget(newOperation, target);
                                generator.generateOperation(Operations.PARA_INSERT, newOperation);
                            }
                        }

                        // modifying the cell attributes, if changeTracking is activated or if change track attributes need to be removed
                        cellAttrs = null;
                        if (self.getChangeTrack().isActiveChangeTracking()) {
                            cellAttrs = { changes: { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null } };
                        } else if (DOM.isChangeTrackNode(cellClone)) {
                            // change tracking is not active, but reference cell has change track attribute -> this attribute needs to be removed
                            cellAttrs = { changes: { inserted: null, removed: null, modified: null } };
                        }

                        if (cellAttrs !== null) {
                            cellPos = _.clone(rowPos);
                            cellNumber = cellNumber !== null ? cellNumber : $(cellClone).prevAll().length;
                            cellPos.push(cellNumber);
                            newOperation = { attrs: cellAttrs, start: _.clone(cellPos) };
                            self.extendPropertiesWithTarget(newOperation, target);
                            generator.generateOperation(Operations.SET_ATTRIBUTES, newOperation);
                        }

                    });
                }

                // no call of implTableChanged in implInsertColumn
                self.setRequiresElementFormattingUpdate(false);  // no call of implTableChanged -> attributes are already set in implSetAttributes
                self.setGUITriggeredOperation(true);

                // apply all collected operations
                this.applyOperations(generator);

                self.setRequiresElementFormattingUpdate(true);
                self.setGUITriggeredOperation(false);

                // setting the cursor position
                selection.setTextSelection(self.getLastOperationEnd());

            }, this); // enterUndoGroup()
        };

        /**
         * Merging table cells in a table in the document.
         */
        this.mergeCells = function () {

            var // the selection object
                selection = self.getSelection(),
                isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.clone(startPos),
                endPosition = null,
                operations = [];

            if (endCol > startCol) {

                for (var i = endRow; i >= startRow; i--) {  // merging for each row

                    var rowPosition = Position.appendNewIndex(tablePos, i);

                    var localStartCol = startCol,
                        localEndCol = endCol;

                    if (!isCellSelection && (i < endRow) && (i > startCol)) {
                        // merging complete rows
                        localStartCol = 0;
                        localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                    }

                    var count = localEndCol - localStartCol,
                        cellPosition = Position.appendNewIndex(rowPosition, localStartCol);

                    newOperation = { name: Operations.CELL_MERGE, start: cellPosition, count: count };
                    self.extendPropertiesWithTarget(newOperation, target);
                    operations.push(newOperation);

                    endPosition = _.clone(cellPosition);
                }

                // apply the operations (undo group is created automatically)
                this.applyOperations(operations);

                endPosition.push(0);
                endPosition.push(0);

                // setting the cursor position
                selection.setTextSelection(endPosition);
            }
        };

        /**
         * Inserting table cells into a table in the document.
         */
        this.insertCell = function () {

            var // the selection object
                selection = self.getSelection(),
                isCellSelection = selection.getSelectionType() === 'cell',
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition(),
                count = 1,  // default, adding one cell in each row
                endPosition = null,
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // current root container node for element
                rootNode = self.getCurrentRootNode(target),
                // created operation
                newOperation = null;

            startPos.pop();  // removing character position and paragraph
            startPos.pop();
            endPos.pop();
            endPos.pop();

            var startCol = startPos.pop(),
                endCol = endPos.pop(),
                startRow = startPos.pop(),
                endRow = endPos.pop(),
                tablePos = _.copy(startPos, true),
                attrs = { cell: {} },
                operations = [];

            for (var i = endRow; i >= startRow; i--) {

                var rowPosition = Position.appendNewIndex(tablePos, i),
                    localEndCol = endCol;

                if (!isCellSelection && (i < endRow) && (i > startCol)) {
                    // removing complete rows
                    localEndCol = Position.getLastColumnIndexInRow(rootNode, rowPosition);
                }

                localEndCol++;  // adding new cell behind existing cell
                var cellPosition = Position.appendNewIndex(rowPosition, localEndCol);
                attrs.cell.gridSpan = 1;  // only 1 grid for the new cell

                newOperation = { name: Operations.CELLS_INSERT, start: cellPosition, count: count, attrs: attrs };
                self.extendPropertiesWithTarget(newOperation, target);
                operations.push(newOperation);

                // Applying new tableGrid, if the current tableGrid is not sufficient
                var tableDomPoint = Position.getDOMPosition(rootNode, tablePos),
                    rowDomPoint = Position.getDOMPosition(rootNode, rowPosition);

                if (tableDomPoint && DOM.isTableNode(tableDomPoint.node)) {

                    var tableGridCount = self.getTableStyles().getElementAttributes(tableDomPoint.node).table.tableGrid.length,
                        rowGridCount = Table.getColSpanSum($(rowDomPoint.node).children());

                    if (rowGridCount > tableGridCount) {

                        localEndCol--;  // behind is evaluated in getTableGridWithNewColumn
                        var insertmode = 'behind',
                            tableGrid = Table.getTableGridWithNewColumn(rootNode, tablePos, localEndCol, insertmode);

                        // Setting new table grid attribute to table
                        newOperation = { name: Operations.SET_ATTRIBUTES, attrs: { table: { tableGrid: tableGrid } }, start: _.clone(tablePos) };
                        self.extendPropertiesWithTarget(newOperation, target);
                        operations.push(newOperation);
                        self.setRequiresElementFormattingUpdate(false);  // no call of implTableChanged -> attributes are already set in implSetAttributes
                    }

                }

                endPosition = _.clone(cellPosition);
            }

            self.setGUITriggeredOperation(true);

            // apply the operations (undo group is created automatically)
            this.applyOperations(operations);

            self.setGUITriggeredOperation(false);
            self.setRequiresElementFormattingUpdate(true);

            endPosition.push(0);
            endPosition.push(0);

            // setting the cursor position
            selection.setTextSelection(endPosition);
        };

        /**
         * Creates operation to split given table in two tables. Split point is row where the cursor is positioned.
         * If cursor is placed in first row, only paragraph is inserted, and whole table shifted for one position down.
         * Row where the cursor is always goes with second newly created table, being the first row in that table.
         * In between two newly created tables, new paragraph is inserted.
         * If there is table in table(s), and cursor is in that table, only that table is split.
         * ODF doesnt support table split.
         */
        this.splitTable = function () {

            var // the undo manager object
                undoManager = self.getUndoManager();

            return undoManager.enterUndoGroup(function () {

                var // operations generator
                    generator = this.createOperationsGenerator(),
                    // the selection object
                    selection = self.getSelection(),
                    //position of the cursor
                    position = selection.getStartPosition(),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(target),
                    // element
                    elementNode = Position.getContentNodeElement(activeRootNode, position.slice(0, -1)),
                    // table node from position
                    tableNode = $(elementNode).closest('table'),
                    // the position of the 'old' table
                    tablePosition = Position.getOxoPosition(activeRootNode, tableNode),
                    // the position of table and row
                    tableAndRowPosition = position.slice(0, tablePosition.length + 1),
                    // cursor position in new table after splitting
                    newPosition = _.clone(tableAndRowPosition),
                    // attributes for the split table operation
                    attrs = null,
                    operationOptions = {};

                if (DOM.isTableNode(tableNode) && !DOM.isExceededSizeTableNode(tableNode)) { // proceed only if it is table node
                    if (_.last(tableAndRowPosition) !== 0) { // split if its not first row, otherwise just insert new paragraph before
                        operationOptions = { start: tableAndRowPosition };
                        self.extendPropertiesWithTarget(operationOptions, target);
                        generator.generateOperation(Operations.TABLE_SPLIT, operationOptions);

                        // update cursor position
                        newPosition[newPosition.length - 2] += 1;
                        newPosition[newPosition.length - 1] = 0;
                    }
                    // Adding change track information only to the inserted paragraph and only, if
                    // change tracking is active.
                    // -> the splitted table cannot be marked as inserted. Rejecting this change track
                    // requires automatic merge of the splitted table, if possible.
                    if (self.getChangeTrack().isActiveChangeTracking()) {
                        attrs = attrs || {};
                        attrs.changes = { inserted: self.getChangeTrack().getChangeTrackInfo(), removed: null };
                    }

                    // #36130 - avoid error in console: Selection.restoreBrowserSelection(): missing text selection range
                    selection.setTextSelection(Position.getFirstPositionInParagraph(activeRootNode, newPosition.slice(0, -1)));

                    operationOptions = { start: newPosition.slice(0, -1), attrs: attrs };
                    self.extendPropertiesWithTarget(operationOptions, target);
                    generator.generateOperation(Operations.PARA_INSERT, operationOptions);

                    // apply all collected operations
                    self.applyOperations(generator);
                    // set cursor in new paragraph between split tables
                    selection.setTextSelection(newPosition, null, { simpleTextSelection: false, splitOperation: false });
                }
            }, this);
        };

        /**
         * Creates one table from two neighbour tables. Conditions are that there is no other element between them,
         * both tables have same table style, and same number of columns (table grids). Condition that must be fulfilled,
         * is that union of this two tables doesn't exceed defined number of columns, row, or cells.
         * ODF doesn't support this feature.
         *
         * @param {Number[]} position
         *  OXO position of first table we try to merge
         * @param {Object} [options]
         * Optional parameters:
         *      @param {Boolean} [options.next]
         *      If set to false, we try to merge passed table position with previous element.
         *      Otherwise with next.
         */
        this.mergeTable = function (position, options) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            return undoManager.enterUndoGroup(function () {

                var // merging is designed to try to merge with next element, so if we are already at next element jump back to previous
                    next = Utils.getBooleanOption(options, 'next', true),
                    // target for operation - if exists, it's for ex. header or footer
                    target = self.getActiveTarget(),
                    // currently active root node
                    activeRootNode = self.getCurrentRootNode(target),
                    // element
                    elementNode = Position.getContentNodeElement(activeRootNode, position),
                    // the 'old' table node
                    tableNode = $(elementNode).closest('table'),
                    // oxo position of table
                    tablePosition = Position.getOxoPosition(activeRootNode, tableNode),
                    // second table node, which we try to merge with first, old one
                    secondTableNode,
                    // operations generator
                    generator = this.createOperationsGenerator(),
                    operationOptions = {};

                // merging is designed to try to merge with next element, so if we are already at next element jump back to previous
                if (!next) {
                    secondTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: false });
                    if (tablePosition[0] !== 0) {
                        tablePosition[0] -= 1;
                    }
                } else {
                    secondTableNode = DOM.getAllowedNeighboringNode(tableNode, { next: true });
                }

                if (DOM.isTableNode(secondTableNode) && Table.mergeableTables(tableNode, secondTableNode)) {
                    // merging is posible only if two tables have the same number of cols and style id
                    operationOptions = { start: tablePosition };
                    self.extendPropertiesWithTarget(operationOptions, target);
                    generator.generateOperation(Operations.TABLE_MERGE, operationOptions);
                    // apply all collected operations
                    self.applyOperations(generator);

                    return true;
                }

            }, this);
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the insertTable operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the table has been inserted successfully.
         */
        this.insertTableHandler = function (operation) {
            return implInsertTable(operation);
        };

        /**
         * The handler for the insertRow operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the row has been inserted successfully.
         */
        this.insertRowHandler = function (operation) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [],
                    undoOperation = null;

                // TODO: create a single DELETE operation for the row range, once this is supported
                _(count).times(function () {
                    undoOperation = { name: Operations.DELETE, start: operation.start };
                    self.extendPropertiesWithTarget(undoOperation, operation.target);
                    undoOperations.push(undoOperation);
                });

                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertRows(operation.start, operation.count, operation.insertDefaultCells, operation.referenceRow, operation.attrs, operation.target);

        };

        /**
         * The handler for the insertColumn operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the column has been inserted successfully.
         */
        this.insertColumnHandler = function (operation) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {

                undoManager.enterUndoGroup(function () {

                    // COLUMNS_DELETE cannot be the answer to COLUMN_INSERT, because the cells of the new column may be inserted
                    // at very different grid positions. It is only possible to remove the new cells with deleteCells operation.
                    var localPos = _.clone(operation.start),
                        rootNode = self.getRootNode(operation.target),
                        table = self.useSlideMode() ? Position.getDOMPosition(rootNode, localPos, true).node : Position.getDOMPosition(rootNode, localPos).node,  // -> this is already the new grid with the new column!
                        allRows = DOM.getTableRows(table),
                        allCellInsertPositions = Table.getAllInsertPositions(allRows, operation.gridPosition, operation.insertMode),
                        cellPosition = null,
                        undoOperation = {};

                    for (var i = (allCellInsertPositions.length - 1); i >= 0; i--) {
                        cellPosition = Position.appendNewIndex(localPos, i);
                        cellPosition.push(allCellInsertPositions[i]);
                        undoOperation = { name: Operations.DELETE, start: cellPosition };
                        self.extendPropertiesWithTarget(undoOperation, operation.target);
                        undoManager.addUndo(undoOperation);
                    }

                    undoManager.addUndo(null, operation);  // only one redo operation

                }, this); // enterUndoGroup()
            }

            return implInsertColumn(operation.start, operation.gridPosition, operation.insertMode, operation.target);
        };

        /**
         * The handler for the splitTable operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the table was splitted successfully.
         */
        this.tableSplitHandler = function (operation) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {
                var tablePos = operation.start.slice(0, -1),
                    undoOperation = { name: Operations.TABLE_MERGE, start: tablePos };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            return implSplitTable(operation.start, operation.target);
        };

        /**
         * The handler for the mergeTable operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the table was merged successfully.
         */
        this.tableMergeHandler = function (operation) {
            return implMergeTable(operation);
        };

        /**
         * The handler for the mergeCell operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the cells were merged successfully.
         */
        this.cellMergeHandler = function (operation) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {
                var content = null,
                    gridSpan = null,
                    undoOperation = { name: Operations.CELL_SPLIT, start: operation.start, content: content, gridSpan: gridSpan };

                self.extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            return implMergeCell(_.copy(operation.start, true), operation.count, operation.target);
        };

        /**
         * The handler for the insertCell operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the table cell were inserted successfully.
         */
        this.cellInsertHandler = function (operation) {

            var // the undo manager object
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {
                var count = Utils.getIntegerOption(operation, 'count', 1, 1),
                    undoOperations = [],
                    undoOperation = null;

                // TODO: create a single DELETE operation for the cell range, once this is supported
                _(count).times(function () {
                    undoOperation = { name: Operations.DELETE, start: operation.start };
                    self.extendPropertiesWithTarget(undoOperation, operation.target);
                    undoOperations.push(undoOperation);
                });
                undoManager.addUndo(undoOperations, operation);
            }

            return implInsertCells(operation.start, operation.count, operation.attrs, operation.target);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class TableOperationMixin

    // constants --------------------------------------------------------------

    // exports ================================================================

    return TableOperationMixin;
});
