/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/stringconvertermixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/tk/utils/deferredutils'
], function (DrawingFrame, DOM, Utils, DeferredUtils) {

    'use strict';

    // mix-in class StringConverterMixin ======================================

    /**
     * A mix-in class for the converters, that convert the document model in
     * the DOM into a string and vice versa. This is used for example for
     * saving the document in the localstorage or for evaluating the string
     * received from the server in the fast load process.
     *
     * @constructor
     */
    function StringConverterMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * Helper function for generating a logging string, that contains
         * all strings of an array in unique, sorted form together with
         * the number of occurences in the array.
         * Input:
         * ['abc', 'def', 'abc', 'ghi', 'def']
         * Output:
         * 'abc (2), def (2), ghi'
         *
         * @param {Array} arr
         *  An array with strings.
         *
         * @returns {String}
         *  The string constructed from the array content.
         */
        function getSortedArrayString(arr) {

            var counter = {},
                arrayString = '';

            _.countBy(arr, function (elem) {
                counter[elem] = counter[elem] ? counter[elem] + 1 : 1;
            });

            _.each(_.keys(counter).sort(), function (elem, index) {
                var newString = elem + ' (' + counter[elem] + ')';
                newString += ((index + 1) < _.keys(counter).length) ? ', ' : '';
                arrayString += newString;
            });

            return arrayString;
        }

        /**
         * Assigning the character attributes at (empty) paragraphs to the text span
         * inside the paragraphs.
         *
         * @param {jQuery} paragraph
         *  The paragraph node.
         *
         * @param {Object} attributes
         *  The object with the character attributes.
         *
         * @param {Array} dataKeys
         *  The collector for all assigned keys.
         */
        function assignCharacterAttributesToSpan(paragraph, attributes, dataKeys) {

            _.each(paragraph.children('span'), function (node) {
                if (DOM.isSpan(node)) {
                    // setting the character attributes to the text span(s) inside the paragraph
                    for (var key in attributes) {
                        $(node).data(key, attributes[key]);
                        dataKeys.push(key);
                    }
                }
            });

        }

        // public methods -----------------------------------------------------

        /**
         * Generating a document string from the editdiv element. jQuery data
         * elements are saved at the sub elements with the attribute 'jquerydata'
         * assigned to these elements. Information about empty paragraphs
         * is stored in the attribute 'nonjquerydata'.
         *
         * @param {jQuery.Deferred} [maindef]
         *  The deferred used to update the progress.
         *
         * @param {Number} [startProgress]
         *  The start value of the notification progress.
         *
         * @param {Number} [endProgress]
         *  The end value of the notification progress.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved or rejected if the document strings
         *  could be generated successfully. All document strings are packed into
         *  an array. Several document strings are generated for each layer, that
         *  needs to be stored in the local storage. This are the page content,
         *  the drawing layer, ... .
         */
        this.getFullModelDescription = function (maindef, startProgress, endProgress) {

            var // collecting all keys from data object for debug reasons
                dataKeys = [],
                // collector for all different saved nodes (page content, drawing layer , ...)
                allNodesCollector = [],
                // collector for all html strings and the extension in the registry for every layer (page content, drawing layer , ...)
                allHtmlStrings = [],
                // the number of chunks, that will be evaluated
                chunkNumber = 5,
                // the current notification progress
                currentNotify = startProgress,
                // the notify difference for one chunk
                notifydiff = Utils.round((endProgress - startProgress) / chunkNumber, 0.01),
                // a selected drawing
                selectedDrawing = null,
                // the padding-bottom value saved at the pagecontent node
                paddingBottomInfo = null,
                // the selection object
                selection = self.getSelection();

            // converting all editdiv information into a string, saving especially the data elements
            // finding all elements, that have a 'data' jQuery object set.
            // Helper function to split the array of document nodes into smaller parts.
            function prepareNodeForStorage(node) {

                var // the jQuery data object of the node
                    dataObject = null,
                    // the return promise
                    def = DeferredUtils.createDeferred(app, 'StringConverterMixin: prepareNodeForStorage');

                // updating progress bar
                if (maindef) {
                    currentNotify += Utils.round(notifydiff, 0.01);
                    maindef.notify(currentNotify); // updating corresponding to the length inside each -> smaller snippets deferred.
                }

                try {

                    // the current node, as jQuery object
                    node = $(node);

                    // simplify restauration of empty text nodes
                    if ((node.is('span')) && (DOM.isEmptySpan(node))) {
                        dataKeys.push('isempty');
                        // using nonjquerydata instead of jquerydata (32286)
                        node.attr('nonjquerydata', JSON.stringify({ isempty: true }));
                    }

                    // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                    if (_.browser.WebKit && DOM.isDrawingFrame(node)) {
                        node.removeData('trackingOptions');
                        node.removeData('tracking-options');
                    }

                    // removing spell check information at node
                    self.getSpellChecker().clearSpellcheckHighlighting(node);

                    // handling jQuery data objects
                    dataObject = node.data();
                    if (_.isObject(dataObject) && !_.isEmpty(dataObject)) {
                        if (dataObject[DOM.DRAWINGPLACEHOLDER_LINK]) { delete dataObject[DOM.DRAWINGPLACEHOLDER_LINK]; }  // no link to other drawing allowed
                        if (dataObject[DOM.DRAWING_SPACEMAKER_LINK]) { delete dataObject[DOM.DRAWING_SPACEMAKER_LINK]; }  // no link to other drawing allowed
                        if (dataObject[DOM.COMMENTPLACEHOLDER_LINK]) { delete dataObject[DOM.COMMENTPLACEHOLDER_LINK]; }  // no link to other comment allowed
                        dataKeys = dataKeys.concat(_.keys(dataObject));
                        node.attr('jquerydata', JSON.stringify(dataObject));  // different key -> not overwriting isempty value
                        // Utils.log('Saving data: ' + this.nodeName + ' ' + this.className + ' : ' + JSON.stringify($(this).data()));

                        // handling images that have the session used in the src attribute
                        // Avoiding error 403 during saving the document in the console (Bug 37640)
                        if (DOM.isImageNode(node)) {
                            var imgNode = node.find('img'),
                                srcAttr = imgNode.attr('src');
                            if (_.isString(srcAttr) && (srcAttr.length > 0)) {
                                imgNode.attr('src', null);
                                imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE, srcAttr.replace(/\bsession=(\w+)\b/, 'session=_REPLACE_SESSION_ID_'));
                            }
                        }
                    }

                    def.resolve();

                } catch (ex) {
                    // dataError = true;
                    Utils.info('quitHandler, failed to save document in local storage (3): ' + ex.message);
                    def.reject();
                }

                return def.promise();
            }  // end of 'prepareNodeForStorage'

            // removing drawing selections
            if (selection.getSelectionType() === 'drawing') {
                selectedDrawing = selection.getSelectedDrawing();
                DrawingFrame.clearSelection(selectedDrawing);
                // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                selectedDrawing.removeData('trackingOptions');
                selectedDrawing.removeData('tracking-options');
            }

            // removing selections around 'additional' text frame selections
            if (selection.isAdditionalTextframeSelection()) {
                selectedDrawing = selection.getSelectedTextFrameDrawing();
                DrawingFrame.clearSelection(selectedDrawing);
                // removing additional tracking handler options, that are specific for OX Text (Chrome only)
                selectedDrawing.removeData('trackingOptions');
                selectedDrawing.removeData('tracking-options');
            }

            // removing artificial selection of empty paragraphs
            if (!_.isEmpty(self.getHighlightedParagraphs())) { self.removeArtificalHighlighting(); }

            // removing a change track selection, if it exists
            if (self.getChangeTrack().getChangeTrackSelection()) { self.getChangeTrack().clearChangeTrackSelection(); }

            // removing classes at a highlighted comment thread (because comment thread is active or hovered), if necessary
            self.getCommentLayer().clearCommentThreadHighlighting();

            // removing an optionally set comment author filter
            self.getCommentLayer().removeAuthorFilter();

            // removing active header/footer context menu, if exists
            if (self.isHeaderFooterEditState()) {
                self.getPageLayout().leaveHeaderFooterEditMode();
            }

            // removing search highlighting
            self.getSearchHandler().clearHighlighting();

            // removing artifical increased size of an implicit paragraph
            if (self.getIncreasedParagraphNode()) {
                $(self.getIncreasedParagraphNode()).css('height', 0);
                self.setIncreasedParagraphNode(null);
            }

            // First layer: Collecting all nodes of the page content
            allNodesCollector.push({ extension: '', selector: DOM.PAGECONTENT_NODE_SELECTOR, allNodes: self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).find('*') });

            // saving specific page content node styles at its first child (style 'padding-bottom')
            paddingBottomInfo = self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).css('padding-bottom');
            self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR).children(':first').attr('pagecontentattr', JSON.stringify({ 'padding-bottom': paddingBottomInfo }));

            // Second layer: Handling the drawing layer (taking care of order in allNodesCollector)
            if (self.getDrawingLayer().containsDrawingsInPageContentNode()) { allNodesCollector.push({ extension: '_DL', selector: DOM.DRAWINGLAYER_NODE_SELECTOR, allNodes: self.getNode().children(DOM.DRAWINGLAYER_NODE_SELECTOR).find('*') }); }

            // Third layer: Handling the header/footer placeholder layer (taking care of order in allNodesCollector)
            if (self.getPageLayout().hasContentHeaderFooterPlaceHolder()) { allNodesCollector.push({ extension: '_HFP', selector: DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR, allNodes: self.getNode().children(DOM.HEADER_FOOTER_PLACEHOLDER_SELECTOR).find('*') }); }

            // Fourth layer: Handling the comment layer (taking care of order in allNodesCollector)
            if (!self.getCommentLayer().isEmpty()) { allNodesCollector.push({ extension: '_CL', selector: DOM.COMMENTLAYER_NODE_SELECTOR, allNodes: self.getNode().children(DOM.COMMENTLAYER_NODE_SELECTOR).find('*') }); }

            // iterating over all nodes (first page content, then drawing layer, ...)
            return self.iterateArraySliced(allNodesCollector, function (oneLayer) {

                return self.iterateArraySliced(oneLayer.allNodes, prepareNodeForStorage, { delay: 'immediate', infoString: 'Text: getFullModelDescription: inner', app: app })
                .done(function () {
                    // collecting the html strings for each layer
                    allHtmlStrings.push({ extension: oneLayer.extension, htmlString: self.getNode().children(oneLayer.selector).html() });
                });
            }, { infoString: 'Text: getFullModelDescription: outer', app: app })
            .then(function () {
                return allHtmlStrings;
            });

        };

        /**
         * Generating the editdiv element from an html string. jQuery data
         * elements are restored at the sub elements, if the attribute 'jquerydata'
         * is assigned to these elements.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.usedLocalStorage=false]
         *      If set to true, the specified html string was loaded from the
         *      local storage. In this case, no browser specific modifications
         *      need to be done on the string.
         *  @param {Boolean} [options.drawingLayer=false]
         *      If set to true, the specified html string is the drawing layer.
         *  @param {Boolean} [options.commentLayer=false]
         *      If set to true, the specified html string is the comment layer.
         *  @param {Boolean} [options.headerFooterLayer=false]
         *      If set to true, the specified html string is the header/footer placeholder layer.
         *
         * @param {String} htmlString
         *  The html string for the editdiv element.
         */
        this.setFullModelNode = function (htmlString, options) {

            var // collecting all keys assigned to the data objects
                dataKeys = [],
                // the data objects saved in the string for the nodes
                dataObject = null,
                // whether the html string was read from the local storage
                usedLocalStorage = Utils.getBooleanOption(options, 'usedLocalStorage', false),
                // whether the html string describes the comment layer (loaded from local storage)
                isCommentLayer = Utils.getBooleanOption(options, 'commentLayer', false),
                // whether the html string describes the drawing layer (loaded from local storage)
                isDrawingLayer = Utils.getBooleanOption(options, 'drawingLayer', false),
                // whether the html string describes the header/footer layer (loaded from local storage)
                isHeaderFooterLayer = Utils.getBooleanOption(options, 'headerFooterLayer', false),
                // eather the page content node, parent of the editdiv, or headerFooterContainer node
                contentNode = null;

            // helper function to get the node in which the html string needs to be inserted
            function getStringParentNode() {

                var // the parent node for each layer
                    parentNode = null;

                if (self.useSlideMode()) {
                    parentNode = self.getNode();
                } else if (isDrawingLayer) {
                    parentNode = self.getDrawingLayer().getDrawingLayerNode();
                } else if (isCommentLayer) {
                    parentNode = self.getCommentLayer().getOrCreateCommentLayerNode({ visible: false });
                } else if (isHeaderFooterLayer) {
                    parentNode = self.getPageLayout().getHeaderFooterPlaceHolder();
                } else {
                    parentNode = self.getNode().children(DOM.PAGECONTENT_NODE_SELECTOR);
                }

                return parentNode;
            }

            // registering import type
            if (!self.isImportFinished()) {
                if (usedLocalStorage) {
                    self.setLocalStorageImport(true);
                } else {
                    self.setFastLoadImport(true);
                }
            }

            if (htmlString) {

                // Avoiding error 403 during loading the document in the console (35405)
                // -> replacing the '_REPLACE_SESSION_ID_' before it is inserted into the DOM
                // -> this is a problem, if the user really uses this replacement string in the dom
                htmlString = htmlString.replace(/\bsession=_REPLACE_SESSION_ID_\b/g, 'session=' + ox.session);

                // setting the content node for each layer
                contentNode = getStringParentNode();

                // setting new string to editdiv node
                contentNode
                    .html(htmlString)
                    .find('[jquerydata], [nonjquerydata]')
                    .each(function () {
                        var node = $(this),
                            imgNode = null,
                            srcAttr = null,
                            characterAttributes = null,
                            key = null;

                        if (node.attr('nonjquerydata')) {
                            try {
                                dataObject = JSON.parse(node.attr('nonjquerydata'));
                                for (key in dataObject) {
                                    if (key === 'isempty') {  // the only supported key for nonjquerydata
                                        DOM.ensureExistingTextNode(this);
                                        dataKeys.push(key);
                                    }
                                }
                                node.removeAttr('nonjquerydata');
                            } catch (ex) {
                                Utils.error('Failed to parse attributes: ' + node.attr('jquerydata') + ' Exception message: ' + ex.message);
                            }
                        }

                        if (node.attr('jquerydata')) {

                            try {
                                dataObject = JSON.parse(node.attr('jquerydata'));

                                // No list style in comments in odt: 38829 (simply removing the list style id)
                                if (app.isODF() && !usedLocalStorage && isCommentLayer && DOM.isParagraphNode(node) && dataObject.attributes && dataObject.attributes.paragraph && dataObject.attributes.paragraph.listStyleId) { delete dataObject.attributes.paragraph.listStyleId; }

                                // collecting all paragraphs with character attributes
                                if (!app.isODF() && !usedLocalStorage && DOM.isParagraphNode(node)) {
                                    // check for character attributes
                                    if ('attributes' in dataObject && 'character' in dataObject.attributes) {
                                        characterAttributes = dataObject.attributes.character;
                                        delete dataObject.attributes.character;
                                        if (_.isEmpty(dataObject.attributes)) { delete dataObject.attributes; }
                                    }
                                }

                                for (key in dataObject) {
                                    node.data(key, dataObject[key]);
                                    dataKeys.push(key);
                                }

                                // assigning character attributes to the first (empty) text span
                                if (characterAttributes) { assignCharacterAttributesToSpan(node, { attributes: { character: characterAttributes } }, dataKeys); }

                                node.removeAttr('jquerydata');
                            } catch (ex) {
                                Utils.error('Failed to parse attributes: ' + node.attr('jquerydata') + ' Exception message: ' + ex.message);
                            }
                        }

                        if (DOM.isImageNode(node)) {
                            imgNode = node.find('img');
                            srcAttr = imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE);
                            if (_.isString(srcAttr) && (srcAttr.length > 0)) {
                                imgNode.attr('src', srcAttr);
                                imgNode.attr(DOM.LOCALSTORAGE_SRC_OVERRIDE, null);
                            }
                        }

                    });

                contentNode.find('table[size-exceed-values]').each(function () {
                    var // the values provided by the backend which specifies
                        // what part of the table exceeds the size
                        values = $(this).attr('size-exceed-values'),
                        // the separate parts
                        parts = null;

                    if (values) {
                        parts = values.split(',');
                        if (parts && parts.length === 3) {
                            DOM.makeExceededSizeTable($(this), parts[0], parts[1], parts[2]);
                        }
                    }
                });

                // setting table cell attribute 'contenteditable' to true for non-MSIE browsers (task 33642)
                // -> this is not necessary, if the document was loaded from the local storage
                if (!usedLocalStorage && !_.browser.IE) { contentNode.find('div.cell').attr('contenteditable', true); }

                // assigning page content attributes, that are saved at its first child
                if (contentNode.children(':first').attr('pagecontentattr')) {
                    dataObject = JSON.parse(contentNode.children(':first').attr('pagecontentattr'));
                    if (dataObject['padding-bottom']) {
                        contentNode.css('padding-bottom', dataObject['padding-bottom']);
                    }
                    contentNode.children(':first').removeAttr('pagecontentattr');
                }

                if (app.isODF()) { // in odt it is possible to have empty fields, text node needs to be ensured, #49844
                    _.each(contentNode.find('.empty-field'), function (fieldNode) {
                        DOM.ensureExistingTextNodeInField(fieldNode.firstChild);
                    });
                }

                if (!_.isEmpty(dataKeys)) { Utils.info('Setting data keys: ' + getSortedArrayString(dataKeys)); }
            }
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class StringConverterMixin

    // exports ================================================================

    return StringConverterMixin;

});
