/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/model/paragraphmixin', [
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/utils/textutils'
], function (DrawingFrame, AttributeUtils, DOM, Operations, Position, Utils) {

    'use strict';

    // mix-in class ParagraphMixin ======================================

    /**
     * A mix-in class for the paragraph handling in the document model.
     *
     * @constructor
     */
    function ParagraphMixin(app) {

        var // self reference for local functions
            self = this;

        // private methods ----------------------------------------------------

        /**
         * After splitting a paragraph with place holders in the new paragraph (that
         * was cloned before), it is necessary to update the place holder nodes in the
         * models. Also range marker and complex field models need to be updated.
         *
         * @param {Node|jQuery} para
         *  The paragraph node.
         */
        function updateAllModels(para) {

            var // whether the existence of the components in the specified node shall be checked
                noCheck = !DOM.isParagraphNode(para);

            // also update the links of the drawings in the drawing layer to their placeholder
            if (noCheck || DOM.hasDrawingPlaceHolderNode(para)) { self.getDrawingLayer().repairLinksToPageContent(para); }
            // also update the links of the comments in the comment layer to their placeholder
            if (noCheck || DOM.hasCommentPlaceHolderNode(para)) { self.getCommentLayer().repairLinksToPageContent(para); }
            // also update the collected range markers
            if (noCheck || DOM.hasRangeMarkerNode(para)) { self.getRangeMarker().updateRangeMarkerCollector(para); }
            // also update the collected complex fields
            if (noCheck || DOM.hasComplexFieldNode(para)) { self.getFieldManager().updateComplexFieldCollector(para); }
            // also update the collected simple fields
            if (noCheck || DOM.hasSimpleFieldNode(para)) { self.getFieldManager().updateSimpleFieldCollector(para); }
        }

        /**
         * Splitting a paragraph at a specified logical position.
         *
         * @param {Number[]} position
         *  The logical position at which the paragraph shall be splitted.
         *
         * @returns {Boolean}
         *  Whether the paragraph has been splitted successfully.
         */
        function implSplitParagraph(position, target) {

            var // the last value of the position array
                offset = _.last(position),
                // the position of the 'old' paragraph
                paraPosition = position.slice(0, -1),
                // container root node of paragraph
                rootNode = self.getRootNode(target),
                // the 'old' paragraph node
                paragraph = (self.useParagraphCache() && !target && self.getParagraphCache()) || Position.getParagraphElement(rootNode, paraPosition),
                // the position of the 'new' paragraph
                newParaPosition = Position.increaseLastIndex(paraPosition),
                // the 'new' paragraph node
                newParagraph = paragraph ? $(paragraph).clone(true).find('div.page-break, ' + DOM.DRAWING_SPACEMAKER_NODE_SELECTOR).remove().end().insertAfter(paragraph) : null, //remove eventual page breaks before inserting to DOM
                // a specific position
                startPosition = null,
                // the length of the paragraph
                paraLength = paragraph ? Position.getParagraphNodeLength(paragraph) : -1,
                // whether the paragraph is splitted at the end
                isLastPosition = (paraLength === offset),
                // whether the paragraph contains floated drawings
                hasFloatedChildren = paragraph ? DOM.containsFloatingDrawingNode(paragraph) : false,
                // Performance: Saving global paragraph cache
                paragraphCacheSafe = (self.useParagraphCache() && self.getParagraphCache()) || null,
                // Performance: Saving data for list updates
                paraAttrs = null,
                // whether the split paragraph contains selected drawing(s)
                updateSelectedDrawing = false,
                // whether the document is in read-only mode
                readOnly = self.getEditMode() !== true,
                // the list label node inside a paragraph
                listLabelNode = null,
                // the current element
                currentElement,
                // the old drawing start position before splitting the paragraph
                drawingStartPosition = null, index = 0,
                // the last text span of the paragraph
                finalTextSpan = null,
                // the selection object
                selection = self.getSelection(),
                // the spell checker object
                spellChecker = self.getSpellChecker();

            // return with error, if paragraph could not be found or offset is bigger than the length of the paragraph
            if (!paragraph || offset > paraLength) { return false; }

            // also not allowing splitting of slides in presentation app
            if (paragraph && self.useSlideMode() && DOM.isSlideNode(paragraph)) { return false; }

            // checking if a selected drawing is affected by this split
            if (readOnly && selection.getSelectionType() === 'drawing') {
                // is the drawing inside the splitted paragraph and is the drawing behind the split?
                if (Utils.containsNode(paragraph, selection.getSelectedDrawing())) {
                    // split happens at position 'position'. Is this before the existing selection?
                    if (Utils.compareNumberArrays(position, selection.getStartPosition(), position.length) < 0) {
                        updateSelectedDrawing = true;
                        drawingStartPosition = _.clone(selection.getStartPosition());
                    }
                }
            }

            // caching current paragraph/table (if paragraph split inside table) for page break calculation
            currentElement = (position.length > 2) ? $(paragraph).parents('table').last() : $(paragraph);

            // Performance: If the paragraph is splitted at the end, the splitting process can be simplified
            if (isLastPosition && paraLength > 0 && DOM.isTextSpan(newParagraph.children(':last'))) {
                // the final textSpan must be reused to save character attributes
                finalTextSpan = newParagraph.children(':last').clone(true).text('');
                if (DOM.isListLabelNode(newParagraph.children(':first'))) { listLabelNode = newParagraph.children(':first'); }
                newParagraph.empty().end().prepend(finalTextSpan);
                DOM.ensureExistingTextNode(finalTextSpan);  // must be done after insertion into the DOM
                if (listLabelNode) { newParagraph.prepend(listLabelNode); }
                startPosition = Position.appendNewIndex(newParaPosition);
            } else {

                // delete trailing part of original paragraph
                if (offset !== -1) {
                    if (self.useParagraphCache()) { self.setParagraphCache(paragraph); } // Performance: implDeleteText can use cached paragraph
                    self.implDeleteText(position, Position.appendNewIndex(paraPosition, -1), { allowEmptyResult: true, keepDrawingLayer: true }, target);
                    // updating absolute paragraph drawings in the second paragraph
                    self.getDrawingLayer().registerUpdateElement(newParagraph);
                    if (hasFloatedChildren) {
                        // delete all image divs that are no longer associated with following floating drawings
                        Position.removeUnusedDrawingOffsetNodes(paragraph);
                    }
                    if (offset === 0) { // if split is at the beginning of paragraph, remove page breaks immediately for better user experience
                        $(paragraph).find('div.page-break').remove();
                        // handle place holder nodes for drawings and comment
                        updateAllModels(newParagraph);
                    }
                }

                // delete leading part of new paragraph
                startPosition = Position.appendNewIndex(newParaPosition);
                if (offset > 0) {
                    if (self.useParagraphCache()) { self.setParagraphCache(newParagraph); } // Performance: implDeleteText cannot use cached paragraph
                    self.implDeleteText(startPosition, Position.increaseLastIndex(startPosition, offset - 1), { keepDrawingLayer: true }, target);
                    if (hasFloatedChildren) {
                        // delete all empty text spans in cloned paragraph before floating drawings
                        // TODO: implDeleteText() should have done this already
                        Position.removeUnusedDrawingOffsetNodes(newParagraph);
                    }
                    // handle place holder nodes for drawings and comment
                    updateAllModels(newParagraph);
                }
                // drawings in the new paragraph need a repaint of the canvas border (also if offset is 0)
                self.trigger('drawingHeight:update', newParagraph.find(DrawingFrame.CANVAS_NODE_SELECTOR));
            }

            // update formatting of the paragraphs
            self.implParagraphChanged(paragraph);
            self.implParagraphChanged(newParagraph);

            // only for text, there are no paragraph borders and color in presentation
            if (!self.useSlideMode()) {
                // only needed for updating paragraphs that have a color or a border
                self.getParagraphStyles().updateParagraphBorderAndColorNodes($(newParagraph).next());
            }

            // invalidating an optional cache with spell results synchronously
            if (spellChecker.hasSpellResultCache(paragraph)) {
                spellChecker.clearSpellResultCache(paragraph);
                spellChecker.clearSpellResultCache(newParagraph);
            }

            if (self.useParagraphCache()) { self.setParagraphCache(paragraphCacheSafe); } // Performance: Restoring global paragraph cache

            // checking paragraph attributes for list styles
            if (self.handleTriggeringListUpdate(paragraph, { checkSplitInNumberedList: true })) {
                $(newParagraph).data('splitInNumberedList', 'true');
            }

            // checking paragraph attributes for list styles
            paraAttrs = AttributeUtils.getExplicitAttributes(paragraph);

            // !!!Notice - We do not send operation for removing this attribute, it is by default set to false for any paragraph in filter
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.pageBreakBefore && paraAttrs.paragraph.pageBreakBefore === true) {
                self.getParagraphStyles().setElementAttributes(newParagraph, { paragraph: { pageBreakBefore: false } });
                newParagraph.removeClass('manual-page-break');
            }
            // !!!Notice - We do not send operation for removing this attribute, it is by default set to false for any paragraph in filter
            if (paraAttrs && paraAttrs.paragraph && paraAttrs.paragraph.pageBreakAfter && paraAttrs.paragraph.pageBreakAfter === true) {
                self.getParagraphStyles().setElementAttributes(newParagraph, { paragraph: { pageBreakAfter: false } });
                newParagraph.removeClass('manual-pb-after');
            }

            // block page breaks render if operation is targeted
            if (target) {
                self.setBlockOnInsertPageBreaks(true);
            }

            if (!isLastPosition && currentElement.data('lineBreaksData')) {
                currentElement.removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            if (newParagraph.data('lineBreaksData')) {
                newParagraph.removeData('lineBreaksData'); // for the cloned paragraph also
            }

            // removing selected classes from the new paragraph (48180)
            newParagraph.removeClass(DOM.PARAGRAPH_NODE_LIST_EMPTY_CLASS + ' ' + DOM.PARAGRAPH_NODE_LIST_EMPTY_SELECTED_CLASS);

            //quitFromPageBreak = true; // quit from any currently running pagebreak calculus - performance optimization
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(paragraph));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            self.setLastOperationEnd(startPosition);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraphs again
            // newly, also in other browsers (new jQuery version?)
            self.repairEmptyTextNodes(newParagraph);

            if (_.browser.IE) {
                self.repairEmptyTextNodes(paragraph); // also repair original paragraph in IE, #37652
            }

            // Performance: Saving paragraph info for following operations
            selection.setParagraphCache(newParagraph, _.clone(newParaPosition), 0);

            if (updateSelectedDrawing) {
                // selecting the previously selected drawing again
                index = position.length - 1;
                drawingStartPosition[index] -= position[index];  // new text position inside paragraph
                drawingStartPosition[index - 1]++;  // new paragraph position
                selection.setTextSelection(drawingStartPosition, Position.increaseLastIndex(drawingStartPosition));
            }

            return true;
        }

        // public methods -----------------------------------------------------

        /**
         * Inserting a paragraph into the document.
         *
         * @param {Number[]} start
         *  The logical position of the paragraph to be inserted.
         */
        this.insertParagraph = function (start) {

            var // the attributes of the inserted paragraph
                attrs = {},
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // created operation
                newOperation = null,
                // the change track object
                changeTrack = self.getChangeTrack();

            // handling change tracking
            if (changeTrack.isActiveChangeTracking()) { attrs.changes = { inserted: changeTrack.getChangeTrackInfo() }; }

            newOperation = { name: Operations.PARA_INSERT, start: _.clone(start), attrs: attrs };
            self.extendPropertiesWithTarget(newOperation, target);

            // applying operation
            self.applyOperations(newOperation);
        };

        /**
         * Creating the operation for splitting a paragraph. A check is necessary, if the paragraph
         * contains leading floated drawings. In this case it might be necessary to update the
         * specified split position. For example the specified split position is [0,1]. If there
         * is a floated drawing at position [0,0], it is necessary to split the paragraph at
         * the position [0,0], so that the floated drawing is also moved into the following
         * paragraph.
         *
         * @param {Number[]} position
         *  The logical split position.
         *
         * @param {Node|jQuery|Null} [para]
         *  The paragraph that will be splitted. If this object is a jQuery collection, uses
         *  the first DOM node it contains. If missing, it will be determined from the specified
         *  logical position. This parameter can be used for performance reasons.
         */
        this.splitParagraph = function (position, para) {

            var
                // currently active root node
                activeRootNode = self.getCurrentRootNode(),
                // the paragraph node, if not specified as parameter
                paragraph = para || Position.getParagraphElement(activeRootNode, position.slice(0, -1)),
                // the last position of the specified logical split position
                offset = _.last(position),
                // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                // the operation options
                operationOptions = {};

            // moving the leading absolute drawings to the new paragraph
            if (DOM.hasAbsoluteParagraphDrawing(paragraph) && offset > 0 && offset <= Position.getLeadingAbsoluteDrawingCount(paragraph)) {
                position[position.length - 1] = 0;
            }

            operationOptions = { name: Operations.PARA_SPLIT, start: _.clone(position) };
            self.extendPropertiesWithTarget(operationOptions, target);
            self.applyOperations(operationOptions);
        };

        /**
         * Creating the operation for merging two paragraphs.
         *
         * @param {Number[]} position
         *  The logical merge position.
         */
        this.mergeParagraph = function (position) {

            var // target for operation - if exists, it's for ex. header or footer
                target = self.getActiveTarget(),
                operationOptions = { name: Operations.PARA_MERGE, start: _.clone(position) };

            self.extendPropertiesWithTarget(operationOptions, target);
            self.applyOperations(operationOptions);
        };

        // operation handler --------------------------------------------------

        /**
         * The handler for the splitParagraph operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the paragraph was splitted successfully.
         */
        this.splitParagraphHandler = function (operation) {

            var // the undo manager
                undoManager = self.getUndoManager();

            if (undoManager.isUndoEnabled()) {

                var paragraphPos = operation.start.slice(0, -1),
                    undoOperation = { name: Operations.PARA_MERGE, start: paragraphPos };

                self.extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }
            return implSplitParagraph(operation.start, operation.target);
        };

        /**
         * The handler for the mergeParagraph operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the paragraphs were merged successfully.
         */
        this.mergeParagraphHandler = function (operation) {

            var // container root node of paragraph
                rootNode = self.getRootNode(operation.target),
                // the paragraph that will be merged with its next sibling
                paragraphInfo = Position.getDOMPosition(rootNode, operation.start, true),
                // current and next paragraph, as jQuery objects
                thisParagraph = null, nextParagraph = null,
                // text position at end of current paragraph, logical position of next paragraph
                paraEndPosition = null, nextParaPosition = null,
                // first child node of next paragraph
                firstChildNode = null,
                // the undo/redo operations
                generator = self.getUndoManager().isUndoEnabled() ? this.createOperationsGenerator() : null,
                // start node from where to run page break calculations
                currentElement,
                // the character attributes
                characterAttributes = null,
                // the list label elements at the following paragraph
                listLabelInNextParagraph = null,
                // undo operation for passed operation
                undoOperation = {};

            // get current paragraph
            if (!paragraphInfo || !DOM.isParagraphNode(paragraphInfo.node)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found at position ' + JSON.stringify(operation.start));
                return false;
            }
            thisParagraph = $(paragraphInfo.node);
            currentElement = thisParagraph;
            paraEndPosition = Position.appendNewIndex(operation.start, Position.getParagraphLength(rootNode, operation.start));

            // get next paragraph
            nextParagraph = thisParagraph.next();

            // but make sure next paragraph is only valid node, and not pagebreak for example
            while (nextParagraph.length > 0 && !nextParagraph.is(DOM.CONTENT_NODE_SELECTOR)) {
                nextParagraph = nextParagraph.next();
            }

            if (!DOM.isParagraphNode(nextParagraph)) {
                Utils.warn('Editor.mergeParagraph(): no paragraph found after position ' + JSON.stringify(operation.start));
                return false;  // forcing an error, if there is no following paragraph
            }
            nextParaPosition = Position.increaseLastIndex(operation.start);

            // generate undo/redo operations
            if (generator) {
                undoOperation = { start: _.clone(paraEndPosition), target: operation.target };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                generator.generateOperation(Operations.PARA_SPLIT, undoOperation);

                // restoring character attributes at the empty paragraphs again
                if (!app.isODF()) {
                    characterAttributes = Utils.getCharacterAttributesFromEmptyTextSpan(nextParagraph);
                    if (characterAttributes && !_.isEmpty(characterAttributes)) { self.getParagraphStyles().setElementAttributes(nextParagraph, { character: characterAttributes }); }
                }

                undoOperation = { start: nextParaPosition };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                generator.generateSetAttributesOperation(nextParagraph, undoOperation, { clearFamily: 'paragraph' });
                self.getUndoManager().addUndo(generator.getOperations(), operation);
            }

            // remove dummy text node from current paragraph
            if (DOM.isDummyTextNode(thisParagraph[0].lastChild)) {
                $(thisParagraph[0].lastChild).remove();
            }

            // remove list label node from next paragraph (taking care of list update (35447)
            listLabelInNextParagraph = nextParagraph.children(DOM.LIST_LABEL_NODE_SELECTOR);
            if (listLabelInNextParagraph.length > 0) {
                listLabelInNextParagraph.remove();
                self.handleTriggeringListUpdate(nextParagraph);
            }

            // append all children of the next paragraph to the current paragraph, delete the next paragraph
            firstChildNode = nextParagraph[0].firstChild;
            thisParagraph.append(nextParagraph.children());
            nextParagraph.remove();

            // when merging par that contains MS hardbreak with type page, mark merged paragraph
            if (thisParagraph.find('.ms-hardbreak-page').length) {
                thisParagraph.addClass('manual-page-break contains-pagebreak');
            }

            // remove one of the sibling text spans at the concatenation point,
            // if one is empty; otherwise try to merge equally formatted text spans
            if (DOM.isTextSpan(firstChildNode) && DOM.isTextSpan(firstChildNode.previousSibling)) {
                if (DOM.isEmptySpan(firstChildNode)) {
                    $(firstChildNode).remove();
                } else if (DOM.isEmptySpan(firstChildNode.previousSibling)) {
                    $(firstChildNode.previousSibling).remove();
                } else {
                    Utils.mergeSiblingTextSpans(firstChildNode);
                }
            }

            // refresh DOM
            self.implParagraphChanged(thisParagraph);

            // checking paragraph attributes for list styles and updating lists, if required
            self.handleTriggeringListUpdate(thisParagraph);

            // new cursor position at merge position
            self.setLastOperationEnd(_.clone(paraEndPosition));

            if ($(currentElement).data('lineBreaksData')) {
                $(currentElement).removeData('lineBreaksData'); //we changed paragraph layout, cached line breaks data needs to be invalidated
            }
            //render pagebreaks after merge
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(thisParagraph));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            // only for text, there are no paragraph borders and color in presentation
            if (!self.useSlideMode()) {
                // only needed for updating paragraphs that have a color or a border
                self.getParagraphStyles().updateParagraphBorderAndColorNodes(thisParagraph.next());
            }

            return true;
        };

        /**
         * The handler for the insertParagraph operation.
         *
         * @param {Object} operation
         *  The operation object.
         *
         * @returns {Boolean}
         *  Whether the paragraph was inserted successfully.
         */
        this.insertParagraphHandler = function (operation) {

            var // the new paragraph
                paragraph = DOM.createParagraphNode(),
                // insert the paragraph into the DOM tree
                inserted,
                // text position at the beginning of the paragraph
                startPosition = null,
                //parents required for page breaks
                parents = paragraph.parents('table'),
                //page breaks
                currentElement = paragraph[0],
                undoOperation,
                // a helper object for modified attributes
                modifiedAttributes = null,
                // the undo manager object
                undoManager = self.getUndoManager();

            // checking valid paragraph operation for presentation app (48608)
            if (self.useSlideMode() && operation.start.length < 3) { return false; }

            if (parents.length > 0) {
                currentElement = parents.last();
            }

            if (operation.target) {
                inserted = self.insertContentNode(_.clone(operation.start), paragraph, operation.target);
            } else {
                inserted = self.insertContentNode(_.clone(operation.start), paragraph);
            }

            // insertContentNode() writes warning to console
            if (!inserted) { return false; }

            // insert required helper nodes
            self.validateParagraphNode(paragraph);

            // in Internet Explorer it is necessary to add new empty text nodes in paragraph again
            // newly, also in other browsers (new jQuery version?)
            self.repairEmptyTextNodes(paragraph);

            startPosition = Position.appendNewIndex(operation.start);

            // removing a following implicit paragraph (for example in a table cell)
            // -> exchanging an implicit paragraph with a non-implicit paragraph
            if (DOM.isImplicitParagraphNode(paragraph.next())) {
                paragraph.next().remove();
            }

            // generate undo/redo operations
            if (undoManager.isUndoEnabled()) {
                undoOperation = { name: Operations.DELETE, start: operation.start };
                self.extendPropertiesWithTarget(undoOperation, operation.target);
                undoManager.addUndo(undoOperation, operation);
            }

            // apply the passed paragraph attributes
            if (_.isObject(operation.attrs)) {
                // No list style in comments in odt: 38829 (simply removing the list style id)
                if (app.isODF() && operation.target && operation.attrs && operation.attrs.paragraph && operation.attrs.paragraph.listStyleId && self.getCommentLayer().isCommentTarget(operation.target)) { delete operation.attrs.paragraph.listStyleId; }

                modifiedAttributes = _.copy(operation.attrs, true);  // not modifying the original object

                if (!app.isODF()) {

                    // assigning character attributes to text spans, not to paragraphs (41250)
                    if ('character' in operation.attrs) {

                        // setting the character attributes to the text span(s) inside the paragraph
                        _.each(paragraph.children('span'), function (node) {
                            if (DOM.isTextSpan(node)) { self.getCharacterStyles().setElementAttributes(node, operation.attrs); }
                        });

                        // removing the character attributes, so that they are not assigned to the paragraph
                        delete modifiedAttributes.character;
                    }

                }

                // saving attributes at paragraph node
                if (!_.isEmpty(modifiedAttributes)) { self.getParagraphStyles().setElementAttributes(paragraph, operation.attrs); }
            }

            // set cursor to beginning of the new paragraph
            // But not, if this was triggered by an insertColumn or insertRow operation (Task 30859)
            if (!self.isGUITriggeredOperation()) { self.setLastOperationEnd(startPosition); }

            // register paragraph for deferred formatting, especially empty paragraphs
            // or update them immediately after the document import has finished (Task 28370)
            if (self.isImportFinished()) {
                self.implParagraphChangedSync($(paragraph));

                // only for text, there are no paragraph borders and color in presentation
                if (!self.useSlideMode()) {
                    // only needed for updating paragraphs that have a color or a border
                    self.getParagraphStyles().updateParagraphBorderAndColorNodes($(paragraph).next());
                }

            } else {
                self.implParagraphChanged(paragraph);
            }

            // updating lists, if required
            if (self.handleTriggeringListUpdate(paragraph, { paraInsert: true })) {
                // mark this paragraph for later list update. This is necessary, because it does not yet contain
                // the list label node (DOM.LIST_LABEL_NODE_SELECTOR) and is therefore ignored in updateLists.
                $(paragraph).data('updateList', 'true');
            }

            //render pagebreaks after insert
            self.insertPageBreaks(currentElement, DrawingFrame.getClosestTextFrameDrawingNode(paragraph));
            if (!self.isPageBreakMode()) { app.getView().recalculateDocumentMargin(); }

            // Performance: Saving paragraph info for following operations
            self.getSelection().setParagraphCache(paragraph, _.clone(operation.start), 0);

            // #39486
            if (operation.target) {
                self.getPageLayout().markParagraphAsMarginal(paragraph, operation.target);
            }

            return true;
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = null;
        });

    } // class ParagraphMixin

    // exports ================================================================

    return ParagraphMixin;

});
