/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Miroslav Dzunic <miroslav.dzunic@open-xchange.com>
 */

define('io.ox/office/textframework/components/field/complexfield', [
    'io.ox/office/tk/utils/dateutils',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/textframework/components/field/basefield'
], function (DateUtils, LocaleData, Utils, Operations, DOM, Position, BaseField) {

    'use strict';

    // class ComplexField =====================================================

    /**
     * An instance of this class represents the model for all complex fields in the
     * edited document.
     *
     * @constructor
     *
     * @extends Field
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function ComplexField(app) {

        var // self reference
            self = this,
            // an object with target string as key and target node as value, the model
            allFields = {},
            // a counter for the complex field ID
            fieldID = 0,
            // the text model object
            model = null,
            // the selection object
            selection = null,
            // page layout object
            pageLayout = null,
            // number formatter object
            numberFormatter = null,
            // range marker object
            rangeMarker = null,
            // change track object
            changeTrack = null;

        // base constructors --------------------------------------------------

        BaseField.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one complex field in the collection (the model).
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         * @param {String} id
         *  The unique id of the complex field node.
         *
         * @param {String} [target]
         *  The target, where the complex field is located.
         */
        function addIntoComplexFieldCollection(field, id, target) {

            var // the complex field node in jQuery format
                $field = $(field),
                // whether the complex field is inside header or footer
                // -> in this case it is useless to save the node itself, but saving
                //    the target allows to find the range with specific id fast
                isMarginalId = target && pageLayout.isIdOfMarginalNode(target);

            // mark field node in comment
            if (target && model.getCommentLayer().isCommentTarget(target)) {
                $field.addClass(DOM.FIELD_IN_COMMENT_CLASS);
            }

            // adding the node also into the type specific collector
            allFields[id] = isMarginalId ? target : $field;
        }

        /**
         * Removing one complex field with specified id from the collection (the model).
         *
         * @param {String} id
         *  The unique id of the complex field node.
         */
        function removeFromComplexFieldCollection(id) {
            delete allFields[id];
        }

        /**
         * Finding in a header or footer specified by the target the complex field
         * with the specified id.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom. If passed,
         *  forces to return node with that index in document, from top to bottom.
         *
         * @returns {jQuery|null}
         *  The range marker node with the specified id and type, or null, if no such
         *  range marker exists.
         */
        function getMarginalComplexField(id, target, index) {

            var // the specified header or footer node
                marginalNode = null,
                // the searched complex field with the specified id
                field = null;

            marginalNode = model.getRootNode(target, index);

            if (marginalNode) {
                field = _.find(marginalNode.find(DOM.COMPLEXFIELDNODE_SELECTOR), function (field) { return DOM.getComplexFieldId(field) === id; });
            }

            return field ? $(field) : null;
        }

        /**
         * Private method to mark all nodes that belong to complex field,
         * or, so to say, are inside range start and range end of a field.
         *
         * @param {HTMLElement|jQuery} marker
         *  One range marker node.
         *
         * @param {String} id
         *  Id of processed field.
         */
        function markComplexFieldForHighlighting(cxField, id) {
            var $node = $(cxField).next(),
                $helperNode;

            if (!rangeMarker.getEndMarker(id)) {
                return; // no range marker inserted yet, return
            }

            while ($node.length && (!DOM.isRangeMarkerEndNode($node) || DOM.getRangeMarkerId($node) !== id)) { // #42084, #42674
                if (!$node.data('fieldId')) {
                    $node.addClass(DOM.COMPLEXFIELDMEMBERNODE_CLASS + ' complex' + id).data('fieldId', id);
                }
                $helperNode = $node.next();
                if (!$helperNode.length) {
                    $helperNode = $node.parent().next().children().first(); // if field is in more than one paragraph, search for range-end in next p
                    if (!$helperNode.length) {
                        Utils.warn('complexField.markComplexFieldForHighlighting(): rangeEnd not found!');
                        return;
                    } else {
                        $node = $helperNode;
                    }
                } else {
                    $node = $helperNode;
                }
            }
        }

        /**
         * Removing empty text spans between the range marker start node and the
         * following complex field node.
         *
         * @param {HTMLElement|jQuery} node
         *  The complex field node.
         */
        function removePreviousEmptyTextSpan(field) {

            var // the node preceding the complex field node
                precedingNode = $(field).prev();

            if (precedingNode.length > 0 && precedingNode.prev().length > 0 && DOM.isEmptySpan(precedingNode) && DOM.isRangeMarkerStartNode(precedingNode.prev())) {
                precedingNode.remove();
            }
        }

        /**
         * Helper function to keep the global fieldID up-to-date. After inserting
         * a new complex field or after loading from local storage, this number needs to be updated.
         * Then a new complex field can be inserted from the client with a valid and unique id.
         *
         * @param {String} id
         *  The id of the complex field node.
         */
        function updateFieldID(id) {

            // does the id end with a number?
            // -> then the fieldID needs to be checked

            var // resulting array of the regular expression
                matches = /(\d+)$/.exec(id),
                // the number value at the end of the id
                number = 0;

            if (_.isArray(matches)) {
                number = parseInt(matches[1], 10);

                if (number >= fieldID) {
                    fieldID = number + 1;
                }
            }
        }

        /**
         * Update of content for complex field: Page number
         *
         * @param {jQuery|Node} node
         *  Complex field node.
         *
         * @param {String} format
         *  Format of the number.
         */
        function updatePageNumberFieldCx(node, format) {
            var number = pageLayout.getPageNumber(node);
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = null;
            var endPos = null;

            if (model.isHeaderFooterEditState()) { // no explicit update of pageNumber field in header/footer!
                return;
            }

            startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            number = self.formatPageFieldInstruction(number, format);

            // TODO: Nested complex fields

            if (startPos && endPos && number) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(number, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, number.length + 1));
                });
            } else {
                Utils.error('complexField.updatePageNumberFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, number);
            }
        }

        /**
        * Update of content for complex field: Number of pages.
        *
        * @param {jQuery|Node} node
        *   Complex field node
        *
        * @param {String} format
        *  Format of the number.
        */
        function updateNumPagesFieldCx(node, format) {
            var numPages = pageLayout.getNumberOfDocumentPages();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = null;
            var endPos = null;

            if (model.isHeaderFooterEditState()) { // no explicit update of numPages field in header/footer!
                return;
            }
            startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            numPages = self.formatPageFieldInstruction(numPages, format, true);

            // TODO: Nested complex fields

            if (startPos && endPos && numPages) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(numPages, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, numPages.length + 1));
                });
            } else {
                Utils.error('complexField.updateNumPagesFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, numPages);
            }
        }

        /**
         * Update of content for complex field: Date
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} formatCode
         *  Format of date
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.time=false]
         *      If it's time format only.
         */
        function updateDateTimeFieldCx(node, formatCode, options) {

            var serial = numberFormatter.convertDateToNumber(DateUtils.makeUTCNow());
            var time = Utils.getBooleanOption(options, 'time', false);
            var localeFormatCode = (formatCode && formatCode !== 'default') ? formatCode : time ? LocaleData.SHORT_TIME : LocaleData.SHORT_DATE;
            var parsedFormat = numberFormatter.getParsedFormat(localeFormatCode);
            var formattedDate = numberFormatter.formatValue(parsedFormat, serial);
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (startPos && endPos) {
                if ($(node).data('datepickerValue')) {
                    $(node).data('datepickerValue', new Date());
                }
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(formattedDate, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, formattedDate.length + 1));
                });
            } else {
                Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
            }

            // TODO: Nested complex fields
        }

        /**
         * Update the content of complex field: FILENAME
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateFileNameFieldCx(node, format) {
            var fileName = app.getFullFileName();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (format && fileName) {
                if ((/Lower/).test(format)) {
                    fileName = fileName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    fileName = fileName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    fileName = Utils.capitalize(fileName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    fileName = Utils.capitalizeWords(fileName);
                }
            }

            // TODO: Nested complex fields

            if (startPos && endPos && fileName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(fileName, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, fileName.length + 1));
                });
            } else {
                Utils.error('complexField.updateFileNameFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, fileName);
            }
        }

        /**
         * Update the content of complex field: AUTHOR
         *
         * @param {jQuery|Node} node
         *  Complex field node
         *
         * @param {String} format
         *  Format of the field: Upper case, lower case, first capitalize, all first letters in words capitalized.
         *
         */
        function updateAuthorFieldCx(node, format) {
            var authorName = app.getClientOperationName();
            var id = DOM.getComplexFieldMemberId(node);
            var startPos = Position.increaseLastIndex(Position.getOxoPosition(selection.getRootNode(), node));
            var endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(id));

            if (format && authorName) {
                if ((/Lower/).test(format)) {
                    authorName = authorName.toLowerCase();
                } else if ((/Upper/).test(format)) {
                    authorName = authorName.toUpperCase();
                } else if ((/FirstCap/).test(format)) {
                    authorName = Utils.capitalize(authorName.toLowerCase());
                } else if ((/Caps/).test(format)) {
                    authorName = Utils.capitalizeWords(authorName);
                }
            }

            // TODO: Nested complex fields

            if (startPos && endPos && authorName) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(authorName, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, authorName.length + 1));
                });
            } else {
                Utils.error('complexField.updateAuthorFieldCx(): Wrong start end postitions, or text: ', startPos, endPos, authorName);
            }
        }

        /**
         * Local helper function that reverts conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} specialField
         */
        function restoreSpecialField(specialField) {
            var content = $(specialField).children().detach();

            $(specialField).data('length', 1).removeClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).after(content);
        }

        /**
         * Local helper used in case of undo of changetracked fields, where range start and range end are not changetracked.
         * Method getComplexFieldIdFromStack needs to have insertRange, insertComplexField, insertRange order.
         * Since ranges are not changetracked, on undo of changetrack operation, only insertComplexField operation comes,
         * and id is not found. This method tries to get id from rangeStart node, that is already in dom.
         *
         * @param  {Array} start
         *  Oxo position where field node should be inserted
         * @param  {String} target
         *  Target id of content node for position.
         * @return {String|Null} Found id, or null.
         */
        function tryGetIDfromPrevRangeNode(start, target) {
            var id = null;
            var prevRangeMarkerPos = Position.decreaseLastIndex(start);
            var startMarkerPoint = Position.getDOMPosition(model.getRootNode(target), prevRangeMarkerPos, true);
            var startMarkerNode = startMarkerPoint && startMarkerPoint.node;

            if (DOM.isRangeMarkerStartNode(startMarkerNode) && DOM.getRangeMarkerType(startMarkerNode) === 'field') {
                id = DOM.getRangeMarkerId(startMarkerNode);
                if (self.getComplexField(id)) { // if field with this id is already in a model, it is not correct id
                    id = null;
                }
            }

            return id;
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the document contains complex fields in the document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one complex field.
         */
        this.isEmpty = function () {
            return _.isEmpty(allFields);
        };

        /**
         * Public method to get all complex fields from model.
         *
         * @returns {Object}
         */
        this.getAllFields = function () {
            return allFields;
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getComplexField = function (id) {

            var // the value saved under the specified id
                field = allFields[id] || null;

            return _.isString(field) ? getMarginalComplexField(id, field) : field;
        };

        /**
         * Getter for target container id of marginal complex field node.
         *
         * @param{String} id
         * Id of the field in collection model.
         *
         * @returns {jQuery|null}
         *  Target container id of marginal complex field node.
         */
        this.getMarginalComplexFieldsTarget = function (id) {

            var // the value saved under the specified id
                field = allFields[id] || null;

            return _.isString(field) ? field : null;
        };

        /**
         * Check, whether the specified target string is a valid target for a complex field.
         *
         * @param {String} target
         *  The target string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id for a complex field node.
         */
        this.isComplexFieldId = function (id) {
            return self.getComplexField(id) !== null;
        };

        /**
         * Provides a unique ID for the complex field.
         *
         * @returns {String}
         *  A unique id.
         */
        this.getNextComplexFieldID = function () {

            var // the next free complex field id
                fieldIdString = 'fld' + fieldID++,
                // the range marker object
                rangeMarker = model.getRangeMarker();

            // checking, if there also exist no range markers with this id
            while (rangeMarker.isUsedRangeMarkerId(fieldIdString)) {
                fieldIdString = 'fld' + fieldID++;
            }

            return fieldIdString;
        };

        /**
         * Public method to remove complex field from model.
         *
         * @param {HTMLElement|jQuery} field
         *  One complex field node.
         *
         */
        this.removeFromComplexFieldCollection = function (node) {
            removeFromComplexFieldCollection(DOM.getComplexFieldId(node));
        };

        /**
         * Handler for insertComplexField operations. Inserts a complex field node into the document DOM.
         *
         * @param {Number[]} start
         *  The logical start position for the new complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been inserted successfully.
         */
        this.insertComplexFieldHandler = function (start, instruction, attrs, target) {

            var // the range marker element
                complexFieldNode = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.COMPLEXFIELDNODE_CLASS),
                // the text span needed for inserting the range marker node
                textSpan = model.prepareTextSpanForInsertion(start, null, target),
                // id of the field, to connect with range markers
                id = model.getFieldManager().getComplexFieldIdFromStack(),
                // the start marker node, that belongs to this complex field
                startMarker = null;

            if (!textSpan) { return false; }

            if (!id) {
                // On undo change-tracked complex field,
                // range start and range end are not handled, and it is possible not to fetch id from stack.
                // Try to fetch range marker and id from previous dom node
                id = tryGetIDfromPrevRangeNode(start, target);

                if (!id) {
                    Utils.error('insertComplexFieldHandler: no id!');
                }
            }
            startMarker = model.getRangeMarker().getStartMarker(id);

            if (attrs && attrs.character &&  !_.isUndefined(attrs.character.autoDateField)) {
                complexFieldNode.attr('data-auto-date', false);
            }

            // saving also the id at the complex field node
            complexFieldNode.attr('data-field-id', id);
            complexFieldNode.data('fieldInstruction', instruction);

            // inserting the complex field node into the DOM
            if (startMarker) {
                if (DOM.isEmptySpan(textSpan) && Utils.getDomNode($(textSpan).prev()) === Utils.getDomNode(startMarker)) { $(textSpan).remove(); }
                complexFieldNode.insertAfter(startMarker);  // no empty span between start marker and complex field node
            } else {
                complexFieldNode.insertAfter(textSpan);
            }

            // setting the (change track) attributes
            if (_.isObject(attrs)) { model.getCharacterStyles().setElementAttributes(complexFieldNode, attrs); }

            // adding the complex field node in the collection of all complex fields
            // -> this also needs to be done after loading document from fast load or using local storage
            addIntoComplexFieldCollection(complexFieldNode, id, target);

            markComplexFieldForHighlighting(complexFieldNode, id);

            // updating the paragraph
            if (model.isImportFinished()) { model.implParagraphChangedSync(complexFieldNode.parent()); }

            // handling the global counter
            updateFieldID(id);

            return true;
        };

        /**
         * Handler for updateComplexField operation.
         * Replaces original complex field DOM node with new one, with new instruction.
         *
         * @param {Number[]} start
         *  The logical start position for the complex field.
         *
         * @param {String} instruction
         *  The instructions of the complex field.
         *
         * @param {Object} attrs
         *  The attributes of the complex field.
         *
         * @param {String} target
         *  The target string corresponding to the specified start position.
         *
         * @returns {Boolean}
         *  Whether the complex field has been updated successfully.
         */
        this.updateComplexFieldHandler = function (start, instruction, attrs, target) {

            var undoManager = model.getUndoManager(),
                undoOperation,
                redoOperation,
                complexFieldNode = Position.getDOMPosition(model.getRootNode(target), start, true),
                oldInstruction,
                id;

            if (!complexFieldNode || !complexFieldNode.node) {
                Utils.error('complexfield.updateComplexFieldHandler(): unable to fetch complex field node!');
                return false;
            }
            complexFieldNode = complexFieldNode.node;

            id = DOM.getComplexFieldId(complexFieldNode);
            oldInstruction = DOM.getComplexFieldInstruction(complexFieldNode);

            if (!id || !oldInstruction) {
                Utils.error('complexfield.updateComplexFieldHandler(): unable to fetch id or instruction!');
                return false;
            }
            // if there is data from datepicker, reset with new value of current date
            if ($(complexFieldNode).data('datepickerValue')) {
                $(complexFieldNode).data('datepickerValue', new Date());
            }
            // saving also the id at the complex field node
            $(complexFieldNode).attr('data-field-id', id);

            if (attrs && attrs.character && !_.isUndefined(attrs.character.autoDateField)) {
                $(complexFieldNode).attr('data-auto-date', attrs.character.autoDateField); // special case: for performance reasons, this is not handled in character styles.
                model.getCharacterStyles().setElementAttributes(complexFieldNode, attrs); // This is needed to remove property set in SetAttributes operation, in data of the element!

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    undoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: { character: { autoDateField: !attrs.autoDateField } }, instruction: oldInstruction };
                    redoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: { character: { autoDateField: attrs.autoDateField } }, instruction: oldInstruction };
                    if (target) {
                        undoOperation.target = target;
                        redoOperation.target = target;
                    }
                    undoManager.addUndo(undoOperation, redoOperation);
                }
            }

            if (instruction) {
                $(complexFieldNode).data('fieldInstruction', instruction);

                // create the undo action
                if (undoManager.isUndoEnabled()) {
                    undoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: {}, instruction: oldInstruction };
                    redoOperation = { name: Operations.COMPLEXFIELD_UPDATE, start: start, attrs: {}, instruction: instruction };
                    if (target) {
                        undoOperation.target = target;
                        redoOperation.target = target;
                    }
                    undoManager.addUndo(undoOperation, redoOperation);
                }
            }

            //markComplexFieldForHighlighting(complexFieldNode, id);

            return true;
        };

        /**
         * Inserting a complex field. This function generates the operations for the ranges and
         * the complex field node. It does not modify the DOM. This function is only executed in
         * the client with edit privileges, not in remote clients.
         *
         * @param {String} fieldType
         *
         * @param {String} fieldFormat
         */
        this.insertComplexField = function (fieldType, fieldFormat) {

            return model.getUndoManager().enterUndoGroup(function () {

                function doInsertComplexField() {
                    var start = selection.getStartPosition(),
                        fieldStart = Position.increaseLastIndex(start),
                        textStart = Position.increaseLastIndex(fieldStart),
                        rangeEndStart = null,
                        generator = model.createOperationsGenerator(),
                        operation = {},
                        insertTextOperation = null,
                        // target for operation - if exists, it's for ex. header or footer
                        target = model.getActiveTarget(),
                        representation = null,
                        instruction = null;

                    model.doCheckImplicitParagraph(start);

                    if (fieldType) {
                        operation.start = fieldStart;

                        if (self.isCurrentDate(fieldType) || self.isCurrentTime(fieldType)) {
                            if (!fieldFormat || fieldFormat === 'default') {
                                fieldFormat = LocaleData.SHORT_DATE;
                            }
                            representation = self.getDateTimeRepresentation(fieldFormat);
                            instruction = fieldType.toUpperCase() + (fieldFormat.length ? ' \\@ "' + fieldFormat + '"' : '');
                            operation.attrs = operation.attrs || {};
                            operation.attrs.character = operation.attrs.character || {};
                            operation.attrs.character = { autoDateField: false };
                        } else if (self.isFileName(fieldType) || self.isAuthor(fieldType)) {
                            representation = self.isFileName(fieldType) ? app.getFullFileName() : app.getClientOperationName();
                            if ((/Lower/i).test(fieldFormat)) {
                                representation = representation.toLowerCase();
                            } else if ((/Upper/i).test(fieldFormat)) {
                                representation = representation.toUpperCase();
                            } else if ((/FirstCap/i).test(fieldFormat)) {
                                representation = Utils.capitalize(representation.toLowerCase());
                            } else if ((/Caps/i).test(fieldFormat)) {
                                representation = Utils.capitalizeWords(representation);
                            }
                            instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        } else if (self.isNumPages(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getNumberOfDocumentPages(), fieldFormat, true);
                            instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        } else if (self.isPageNumber(fieldType)) {
                            representation = self.formatPageFieldInstruction(pageLayout.getPageNumber(), fieldFormat);
                            instruction = 'PAGE' + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                        }

                        if (representation.length) {
                            operation.instruction = instruction;
                            model.extendPropertiesWithTarget(operation, target);

                            insertTextOperation = { start: textStart, text: representation };

                            // modifying the attributes, if changeTracking is activated
                            if (changeTrack.isActiveChangeTracking()) {
                                operation.attrs = operation.attrs || {};
                                operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                                insertTextOperation.attrs = { changes: { inserted: changeTrack.getChangeTrackInfo(), removed: null } };

                            }
                            generator.generateOperation(Operations.RANGE_INSERT, { start: start, type: 'field', position: 'start' });
                            generator.generateOperation(Operations.COMPLEXFIELD_INSERT, operation);
                            generator.generateOperation(Operations.TEXT_INSERT, insertTextOperation);

                            rangeEndStart = Position.increaseLastIndex(textStart, representation.length);
                            generator.generateOperation(Operations.RANGE_INSERT, { start: rangeEndStart, type: 'field', position: 'end' });

                            model.applyOperations(generator.getOperations());

                            selection.setTextSelection(Position.increaseLastIndex(rangeEndStart));
                        } else {
                            Utils.warn('complexField.insertComplexField(): representation missing!');
                        }
                    }
                }

                if (selection.hasRange()) {
                    return model.deleteSelected()
                    .done(function () {
                        doInsertComplexField();
                    });
                }

                doInsertComplexField();
                return $.when();

            });
        };

        /**
         * Public method that updates format of selected complex field.
         *
         * @param {String} fieldType
         *  Type of the field.
         * @param {String} fieldFormat
         *  New format that field will have.
         * @param {Array} fieldNodePos
         *  Oxo position of complex field node itself, between range markers.
         * @param {String} fieldId
         *  Id of updated field
         */
        this.updateComplexFieldFormat = function (fieldType, fieldFormat, fieldNodePos, fieldId) {
            var
                generator = model.createOperationsGenerator(),
                operation = {},
                target = model.getActiveTarget(),
                representation = null,
                instruction = null,
                insertTextOperation = null,
                textStart = Position.increaseLastIndex(fieldNodePos),
                startPos = fieldNodePos,
                endPos = Position.getOxoPosition(selection.getRootNode(), rangeMarker.getEndMarker(fieldId));

            if (fieldType && fieldNodePos.length) {
                operation.start = fieldNodePos;

                if (self.isCurrentDate(fieldType) || self.isCurrentTime(fieldType)) {
                    if (!fieldFormat || fieldFormat === 'default') {
                        fieldFormat = LocaleData.SHORT_DATE;
                    }
                    representation = self.getDateTimeRepresentation(fieldFormat);
                    instruction = fieldType.toUpperCase() + (fieldFormat.length ? ' \\@ "' + fieldFormat + '"' : '');
                } else if (self.isFileName(fieldType) || self.isAuthor(fieldType)) {
                    representation = self.isFileName(fieldType) ? app.getFullFileName() : app.getClientOperationName();
                    if ((/Lower/i).test(fieldFormat)) {
                        representation = representation.toLowerCase();
                    } else if ((/Upper/i).test(fieldFormat)) {
                        representation = representation.toUpperCase();
                    } else if ((/FirstCap/i).test(fieldFormat)) {
                        representation = Utils.capitalize(representation.toLowerCase());
                    } else if ((/Caps/i).test(fieldFormat)) {
                        representation = Utils.capitalizeWords(representation);
                    }
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                } else if (self.isNumPages(fieldType)) {
                    representation = self.formatPageFieldInstruction(pageLayout.getNumberOfDocumentPages(), fieldFormat, true);
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                } else if (self.isPageNumber(fieldType)) {
                    representation = self.formatPageFieldInstruction(pageLayout.getPageNumber(), fieldFormat);
                    instruction = fieldType.toUpperCase() + ((fieldFormat.length && fieldFormat !== 'default') ? ' \\* ' + fieldFormat : '');
                }

                if (representation.length) {
                    operation.instruction = instruction;
                    //operation.id = fieldId;
                    model.extendPropertiesWithTarget(operation, target);

                    insertTextOperation = { start: textStart, text: representation };

                    // modifying the attributes, if changeTracking is activated
                    if (changeTrack.isActiveChangeTracking()) {
                        operation.attrs = operation.attrs || {};
                        operation.attrs.changes = { inserted: changeTrack.getChangeTrackInfo(), removed: null };
                        insertTextOperation.attrs = { changes: { inserted: changeTrack.getChangeTrackInfo(), removed: null } };
                    }

                    if (startPos && endPos) {
                        return model.getUndoManager().enterUndoGroup(function () {

                            selection.setTextSelection(textStart, endPos);
                            model.deleteSelected();

                            generator.generateOperation(Operations.COMPLEXFIELD_UPDATE, operation);
                            generator.generateOperation(Operations.TEXT_INSERT, insertTextOperation);
                            model.applyOperations(generator.getOperations());

                            selection.setTextSelection(Position.increaseLastIndex(textStart, representation.length + 1));
                        });
                    } else {
                        Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
                    }
                } else {
                    Utils.warn('complexField.updateComplexField(): representation missing!');
                }
            }
        };

        /**
         * After splitting a paragraph, it is necessary, that all complex field nodes in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateComplexFieldCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all complex fields inside the paragraph
                nodeFields = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            nodeFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the complex field nodes in the collection objects
            _.each(nodeFields, function (oneField) {
                if (!checkMarginal || !DOM.isMarginalNode(oneField)) {
                    // simply overwriting the old complex fields with the new values
                    addIntoComplexFieldCollection(oneField, DOM.getComplexFieldId(oneField));
                }
            });
        };

        /**
         * After load from local storage or fast load the collectors for the complex fields need to be filled.
         *
         * @param {Boolean} usedLocalStorage
         *  Whether the document was loaded from the local storage.
         *
         * @param {Boolean} usedFastLoad
         *  Whether the document was loaded with the fast load process.
         */
        this.refreshComplexFields = function (usedLocalStorage, usedFastLoad) {

            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all range marker nodes in the document
                allComplexFieldNodes = pageContentNode.find(DOM.COMPLEXFIELDNODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = pageLayout.getHeaderFooterPlaceHolder().children(),
                // collector for all complex nodes in comment layer
                commentFieldNodes = $(DOM.getCommentLayerNode(model.getNode())).find(DOM.COMPLEXFIELDNODE_SELECTOR),
                // the target string node
                target = '',
                // whether an update of header and footer is required
                updateMarginalNodes = false;

            // helper function to add a collection of range marker nodes into the model
            function addMarginalNodesIntoCollection(collection) {
                _.each(collection, function (oneComplexField) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField), target);
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) {
                        removePreviousEmptyTextSpan(oneComplexField);
                        updateMarginalNodes = true;
                    }
                });
                pageLayout.replaceAllTypesOfHeaderFooters();
            }

            // reset model
            allFields = {};

            // 1. Search in page content node
            // 2. Search in header/footer template node

            // adding all range marker nodes into the collector (not those from header or footer)
            _.each(allComplexFieldNodes, function (oneComplexField) {
                if (!DOM.isMarginalNode($(oneComplexField).parent())) {
                    addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField));
                    updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                    if (usedFastLoad) { removePreviousEmptyTextSpan(oneComplexField); }
                }
            });

            // adding the range markers that are located inside header or footer (using the template node)
            _.each(allMargins, function (margin) {

                var // a collector for all range marker nodes inside one margin
                    allMarginComplexFieldNodes = $(margin).find(DOM.COMPLEXFIELDNODE_SELECTOR);

                if (allMarginComplexFieldNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    addMarginalNodesIntoCollection(allMarginComplexFieldNodes);
                }
            });

            // adding fields that are in comment layer
            _.each(commentFieldNodes, function (oneComplexField) {
                addIntoComplexFieldCollection(oneComplexField, DOM.getComplexFieldId(oneComplexField));
                updateFieldID(DOM.getComplexFieldId(oneComplexField)); // updating the value for the global field id, so that new fields get an increased number.
                if (usedFastLoad) {
                    removePreviousEmptyTextSpan(oneComplexField);
                }
                // mark field node
                $(oneComplexField).addClass(DOM.FIELD_IN_COMMENT_CLASS);
            });

            // if empty text spans were removed in header or footer node, an update of the header and
            // footer nodes in the document is required
            if (updateMarginalNodes) {
                // TODO:
                // Refreshing the marginal nodes in the document with the nodes from the template node (not available yet)
            }
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all complex field nodes
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeAllInsertedComplexFields = function (para) {

            var // all range markers inside the paragraph
                nodeFields = $(para).find(DOM.COMPLEXFIELDNODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(nodeFields, function (field) {

                var // the id of the marker node
                    fieldId = DOM.getComplexFieldId(field);

                if (_.isString(fieldId)) {
                    removeFromComplexFieldCollection(fieldId);
                }
            });
        };

        /**
         * When pasting content, that contains complex fields, it is necessary, that the
         * complex fields get a new unique id.
         *
         * @param {Object[]} operations
         *  The collection with all paste operations.
         */
        this.handlePasteOperationTarget = function (operations) {

            var // a container for all complex field ids
                allIDs = [],
                // a container for all range marker ids of complex fields
                allRangeMarkerIDs = {};

            // replacing the complex field ids
            function replaceID(ops, id1, id2) {

                _.each(ops, function (op) {
                    if (op) {
                        if (op.id === id1) { op.id = id2; }
                        if (op.target === id1) { op.target = id2; }
                    }
                });
            }

            // collecting all 'id's of insertComplexField operations and assign new target values
            _.each(operations, function (operation) {
                if (operation && operation.name === Operations.COMPLEX_FIELD_INSERT) {
                    allIDs.push(operation.id);
                } else if (operation && operation.name === Operations.RANGE_INSERT && operation.type === 'field') {
                    allRangeMarkerIDs[operation.id] = 1;
                }
            });

            // iterating over all registered complex field ids
            if (allIDs.length > 0) {
                _.each(allIDs, function (oldID) {
                    var newID = self.getNextComplexFieldID();
                    // iterating over all operations, so that also the range markers will be updated
                    replaceID(operations, oldID, newID);
                    delete allRangeMarkerIDs[oldID];
                });
            }

            // are there still unmodified range marker targets (maybe only the start marker was pasted)
            _.each(_.keys(allRangeMarkerIDs), function (oldID) {
                var newID = self.getNextComplexFieldID();
                // iterating over all operations
                replaceID(operations, oldID, newID);
            });

        };

        /**
         * Checks instruction, and depending of extracted field type,
         * updates passed complex field node.
         *
         * @param {jQuery|Node} node
         *  Node that is going to be updated.
         *
         * @param {String} instruction
         *  Field instruction.
         */
        this.updateByInstruction = function (node, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type;
            instruction = dividedData.instruction;

            if (type) {
                if (self.isNumPages(type)) {
                    updateNumPagesFieldCx(node, instruction);
                } else if (self.isPageNumber(type)) {
                    updatePageNumberFieldCx(node, instruction);
                } else if (self.isCurrentDate(type)) {
                    updateDateTimeFieldCx(node, instruction);
                } else if (self.isCurrentTime(type)) {
                    updateDateTimeFieldCx(node, instruction, { time: true });
                } else if (self.isFileName(type)) {
                    updateFileNameFieldCx(node, instruction);
                } else if (self.isAuthor(type)) {
                    updateAuthorFieldCx(node, instruction);
                }
            }
        };

        /**
         * Update date and time fields before running download or print action.
         *
         * @param {jQuery|Node} field
         *  Field to be updated.
         *
         * @param {String} instruction
         *  Instruction for updating.
         */
        this.updateDateTimeFieldOnDownload = function (field, instruction) {
            var dividedData = self.cleanUpAndExtractType(instruction),
                type = dividedData.type;
            instruction = dividedData.instruction;

            if (type && (self.isCurrentDate(type) || self.isCurrentTime(type)) && (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field))) { // #42093
                updateDateTimeFieldCx(field, instruction, { time: self.isCurrentTime(type) });
            }
        };

        /**
         * Converts normal complex field to special complex field, when field is inside header/footer and has type PAGE.
         * Content of field is detached and inserted inside complex field.
         * Inverse function from this function is restoreSpecialFields.
         *
         * @param {String} fieldId
         *  Id of the field
         *
         * @param {String} marginalTarget
         *  Id of the marginal target node
         *
         * @param {String} type
         *  Type of field: num pages or page number
         */
        this.convertToSpecialField = function (fieldId, marginalTarget, type) {
            var innerContent = $(),
                rootNode = model.getRootNode(marginalTarget),
                rangeStart = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_STARTTYPE_SELECTOR),
                rangeEnd = rootNode.find('[data-range-id="' + fieldId + '"]').filter(DOM.RANGEMARKER_ENDTYPE_SELECTOR),
                fieldNode = null,
                startPos = Position.getOxoPosition(rootNode, rangeStart),
                endPos = Position.getOxoPosition(rootNode, rangeEnd),
                innerLen = _.last(endPos) - _.last(startPos),
                isNumPages = (/NUMPAGES/i).test(type),
                className = isNumPages ? 'cx-numpages' : 'cx-page',
                replacementNode,
                helperNode;

            if (innerLen > 0) {
                innerLen -= 1;
            } else {
                Utils.warn('complexField.convertToSpecialField(): inner length is < 1');
            }

            if (rangeStart.length) {
                fieldNode = rangeStart.next();
                if (!DOM.isComplexFieldNode(fieldNode)) {
                    Utils.warn('complexField.convertToSpecialField(): cx field node not found after range start!');
                    return;
                }
            }

            helperNode = rangeEnd.prev();
            while (helperNode.length && (!DOM.isComplexFieldNode(helperNode) || DOM.getComplexFieldId(helperNode) !== fieldId)) {
                innerContent = innerContent.add(helperNode);
                helperNode = helperNode.prev();
            }
            innerContent.detach().empty();
            fieldNode.append(innerContent).data('length', innerLen).addClass(DOM.SPECIAL_FIELD_NODE_CLASSNAME).addClass(className);

            if (!DOM.isMarginalPlaceHolderNode(rootNode.parent())) {
                replacementNode = rootNode.children().clone(true);
                replacementNode.find('[contenteditable="true"]').attr('contenteditable', false);
                pageLayout.getHeaderFooterPlaceHolder().children('[data-container-id="' + marginalTarget + '"]').empty().append(replacementNode);
            }
        };

        /**
         * Restoring special field to normal is required before delete operation.
         *
         *@param {jQuery} specialField
         */
        this.restoreSpecialField = function (specialField) {
            restoreSpecialField(specialField);
        };

        /**
         * Check if there are special fields of type page in header, inside given searchNode,
         * and if yes, restore them before deleting them.
         *
         * @param {Node} searchNode
         *  Node which we search for special fields.
         *
         * @param {Number} startOffset
         *  Start offset of given node.
         *
         * @param {Number} endOffset
         *  End offset of given node.
         */
        this.checkRestoringSpecialFields = function (searchNode, startOffset, endOffset) {

            if (DOM.isMarginalNode(searchNode) && $(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) {
                _.each($(searchNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                    var fieldPos = Position.getOxoPosition(selection.getRootNode(), specialField, 0);
                    if (Position.isNodePositionInsideRange(fieldPos, startOffset, endOffset)) {
                        restoreSpecialField(specialField);
                    }
                });
            }
        };

        /**
         * Check if there are special fields in container node, and revert conversion from special field to normal,
         * which is necessary before creating undo operations.
         *
         * @param {jQuery|Node} containerNode
         *  Can be whole header or footer, not restricted to paragraph parts.
         *  For that, see function: checkRestoringSpecialFields
         */
        this.checkRestoringSpecialFieldsInContainer = function (containerNode) {
            _.each($(containerNode).find(DOM.SPECIAL_FIELD_NODE_SELECTOR), function (specialField) {
                restoreSpecialField(specialField);
            });
        };

        /**
         * After cloning headers&footers special page fields have to be updated with proper page number.
         */
        this.updateCxPageFieldsInMarginals = function () {
            var $pageContentNode = null,
                firstHeadNum = 1,
                lastFootNum = null,
                // first header in doc
                $firstHeaderFields = null,
                // last footer in doc
                $lastFooterFields = null,
                numPagesNodes = null;

            if (self.isEmpty() || !pageLayout.getHeaderFooterPlaceHolder().find(DOM.SPECIAL_FIELD_NODE_SELECTOR).length) {
                return; // no special fields in the document, don't make unnecessary loops
            }

            // performance: assign variables only if really needed
            $pageContentNode = DOM.getPageContentNode(model.getNode());
            lastFootNum = pageLayout.getNumberOfDocumentPages();
            // first header in doc - page number always 1
            $firstHeaderFields = pageLayout.getFirstHeaderWrapperNode().children(DOM.HEADER_SELECTOR).find('.cx-page');
            // last footer in doc gets total page count as page number
            $lastFooterFields = pageLayout.getLastFooterWrapperNode().children(DOM.FOOTER_SELECTOR).find('.cx-page');

            _.each($firstHeaderFields, function (headerField) {
                var $headerField = $(headerField),
                    headerFormat = DOM.getComplexFieldInstruction($headerField);

                $headerField.children().empty().html(self.formatPageFieldInstruction(firstHeadNum, headerFormat));
            });

            _.each($lastFooterFields, function (footerField) {
                var $footerField = $(footerField),
                    footerFormat = DOM.getComplexFieldInstruction($footerField);

                $footerField.children().empty().html(self.formatPageFieldInstruction(lastFootNum, footerFormat));
            });

            _.each(pageLayout.getPageBreaksCollection(), function (pageBreak) {
                var $pageBreak = $(pageBreak),
                    headerNumber = $pageBreak.data('page-num') + 1,
                    footerNumber = $pageBreak.data('page-num'),
                    $headerFields = $pageBreak.children(DOM.HEADER_SELECTOR).find('.cx-page'),
                    $footerFields = $pageBreak.children(DOM.FOOTER_SELECTOR).find('.cx-page');

                _.each($headerFields, function (headerField) {
                    var $headerField = $(headerField),
                        headerFormat = DOM.getComplexFieldInstruction($headerField);

                    $headerField.children().empty().html(self.formatPageFieldInstruction(headerNumber, headerFormat));
                });

                _.each($footerFields, function (footerField) {
                    var $footerField = $(footerField),
                        footerFormat = DOM.getComplexFieldInstruction($footerField);

                    $footerField.children().empty().html(self.formatPageFieldInstruction(footerNumber, footerFormat));
                });
            });

            // update number of pages fields, which all have same value
            if (pageLayout.getHeaderFooterPlaceHolder().find('.cx-numpages').length) {
                numPagesNodes = $pageContentNode.find(DOM.PAGE_BREAK_SELECTOR).add(pageLayout.getFirstHeaderWrapperNode()).add(pageLayout.getLastFooterWrapperNode()).find('.cx-numpages');
                _.each(numPagesNodes, function (fieldNode) {
                    var $fieldNode = $(fieldNode),
                        fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                    $fieldNode.children().empty().html(self.formatPageFieldInstruction(lastFootNum, fieldFormat, true));
                });
            }
        };

        /**
         * Function called after importing document.
         * It loops all fields in collection and:
         *  - marks them for highlighting,
         *  - updates date and time fields
         *  - converts page fields in header&footer to special fields
         */
        this.updateDateTimeAndSpecialFields = function () {
            var updatePageFields = false,
                returnOriginalCursorPos = false,
                startPos = selection.getStartPosition(),
                endPos = selection.getEndPosition();

            _.each(allFields, function (entry, fieldId) {
                var field = self.getComplexField(fieldId),
                    instruction = DOM.getComplexFieldInstruction(field),
                    dividedData = self.cleanUpAndExtractType(instruction),
                    type = dividedData.type,
                    format = null;

                instruction = dividedData.instruction;

                markComplexFieldForHighlighting(field, fieldId);

                if (type) {
                    if ((self.isCurrentDate(type) || self.isCurrentTime(type)) && model.getEditMode()) {
                        format = instruction;

                        if (DOM.isAutomaticDateField(field) && (!DOM.isMarginalNode(field) || !DOM.isInsideHeaderFooterTemplateNode(model.getNode(), field))) { // update only if not fixed field, #42093
                            // if previously updated field is marginal, and this is not, leave header edit state
                            selection.swichToValidRootNode(field);
                            updateDateTimeFieldCx(field, format, { time: self.isCurrentTime(type) });
                            if (pageLayout.isIdOfMarginalNode(entry)) {
                                markComplexFieldForHighlighting(self.getComplexField(fieldId), fieldId);
                                pageLayout.getHeaderFooterPlaceHolder().children('.' + entry).empty().append(selection.getRootNode().children().clone(true));
                            }
                            returnOriginalCursorPos = true;
                        }
                    } else if (_.isString(entry) && pageLayout.isIdOfMarginalNode(entry) && self.isPageNumber(type)) {
                        if (!DOM.isSpecialField(field)) {
                            self.convertToSpecialField(fieldId, entry, type);
                        }
                        updatePageFields = true;
                    }
                } else {
                    Utils.warn('complexField.updateDateTimeAndSpecialFields: invalid type: ', type);
                }
            });
            if (updatePageFields) {
                // update reference in collection because of replaced node
                pageLayout.headerFooterCollectionUpdate();
                pageLayout.replaceAllTypesOfHeaderFooters(); // complex fields in header/footers are refreshed in this method call, if any
            }
            if (returnOriginalCursorPos) {
                // if there was update of date fields, return cursor to first document position
                if (model.isHeaderFooterEditState()) {
                    pageLayout.leaveHeaderFooterEditMode();
                }
                model.setActiveTarget();
                selection.setNewRootNode(model.getNode());
                selection.setTextSelection(startPos, endPos);
            }
        };

        /**
         * Public method for updating current page number field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Boolean} isHeader
         *  If node is header or not.
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageNumInCurrentNodeCx = function ($node, isHeader, pageCount) {
            if (isHeader) {
                if (DOM.isHeaderWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, 1);
                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            } else {
                if (DOM.isFooterWrapper($node.parent())) {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader, pageCount);

                } else {
                    self.updatePageNumInCurrentMarginalCx($node, isHeader);
                }
            }
        };

        /**
         *
         *
         * @param $node
         * @param pageNum
         */
        this.updatePageNumInCurrentMarginalCx = function ($node, isHeader, pageNum) {
            _.each($node.find('.cx-page'), function (field) {
                var $field = $(field),
                    format = DOM.getComplexFieldInstruction($field),
                    pageNumber = pageNum || ($(field).parentsUntil('.pagecontent', '.page-break').data('page-num') + (isHeader ? 1 : 0));

                $field.children().empty().html(self.formatPageFieldInstruction(pageNumber, format));
            });
        };

        /**
         * Public method for updating number of pages field(s) in given node.
         *
         * @param {jQuery} $node
         * @param {Number} pageCount
         *  Number of total pages in document.
         */
        this.updatePageCountInCurrentNodeCx = function ($node, pageCount) {
            _.each($node.find('.cx-numpages'), function (fieldNode) {
                var $fieldNode = $(fieldNode),
                    fieldFormat = DOM.getComplexFieldInstruction($fieldNode);

                $fieldNode.children().empty().html(self.formatPageFieldInstruction(pageCount, fieldFormat, true));
            });
        };

        /**
        * Update date field with value from popup, datepicker or input field.
        * @param {jQuery} node
        *   Field node that is going to be updated
        * @param {String} fieldValue
        *   Value with which will field be updated.
        */
        this.updateDateFromPopupValue = function (node, fieldValue) {
            var formattedDate = fieldValue,
                id = DOM.getComplexFieldMemberId(node),
                isMarginal = DOM.isMarginalNode(node),
                rootNode = isMarginal ? DOM.getClosestMarginalTargetNode(node) : selection.getRootNode(),
                startPos = Position.increaseLastIndex(Position.getOxoPosition(rootNode, node)),
                endPos = Position.getOxoPosition(rootNode, rangeMarker.getEndMarker(id));

            if (startPos && endPos) {
                return model.getUndoManager().enterUndoGroup(function () {
                    selection.setTextSelection(startPos, endPos);
                    model.deleteSelected();
                    model.insertText(formattedDate, startPos);

                    selection.setTextSelection(Position.increaseLastIndex(startPos, formattedDate.length + 1));
                });
            } else {
                Utils.error('complexField.updateDateTimeFieldCx(): Wrong start and end postitions: ', startPos, endPos);
            }
        };

        /**
         * Public method to mark all nodes that belong to complex field,
         * and are inside range start and range end of a field.
         *
         * @param {HTMLElement|jQuery} field
         *  Complex field node.
         *
         * @param {String} id
         *  Id of processed field.
         */
        this.markComplexFieldForHighlighting = function (field, id) {
            markComplexFieldForHighlighting(field, id);
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {
            model = app.getModel();
            selection = model.getSelection();
            pageLayout = model.getPageLayout();
            numberFormatter = model.getNumberFormatter();
            rangeMarker = model.getRangeMarker();
            changeTrack = model.getChangeTrack();
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            model = selection = pageLayout = numberFormatter = rangeMarker = changeTrack = null;
        });

    } // class ComplexField

    // constants --------------------------------------------------------------

    // export =================================================================

    // derive this class from class BaseField
    return BaseField.extend({ constructor: ComplexField });
});
